/***************************************************************************

Another ScottFree Driver, revision 1.20
Derived from Alan Cox's "ScottFree" revision 1.14,
that work (C) 1993, 1994, 1995 Swansea University Computer Society.
This driver (C) 1998 Robert Schneck.

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License along
with this program (see the file 'license.txt'); if not, write to the Free
Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

Statement: Everything in this program has been deduced or obtained
solely from published material.  No game interpreter code has been
disassembled.

***************************************************************************/

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>
#include <stdarg.h>
#include <time.h>

#include "asfdmain.h"
#include "asfddos.h"

/* Information stored in the adventure database */
struct {
  short magicHead;
  short maxItem;
  short maxAction;
  short maxWord; /* Smaller of verb/noun is padded to same size */
  short maxRoom;
  short maxCarry;
  short startRoom;
  short numTreasures;
  short wordLength;
  short initialLightTime;
  short maxMessage;
  short treasureRoom;
  
  Action *action;
  
  char **verb;
  char **noun;
  
  Room *room;
  
  char **message;
  
  Item *item;
  
  short version;
  short adventureNumber;
  short magicTail;
} database;


/* Information particular to this game */
struct {
  short playerRoom;
  short *itemLoc;
  short lightTime;
  
  long bitFlags;  /* number of flags unknown.  Seems <=32 */
  
  short counter[MAX_COUNTER+1];  /* range unknown */
  short currentCounter;
  
  /* probably currentSavedRoom should just be savedRoom[0] */
  short currentSavedRoom;
  short savedRoom[MAX_COUNTER+1]; /* range unknown */
} game;


/* Information for driver */

/*  from last look, for determining whether to redraw */
struct {
  short playerRoom;
  short *itemLoc;
  boolean dark;
  boolean clear; /* was there a clear screen (or other overriding redraw)? */
} lastLook;

/* command-line parameters */
int options;

/* TRUE if parsing an multi-line action */
boolean continuation;
/* used by "get all" to stop when carring capacity message displayed once */
boolean printedTooMuch;
/* used to get items for which autoGet is not in the noun list */
char *getNoun;

/* current position of output */
extern int outputPos;

/* current command */
char verb[MAX_INPUT_WORD_LENGTH+1];
char noun[MAX_INPUT_WORD_LENGTH+1];
short verbNum;
short nounNum;


void TitleScreen() {
  ClearScreen();
  PrintTitle("Another Scott Free Driver, revision 1.20\n");
  PrintTitle("Derived from Alan Cox's \"Scott Free\" revision 1.14, \n");
  PrintTitle("that work (C) 1993, 1994, 1995 ");
  PrintTitle("Swansea University Computer Society.\n");
  PrintTitle("This driver (C) 1998 Robert Schneck.\n");
  PrintTitle("Distributed under the GNU General Public License, and with absolutely no warranty; see the license for details.\n\n");
}


boolean GetBit(short bit) {
  return ((game.bitFlags&(1L<<bit))!=0);
}
void SetBit(short bit) {
  game.bitFlags|=(1L<<bit);
}
void ClearBit(short bit) {
  game.bitFlags&=~(1L<<bit);
}

boolean GetOption(int option) {
  return ((options&option)!=0);
}
void SetOption(int option) {
  options|=option;
}
void ClearOption(int option) {
  options&=~option;
}
void ToggleOption(int option) {
  options^=option;
}


/* True n% of the time */
boolean RandomPercent(short n) {
  unsigned int rv = rand()<<6;
  rv %=100;
  return (rv<n);
}

/* Current number of items carried */
short CountCarried() {
  short i;
  short count=0;
  for (i=0;i<=database.maxItem;i++) {
    if (game.itemLoc[i] == CARRIED) {
      count++;
    }
  }
  return count;
}

boolean CondInRoom(short val) {
  return (game.playerRoom==val);
}
boolean CondHere(short val) {
  return (game.itemLoc[val]==game.playerRoom);
}
boolean CondCarried(short val) {
  return (game.itemLoc[val]==CARRIED);
}
boolean CondDestroyed(short val) {
  return (game.itemLoc[val]==DESTROYED);
}
boolean CondAvailable(short val) {
  return (CondHere(val)||CondCarried(val));
}
boolean CondOriginalRoom(short val) {
  return (game.itemLoc[val]==database.item[val].initialLoc);
}

boolean Dark() {
  return (GetBit(DARKBIT) && !CondAvailable(LIGHT_SOURCE));
}


/* TRUE if we should Look() again */
boolean Redraw() {
  short i;
  
  if (lastLook.clear) {
    return TRUE; /* not sure I should redraw after clear. 
		    seems good though */
  }
  if (lastLook.dark != Dark()) {
    return TRUE;
  }
  if (Dark()) {
    return FALSE; /* since you can't see the change... */
  }
  if (lastLook.playerRoom != game.playerRoom) {
    return TRUE;
  }
  for (i=0;i<=database.maxItem;i++) {
    if ((lastLook.itemLoc[i]==game.playerRoom) !=
        (game.itemLoc[i]==game.playerRoom)) {
      return TRUE;
    }
  }
  return FALSE;
}


/* Fatal error message */
void Fatal(char *message) {
  PrintDiag(message);
  PrintDiag(".\n\n");
  Exit (1);
}

/* Allocate memory, fatal error if not enough */
void *MemAlloc(int size) {
  void *t=(void *)malloc(size);
  if(t==NULL) {
    Fatal("Out of memory");
  }
  return(t);
}


void BadCommandLine(char *name) {
  PrintDiag("Usage:\n");
  PrintDiag(name);
  PrintDiag(" [-a] [-l] [-d] [-s] [-p] [-t] <databasename> [save-file]\n");
  PrintDiag("\nSpecifying save-file causes a restore of the save to take place.\n\n");
  PrintDiag(" -a (ALL) toggles use of GET ALL/DROP ALL (default on)\n");
  PrintDiag(" -l (LOOK) toggles automatic look when the description changes (default on)\n");
  PrintDiag(" -d (DIM) toggles whether the driver should print light-running-out messages\n");
  PrintDiag("    each turn or every 5 turns (default each turn)\n");
  PrintDiag(" -s (SPACE) toggles the use of space or newline after game messages\n");
  PrintDiag("    (default newline)\n");
  PrintDiag(" -p (PREHISTORIC LAMP) toggles whether the driver destroys the light source\n");
  PrintDiag("    when the light runs out (default off, the game should take care of it)\n");
  PrintDiag(" -t (TRACE) toggles trace messages when actions are performed (default off)\n");
  Exit(1);
}

void ParseArguments(int *argc, char **argv[]) {
  /* switches */
  SetOption(ALL_OPTION);
  SetOption(LOOK_OPTION);

  while((*argv)[1]) {
    if(*(*argv)[1]!='-') {
      break;
    }
    switch((*argv)[1][1]) {
    case 'a':ToggleOption(ALL_OPTION); break;
    case 'l':ToggleOption(LOOK_OPTION); break;
    case 's':ToggleOption(SPACE_OPTION); break;
    case 'd':ToggleOption(DIM_OPTION); break;
    case 'p':ToggleOption(PREHISTORIC_LAMP_OPTION); break;
    case 't':ToggleOption(TRACE_OPTION); break;
    default:
      BadCommandLine((*argv)[0]);
    }
    if((*argv)[1][2]!=0) {
      BadCommandLine((*argv)[0]);
    }
    (*argv)[1] = (*argv)[0];
    (*argv)++;
    (*argc)--;
  }
  if(*argc!=2 && *argc!=3) {
    BadCommandLine((*argv)[0]);
  }
}


/* read (and allocate memory) for a string in a ScottFree database */
char *ReadString(FILE* f) {
  char temp[1024];
  char* result;
  int c;
  int i = 0;
  
  /* skip white */
  do {
    c=fgetc(f);
  } while (c!=EOF && isspace(c));
  
  if(c!='"') {
    Fatal("Invalid database (initial quote missing)");
  }
  
  /* Put up till next " in temp */
  while(TRUE) {
    c=fgetc(f);
    if(c==EOF) {
      Fatal("Invalid database (EOF in string)");
    }
    if(c=='"') {
      c = fgetc(f);
      /* seems that "" -> " in string */
      if(c!='"') {
        ungetc(c,f);
        break;
      }
    }
    if(c==0x60) {
      /* apparently backtick also gives a double quote */
      c='"';
    }
    
    temp[i++]=(char)c;
  }
  
  temp[i]=0; /* close the string */
  /* allocate memory for it */
  result=MemAlloc(i+1);
  memcpy(result,temp,i+1);
  return result;
}

/* round a location to 0-255 */
void EightBit (short *loc) {
  while((*loc)<0) {
    (*loc)+=256;
  }
  (*loc)%=256;
}

void LoadDatabase(char *filename) {
  FILE* f;
  int i,j;
  char* t;
  
  short vocab;
  short cond[5];
  short act[2];
  
  f = fopen(filename,"r");
  if (f==NULL) {
    perror(filename);
    Exit(1);
  }
  
  /* get header */
  if(fscanf(f,"%hd %hd %hd %hd %hd %hd %hd %hd %hd %hd %hd %hd",
            &database.magicHead, &database.maxItem, &database.maxAction,
            &database.maxWord, &database.maxRoom, &database.maxCarry,
            &database.startRoom, &database.numTreasures,
            &database.wordLength, &database.initialLightTime,
            &database.maxMessage, &database.treasureRoom)!=12) {
    Fatal("Invalid database (bad header)");
  }
  EightBit(&database.startRoom);
  EightBit(&database.treasureRoom); /* these probably not necessary */
  
  /* allocate memory */
  database.item=(Item *)MemAlloc(sizeof(Item)*(database.maxItem+1));
  database.action=(Action *)MemAlloc(sizeof(Action)*(database.maxAction+1));
  database.room=(Room *)MemAlloc(sizeof(Room)*(database.maxRoom+1));
  database.verb=(char**)MemAlloc(sizeof(char*)*(database.maxWord+1));
  database.noun=(char**)MemAlloc(sizeof(char*)*(database.maxWord+1));
  database.message=(char**)MemAlloc(sizeof(char*)*(database.maxMessage+1));
  /* also for these */
  game.itemLoc=(short *)MemAlloc(sizeof(short)*(database.maxItem+1));
  lastLook.itemLoc=(short *)MemAlloc(sizeof(short)*(database.maxItem+1));
  
  /* load action lines */
  for(i=0;i<=database.maxAction;i++) {
    if(fscanf(f,"%hd %hd %hd %hd %hd %hd %hd %hd",
              &vocab,&cond[0],&cond[1],&cond[2],&cond[3],&cond[4],
              &act[0],&act[1])!=8) {
      Fatal("Invalid database (bad action line)");
    }
    database.action[i].verb = vocab/150;
    database.action[i].noun = vocab%150;
    for(j=0;j<5;j++) {
      database.action[i].condition[j] = cond[j]%20;
      database.action[i].value[j] = cond[j]/20;
    }
    for(j=0;j<2;j++) {
      database.action[i].action[j*2] = act[j]/150;
      database.action[i].action[j*2+1] = act[j]%150;
    }
  }
  
  /* read words */
  for(i=0;i<=database.maxWord;i++) {
    database.verb[i] = ReadString(f);
    database.noun[i] = ReadString(f);
  }
  
  /* read rooms */
  for(i=0;i<=database.maxRoom;i++) {
    fscanf(f,"%hd %hd %hd %hd %hd %hd",
           &database.room[i].exit[0],&database.room[i].exit[1],
           &database.room[i].exit[2],&database.room[i].exit[3],
           &database.room[i].exit[4],&database.room[i].exit[5]);
    for(j=0;j<6;j++) {
      EightBit(&database.room[i].exit[j]); /* probably not necessary */
    }
    database.room[i].text = ReadString(f);
  }
  
  /* read messages */
  for(i=0;i<=database.maxMessage;i++) {
    database.message[i] = ReadString(f);
  }
  
  /* read items */
  for(i=0;i<=database.maxItem;i++) {
    database.item[i].text = ReadString(f);
    database.item[i].autoGet = strchr(database.item[i].text,'/');
    /* some games use // to indicate no auto get */
    /* I don't know why we worry about *, what I do is assumed from
       my reading of ScottFree */
    if(database.item[i].autoGet) {
      *(database.item[i].autoGet) = 0;
      (database.item[i].autoGet)++;
      t=strchr(database.item[i].autoGet,'/');
      if(t!=NULL) {
        *t = 0;
      }
      if(database.item[i].autoGet[0]==0 || database.item[i].autoGet[0]=='*') {
        database.item[i].autoGet = NULL;
      }
    }
    fscanf(f,"%hd",&database.item[i].initialLoc);
    /* round locations to 0-255 (necessary for -1 => 255) */
    EightBit(&database.item[i].initialLoc);
  }
  
  /* read comments */
  for(i=0;i<=database.maxAction;i++) {
    database.action[i].comment = ReadString(f);
  }
  
  /* read tail */
  fscanf(f,"%hd %hd %hd",
         &database.version,&database.adventureNumber,&database.magicTail);
  
  fclose(f);
}


/* save and restore in Scott Free savefile format */
void SaveGame() {
  char buf[MAX_INPUT_LENGTH+1];
  FILE *f;
  int i;
  
  PrintDiag("Filename: ");
  Input(buf);
  f=fopen(buf,"w");
  if(f==NULL) {
    PrintDiag("Unable to create save file.\n");
    return;
  }
  
  for(i=0;i<=MAX_COUNTER;i++) {
    fprintf(f,"%hd %hd\n",game.counter[i],game.savedRoom[i]);
  }
  fprintf(f,"%ld %hd %hd %hd %hd %hd\n",
          game.bitFlags,GetBit(DARKBIT)?1:0,
          game.playerRoom,
          game.currentCounter,game.currentSavedRoom,
          game.lightTime);
  for(i=0;i<=database.maxItem;i++) {
    fprintf(f,"%hd\n",game.itemLoc[i]);
  }
  fclose(f);
  PrintDiag("Saved.\n");
}


/* return TRUE iff restore successful */
boolean RestoreGame(char *filename) {
  FILE *f;
  int i;
  short temp;
  
  f = fopen(filename,"r");
  if(f==NULL) {
    PrintDiag("Unable to restore game.\n");
    return FALSE;
  }
  
  for (i=0;i<=MAX_COUNTER;i++) {
    fscanf(f,"%hd %hd\n",&game.counter[i],&game.savedRoom[i]);
  }
  fscanf(f,"%ld %hd %hd %hd %hd %hd\n",
         &game.bitFlags,&temp,
         &game.playerRoom,
         &game.currentCounter,&game.currentSavedRoom,
         &game.lightTime);
  /* backward compatibility */
  if (temp==1) {
    SetBit(DARKBIT);
  }
  for(i=0;i<=database.maxItem;i++) {
    fscanf(f,"%hd\n",&game.itemLoc[i]);
  }
  fclose(f);
  
  return(TRUE);
}


/* Initialize a new game */     
void InitGame() {
  short i;
  
  game.playerRoom = database.startRoom;
  for(i=0;i<=database.maxItem;i++) {
    game.itemLoc[i] = database.item[i].initialLoc;
  }
  game.lightTime = database.initialLightTime;
  /* rest inits to zero since global */
}


/* Look */
void Look() {
  static char *exitName[6]= {
    "North","South","East","West","Up","Down"
  };
  
  char *roomText;
  char lastChar;
  short i;
  boolean flag;
  
  /* if too dark, say so */
  if(Dark()) {
    PrintLook(DM_TOO_DARK_TO_SEE);
    PrintLook("\n");
  }
  else {
    /* describe room */
    roomText = database.room[game.playerRoom].text;
    if(*roomText=='*') {
      PrintLook(roomText+1);
    }
    else {
      PrintLook(DM_ROOM_PREFIX);
      PrintLook(roomText);
    }
    
    /* punctuate if necessary */
    lastChar = roomText[strlen(roomText)-1];
    if (lastChar!='.' && lastChar!='!' && lastChar!='?') {
      PrintLook(".");
    }
    PrintLook("\n");
    
    /* describe items */
    flag = FALSE;
    for (i=0; i<=database.maxItem; i++) {
      if (game.itemLoc[i]==game.playerRoom) {
        flag=TRUE;
        PrintLook(database.item[i].text);
        PrintLook(". ");
      }
    }
    if (flag) {
      PrintLook("\n");
    }
    
    /* describe exits */
    flag = FALSE;
    for (i=0; i<6; i++) {
      if (database.room[game.playerRoom].exit[i]!=0) {
        flag=TRUE;
        PrintLook(exitName[i]);
        PrintLook(" ");
      }
    }
    if (flag) {
      PrintLook("\n");
    }
  }
  
  /* store the current state in lastLook to determine if redraw needed */
  lastLook.playerRoom = game.playerRoom;
  for (i=0;i<=database.maxItem;i++) {
    lastLook.itemLoc[i] = game.itemLoc[i];
  }
  lastLook.dark = Dark();
  lastLook.clear = FALSE;
}


/* Find the number of the noun or verb entered */
short WhichWord(char* word,char **list) {
  short i;
  char* currWord;
  short lastNonSynonym;

  /* this seems helpful (in Buckaroo Banzai) */
  if(strlen(word)==0) {
    return -1;
  }
  
  /* start at 1 */
  for (i=1;i<=database.maxWord;i++) {
    currWord = list[i];
    if(*currWord=='*') {
      currWord++;
    }
    else {
      lastNonSynonym = i;
    }
    if(strncasecmp(word,currWord,database.wordLength)==0) {
      return lastNonSynonym;
    }
  }
  return -1;
}

/* Find the number of an item corresponding to the given noun,
   at this location */
/* nounNum must be >=0 or -3 */
/* if nounNum == -3 we use global getNoun (in case there are autoGets not in
   the noun list */
short WhichItem(short nounNum,short loc) {
  short i;
  int result; /* num if item;
                 -1 if no such item;
                 -2 if item not at this location;
                 -3 if item carried (unless loc==CARRIED) */
  
  result = -1;
  for (i=0; i<=database.maxItem; i++) {
    if(database.item[i].autoGet &&
       strncasecmp(database.item[i].autoGet,
                   (nounNum==-3?getNoun:database.noun[nounNum]),
                   database.wordLength)==0) {
      if (game.itemLoc[i]==loc) {
        return i;
      }
      if (CondCarried(i)) {
        result=-3;
      }
      if (result==-1) {
        result = -2;
      }
    }
  }
  return result;
}

/* I don't know if there can be an item autoGet text which is not
   in the noun database.  This should take care of it though.
   Return TRUE if the noun is an autoGet */
boolean NounIsAutoget () {
  short i;
  for (i=0; i<=database.maxItem; i++) {
    if(database.item[i].autoGet &&
       strncasecmp(database.item[i].autoGet,
                   noun,
                   database.wordLength)==0) {
      return TRUE;
    }
  }
  return FALSE;
}

/* Ask for and accept the player's input; preprocess */
void GetInput() {
  char buf[MAX_INPUT_LENGTH + 1];
  char dummy[MAX_INPUT_WORD_LENGTH + 1];
  int num;
  
  do {
    do {
      strcpy(verb,"");
      strcpy(noun,"");
      PrintCommand(DM_WHAT_NOW);
      Input(buf);
      /* Could use MAX_INPUT_WORD_LENGTH here ? */
      num=sscanf(buf,INPUT_SCANF_FORMAT,verb,noun,dummy);
    } while (num==EOF);
    
    if (num==3) {
      PrintMessage(DM_TOO_MANY_WORDS);
      continue;
    }
    
    /* abbreviations */
    if (strlen(verb)==1) {
      switch(tolower(*verb)) {
      case 'n': strcpy(verb,"NORTH");break;
      case 'e': strcpy(verb,"EAST");break;
      case 's': strcpy(verb,"SOUTH");break;
      case 'w': strcpy(verb,"WEST");break;
      case 'u': strcpy(verb,"UP");break;
      case 'd': strcpy(verb,"DOWN");break;
      case 'i': strcpy(verb,"INVENTORY");break;
      case 'l': strcpy(verb,"LOOK");break;
      }
    }
    
    /* directions as verbs */
    nounNum=WhichWord(verb,database.noun);
    if (nounNum>=1 && nounNum<=6) {
      verbNum=GO_VERB;
      break;
    }
    else {
      verbNum=WhichWord(verb,database.verb);
      nounNum=WhichWord(noun,database.noun);
    }
    
    /* no such verb */
    if (verbNum==-1) {
      PrintMessageString(DM_NO_VERB,verb);
      continue;
    }
    
    /* Use -3 to refer to nouns that are an autoGet but not in
       the noun list; don't know if this is necessary */
    if(nounNum==-1 && NounIsAutoget()) {
      nounNum = -3;
      getNoun = noun;
    }
    
    /* Use -2 to refer to ALL */
    if (GetOption(ALL_OPTION) && nounNum==-1 &&
        strncasecmp(noun,"ALL",database.wordLength)==0) {
      nounNum = -2;
    }
    
    /* no such noun */
    /* after this, nounNum==-1 means no noun */
    if (nounNum == -1 && *noun!=0) {
      PrintMessageString(DM_NO_NOUN,noun);
      continue;
    }
    
    break;
  } while (TRUE);
}


/* get this item (-1 if no such item, -2 if not here, -3 if carried */
/* return TRUE when successful */
boolean Get(short item) {
  /* is this really the correct order of events ? */
  if (CountCarried()>=database.maxCarry) {
    OutReset(); /* not sure this is needed */
    PrintMessage(DM_GOT_TOO_MUCH);
    printedTooMuch = TRUE;
    return FALSE;
  }
  if (item==-1) {
    PrintMessage(DM_BEYOND_POWER);
    return FALSE;
  }
  if (item==-2) {
    PrintMessage(DM_ITEM_NOT_HERE);
    return FALSE;
  }
  if (item==-3) {
    PrintMessage(DM_ALREADY_HAVE);
    return FALSE;
  }
  
  game.itemLoc[item] = CARRIED;
  return TRUE;
}


/* drop this item (-1 if no such item, -2 if not carried) */
void Drop(short item) {
  if (item==-1) {
    PrintMessage(DM_BEYOND_POWER);
    return;
  }
  if (item==-2) {
    PrintMessage(DM_NOT_CARRIED);
    return;
  }
  game.itemLoc[item] = game.playerRoom;
  PrintMessage(DM_OKAY);
}


void Quit() {
  /* If an auto action ends the game, we'll need this look */
  if (Redraw()) {
    Look();
  }
  PrintTitle(DM_ADVENTURE_OVER);
  Exit(0);
}


/* Send the player to the last room.  Then an auto action should do 
   whatever is necessary.  There is sometimes a way out (like Adventureland;
   and less obviously in Pirate Adventure) */
void KillPlayer() {
  PrintMessage(DM_DEAD);
  ClearBit(DARKBIT);
  game.playerRoom=database.maxRoom;
  Look();
}


void Score() {
  short i;
  int count = 0;
  
  for (i=0; i<=database.maxItem; i++) {
    if (game.itemLoc[i]==database.treasureRoom &&
        *(database.item[i].text)=='*') {
      count++;
    }
  }
  PrintMessageNum(DM_STORED_TREASURES,count);
  PrintMessageNum(DM_SCORE_SCALE,(count*100)/database.numTreasures);
  if (count==database.numTreasures) {
    PrintMessage(DM_YOU_WON);
    Quit();
  }
}


void Inventory() {
  short i;
  boolean flag = FALSE;
  PrintMessage(DM_CARRYING);
  for (i=0;i<=database.maxItem;i++) {
    if(game.itemLoc[i]==CARRIED) {
      PrintMessage(database.item[i].text);
      PrintMessage(". ");
      flag = TRUE;
    }
  }
  if (!flag) {
    PrintMessage(DM_NOTHING);
  }
  else {
    PrintMessage("\n");
  }
}


void FillLamp() {
  game.lightTime = database.initialLightTime;
  game.itemLoc[LIGHT_SOURCE] = CARRIED;
  ClearBit(LIGHTOUTBIT);
}


/* I hacked this in here to try to get acceptable spacing */
void OutReset() {
  if(outputPos>0) {
    PrintMessage("\n");
  }
}


/* attempt to perform the action; return TRUE if conditions met */
boolean PerformActionLine(short act) {
  short param[5]; /* parameters */
  short pNum = 0; /* number of current parameter */
  short i;
  short val;
  short temp;
  
  /* scan conditions */
  for (i=0;i<5;i++) {
    val = database.action[act].value[i];
    switch(database.action[act].condition[i]) {
    case 0:
      param[pNum++]=val;
      break;
    case 1: if(!CondCarried(val)) {return FALSE;} break;
    case 2: if(!CondHere(val)) {return FALSE;} break;
    case 3: if(!CondAvailable(val)) {return FALSE;} break;
    case 4: if(!CondInRoom(val)) {return FALSE;} break;
    case 5: if(CondHere(val)) {return FALSE;} break;
    case 6: if(CondCarried(val)) {return FALSE;} break;
    case 7: if(CondInRoom(val)) {return FALSE;} break;
    case 8: if(!GetBit(val)) {return FALSE;} break;
    case 9: if(GetBit(val)) {return FALSE;} break;
    case 10:if(CountCarried()==0) {return FALSE;} break;
    case 11:if(CountCarried()>0) {return FALSE;} break;
    case 12:if(CondAvailable(val)) {return FALSE;} break;
    case 13:if(CondDestroyed(val)) {return FALSE;} break;
    case 14:if(!CondDestroyed(val)) {return FALSE;} break;
    case 15:if(game.currentCounter > val) {return FALSE;} break;
    case 16:if(game.currentCounter <= val) {return FALSE;} break;
    case 17:if(!CondOriginalRoom(val)) {return FALSE;} break;
    case 18:if(CondOriginalRoom(val)) {return FALSE;} break;
    case 19:if(game.currentCounter!=val) {return FALSE;} break;
      /* Alan Cox comments: only seen in Brian Howarth games so far */
    }
  }
  
  /* if passed conditions, perform actions */
  if(GetOption(TRACE_OPTION)) {
    PrintDiagNum("(DID: %d",act);
    if(*(database.action[act].comment)!=0) {
      PrintDiag("  ");
      PrintDiag(database.action[act].comment);
    }
    PrintDiag(")");
    if (GetOption(SPACE_OPTION)) {
      PrintDiag(" ");	
    }
    else {
      PrintDiag("\n");
    }
  }
  pNum = 0;
  for (i=0; i<4; i++) {
    val = database.action[act].action[i];
    if ((val>=1 && val<=51) || val>=102) {
      if (val>=102) {
	val-=50;
      }
      PrintMessage(database.message[val]);
      if (GetOption(SPACE_OPTION)) {
	if(!isspace(database.message[val]
		    [strlen(database.message[val])-1])) {
	  PrintMessage(" ");	
	}
      }
      else {
	PrintMessage("\n");
      }
    }
    else switch(val) {
    case 0:/* NOP */ break;
      /* The VIC-20 driver aborts an action line after a Get fails.
         From looking at the games I could get in ScottFree format
         with "scottdec" I'm sure this is intended for all of them
         (the one possible exception could be "MAKE FUEL" in Buckaroo
         Banzai).  Not sure how it should affect a continuation; I
         couldn't find any examples. */
    case 52:if (!Get(param[pNum++])) {return TRUE;} break;
    case 53:game.itemLoc[param[pNum++]]=game.playerRoom; break;
    case 54:game.playerRoom=param[pNum++]; break;
    case 55:game.itemLoc[param[pNum++]]=DESTROYED; break;
    case 56:SetBit(DARKBIT); break;
    case 57:ClearBit(DARKBIT); break;
    case 58:SetBit(param[pNum++]); break;
      /* 59 == 55 ? */
    case 59:game.itemLoc[param[pNum++]]=DESTROYED; break;
    case 60:ClearBit(param[pNum++]); break;
    case 61:OutReset(); KillPlayer(); break;
    case 62:game.itemLoc[param[pNum]]=param[pNum+1]; pNum+=2; break;
    case 63:OutReset(); Quit(); break;
    case 64:OutReset(); Look(); break;
    case 65:OutReset(); Score(); break;
    case 66:OutReset(); Inventory(); break;
    case 67:SetBit(0); break;
    case 68:ClearBit(0); break;
    case 69:FillLamp(); break;
    case 70:ClearScreen(); break;
    case 71:OutReset(); SaveGame(); break;
    case 72:
      temp=game.itemLoc[param[pNum]];
      game.itemLoc[param[pNum]]=game.itemLoc[param[pNum+1]];
      game.itemLoc[param[pNum+1]]=temp;
      pNum+=2;
      break;
    case 73:continuation=TRUE; break;
    case 74:game.itemLoc[param[pNum++]]=CARRIED; break;
    case 75:
      game.itemLoc[param[pNum]]=game.itemLoc[param[pNum+1]];
      pNum+=2;
      break;
      /* 76 == 64 ? */
    case 76:OutReset(); Look(); break;
      /* Limit counters at -1 or 0 ? */
    case 77:if(game.currentCounter>=0) {game.currentCounter--;} break;
      /* added this spacing after "Ghost Town" */
    case 78:PrintMessageNum("%d ",game.currentCounter); break;
    case 79:game.currentCounter=param[pNum++]; break;
    case 80:
      temp = game.playerRoom;
      /* probably currentSavedRoom should just be savedRoom[0] */
      game.playerRoom = game.currentSavedRoom;
      game.currentSavedRoom = temp;
      break;
    case 81:
      /* Alan Cox comments:
         This is somewhat guessed. Claymorgue always
         seems to do select counter n, thing, select counter n,
         but uses one value that always seems to exist. Trying
         a few options I found this gave sane results on ageing */
      temp = game.currentCounter;
      game.currentCounter = game.counter[param[pNum]];
      game.counter[param[pNum]] = temp;
      pNum++;
      break;
    case 82:game.currentCounter+=param[pNum++]; break;
      /* Limit counters at -1 or 0 ? */
    case 83:/* Alan Cox comments: Note: This seems to be needed. I don't yet
               know if there is a maximum value to limit too */
      game.currentCounter-=param[pNum++];
      if(game.currentCounter<-1) {game.currentCounter=-1;}
      break;
      /* maybe a space would be useful here too */
    case 84:PrintMessage(noun); PrintMessage(" "); break;
    case 85:PrintMessage(noun); PrintMessage("\n"); break;
    case 86:PrintMessage("\n"); break;
    case 87:
      temp = game.playerRoom;
      game.playerRoom = game.savedRoom[param[pNum]];
      game.savedRoom[param[pNum]] = temp;
      pNum++;
      break;
    case 88:Delay(); break;
    case 89:pNum++; /* SAGA draw picture n */ break;
    default:
      OutReset();
      PrintDiagNum("Unknown action %d!\n",val);
      break;
    }
  }
  return TRUE;
}


/* perform all automatic actions */
void AutoActions () {
  short i;
  
  for (i=0;i<database.maxAction;i++) {
    if (database.action[i].verb!=0 || database.action[i].noun!=0) {
      continuation = FALSE;
      OutReset();
    }
    if ((database.action[i].verb==0
         && RandomPercent(database.action[i].noun))
        || continuation) {
      PerformActionLine(i);
    }
  }
  continuation=FALSE;
  OutReset();
}


boolean MoveCommand() {
  short next;
  
  if(nounNum==-1) {
    PrintMessage(DM_NO_DIRECTION);
    return TRUE;
  }
  if(nounNum>=1 && nounNum<=6) {
    if (Dark()) {
      PrintMessage(DM_DANGEROUS_DARK);
    }
    next = database.room[game.playerRoom].exit[nounNum-1];
    if(next!=0) {
      game.playerRoom = next;
      PrintMessage(DM_OKAY);
      lastLook.clear = TRUE; /* this is here because of moving in some
		   directions will take you back to the same room;
		   we don't want a failure to auto-look to give that
		   away */
      return TRUE;
    }
    if(Dark()) {
      PrintMessage(DM_BROKE_NECK);
      KillPlayer();
      return TRUE;
    }
    PrintMessage(DM_NO_WAY);
    return TRUE;
  }
  return FALSE;
}

/* GET noun */
/* return false if output still needed */
/* notYet is true iff some action might correspond */
boolean GetCommand(boolean notYet) {
  short i;
  short startRoom;
  boolean *startHere;
  boolean flag;
  short theItem;
  
  startHere = (boolean *) MemAlloc(sizeof(boolean)*database.maxItem);
  /* no noun */
  if (nounNum==-1) {
    PrintMessage(DM_NO_ITEM);
    return TRUE;
  }
  /* get all */
  if (nounNum==-2) {
    if (Dark()) {
      PrintMessage(DM_GET_ALL_DARK);
      return TRUE;
    }
    
    flag = FALSE;
    printedTooMuch = FALSE;
    startRoom = game.playerRoom;
    /* only get things that started off here (and gettable) */
    for (i=0;i<=database.maxItem;i++) {
      startHere[i] = (CondHere(i) && database.item[i].autoGet) ;
    }
    for (i=0;i<=database.maxItem;i++) {
      /* if it's still here, try to get it */
      if (startHere[i] && CondHere(i)) {
        flag=TRUE;
        PrintMessage(database.item[i].text);
        PrintMessage(": ");
        nounNum = WhichWord(database.item[i].autoGet,database.noun);
	getNoun = database.item[i].autoGet;
        PerformCommand();
        /* if the command doesn't give any output this may be a problem */
	/* but now taken care of by OutReset() in PerformCommand() */
        /* stop if we move, it gets dark, or hit too much */
        if (game.playerRoom != startRoom || Dark() || printedTooMuch) {
          break;
        }
      }
    }
    if (!flag) {
      PrintMessage(DM_NO_GET_ALL);
    }
    return TRUE;
  }

  theItem = WhichItem(nounNum,game.playerRoom);
  if(notYet && (theItem<0)) {
    return FALSE;
  }
  if(Get(theItem)) {
    PrintMessage(DM_OKAY);
  }
  return TRUE;
}

/* DROP noun*/
/* return false if output still needed */
/* notYet is true iff some action might correspond */
boolean DropCommand(boolean notYet) {
  short i;
  short startRoom;
  boolean startDark;
  boolean *startCarried;
  boolean flag;
  short theItem;

  startCarried = (boolean *)MemAlloc(sizeof(boolean)*database.maxItem);
  
  /* no noun */
  if (nounNum==-1) {
    PrintMessage(DM_NO_ITEM);
    return TRUE;
  }
  /* drop all */
  if (nounNum==-2) {
    flag = FALSE;
    startDark = Dark();
    startRoom = game.playerRoom;
    /* only drop things that started off carried (and dropable) */
    for (i=0;i<=database.maxItem;i++) {
      startCarried[i] = (CondCarried(i) && database.item[i].autoGet) ;
    }
    for (i=0;i<=database.maxItem;i++) {
      /* if still carried, try to drop it */
      if (startCarried[i] && CondCarried(i)) {
        flag=TRUE;
        PrintMessage(database.item[i].text);
        PrintMessage(": ");
        nounNum = WhichWord(database.item[i].autoGet,database.noun);
	getNoun = database.item[i].autoGet;
        PerformCommand();
        /* stop if we move or dark changes*/
        if (game.playerRoom != startRoom || startDark != Dark()) {
          break;
        }
      }
    }
    if (!flag) {
      PrintMessage(DM_NO_DROP_ALL);
    }
    return TRUE;
  }
  
  theItem = WhichItem(nounNum,CARRIED);
  if (notYet && theItem<0) {
    return FALSE;
  }
  
  Drop(theItem);
  return TRUE;
}

/* Perform the current command */
void PerformCommand () {
  short i;
  int result; 
  /* -1 if no action match;
     0 if action match, but conditions not met;
     1 if action completed. */
  
  /* should this really be before the action perusal? */
  if (verbNum==GO_VERB) {
    if (MoveCommand())
      return;
  }
  
  /* see if we match any of the game's actions */
  result = -1;
  for (i=0;i<=database.maxAction;i++) {
    /* stop continuations once reached non-zero line */
    if ((database.action[i].verb!=0 || database.action[i].noun!=0) 
        && continuation) {
      break;
    }
    if ((verbNum==database.action[i].verb &&
         (nounNum==database.action[i].noun || database.action[i].noun==0))
        || continuation) {
      if (result==-1) {
        result = 0;
      }
      if (PerformActionLine(i)) {
        result = 1;
        if (!continuation) {
          break;
        }
      }
    }
  }
  continuation = FALSE;
  
  if (result==1) {
    OutReset();
    return;
  }
  
  /* auto get and drop */
  if (verbNum==GET_VERB) {
    if (GetCommand(result==0)) {
      return;
    }
  }
  if (verbNum==DROP_VERB) {
    if (DropCommand(result==0)) {
      return;
    }
  }
  
  switch(result) {
  case -1:PrintMessage(DM_DONT_UNDERSTAND); break;
  case 0:PrintMessage(DM_NOT_YET); break;
  }
}


/* countdown to dark */
void WorkLight() {
  /* Brian Howarth games seem to use -1 light time for forever */
  if(!CondDestroyed(LIGHT_SOURCE) &&
     game.lightTime != -1) {
    game.lightTime--;
    
    if (game.lightTime < 1) {
      SetBit(LIGHTOUTBIT);
      if(CondAvailable(LIGHT_SOURCE)) {
        PrintMessage(DM_LIGHT_OUT);
        Look();
      }
      
      /* If we need to do the work of destroying the lamp ourselves */
      if (GetOption(PREHISTORIC_LAMP_OPTION)) {
        game.itemLoc[LIGHT_SOURCE] = DESTROYED;
      }
    }
    else if (game.lightTime<25) {
      if(CondAvailable(LIGHT_SOURCE)) {
        if (!GetOption(DIM_OPTION) || game.lightTime%5==0) {
          PrintMessageNum(DM_LIGHT_COUNTDOWN,game.lightTime);
        }
      }
    }
  }
}


void MainLoop () {
  /* Seems better to leave this out (since some auto actions will be like
     continuations, as "drop mirror" in Adventureland); the only real
     problem is when the auto action ends the game.  So I put in a look
     in the Quit() function */
  /*
    if (GetOption(LOOK_OPTION) && Redraw()) {
    Look();
    }
    */
  AutoActions();
  if (GetOption(LOOK_OPTION) && Redraw()) {
    Look();
  }
  GetInput();
  PerformCommand();
  WorkLight();
}

/*** main ***/
void main (int argc,char *argv[]) {
  ParseArguments(&argc,&argv);
  /* after ParseArguments argv[1] and argv[2] hold game name and save name */
  LoadDatabase(argv[1]);
  
  Initialize();
  TitleScreen();
  /* if no game to restore, or if restore fails, initialize a new game */
  if(argc!=3 || !RestoreGame(argv[2])) {
    InitGame();
  }
  
  Look();
  while(TRUE) {
    MainLoop();
  }
}
