//-----------------------------------------------------------------------------
// CyScott: Scott Adams adventure game interpreter   ported by Athlor         |
//-----------------------------------------------------------------------------
#include <CyWin.h>
#include "epforms.h" // Ernest Pazerra forms
#define TRUE 1
#define FALSE 0
#define LED (*(char *)0xFFFF61) // Green LED register
#define MyLoc (GameHeader.PlayerRoom)
#define LIGHT_SOURCE 9 // Always 9 how odd
#define CARRIED 255    // Carried
#define DESTROYED 0    // Destroyed
#define DARKBIT 15
#define LIGHTOUTBIT 16 // Light gone out

typedef struct
{
 int Unknown;
 int NumItems;
 int NumActions;
 int NumWords; // Smaller of verb/noun is padded to same size
 int NumRooms;
 int MaxCarry;
 int PlayerRoom;
 int Treasures;
 int WordLen;
 int LightTime;
 int NumMessages;
 int TreasureRoom;
} Header;

long AC_Vocab[255];
long AC_Cond[255][5];
long AC_Act[255][2];

typedef struct
{
 char *Text;
 int Exits[6];
} Room;

typedef struct
{
 char *Text;
 int Loc; // WARNING: THESE TWO MUST BE 8 BIT VALUES
 int InitialLoc;
 char *AutoGet;
} Item;

char nam[48]; // Filename (.dat)
Header GameHeader;
Item *Items;
Room *Rooms;
char **Verbs;
char **Nouns;
char **Messages;
int LightRefill;
char NounText[16];
int Counters[16];  // Range unknown
int CurrentCounter;
int SavedRoom;
int RoomSaved[16]; // Range unknown
int Redraw;        // Update item window
long BitFlags=0;   // Might be >32 flags - I haven't seen >32 yet
char *cs="CyScott v0.6";
char tnum[8];
char tmp[1024];
char x=0; // Exit code
int getc(void);
void strupr(char *);
void cls(void);

void Fatal(char *);
void *MemAlloc(int);
int RandomPercent(int);
int CountCarried(void);
char *MapSynonym(char *);
int MatchUpItem(char *,int);
char *ReadString(void);
int Readnum(void);
void Writenum(long);
void LoadDatabase(void);
void Wrap(char *);
void Look(void);
int WhichWord(char *,char **);
void GetInput(int *,int *);
void SaveGame(void);
void LoadGame(char *);
int PerformLine(int);
int PerformActions(int,int);
char isupper(char);
char tolower(char);
char isspace(char);

struct MSequence mus;
struct Message* ptr_msg; // Structures
struct FileInput f_in;
struct FileOutput f_out;
struct module_t main_module;

//-----------------------------------------------------------------------------
// Cybiko Specific:  clrln, Scroll, puts, Wrap, _getch, gets and printf       |
//-----------------------------------------------------------------------------
void clrln(void)
{
  TGraph_set_color(main_module.m_gfx,CLR_WHITE);
  TGraph_fill_rect(main_module.m_gfx,0,91,160,9); // Erase bottom line
  TGraph_set_color(main_module.m_gfx,CLR_BLACK);
}

void Scroll(void)
{
  TGraph_scroll(main_module.m_gfx,0,0,160,100,0,-9);
  clrln();
}

int puts(char *s)
{
  Scroll();
  Graphics_draw_text(main_module.m_gfx,s,0,90); // Print Text
  DisplayGraphics_show(main_module.m_gfx); // Update Display
}

void Wrap(char *s)
{
 char l[64];int i;int lsp;
  while(DisplayGraphics_string_width(main_module.m_gfx,s)>155)
  {
    i=0; l[i]=0;
    do{
      if(*s=='\n')  *s=32; // Change \n to space (.dat files contain \n)
      if(*s==32)  lsp=i; // Remember last space
      l[i++]= *s++; l[i]=0;
    }while(DisplayGraphics_string_width(main_module.m_gfx,l)<155/*&&(l[i-1]!='\n')*/);
    /*if(*(s-1)!='\n')*/ {s=s-(i-lsp)+1;i=lsp;l[i++]='\n'; l[i]=0;}
    puts(l);
  }
  puts(s);
}

int _getch(void)
{
 int key;bool pressed=FALSE;
  do{
    ptr_msg=cWinApp_get_message(main_module.m_process,0,1,MSG_USER);
    switch(ptr_msg->msgid)
    {
      case MSG_KEYDOWN:
        key=Message_get_key_param(ptr_msg)->scancode; // Allows spec keys
        switch(key)
        {
          case KEY_ENTER: case KEY_BACKSPACE: case KEY_RIGHT: case KEY_SELECT:
          case KEY_UP: case KEY_DOWN: case KEY_LEFT: case KEY_DEL: case KEY_TAB:
            pressed=TRUE;
          case KEY_SHIFT: case KEY_CONTROL: break;
          default: pressed=TRUE;key=Message_get_key_param(ptr_msg)->ch;break;
        }
        break;
      default: cWinApp_defproc(main_module.m_process,ptr_msg); // Process remaining
    }
    Message_delete(ptr_msg); // Delete processed message
  }while(!pressed);
 return key;
}

void gets(char *lbuf)
{
int key,i=0;bool done=FALSE;
  Scroll();
  lbuf[0]=0;
  do{
    key=_getch(); // Wait until key is pressed
    if(key==264) {lbuf[0]='w';lbuf[1]=0;break;} // Control pad hack
    if(key==265) {lbuf[0]='n';lbuf[1]=0;break;}
    if(key==266) {lbuf[0]='e';lbuf[1]=0;break;}
    if(key==267) {lbuf[0]='s';lbuf[1]=0;break;}
    switch(key)
    {
      case KEY_ENTER: lbuf[i]=0;done=TRUE;break;
      case KEY_BACKSPACE: if(i) lbuf[--i]=0;break;
      default: lbuf[i++]=(char)key;lbuf[i]=0;break;
    }
    clrln();
    Graphics_draw_text(main_module.m_gfx,lbuf,0,91);
    DisplayGraphics_show(main_module.m_gfx); // Update Display
  }while(!done);
}

int printf(char *tmp, ...)
{
 char tbuf[99]; char *t=tbuf;
  vsprintf(tbuf, tmp, &tmp+1);
  Wrap(tbuf);
}

//-----------------------------------------------------------------------------
// Main - Program Entry Point                                                 |
//-----------------------------------------------------------------------------
long main(int argc, char* argv[], bool start)
{
 int vb,no;
  init_module(&main_module); // Initialize module
  vb=FileListForm(cs,"*.dat",nam,main_module.m_process,5,5,150,90); //EP
  if(vb==-1) return 0L; // ESC pressed
  MSequence_ctor(&mus,"tol.mus");
  MSequence_play_background(&mus);
  draw_lib(0,0,0,0,BM_NORMAL); // Display title bmp
  DisplayGraphics_show(main_module.m_gfx); // Update Display
  LoadDatabase();

  if(MessageBox(cs,"Load a saved game?",mbYes|mbNo,main_module.m_process)==mrYes)
  {
    vb=FileListForm("Saved Game","*.sag",nam,main_module.m_process,5,5,150,90);
    if(vb>-1) LoadGame(nam);
  }
  cls();
  puts(cs);puts("ported by Athlor");
  srand(clock());
  Look();
  do{
    PerformActions(0,0);
    if(Redraw!=0) {Look();Redraw=0;}
    GetInput(&vb,&no);
    switch(PerformActions(vb,no))
    {
      case -1: puts("I don't understand your command.");break;
      case -2: puts("I can't do that yet.");break;
    }
    if(Items[LIGHT_SOURCE].Loc/*==-1*/!=DESTROYED && GameHeader.LightTime!= -1)
    { // Brian Howarth games seem to use -1 for forever
      GameHeader.LightTime--;
      if(GameHeader.LightTime<1)
      {
        BitFlags|=(1L<<LIGHTOUTBIT);
        if(Items[LIGHT_SOURCE].Loc==CARRIED||Items[LIGHT_SOURCE].Loc==MyLoc)
          puts("Light has run out!");
        Items[LIGHT_SOURCE].Loc=DESTROYED; // Optional lamp disappears
      }
      else if(GameHeader.LightTime<25)
        if(Items[LIGHT_SOURCE].Loc==CARRIED||Items[LIGHT_SOURCE].Loc==MyLoc)
          printf("Light runs out in %d turns.",GameHeader.LightTime);
    }
  }while(x==0);
  MSequence_dtor(&mus,LEAVE_MEMORY);
  free(Messages);
  free(Rooms);
  free(Nouns);
  free(Verbs);
  free(Items);
 return 0L;
} // End Main

//---------------------------------------------------------------------CyScott-
void Fatal(char *z)
{
  cprintf("%s.\n",z); // Debug
  beep(0);
  x=1;
}

void *MemAlloc(int size)
{
 void *t=(void *)malloc(size);
  if(t==NULL) Fatal("Out of memory");
 return t;
}

int RandomPercent(int n)
{
 int rv=random(100)+1; // rv=rand()<<6;
  rv%=100;
  if(rv<n) return 1;
 return 0;
}

int CountCarried(void)
{
 int ct=0,n=0;
  while(ct<=GameHeader.NumItems)
  {
    if(Items[ct].Loc==CARRIED) n++;
    ct++;
  }
 return n;
}

char *MapSynonym(char *word)
{
 int n=1;
 char *tp;
 static char lastword[16]; // Last non synonym
  while(n<=GameHeader.NumWords)
  {
    tp=Nouns[n];
    if(*tp=='*') tp++;
    else strcpy(lastword,tp);
    if(strncmp(word,tp,GameHeader.WordLen)==0) return lastword;
    n++;
  }
 return 0;
}

int MatchUpItem(char *text, int loc)
{
 char *word=MapSynonym(text);
 int ct=0;
  if(word==NULL) word=text;
  while(ct<=GameHeader.NumItems)
  {
    if(Items[ct].AutoGet&&Items[ct].Loc==loc&&strncmp(Items[ct].AutoGet,word,GameHeader.WordLen)==0)
      return ct;
    ct++;
  }
 return -1;
}

char *ReadString(void)
{
 char *t;
 int c,nc,sr=0,ct=0;
oops:
  do{
    c=getc();
  }while(c != -1 && isspace(c));
  if(c!='"') Fatal("Initial quote expected");
  do{
    if(sr==0) c=getc(); else {c=sr;sr=0;}
    if(c== -1) Fatal("EOF in string");
    if(c=='"')
    {
      nc=getc();
      if(nc!='"')
      {
        sr=nc;
        break;
      }
    }
    if(c==0x60) c='"'; // pdd
    tmp[ct++]=c;
  }while(1);
  tmp[ct]=0;
  t=MemAlloc(ct+1);
  memcpy(t,tmp,ct+1);
  cprintf("%s\n",t); // Debug
  return t;
}

int Readnum(void)
{
 int c,i=0;
  while((c=getc())!='\n') tnum[i++]=c;
  tnum[i]='\0';
  cprintf("%s\n",tnum); // Debug
 return xtoi(tnum);
}

void LoadDatabase(void)
{
 int mc,pr,tr,wl,lt,mn,trm,ct;
 Room *rp;
 Item *ip;
  LED |= 8; // Turn green LED ON
  FileInput_ctor_Ex(&f_in, nam); // Create FileInput obj
  cprintf("Reading Header\n");
  GameHeader.Unknown=Readnum();
  GameHeader.NumItems=Readnum();
  GameHeader.NumActions=Readnum();
  GameHeader.NumWords=Readnum();
  GameHeader.NumRooms=Readnum();
  GameHeader.MaxCarry=Readnum();
  GameHeader.PlayerRoom=Readnum();
  GameHeader.Treasures=Readnum();
  GameHeader.WordLen=Readnum();
  GameHeader.LightTime=Readnum();
  GameHeader.NumMessages=Readnum();
  GameHeader.TreasureRoom=Readnum();
   //LightRefill=Readnum();

  Items=(Item *)MemAlloc(sizeof(Item)*(GameHeader.NumItems+1));
  cprintf("Alloc %d items\n",GameHeader.NumItems);
  Verbs=(char **)MemAlloc(sizeof(char *)*(GameHeader.NumWords+1));
  cprintf("Alloc %d verbs\n",GameHeader.NumWords);
  Nouns=(char **)MemAlloc(sizeof(char *)*(GameHeader.NumWords+1));
  cprintf("Alloc %d nouns\n",GameHeader.NumWords);
  Rooms=(Room *)MemAlloc(sizeof(Room)*(GameHeader.NumRooms+1));
  cprintf("Alloc %d rooms\n",GameHeader.NumRooms);
  Messages=(char **)MemAlloc(sizeof(char *)*(GameHeader.NumMessages+1));
  cprintf("Alloc %d messages\n",GameHeader.NumMessages);
  ct=0; // Load the actions
  cprintf("Reading %d actions\n",GameHeader.NumActions);
  while(ct<GameHeader.NumActions+1)
  {
    cprintf("Element: %d\n",ct);
    AC_Vocab[ct]=Readnum();
    AC_Cond[ct][0]=Readnum();
    AC_Cond[ct][1]=Readnum();
    AC_Cond[ct][2]=Readnum();
    AC_Cond[ct][3]=Readnum();
    AC_Cond[ct][4]=Readnum();
    AC_Act[ct][0]=Readnum();
    AC_Act[ct][1]=Readnum();
    ct++;
  }
  ct=0;
  cprintf("\nReading %d word pairs.\n",GameHeader.NumWords);
  while(ct<GameHeader.NumWords+1)
  {
    Verbs[ct]=ReadString();
    Nouns[ct]=ReadString();
    ct++;
  }
  ct=0;
  rp=Rooms;
  cprintf("Reading %d rooms.\n",GameHeader.NumRooms);
  while(ct<GameHeader.NumRooms+1)
  {
    rp->Exits[0]=Readnum();
    rp->Exits[1]=Readnum();
    rp->Exits[2]=Readnum();
    rp->Exits[3]=Readnum();
    rp->Exits[4]=Readnum();
    rp->Exits[5]=Readnum();
    rp->Text=ReadString();
    ct++;
    rp++;
  }
  ct=0;
  cprintf("Reading %d messages.\n",GameHeader.NumMessages);
  while(ct<GameHeader.NumMessages+1)
    Messages[ct++]=ReadString();
  ct=0;
  cprintf("Reading %d items.\n",GameHeader.NumItems);
  ip=Items;
  while(ct<GameHeader.NumItems+1)
  {
    ip->Text=ReadString();
    ip->AutoGet=strchr(ip->Text,'/');
    // Some games use // to mean no auto get/drop word!
    if(ip->AutoGet && strcmp(ip->AutoGet,"//") && strcmp(ip->AutoGet,"/*"))
    {
      char *t;
      *ip->AutoGet++=0;
      t=strchr(ip->AutoGet,'/');
      if(t!=NULL) *t=0;
    }
    ip->Loc=Readnum();
    ip->InitialLoc=ip->Loc;
    ip++;
    ct++;
  }
/*ct=0;
   while(ct<GameHeader.NumActions+1)  // Discard Comment Strings
   {
      free(ReadString());
      ct++;
   }
  ct=Readnum();
  cprintf("Version %d.%02d of Adventure ",ct/100,ct%100);
  ct=Readnum();*/
  cprintf("%d.\nLoad Complete.\n",ct); // Debug
  FileInput_dtor(&f_in, LEAVE_MEMORY); // Destroy FileInput obj
  LED &= 247; // Turn green LED OFF
}
    
void Look(void)
{
 char *ExitNames[6]={"North","South","East","West","Up","Down"};
 Room *r;
 int ct,f;
  cprintf("Location# %d\n",MyLoc); // Debug
  if((BitFlags&(1L<<DARKBIT))&&Items[LIGHT_SOURCE].Loc!=CARRIED&&Items[LIGHT_SOURCE].Loc!= MyLoc)
  {
    puts("You can't see. It is too dark!");
    return;
  }
  r=&Rooms[MyLoc];
  if(*r->Text=='*') Wrap(r->Text+1);
  else printf("You are in a %s",r->Text);
  ct=0;
  f=0;
  strcpy(tmp,"Exits: ");
  while(ct<6)
  {
    if(r->Exits[ct]!=0)
    {
      if(f==0) f=1;
      else strcat(tmp,", ");
      strcat(tmp,ExitNames[ct]);
    }
    ct++;
  }
  if(f==0) strcat(tmp,"none");
  strcat(tmp,".");
  Wrap(tmp);
  ct=0;
  f=0;
  while(ct<=GameHeader.NumItems)
  {
    if(Items[ct].Loc==MyLoc)
    {
      if(f==0)
      {
        puts("You can also see:");
        f++;
      }
      Wrap(Items[ct].Text);
    }
    ct++;
  }
}

int WhichWord(char *word, char **list)
{
 int n=1,ne=1;
 char *tp;
  while(ne<=GameHeader.NumWords)
  {
    tp=list[ne];
    if(*tp=='*') tp++;
    else n=ne;
    if(strncmp(word,tp,GameHeader.WordLen)==0) return n;
    ne++;
  }
 return -1;
}

void GetInput(int *vb, int *no)
{
 char verb[10],noun[10],buf[64]; // ORG 256 but it bombed
 int vc,nc,num=0;
  do{
    do{
      puts("Tell me what to do ?");
      gets(buf);
      strncpy(verb,buf,strrchr(buf, ' ')-buf);
      strcpy(noun,strrchr(buf, ' ')+1);
      cprintf("Verb: %s  Len: %d\n",verb, strlen(verb)); // Debug
      cprintf("Noun: %s  Len: %d\n",noun, strlen(noun)); // Debug
      if(strlen(verb)) {num++; strupr(verb);}
      if(strlen(noun)) {num++; strupr(noun);}
    }while(num==0||*buf=='\n');
    if(num==1) *noun=0;
    if(*noun==0 && strlen(verb)==1)
    {
      switch(isupper(*verb)?tolower(*verb):*verb)
      {
        case 'n': strcpy(verb,"NORTH");break; // Shortcuts
        case 'e': strcpy(verb,"EAST");break;
        case 's': strcpy(verb,"SOUTH");break;
        case 'w': strcpy(verb,"WEST");break;
        case 'u': strcpy(verb,"UP");break;
        case 'd': strcpy(verb,"DOWN");break;
        case 'i': strcpy(verb,"INVENTORY");break;
        case 'l': strcpy(verb,"LOOK");break;
      }
    }
    nc=WhichWord(verb,Nouns);
    if(nc>=1&&nc<=6) vc=1; // Scott Adams system hack to avoid typing 'go'
    else
    {
      vc=WhichWord(verb,Verbs);
      nc=WhichWord(noun,Nouns);
    }
    *vb=vc;
    *no=nc;
    if(vc==-1) printf("\"%s\" is a word I don't know.",verb);
  }while(vc==-1);
  strcpy(NounText,noun); // Needed by GET/DROP hack
}

void Writenum(long n)
{
 sprintf(tnum,"%d\n",n);
 FileOutput_write(&f_out,tnum,strlen(tnum));
}

////////////////////////////////////////////////////////////////////////////////////////////////
void SaveGame(void)
{
 int ct=InputBox(cs,"Saved game name:",mbOk|mbCancel,16,tmp,main_module.m_process);
  if(ct==mrCancel) return;
  cls();
  strcat(tmp,".sag");
  FileOutput_ctor_Ex(&f_out,tmp,TRUE);
  if(FileOutput_is_bad(&f_out))
  {
    MessageBox(cs,"Unable to save game.",mbOk,main_module.m_process);
    cls();
    return;
  }
  cls();
  for(ct=0;ct<16;ct++)
  {
    Writenum(Counters[ct]);
    cprintf("Counters[%d]: %d\n",ct,Counters[ct]);
    Writenum(RoomSaved[ct]);
    cprintf("RoomSaved[%d]: %d\n",ct,RoomSaved[ct]);
  }
  Writenum(BitFlags);
  cprintf("BitFlags: %d\n",BitFlags);
  Writenum((BitFlags&(1L<<DARKBIT))?1:0);
  cprintf("DarkFlag: %d\n",(BitFlags&(1L<<DARKBIT))?1:0);
  Writenum(MyLoc);
  cprintf("MyLoc: %d\n",MyLoc);
  Writenum(CurrentCounter);
  cprintf("CurrentCounter: %d\n",CurrentCounter);
  Writenum(SavedRoom);
  cprintf("SavedRoom: %d\n",SavedRoom);
  Writenum(GameHeader.LightTime);
  cprintf("LightTime: %d\n",GameHeader.LightTime);
  for(ct=0;ct<=GameHeader.NumItems;ct++)
  {
    Writenum(Items[ct].Loc);
    cprintf("Items[%d].Loc: %d\n",ct,Items[ct].Loc);
  }
  Writenum(0); // Pad extra 0
  FileOutput_dtor(&f_out,LEAVE_MEMORY);
}

void LoadGame(char *name)
{
 int ct,DarkFlag,Ioldpos;
  FileInput_ctor_Ex(&f_in,name);
  if(FileInput_is_bad(&f_in))
  {
    MessageBox(cs,"Unable to restore game.",mbOk,main_module.m_process);
    cls();
    return;
  }
  cls();
  cprintf("Loading saved game: %s\n",name);
  for(ct=0;ct<16;ct++)
  {
    Counters[ct]=Readnum();
    cprintf("Counters[%d]: %d\n",ct,Counters[ct]);
    RoomSaved[ct]=Readnum();
    cprintf("RoomSaved[%d]: %d\n",ct,RoomSaved[ct]);
  }
  BitFlags=Readnum();
  cprintf("BitFlags: %d\n",BitFlags);
  DarkFlag=Readnum();
  cprintf("DarkFlag: %d\n",DarkFlag);
  MyLoc=Readnum();
  cprintf("MyLoc: %d\n",MyLoc);
  CurrentCounter=Readnum();
  cprintf("CurrentCounter: %d\n",CurrentCounter);
  SavedRoom=Readnum();
  cprintf("SavedRoom: %d\n",SavedRoom);
  GameHeader.LightTime=Readnum();
  cprintf("LightTime: %d\n",GameHeader.LightTime);
  if(DarkFlag) BitFlags|=(1L<<15); // Backward compatibility
  for(ct=0;ct<=GameHeader.NumItems;ct++)
  {
    Items[ct].Loc=Readnum();
    cprintf("Items[%d].Loc: %d\n",ct,Items[ct].Loc);
  }
  FileInput_dtor(&f_in,LEAVE_MEMORY);
}

int PerformLine(int ct)
{
 int continuation=0,param[5],pptr=0,act[4],cc=0;
  while(cc<5)
  {
    int cv=AC_Cond[ct][cc],dv=cv/20;
    cv%=20;
    switch(cv)
    {
      case 0: param[pptr++]=dv;break; // Parameter
      case 1: if(Items[dv].Loc!=CARRIED) return 0;break; // Item <arg> carried
      case 2: if(Items[dv].Loc!=MyLoc) return 0;break; // Item <arg> in room with player
      case 3: if(Items[dv].Loc!=CARRIED &&Items[dv].Loc!=MyLoc) return 0;break; // Item <arg> carried or in room with player
      case 4: if(MyLoc!=dv) return 0;break; // In room <arg>
      case 5: if(Items[dv].Loc==MyLoc) return 0;break; // Item <arg> not in room with player
      case 6: if(Items[dv].Loc==CARRIED) return 0;break; // Item <arg> not carried
      case 7: if(MyLoc==dv) return 0;break; // Not in room <arg>
      case 8: if((BitFlags&(1L<<dv))==0) return 0;break; // BitFlag <arg> is set
      case 9: if(BitFlags&(1L<<dv)) return 0;break; // BitFlag <arg> is cleared
      case 10: if(CountCarried()==0) return 0;break; // Something carried (arg unused)
      case 11: if(CountCarried()) return 0;break; // Nothing carried (arg unused)
      case 12: if(Items[dv].Loc==CARRIED||Items[dv].Loc==MyLoc) return 0;break; // Item <arg> not carried nor in room with player
      case 13: if(Items[dv].Loc==0) return 0;break; // Item <arg> is in game [not in room 0]
      case 14: if(Items[dv].Loc) return 0;break; // Item <arg> is not in game [in room 0]
      case 15: if(CurrentCounter>dv) return 0;break; // CurrentCounter <= <arg>
      case 16: if(CurrentCounter<=dv) return 0;break; // CurrentCounter >= <arg>
      case 17: if(Items[dv].Loc!=Items[dv].InitialLoc) return 0;break; // Object still in initial room
      case 18: if(Items[dv].Loc==Items[dv].InitialLoc) return 0;break; // Object not in initial room
      case 19: if(CurrentCounter!=dv) return 0;break; // CurrentCounter = <arg> (Only seen in Brian Howarth games)
    }
    cc++;
  }
  act[0]=AC_Act[ct][0]; // Actions
  act[2]=AC_Act[ct][1];
  act[1]=act[0]%150;
  act[3]=act[2]%150;
  act[0]/=150;
  act[2]/=150;
  cc=0;
  pptr=0;
  while(cc<4)
  {
    if(act[cc]>=1 && act[cc]<52) Wrap(Messages[act[cc]]);
    else if(act[cc]>101) Wrap(Messages[act[cc]-50]);
    else
      switch(act[cc])
      {
        case 0: break; // NOP
        case 52: // Get item <arg>. Checks if you can carry it first
          if(CountCarried()==GameHeader.MaxCarry)
          {
            puts("You are carrying too much.");
            break;
          }
          if(Items[param[pptr]].Loc==MyLoc) Redraw=1;
          Items[param[pptr++]].Loc= CARRIED;break;
        case 53: Redraw=1;Items[param[pptr++]].Loc=MyLoc;break; // Drops item <arg>
        case 54: Redraw=1;MyLoc=param[pptr++];break; // Moves to room <arg>
        case 55: if(Items[param[pptr]].Loc==MyLoc) Redraw=1; // Item <arg> is removed from
                           Items[param[pptr++]].Loc=0;break; // the game (put in room 0)
        case 56: BitFlags|=1L<<DARKBIT;break; // The darkness flag is set
        case 57: BitFlags&=~(1L<<DARKBIT);break; // The darkness flag is cleared
        case 58: BitFlags|=(1L<<param[pptr++]);break; // Bitflag <arg> is set
        case 59: if(Items[param[pptr]].Loc==MyLoc) Redraw=1; // The same as 55 (it seems -
                           Items[param[pptr++]].Loc=0;break; // I'm cautious about this)
        case 60: BitFlags&=~(1L<<param[pptr++]);break; // BitFlag <arg> is cleared
        case 61: vibrate(30);beep(3);puts("You are dead."); BitFlags&=~(1L<<DARKBIT);
                 MyLoc=GameHeader.NumRooms; // It seems to be what the code says!
                 Look();vibrate(0);break;
        case 62: // Item <arg> put in room <arg>
          {int i=param[pptr++]; // Bug fix for some systems - before it could get parameters wrong
           Items[i].Loc=param[pptr++];Redraw=1;break;}
        case 63:
         doneit: puts("GAME OVER!");x=2;beep(1);break;
        case 64: Look();break; // Describe room
        case 65: // Score
        {
          int ct=0,n=0;
          while(ct<=GameHeader.NumItems)
          {
            if(Items[ct].Loc==GameHeader.TreasureRoom && *Items[ct].Text=='*')  n++;
            ct++;
          }
          printf("You have stored %d treasures. Scale of 0 to 100, that rates %d.",
                                                   n,(n*100)/GameHeader.Treasures);
          if(n==GameHeader.Treasures)
          {
            MSequence_play_background(&mus);
            vibrate(70);
            Wrap("Congradulations! Well done, adventurer."); // You WIN
            vibrate(0);
            goto doneit;
          }
          break;
        }
        case 66: // Inventory
        {
          int ct=0,f=0;
          strcpy(tmp,"You are carrying: ");
          while(ct<=GameHeader.NumItems)
          {
            if(Items[ct].Loc==CARRIED)
            {
              if(f==1) strcat(tmp,", ");
              f=1;
              strcat(tmp,Items[ct].Text);
            }
            ct++;
          }
          if(f==0) strcat(tmp,"nothing");
          strcat(tmp,".");
          Wrap(tmp);
          break;
        }
        case 67: BitFlags|=(1L<<0);break;  // Shift 0 ???  // BitFlag 0 is set
        case 68: BitFlags&=~(1L<<0);break; // Shift 0 ???  // BitFlag 0 is cleared
        case 69: GameHeader.LightTime=LightRefill;if(Items[LIGHT_SOURCE].Loc==MyLoc) Redraw=1;
                 Items[LIGHT_SOURCE].Loc=CARRIED;BitFlags&=~(1L<<LIGHTOUTBIT);break;
        case 70: cls();break; // Clear screen
        case 71: SaveGame();Redraw=1;break; // Save game
        case 72: // Swap item <arg1> and item <arg2> locations
        {
          int i1=param[pptr++],i2=param[pptr++],t=Items[i1].Loc;
          if(t==MyLoc||Items[i2].Loc==MyLoc) Redraw=1;
          Items[i1].Loc=Items[i2].Loc;
          Items[i2].Loc=t;
          break;
        }
        case 73: continuation=1;break; // Continue with next line (next line starts verb 0 noun 0)
        case 74: if(Items[param[pptr]].Loc==MyLoc) Redraw=1; // Take item <arg> - no check is
                 Items[param[pptr++]].Loc= CARRIED;break;    // done too see if it can be carried.
        case 75: // Put item <arg1> with item <arg2>
        {

          int i1=param[pptr++],i2=param[pptr++];
          if(Items[i1].Loc==MyLoc) Redraw=1;
          Items[i1].Loc=Items[i2].Loc;
          if(Items[i2].Loc==MyLoc) Redraw=1;
          break;
        }
        case 76: Look();break; // Look (same as 64 ?? - check)
        case 77: if(CurrentCounter>=0) CurrentCounter--;break; // Decrement current counter. Will not go below 0
        case 78: printf("%d",CurrentCounter);break; // Print Counter (Moves before oxygen runs out etc)
        case 79: CurrentCounter=param[pptr++];break; // Set current counter value
        case 80: {int t=MyLoc=SavedRoom;SavedRoom=t;Redraw=1;break;} // Swap location with current location-swap flag
        case 81: // Select a counter. Current counter is swapped with backup counter <arg>
        {
         // This is somewhat guessed. Claymorgue always seems to do select counter n, thing, select counter n,
         // but uses one value that always seems to exist. Trying a few options I found this gave sane results on ageing
          int t=param[pptr++],c1=CurrentCounter;
          CurrentCounter=Counters[t];
          Counters[t]=c1;
          break;
        }
        case 82: CurrentCounter+=param[pptr++];break; // Add <arg> to current counter
        case 83: CurrentCounter-=param[pptr++]; // Subtract <arg> from current counter
          // Note: This seems to be needed. I don't know if there is a maximum value to limit to
          if(CurrentCounter< -1) CurrentCounter= -1;break;
        case 84: puts(NounText);break; // Echo noun player typed without CR
        case 85: puts(NounText);puts("");break; // Echo the noun the player typed
        case 86: puts("");break; // CR
        case 87: // Swap current location value with backup location-swap value <arg>
        {        // Changed this to swap location<->roomflag[x] not roomflag 0 and x
          int p=param[pptr++],sr=MyLoc;MyLoc=RoomSaved[p];RoomSaved[p]=sr;Redraw=1;break;
        }
        case 88: sleep(2000);break; // Wait 2 sec
        case 89: pptr++;break;
          // SAGA draw picture n Spectrum Seas of Blood - start combat ?
          // Poking this into older spectrum games causes a crash
        default:
          cprintf("Unknown action %d [Param begins %d %d]\n",act[cc],param[pptr],param[pptr+1]);
          break;
      }
      cc++;
  }
 return continuation+1;
}

int PerformActions(int vb,int no)
{
 static int disable_sysfunc=0; // Recursion lock
 int d=BitFlags&(1L<<DARKBIT),ct=0,fl,doagain=0;
  if(vb==1 && no == -1)
  {
    puts("Give me a direction too.");
    return 0;
  }
  if(vb==1 && no>=1 && no<=6)
  {
    int nl;
    if(Items[LIGHT_SOURCE].Loc==MyLoc||Items[LIGHT_SOURCE].Loc==CARRIED) d=0;
    if(d) puts("Dangerous to move in the dark!");
    nl=Rooms[MyLoc].Exits[no-1];
    if(nl!=0)
    {
      MyLoc=nl;
      puts("O.K.");
      Look();
      return 0;
    }
    if(d)
    {
      puts("You fell down and broke your neck.");
      beep(0);
      x=3;
    }
    puts("You can't go in that direction.");
    return 0;
   }
   fl= -1;
   while(ct<=GameHeader.NumActions)
   {
     int nv,vv=AC_Vocab[ct]; // Think this is now right. If a line we run has an action73
     if(vb!=0 && (doagain&&vv!=0)) break;    // run all following lines with vocab of 0,0
     if(vb!=0 && !doagain && fl== 0) break;  // Added minor cockup fix 1.11
     nv=vv%150;
     vv/=150;
     if((vv==vb)||(doagain&&AC_Vocab[ct]==0))
     {
       if((vv==0&&RandomPercent(nv))||doagain||(vv!=0&&(nv==no||nv==0)))
       {
         int f2;
         if(fl== -1) fl= -2;
         if((f2=PerformLine(ct))>0)
         {
           fl=0;
           if(f2==2) doagain=1;
           if(vb!=0&&doagain==0) return 0;
         }
       }
     }
     ct++;
     if(AC_Vocab[ct]!=0) doagain=0;
  }
  if(fl!=0&&disable_sysfunc==0)
  {
    int i;
    if(Items[LIGHT_SOURCE].Loc==MyLoc||Items[LIGHT_SOURCE].Loc==CARRIED) d=0;
    if(vb==10||vb==18) // GET and
    {
      if(vb==10) // Yes they really _are_ hardcoded values
      {
        if(strcmp(NounText,"ALL")==0)
        {
          int ct=0,f=0;          
          if(d)
          {
            puts("It is dark.");
            return 0;
          }
          while(ct<=GameHeader.NumItems)
          {
            if(Items[ct].Loc==MyLoc && Items[ct].AutoGet!=NULL && Items[ct].AutoGet[0]!='*')
            {
              no=WhichWord(Items[ct].AutoGet,Nouns);
              disable_sysfunc=1;     // Don't recurse into auto get !
              PerformActions(vb,no); // Recursively check each items table code
              disable_sysfunc=0;
              if(CountCarried()==GameHeader.MaxCarry)
              {
                puts("You are carrying too much.");
                return 0;
              }
              Items[ct].Loc= CARRIED;
              Redraw=1;
              Wrap(Items[ct].Text);
              puts("O.K.");
              f=1;
            }
            ct++;
          }
          if(f==0) puts("Nothing taken.");
          return 0;
        }
        if(no==-1)
        {
          puts("What ?");
          return 0;
        }
        if(CountCarried()==GameHeader.MaxCarry)
        {
          puts("You are carrying too much.");
          return 0;
        }
        i=MatchUpItem(NounText,MyLoc);
        if(i==-1)
        {
          puts("It's beyond your power to do that.");
          return 0;
        }
        Items[i].Loc=CARRIED; // Get single object
        cprintf("Item @ %d\n",Items[i].Loc); // Debug
        puts("O.K.");
        Redraw=1;
        return 0;
      }
      if(vb==18) // DROP
      {
        if(strcmp(NounText,"ALL")==0)
        {
          int ct=0,f=0;
          while(ct<=GameHeader.NumItems)
          {
            if(Items[ct].Loc==CARRIED&&Items[ct].AutoGet&&Items[ct].AutoGet[0]!='*')
            {
              no=WhichWord(Items[ct].AutoGet,Nouns);
              disable_sysfunc=1;
              PerformActions(vb,no);
              disable_sysfunc=0;
              Items[ct].Loc=MyLoc;
              Wrap(Items[ct].Text);
              puts("O.K.");
              Redraw=1;
              f=1;
            }
            ct++;
          }
          if(f==0) puts("Nothing dropped.");
          return 0;
        }
        if(no==-1)
        {
          puts("What ?");
          return 0;
        }
        i=MatchUpItem(NounText,CARRIED);
        if(i==-1)
        {
          puts("It's beyond your power to do that.");
          return 0;
        }
        Items[i].Loc=MyLoc;
        puts("O.K.");
        Redraw=1;
        return 0;
      }
    }
  }
 return fl;
}

int getc(void)
{
 int g=FileInput_read_byte(&f_in);
  if(FileInput_is_eof(&f_in)) {g = -1;beep(1);}  // Cybiko EOF
 return g;
}

void strupr(char *s)
{
  while(*s)
  {
    if(*s>='a' && *s<='z') *s=*s-32;
    s++;
  }
}

void cls(void)
{
  cWinApp_clear_screen(); // Clear screen
  Graphics_set_font(main_module.m_gfx, mini_normal_font); // Set mini font
  puts("");
}

char isupper(char c)
{
 return c>='A' && c<='Z';
}
char tolower(char c)
{
 return isupper(c) ? c+32 : c;
}
char isspace(char c)
{
 return c==' ' || c=='\t' || c=='\n';
}

