#charset "us-ascii"
/* 
 *  Copyright (c) 2006 by Kevin Forchione. All rights reserved.
 *   
 *  This file is part of the TADS 3 Relation Library Extension
 *
 *  set.t
 *
 *--------------------------------------------------------------------
 *  THE TADS 3 SET FILE
 *--------------------------------------------------------------------
 */

#include "relation.h"
#include "bignum.h"

/*
 *  A SetObject can be thought of as any collection of distinct 
 *  things considered as a whole. 
 */
class SetObject: object
{
    name        = nil
    coll        = nil
    elmKind     = nil
    
    /*
     *  Returns the arity of the SetObject. By default this is 0.
     */
    arity       = 0

    /*
     *  Returns the size of the SetObject
     */
    cardinality() { return coll.length(); }

    /*
     *  Returns a list representation of the elements of this set.
     */
    toList() { return coll.toList(); }

    /*
     *  Returns true if the element is a member 
     *  of this set; otherwise returns nil.
     */
    hasElement(element)
    {
        return coll.indexOf(element) != nil;
    }

    /* 
     *  Two sets are said to be identical if they have the
     *  same meaning and they are equal.
     */
    isIdentical(other) { return (name == other.name) && equals(other); }

    /*
     *  Returns true if this set equals other. By definition
     *  two sets are said to be equal if they are subsets of
     *  each other.
     */
    equals(other)
    {
        return isSubsetOf(other) && other.isSubsetOf(self);
    }

    isSubsetOf(other)
    {
        foreach (local element in coll)
            if (!other.hasElement(element))
                return nil;

        return true;
    }

    isSupersetOf(other) { return other.isSubsetOf(self); }


    /*
     *  Removes elements from this set that match
     *  value. 
     *
     *  Returns a set of all elements removed.
     */
    remove([values])
    {
        local vec, remSet; 

        vec = new Vector(coll.length());

        foreach (local elm in coll)
        {
            if (values.length() == 0 || values.indexOf(elm))
            {
                vec.append(elm);
                coll.removeElement(elm);
            }
        }

        /*
         *  Recalculate the relation set arity
         */
        setArity();

        /*
         *  Create a relation set of removed relations
         */
        remSet      = createCopy();
        remSet.coll = new Vector(vec.length());
        vec.forEach(new function(rel)
        {
            remSet.addElement(rel);
        });

        remSet.setArity();

        return remSet;
    }

    /*
     *  Creates a copy of this SetObject.
     */
    createCopy()
    {
        return createInstance(name, toList()...);
    }

    /*
     *  Returns a subset of this set as indicated by the
     *  func parameter, which is passed each element of the
     *  set. If func returns true then this element will be
     *  included in the subset; otherwise it will not be.
     */
    subset(func)
    {
        local subSet, ret;

        ret = coll.subset(func);

        subSet  = createInstance('S(' + name + ')', ret.toList()...);
        subSet.setArity();

        return subSet;
    }

    search([values])
    {
        local srchSet, ret, str;

        ret     = searchIter(values...);
        str     = buildName(values...);

        srchSet = createInstance(str, ret...);
        srchSet.setArity();

        return srchSet;
    }

    searchIter([values])
    {
        local vec;

        if (values.length())
            vec = new Vector(values.length());
        else vec = new Vector(cardinality());
        
        foreach (local elm in coll)
        {
            if (values.length() == 0 || values.indexOf(elm))
            {
                vec.appendUnique(elm);
            }
        }

        return vec.toList();
    }

    /*
     *  A method for Complements
     */
    complement([sets])
    {
        local other, name, set;

        if (sets.length() == 0)
            return self;

        other   = sets.car();

        name    = '(' + other.name + ' - ' + self.name + ')';
        set     = createInstance(name, other.coll.toList()...);
        coll.forEach({x: set.removeElement(x)});
        return set.complement(sets.cdr()...);
    }

    /*
     *  A method for Conjunctions
     */
    intersect([sets])
    {
        local other, name, set;

        if (sets.length() == 0)
            return self;

        other   = sets.car();

        name    = '(' + self.name + ' * ' + other.name + ')';
        set     = createInstance(name, 
            coll.toList().intersect(other.coll.toList())...);
        return set.intersect(sets.cdr()...);
    }

    /*
     *  A method for Disjunctions
     */
    union([sets])
    {
        local other, name, set;

        if (sets.length() == 0)
            return self;

        other   = sets.car();

        name    = '(' + self.name + ' + ' + other.name + ')';
        set     = createInstance(name, coll.toList()...);
        other.coll.forEach({x: set.addElement(x)});
        return set.union(sets.cdr()...);
    }

    /*
     *  A method for Exclusive Disjunctions
     */
    xor([sets])
    {
        local other, name, set;

        if (sets.length() == 0)
            return self;

        other       = sets.car();

        name        = '(' + self.name + ' ^ ' + other.name + ')';
        set         = (self.intersect(other).complement(self)).union(other.intersect(self).complement(other));
        set.name    = name;
        return set.xor(sets.cdr()...);
    }

    /*
     *  A method for creating the set's powerset
     */
    powerset()
    {
        local len, vec, pSize, string, subset;
        
        pSize   = (new BigNumber(2).raiseToPower(cardinality())).roundToDecimal(0);
        pSize   = toInteger(pSize);
        vec     = new Vector(pSize);
        
        vec.append(createInstance(name));

        for (local i = 1; i < pSize; ++i)
        {
            subset  = createInstance(name);
            string  = toString(i, 2);
            len     = string.length();
            for (local j = 1; j <= len; ++j)
            {
                if (string.substr(j,1) == '1')
                    subset.addElement(coll[len + 1 - j]);              
            }
            subset.setArity();
            vec.append(subset);
        }

        return new SetObject('P(' + name + ')', vec.toList()...);
    }
    
    setArity() { arity = 0; }
    
    addElement(element)
    {
        if (elmKind && !element.ofKind(elmKind))
            throw new SetException('element ' + element.name + ' not valid 
                for this set class. ');

        coll.appendUnique(element);
    }

    removeElement(element)
    {
        coll.removeElement(element);
    }

    /*
     *  This method initializes the internal details of 
     *  a relation set. It is directly called for statically-
     *  defined relation sets.
     */
    initInstance(name)
    {
        self.name   = name;
        arity       = 0;
        coll        = new Vector(100);
    }

    construct(name, [elements])
    {
        initInstance(name);
        elements.forEach({element: addElement(element)});
    }
}

class SetException: RuntimeError
{
    construct(msg)
    {
        exceptionMessage    = msg;
        inherited(0);
    }
}