#charset "us-ascii"
/* 
 *  Copyright (c) 2006 by Kevin Forchione. All rights reserved.
 *   
 *  This file is part of the TADS 3 Relation Library Extension
 *
 *  relation_set.t
 *
 *--------------------------------------------------------------------
 *  THE TADS 3 RELATION SET FILE
 *--------------------------------------------------------------------
 */

#include "relation.h"

/*
 *  Relation sets are abstract datatypes designed to store instances 
 *  of relations. The RelationSetObject class defines methods for adding, 
 *  removing and searching for relations encapsulated by a specific 
 *  relation set instance, as well as methods for creating copies 
 *  and instances of itself.  
 */
class RelationSetObject: RelationBase, SetObject
{
    /*
     *  A relation set is a collection of relation objects
     */
    elmKind     = RelationObject

    toMatrix()
    {
        return searchIter([]);
    }

    /*
     *  Returns a list representation of the relatins of this set.
     */
    toList()
    {
        local vec;

        vec = new Vector(coll.length());

        foreach (local rel in coll)
            vec.append(rel.parms);

        return vec.toList();
    }

    /*
     *  Removes relations from this relation set that match
     *  mask. If flg is SearchFirst then only the first
     *  match will be removed; otherwise all matches 
     *  will be removed.
     *
     *  Returns a relation set of all relations removed.
     */
    remove(mask, [flg])
    {
        local vec, remSet; 

        flg = flg.car();

        vec = new Vector(coll.length());

        __RemoveLoop:
        foreach (local rel in coll)
        {
            if (rel.match(mask))
            {
                vec.append(rel);
                coll.removeElement(rel);
                switch(flg)
                {
                    case SearchFirst:
                        break __RemoveLoop;

                    case SearchAll:
                        break;

                    default:
                        break;
                }
            }
        }

        /*
         *  Recalculate the relation set arity
         */
        setArity();

        /*
         *  Create a relation set of removed relations
         */
        remSet  = createCopy();
        remSet.coll = new Vector(vec.length());
        vec.forEach(new function(rel)
        {
            remSet.addElement(rel);
        });
        remSet.setArity();

        return remSet;
    }

    /*
     *  Creates a copy of this RelationSetObject.
     */
    createCopy()
    {
        return createInstance(name, toMatrix()...);
    }

    /*
     *  Returns a subset of this set as indicated by the
     *  func parameter, which is passed each element of the
     *  set. If func returns true then this element will be
     *  included in the subset; otherwise it will not be.
     */
    subset(func)
    {
        local subSet, ret;

        ret = coll.subset(func);

        subSet  = createInstance('S(' + name + ')');
        ret.forEach({rel: subSet.addElement(rel)});
        subSet.setArity();

        return subSet;
    }

    search(mask, [flg])
    {
        local ret, str;

        ret     = searchIter(mask, flg...);
        str     = buildName(mask);

        return createInstance(str, ret...);
    }

    searchIter(mask, [flg])
    {
        local vec, parms;

        flg = flg.car();
        
        if (mask.length())
            vec = new Vector(mask.length()).fillValue([], 1, mask.length());
        else vec = new Vector(arity).fillValue([], 1, arity);
        
        __SearchLoop:
        foreach (local rel in coll)
        {
            if (rel.match(mask))
            {
                if (mask.length() == 0)
                    parms = new Vector(arity, rel.parms).fillValue(null, 
                        rel.parms.length() + 1, 
                        arity - rel.parms.length()).toList();
                else parms = rel.parms;
                accum(vec, parms);
                switch(flg)
                {
                    case SearchFirst:
                        break __SearchLoop;

                    case SearchAll:
                        break;

                    default:
                        break;
                }
            }
        }

        return vec.toList();
    }

    /*
     *  Adds the element of parms to the corresponding
     *  index location of the vector.
     */
    accum(vec, parms)
    {
        for (local i = 1; i <= parms.length(); ++i)
        {
            vec[i] += parms[i];
        }
    }

    /* 
     *  Returns true if the Relation set has the element
     *  represented by the parms argument; otherwise 
     *  the method returns nil. Note that unlike searchIter()
     *  this method looks for an exact match of parms. 
     */
    hasElement([parms])
    {
        foreach (local rel in coll)
        {
            if (rel.parms == parms)
                return true;
        }

        return nil;
    }

    /*
     *  Returns true if this set is a subset of other; otherwise
     *  it returns nil. This set is considered a subset of other
     *  if each of its relation parms matches a relation parm of 
     *  other. Order and multiplicity of elements is irrelevant
     *  in determination of subsets.
     */
    isSubsetOf(other)
    {
        local list1, list2;

        list1   = toList();
        list2   = other.toList();
        
        foreach (local elm in list1)
        {
            if (list2.indexOf(elm) == nil)
                return nil;
        }

        return true;
    }

    /*
     *  A method for Complements 
     */
    complement([sets])
    {
        local other, name, set, vec, mask;

        if (sets.length() == 0)
            return self;

        other   = sets.car();

        name    = '(' + other.name + ' - ' + self.name + ')';
        vec     = new Vector(1).copyFrom(other.toMatrix(), 1, 1, 1);
        set     = createInstance(name, vec.toList()...);
        foreach (local elm in coll)
        {
            mask = elm.parms;
            if (mask.length() < set.arity)
                mask = new Vector(set.arity, mask).fillValue(null, 
                    mask.length() + 1, set.arity - mask.length()).toList();

            set.remove(mask);
        }
        
        return set.complement(sets.cdr()...);
    }

    /*
     *  A method for Disjunctions
     */
    union([sets])
    {
        local other, name, set, matrix, vec;

        if (sets.length() == 0)
            return self;

        other   = sets.car();

        if (!other.ofKind(RelationSetObject))
            return other.union(self, sets.cdr()...);

        name    = '(' + self.name + ' + ' + other.name + ')';
        matrix  = new Vector(10);
        matrix.append(toMatrix());
        matrix.append(other.toMatrix());
        matrix  = normalize(matrix.toList()...);
        vec = new Vector(1).copyFrom(matrix, 1, 1, 1);
        for (local i = 2; i <= matrix.length(); ++i)
        {
            for (local j = 1; j <= matrix[i].length(); ++j)
            {
                vec[1][j] += matrix[i][j];
            }
        }
        
        if (vec.length())
            vec = vec[1];
        else vec = [];
        set     = createInstance(name, vec...);
        return set.union(sets.cdr()...);
    }

    /*
     *  A method for Conjunctions
     */
    intersect([sets])
    {
        local other, name, set, matrix, vec;

        if (sets.length() == 0)
            return self;

        other   = sets.car();

        if (!other.ofKind(RelationSetObject))
            return other.intersect(self, sets.cdr()...);

        name    = '(' + self.name + ' * ' + other.name + ')';
        matrix  = new Vector(10);
        matrix.append(toMatrix());
        matrix.append(other.toMatrix());
        matrix  = normalize(matrix.toList()...);
        vec = new Vector(1).copyFrom(matrix, 1, 1, 1);
        for (local i = 2; i <= matrix.length(); ++i)
        {
            for (local j = 1; j <= matrix[i].length(); ++j)
            {
                vec[1][j] = vec[1][j].intersect(matrix[i][j]);
            }
        }
        
        if (vec.length())
            vec = vec[1];
        else vec = [];
        set     = createInstance(name, vec...);
        return set.intersect(sets.cdr()...);
    }

    /*
     *  Normalizes the matrices so that they are all
     *  of the same dimension for use in Or() and And() functions.
     */
    normalize([matrix])
    {
        local maxLen, val;
        
        maxLen = 0;

        matrix.forEach(new function(r)
        {
            if (r.length() > maxLen)
                maxLen = r.length();
        });
        for (local i = 1; i <= matrix.length(); ++i)
        {
            if (matrix[i].length() < maxLen)
            {
                if (matrix[i].length() > 0)
                    val = new Vector(matrix[i][1].length()).fillValue(null, 
                    1, matrix[i][1].length()).toList();
                matrix[i] = new Vector(maxLen, matrix[i]).fillValue(val,
                    matrix[i].length() + 1, maxLen - matrix[i].length()).toList();
            }
        }

        return matrix;
    }

    /*
     *  Computes the arity of the relation set as
     *  the maximum of arity of its relations.
     */
    setArity()
    {
        arity = 0;
        coll.forEach(new function(rel)
        {
            if (rel.arity > arity)
                arity = rel.arity();
        });
    }

    /*
     *  Loads the relation set from the matrix provided.
     */
    loadSet([matrix])
    {
        local maxLen, parms;

        maxLen = 0;
        matrix.forEach(new function(p)
        {
            if (p.length() > maxLen)
                maxLen = p.length();
        });
        for (local i = 1; i <= matrix.length(); ++i)
        {
            if (matrix[i].length() < maxLen)
            {
                matrix[i] = new Vector(maxLen, matrix[i]).fillValue(null, 
                    matrix[i].length() + 1, maxLen - matrix[i]).toList();
            }
        }

        for (local j = 1; j <= maxLen; ++j)
        {
            parms = [];
            for (local i = 1; i <= matrix.length(); ++i)
            {
                parms += matrix[i][j];
            }
            new RelationObject(self, parms...);
        }
    }


    /*
     *  This method initializes the internal details of 
     *  a relation set. It is directly called for statically-
     *  defined relation sets.
     */
    initInstance(name)
    {
        self.name   = name;
        arity       = 0;
        coll        = new Vector(100);
    }

    /*
     *  This method is called when using the MakeRelSet() macro
     *  for creating dynamically-defined relation sets.
     */
    construct(name, [matrix])
    {
        initInstance(name);
        if (matrix.length())
            loadSet(matrix...);
    }
}