#charset "us-ascii"

/*
 *  Copyright (c) 2001-2004 by Kevin Forchione. All rights reserved.
 *  Based on code provided by Michael J. Roberts.
 *
 *  This file is part of the PROTEUS, the TADS 3 Utility Classes Package
 *
 *  Symbol.t
 *
 *  Provides a basic implementation for mapping from symbol to string
 *  and string to symbol using the TADS 3 Symbol Table generated at pre-
 *  compile time.
 */

/* include the Proteus header */
#include "proteus.h"

/* ------------------------------------------------------------------------ */
/*
 *  Library Pre-Initializer.  This object performs the following
 *  initialization operations immediately after compilation is completed:
 *
 *  Stash a reference to the global symbol table and build a reverse 
 *  symbol table.
 *
 *  This object is named so that other libraries and/or user code can
 *  create initialization order dependencies upon it.  
 */
symbolTablePreinit: PreinitObject
{
    /*
     *  Map a string value to its symbol value. 
     */
    getSymbol(val)
    {
        if (dataType(val) == TypeSString)
        {
            /*
             *  Modifications to objects may be preceded by spaces.
             *  We might need to pad the beginning of the string.
             */
            if (toInteger(val) > 0)
            {
                local newStr = '';

                while (val.length() % 4 != 0)
                {
                    newStr = ' ' + val;
                    val = newStr;
                }
            }
            return symtab[val];
        }
        else 
        {
            if (isSymKeyPresent(val))
                return val;
            else return nil;   
        }
    }

    getSString(val)
    {
        local str;

        if (dataType(val) == TypeSString)
        {
            if (isStrKeyPresent(val))
                return val;
            else return '';
        }
        else
        {
            str = reverseSymtab[val];

            return (str ? str : '');
        }
    }

    isKeyPresent(val)
    {
        if (dataType(val) == TypeSString)
            return isStrKeyPresent(val);
        else
            return isSymKeyPresent(val);
    }

    isStrKeyPresent(val)
    {
        return symtab.isKeyPresent(val);
    }

    isSymKeyPresent(val)
    {
        return reverseSymtab.isKeyPresent(val);
    }

    execute()
    {
        /* store this object for use by our global symbols macro */
        gSymbols = self;

        /*  store a reference to the global symbol table */
        symtab = t3GetGlobalSymbols();
        
        /*
         *  building a reverse mapping - in other words, building a
         *  LookupTable that uses symbol values as the keys and symbol
         *  strings as the associated values.
         */
        reverseSymtab = new LookupTable();
        symtab.forEachAssoc({str, sym: reverseSymtab[sym] = str});
    }

    /* string-to-symbol lookup table */
    symtab			= nil

    /* symbol-to-string lookup table */
    reverseSymtab	= nil
    
    /* execute this preinit after the symbol table has been built */
    execAfterMe = [adv3LibPreinit]
}

modify Object
{
	/*
	 *	Determine if the object has been dynamically-created.
	 */
	isDynamOrAnon()
	{
		return (!gSymbols.isSymKeyPresent(self));
	}

    /*
     *  Determine if the object is an unnamed internal. This
     *  occurs when an object has been modified by the 'modify'
     *  keyword.
     */
    isUnnamedInternal()
    {
        if (isDynamOrAnon())
            return nil;

        if (toInteger(gSymbols.getSString(self)) > 0)
            return true;

        return nil;
    }
}