#charset "us-ascii"

/*
 *   MegaTADS. Additions and modifications to the actor.t file.
 *
 *   Copyright 2007, Krister Fundin.
 */

#include <adv3.h>

#include "mega.h"

/* ---------------------------------------------------------------------- */
/*
 *   The Actor class.
 */
modify Actor

#ifdef MEGA_UNACTOR_ON

    /* put all known Unactors in scope for this actor */
    getExtraScopeItems(actor)
    {
        /* get any inherited scope items */
        local lst = inherited(actor);

        /*
         *   if we are not the actor for which the scope is being
         *   determined, then skip the Unactors
         */
        if (actor != self)
            return lst;

        /* go through all Unactors in the story */
        forEachInstance(Unactor, new function(cur)
        {
            /*
             *   for an Unactor to be in scope, we must either have seen its
             *   target actor or otherwise know about it
             */
            if (knowsAbout(cur.targetObj) || hasSeen(cur.targetObj))
                lst += cur;
        });

        /* return the updated list */
        return lst;
    }

#endif

#ifdef MEGA_INTERJECT_ON

    /*
     *   action handling for saying an interjection explicitly to this
     *   actor, I.E. by typing SAY HELLO TO BOB
     */
    dobjFor(SayInterjectionTo)
    {
        preCond = [canTalkToObj]
        verify()
        {
            /* we can't talk to ourselves */
            verifyNotSelf(&cannotTalkToSelf);
        }
        action()
        {
            /*
             *   call the special 'say' property of the interjection on the
             *   saying actor with ourselves as the explicit target
             */
            gActor.(gAction.itrjc_.sayProp)(self);
        }
    }

#endif

#ifdef MEGA_SAYTOPIC_ON

    /*
     *   Say a literal phrase to someone. We'll try to match the phrase
     *   against an interjection, and if that fails, we'll look for an
     *   applicable SayTopic.
     */
    sayLiteral(txt, actor)
    {
        /* see if the literal matches an interjection */
        try
        {
            /* tokenize and parse */
            local toks = cmdTokenizer.tokenize(txt);
            local match = interjection.parseTokens(toks, cmdDict);

            /* see if there we're any matching interjections */
            if (match != [])
            {
                /*
                 *   There were. Call the special 'say' property of the
                 *   interjection on the speaker (I.E. ourselves), then
                 *   consider the literal handled.
                 */
                self.(match[1].sayProp)(actor);
                return;
            }
        }
        catch (TokErrorNoMatch tokExc)
        {
            /* tokenization failed, so we can't have an interjection */
        }

        /*
         *   Say the literal text to the other actor, possible looking for a
         *   a default one. Note that the topic we pass along is simply the
         *   literal text itself.
         */
        sayToActor(actor, txt, sayLiteralConvType);
    }

    defaultSayLiteralResponse(actor)
    {
        defaultReport(&noResponseFromMsg, self);
    }

    /* action handling for saying a literal explicitly to this actor */
    dobjFor(SayLiteralTo)
    {
        preCond = [canTalkToObj]
        verify()
        {
            /* we can't talk to ourselves */
            verifyNotSelf(&cannotTalkToSelf);
        }
        action()
        {
            /* let the actor handle the action */
            gActor.sayLiteral(gAction.getLiteral(), self);
        }
    }

#endif

;

/* ---------------------------------------------------------------------- */
/*
 *   Add conversational handling for saying literals.
 */

#ifdef MEGA_SAYTOPIC_ON

/* add a property in which to store SayTopics for actors */
modify ActorTopicDatabase
    sayTopics = nil
;

/*
 *   The SayTopic class. Most other topics match either just objects or both
 *   objects and text, but this only only matches text.
 */
class SayTopic: TopicEntry
    /* we go in the sayTopics property */
    includeInList = [&sayTopics]

    /* these properties work the same as for a TopicMatchTopic */
    matchPattern = nil
    matchExactCase = nil

    /* match a topic */
    matchTopic(fromActor, topic)
    {
        /*
         *   if we don't the pattern to be case sensitive, then convert the
         *   topic text to lower-case before trying to match it
         */
        if (!matchExactCase)
            topic = topic.toLower();

        /*
         *   return our match score if the topic text matches the pattern;
         *   otherwise return nil
         */
        return (rexMatch(matchPattern, topic) != nil ? matchScore : nil);
    }

    /*
     *   a match is always possible for this topic, since the player is free
     *   to type anything
     */
    isMatchPossible(actor, scopeList) { return true; }
;

/*
 *   A default SayTopic.
 */
class DefaultSayTopic: SayTopic
    /* we match anything */
    matchTopic(fromActor, topic) { return matchScore; }

    /*
     *   we're a catch-all topic, so give us the low match score, the only
     *   one lower being that of a DefaultAnyTopic
     */
    matchScore = 1
;

/*
 *   Modify the DefaultAnyTopic so that it also matches literals that
 *   weren't caught by a SayTopic or DefaultSayTopic.
 */
modify DefaultAnyTopic
    includeInList = static (inherited + &sayTopics)
;

/*
 *   A conversation type for saying literals. This is pretty much boiler-
 *   plate code.
 */
sayLiteralConvType: ConvType
    unknownMsg = &sayLiteralMsg
    topicListProp = &sayTopics
    defaultResponseProp = &defaultSayLiteralResponse
    defaultResponse(db, other, topic)
        { db.defaultSayLiteralResponse(other); }
;

#endif

/* ---------------------------------------------------------------------- */
/*
 *   The Unactor class.
 */

#ifdef MEGA_UNACTOR_ON

class Unactor: NameAsOther, Unthing
    /* our target actor */
    targetObj = nil

    /* customize our notHereMsg */
    notHereMsg()
    {
        local info, srcLoc;

        /* if there's no actor, then use the plain message */
        if (gActor == nil)
            return &unactorNotHereMsg;

        /* get the follow information for our target actor */
        info = gActor.getFollowInfo(targetObj);

        /*
         *   If there were no follow information, then we have no way of
         *   knowing where the missing actor is. Just use the plain message.
         */
        if (info == nil)
            return &unactorNotHereMsg;

        /*
         *   The following code is mostly the same as the code for
         *   following. If if would be possible to follow our target actor,
         *   then we use unactorHasLeftMsg. If we have seen the target actor
         *   in a different location, then we use unactorLastSeenMsg. If the
         *   the target actor has left but we don't know where, then we use
         *   the plain unactorNotHereMsg again.
         */
        srcLoc = info.sourceLocation.effectiveFollowLocation;

        if (info.connector == nil)
        {
            if (gActor.canSee(srcLoc))
                return &unactorNotHereMsg;
            else
                return lastSeenMsg(srcLoc);
        }

        if (gActor.effectiveFollowLocation != srcLoc)
        {
            if (!gActor.canSee(srcLoc))
                return lastSeenMsg(srcLoc);
        }

        return &unactorHasLeftMsg;
    }

    /*
     *   Return a message saying where our target actor was last seen. This
     *   message takes two parameters, so we can't just return the message
     *   property from notHereMsg. We have to generate it in place and
     *   return the resulting string instead.
     */
    lastSeenMsg(srcLoc)
    {
        return MessageResult.resolveMessageText([self], &unactorLastSeenMsg,
                                                [self, srcLoc]);
    }

    /*
     *   Use our "not here" message for ALL actions. Note that we use
     *   inacessible instead of illogical. This is because the Unactor won't
     *   be visible to or touchable by the actor. If we used illogical, then
     *   the pre-conditions that use inacessible would go before us, so we
     *   have to use inacessible ourselves to trump this.
     */
    dobjFor(All)
    {
        verify() { inaccessible(notHereMsg, self); }
    }
    iobjFor(All)
    {
        verify() { inaccessible(notHereMsg, self); }
    }

    /*
     *   as far as the player is concerned, we are equivalent to our target
     *   actor
     */
    getIdentityObject()
    {
        return targetObj;
    }
;

#endif

