#charset "us-ascii"

/* Copyright (c) 2004 M.D. Dollahite.  All Rights Reserved. */
/*
 *   Library Extention: obstructing connectors
 *
 *   Allow sense connectors to occlude parts of the sense tree.
 *   Based on work by Steve Breslin.
 */
 
#include <adv3.h>

/*--------------------------------------------------------------------------*/
/*
 *   Obstructor: This is a mix-in class that can be used with either
 *   SenseConnector or OneWaySenseConnector to allow it to block
 *   some of the objects sensed through it.
 */
class Obstructor: object
    /*
     *   Do we obstruct the given object in the given sense?
     *   This returns true if the object is to be filtered, nil if not.
     */
    obstructObj(obj, sense)
    {
        /* by default, we don't obstruct anything */
        return nil;
    }
;

/*--------------------------------------------------------------------------*/
/*
 *   Modify some library and CSC objects to make them compatible with
 *   with Obstructor.
 */
modify libGlobal
    /*
     *   curObstructor is the current Obstructor. It is
     *   used to filter out any objects (and their containment children)
     *   from the connector's locations (outward from the POV).
     */
    curObstructor = nil
;

/* 
 *   We modify Thing to check for the current obstructor, if one
 *   currently exists, so we don't add to the sense path objects which
 *   are obstructed by the obstructor.
 *
 *   Thus we modify sensePathToLoc(), sensePathWithin(), and
 *   sensePathWithout(). Note that we don't have to modify
 *   sensePathToContents(), because that's only an intermediary step, and
 *   doesn't update sense path information itself.
 */
modify Thing

    sensePathToLoc(sense, trans, obs, fill)
    {
        if (location != nil)
        {
            /* 
             *   check if my location is being filtered by the current
             *   obstructor. if it is, we do nothing. otherwise,
             *   we proceed as usual.
             */
            if (!libGlobal.curObstructor
                || !libGlobal.curObstructor.
                    obstructObj(location, sense))
            {
                location.sensePathFromWithin(self, sense, trans, obs, fill);
            }
        }
    }

    sensePathFromWithin(fromChild, sense, trans, obs, fill)
    {
        /* 
         *   if I'm obstructed by the current obstructor, I don't
         *   add myself (or my contents) to the sense path.
         */
        if (libGlobal.curObstructor
            && libGlobal.curObstructor.obstructObj(self, sense))
            return;

        /* otherwise, proceed as usual. */
        inherited(fromChild, sense, trans, obs, fill);
    }

    sensePathFromWithout(fromParent, sense, trans, obs, fill)
    {
        /* 
         *   if I'm obstructed by the current obstructor, I don't
         *   add myself (or my contents) to the sense path.
         */
        if (libGlobal.curObstructor
            && libGlobal.curObstructor.obstructObj(self, sense))
            return;

        /* otherwise, proceed as usual. */
        inherited(fromParent, sense, trans, obs, fill);
    }
;

/*
 *   SenseConnector needs minor changes to sensePathFromWithout()
 *   to behave properly in the presence of Obstructors.
 */
modify SenseConnector
    /*
     *   Build a sense path from a container to me
     */
    sensePathFromWithout(fromParent, sense, trans, obs, fill)
    {
        /*
         *   if I'm obstructed by the current obstructor, I don't
         *   add myself (or my other containers (in my locationList)) to
         *   the sense path.
         */
        if (libGlobal.curObstructor
            && libGlobal.curObstructor.obstructObj(self, sense))
            return;

        /* 
         *   if there's better transparency along this path than along any
         *   previous path we've used to visit this item, take this path 
         */
        if (transparencyCompare(trans, tmpTrans_) > 0)
        {
            local transThru;
            
            /* remember the new path to this point */
            tmpTrans_ = trans;
            tmpObstructor_ = obs;

            /* we're coming to this object from outside */
            tmpPathIsIn_ = true;

            /* transmit to my contents */
            sensePathToContents(sense, trans, obs, fill);

            /*
             *   We must transmit this energy to each of our other
             *   parents, possibly reduced for traversing our connector.
             *   Calculate the new level after traversing our connector. 
             */
            transThru = transparencyAdd(trans, transSensingThru(sense));

            /* if we changed the transparency, we're the obstructor */
            if (transThru != trans)
                obs = self;

            /* 
             *   if there's anything left, transmit it to the other
             *   containers 
             */
            if (transThru != opaque)
            {
                /* remember the old obstructor. */
                local oldObstr = libGlobal.curObstructor;

                /* reset the current obstructor. */
                libGlobal.curObstructor = (ofKind(Obstructor) ? self : nil);

                /* transmit to each container except the source */
                foreach (local cur in locationList)
                {
                    /* if this isn't the sender, transmit to it */
                    if (cur != fromParent)
                        cur.sensePathFromWithin(self, sense,
                                                transThru, obs, fill);
                }

                /* restore the old obstructor */
                libGlobal.curObstructor = oldObstr;
            }
        }
    }
;

/*
 *   OneWaySenseConnector needs minor changes to sensePathFromWithout()
 *   to behave properly in the presence of Obstructors.
 */
modify OneWaySenseConnector
    /*
     *   Build a sense path from a container to me
     */
    sensePathFromWithout(fromParent, sense, trans, obs, fill)
    {
        /*
         *   if I'm obstructed by the current obstructor, I don't
         *   add myself (or my other containers (in my locationList)) to
         *   the sense path.
         */
        if (libGlobal.curObstructor
            && libGlobal.curObstructor.obstructObj(self, sense))
            return;

        /*
         *   if there's better transparency along this path than along any
         *   previous path we've used to visit this item, take this path
         */
        if(transparencyCompare(trans, tmpTrans_) > 0)
        {
            local transThru;
            
            /* remember the new path to this point */
            tmpTrans_ = trans;
            tmpObstructor_ = obs;

            /* we're coming to this object from outside */
            tmpPathIsIn_ = true;

            /* transmit to my contents */
            sensePathToContents(sense, trans, obs, fill);

            /*
             *   We must transmit this energy to each of our connectionList
             *   objects, possibly reduced for traversing our connector.
             *   Calculate the new level after traversing our connector.
             */
            transThru = transparencyAdd(trans, transSensingThru(sense));

            /* if we changed the transparency, we're the obstructor */
            if(transThru != trans)
                obs = self;

            /*
             *   if there's anything left, transmit it to our connectionList
             */
            if(transThru != opaque)
            {
                /* remember the old obstructor. */
                local oldObstr = libGlobal.curObstructor;

                /* reset the current obstructor. */
                libGlobal.curObstructor = (ofKind(Obstructor) ? self : nil);

                /* transmit to each connection */
                for(local clst = connectionList, local i = 1,
                    local len = clst.length() ; i <= len ; ++i)
                {
                    if(connectionList[i] != fromParent)
                        connectionList[i].sensePathFromWithin(self, sense, transThru, obs, fill);
                }

                /* restore the old obstructor */
                libGlobal.curObstructor = oldObstr;
            }
        }
    }
;

