#charset "us-ascii"

/*
 *   TADS 3 Action Report Combiner. Report tracing. This file, which
 *   requires reflect.t, provides a way of getting more information about
 *   printed reports during debugging.
 *
 *   Copyright 2006, Krister Fundin (fundin@yahoo.com)
 */

#ifdef __DEBUG

#include <adv3.h>
#include <reflect.h>

/* ---------------------------------------------------------------------- */
/*
 *   Modify combineReportsTransform. Since this is supposed to be the last
 *   transformation on the transcript, this should be an ideal place to
 *   carry out report tracing. Plus, we only have to modify code from our
 *   own extension, which should be relatively safe.
 */
modify combineReportsTransform
    /* apply this transformation */
    applyTransform(trans, vec)
    {
        /* inherit the usual behaviour */
        inherited(trans, vec);

        /*
         *   if report tracing is on and there are any reports at all, trace
         *   the reports
         */
        if (reportTracingOn && vec.length != 0)
            traceReports(vec);
    }

    /*
     *   Trace all reports going through us. We'll replace the reports with
     *   a more detailed breakdown, including report classes, actions and
     *   message properties as well as the actual text.
     */   
    traceReports(vec)
    {
        /* get the global reflection object, for brevity */
        local reflect = mainGlobal.reflectionObj;

        /*
         *   remember the most recent action, since printing it over and
         *   over again would be tiresome
         */
        local lastAction = nil;

        /* start building our trace report */
        local str = '';

        /*
         *   Go through all the reports. Ideally, we would just call a
         *   method on each report and have it print its own details, but we
         *   prefer not to modify more things than we have to.
         */
        foreach (local cur in vec)
        {
            /*
             *   Add the name of this report. Most reports are anonymous
             *   instances of a single class, so just extract this class and
             *   skip the actual name.
             */
            str += reflect.valToSymbol(cur.getSuperclassList()[1]);
            str += '\n';

            /*
             *   for CustomReports, the argument list contains all relevant
             *   information, so just print the argument list
             */
            if (cur.ofKind(CustomReportBase))
            {
                str += '\targs_ = ';
                str += reflect.valToSymbol(cur.args_);
                str += '\n';

                continue;
            }

            /*
             *   for command reports and implicit announcements, we can
             *   print the action that generated the report, but don't
             *   bother if the action was an EventAction, which means
             *   daemons and such
             */
            if ((cur.ofKind(ImplicitActionAnnouncement)
                    || cur.ofKind(CommandReportMessage))
                && cur.action_ != nil
                && !cur.action_.ofKind(EventAction))
            {
                /* get the action */
                local action = cur.action_;

                /*
                 *   if the action is the same as for the last report, then
                 *   we don't have to print all the details again
                 */
                if (action == lastAction)
                {
                    str += '\taction_ = (dito)\n';
                }
                else
                {
                    /* start with action class */
                    str += '\taction_ = ';
                    str += reflect.valToSymbol(action.baseActionClass);
                    str += '\n';

                    /* print the direct objekt, if any */
                    if (action.dobjCur_ != nil
                        && action.dobjList_.length == 1)
                    {
                        str += '\t\tdobjCur_ = ';
                        str += reflect.valToSymbol(action.dobjCur_);
                        str += '\n';
                    }

                    /* print the indirect object, if any */
                    if (action.iobjCur_ != nil
                        && action.iobjList_.length == 1)
                    {
                        str += '\t\tiobjCur_ = ';
                        str += reflect.valToSymbol(action.iobjCur_);
                        str += '\n';
                    }

                    /* remember this action */
                    lastAction = action;
                }
            }

            /* for command reports, add some more information */
            if (cur.ofKind(CommandReportMessage))
            {
                /* add the message property, if any */
                if (cur.messageProp_ != nil)
                {
                    str += '\tmessageProp_ = &amp;';
                    str += reflect.valToSymbol(cur.messageProp_);
                    str += '\n';
                }

                /* add the message text */
                str += '\tmessageText_ = ';
                str += reflect.valToSymbol(cur.messageText_).htmlify();
                str += '\n';
            }
        }

        /*
         *   Finally, add a line to mark the end of the trace. During a
         *   single turn, several transcripts can be used, and when each one
         *   is finished, its reports are displayed and a new trace is made.
         *   Since we can't combine reports from two different traces, it's
         *   important that we know where one trace ends and another begins.
         */
        str += '*** end ***\n';

        /* clear out the report vector */
        vec.setLength(0);

        /* add a single report using the string we constructed */
        vec.append(new MainCommandReport(str));
    }

    /* flag: is report tracing on? */
    reportTracingOn = nil
;

/*
 *   In case reflect.t isn't available, make sure that the compiler knows
 *   about the valToSymbol property.
 */
property valToSymbol;

/* ---------------------------------------------------------------------- */
/*
 *   Define verbs and actions for turning report tracing on and off.
 */
DefineSystemAction(ReportTracingMode)
    execSystemAction()
    {
        if (mainGlobal.reflectionObj == nil)
        {
            "Report tracing requires reflect.t! ";
        }
        else
        {
            local newMode = !combineReportsTransform.reportTracingOn;

            combineReportsTransform.reportTracingOn = newMode;

            "Report tracing mode is now <<newMode ? 'on' : 'off'>>. ";
        }
    }
;

grammar predicate(ReportTracingMode):
    'report-debug'
    : ReportTracingModeAction
;

#endif /* __DEBUG */

