#charset "us-ascii"

#include <adv3.h>

/* 

ConSpace Pathfinder
2005 Steve Breslin

This is an internal mechanism of the ConSpace library, and should be of
little practical interest to game authors.

We use this mechanism in two cases. (1) When an actor wants to interact
with a multiLoc item (as in "STAND NEAR THE BRIDGE"), we use this
mechanism to find which of the multiLoc's locations is closest to the
actor. (2) When an actor moves through a space, we want to know which
rooms he traverses, so that we are able to trigger travel side-effects
along the way, and provide movement messages (which are designed to give
the player a sense of movement through the intervening space between his
initial location and destination).

This is a breadth-first pathfinder for finding the shortest path between
a "start object" and a "target object," with the constraint that the
path must be within a single connection group joining the start and the
target. We return a vector of rooms, starting with the first room and
ending with the target room. If no path is found, and the there's a common
connection group enclosing a/the start and target, we return a direct path.
If there is no such common connection group, we return nil.

We only consider compass-wise connections between rooms when we search
for a path, but it is possible to connect rooms by containment (for
example, by a MultiLoc NestedRoom). These sort of sophisticated
connections are not considered by this algorithm: so any connection of
this sort will be considered "direct" for our purposes.

We allow the "target" to be a list (or vector) of objects, in which case
we return the path to the nearest object in the list (under the
constraint above). This is very simple: we finish the search when we
reach any room in our list of targets.

We also allow the start-object's outermost-room to be a multiLoc, which
means in effect that we expand the search from a number of starting
points, which are in this case the locationList of the multiLoc. This is
very simple: we just add all the locations to the start of the search
queue.

(We considered preprocessing paths, using the Floyd search, but
concluded that this would limit the power of the whole system: the map
(and perhaps even the rooms' connectionGroups) should be allowed to
change at runtime. The fastest algorithm for runtime searches is the
breadth-first search. Dijkstra is for weighted graphs, which we don't
have in conventional IF maps. The bi-directional search is for
non-directed graphs -- so that would assume we do not have any one-way
travel connectors. A heuristic-guided search might be faster, but it's
difficult to discover a good heuristic, given our constraints -- so this
didn't seem worth the trouble. A simple breadth-first search is also
quite nice, insofar as it is easily adapted to search from multiple
starting points towards multiple targets.)

It's a breadth-first search, which means it starts in the start-room,
and expands the search outward, rather like concentric circles. On the
first iteration, all the rooms connected to the start room are added to
the queue. On the second iteration, all the rooms connected to those
rooms are added, and so on. (In a conventional breadth-first search, we
don't add rooms that are already added, since this would create a
"circular search": bad recursion. But our case is a bit more
complicated, as we explain below.) We do this until we find a target
room. This ensures that we find a shortest-path: if for example we had
to do five iterations to reach a target room, then obviously the target
room cannot be less than 5 moves away from the start room.

We add a constraint to the traditional breadth-first search: that the
path must remain within at least one connectionGroup from beginning to
end. This means not only that the path must not only proceed through
rooms that are in connection groups common to "start" and "target," but
also that the path must also have a single connection group running all
the way along the path.

Thus, for each room in the queue, we keep record of its "group history
list." This is the intersection of the parent's "group history" and the
current room's connectionGroup list. (If it's the first room in the
path, it has no parent, so it's just the (start) room's
connectionGroup.)

Before we add a room to the queue, we first calculate its "group
history": if there's no intersection, we don't add the room. (Second, as
an optimization, we check that there's some target whose connectionGroup
intersects with the room's group history.)

Another (perhaps slightly trickier) consequence of this: because two
different routes to a single room might have different "group history,"
we *will* add a room that's already in the queue, but only if its "group
history" is significantly different. We still favor the shorter path,
but if for any reason its group history makes it unworkable as a
solution, we want to keep the second route on the table.

For example, if we reach roomJ staying within both connectionGroup1 and
connectionGroup2, and later we reach roomJ staying within
connectionGroup3, we add roomJ again with this additional information.
However, if we reach roomJ again staying within connectionGroup1, we
won't add roomJ again, since we've already visited roomJ via a path that
was within connectionGroup1: this new path cannot go anywhere that is
unreachable from an earlier path.

As an optimization, we filter out any start rooms whose connectionGroup
does not intersect with any targets' connectionGroup, and vice versa.

If we fail to find a path, but we know there's a connection group
connecting the two rooms, we return a direct path, assuming the connection
is one of spacial contiguity, and therefore direct.

*/

CSPathfinder: object
    findPath(startList, targetList)
    {
        /* We'll build vectors of all the connectionGroups in the start
         * and target rooms.
         */
        local allTargetGroups, allStartGroups;

        /* we convert startList to a list, if it is a single object.
         */
        if(dataType(startList) == TypeObject)
            startList = [startList];

        /* we factor the elements of startList down to their outermost rooms */
        startList = startList.mapAll({x: x.getOutermostRoom});

        /* if any elements in the startList are multiLocs, remove the
         * multiLoc and add the multiLoc's locations to the startList.
         *
         * we could do this recursively, but nested multiLocs should be
         * fairly rare.
         */
        foreach(local start in startList)
        {
            if(start.locationList)
            {
                startList -= start;
                startList += start.locationList;
            }
        }

        /* Remove any redundant entries. */
        startList = startList.getUnique();

        /* We perform the same routine on targetList. */
        if(dataType(targetList) == TypeObject)
            targetList = [targetList];

        targetList = targetList.mapAll({x: x.getOutermostRoom});

        foreach(local target in targetList)
        {
            if(target.locationList)
            {
                targetList -= target;
                targetList += target.locationList;
            }
        }

        /* Remove any redundant entries. */
        targetList = targetList.getUnique();

        /* If there's a direct intersection of startList and targetList,
         * we return this single-element list.
         */
        if(startList.intersect(targetList).length() == 1)
            return startList.intersect(targetList);

        /* We finish by paring down the startList and targetList if
         * possible.
         *
         * For each object in the targetList, we check that it shares
         * some connectionGroup with some element in the startList.
         * If not, we cannot reach it, so we remove it from
         * consideration.
         *
         * Likewise, for each object in the startList whose
         * connectionGroup shares nothing in common with any of the
         * targetList's connectionGroups. We'll remove such objects from
         * the startList.
         */
        allTargetGroups = [];
        allStartGroups = [];
        foreach(local target in targetList)
        {
            if(!target.connectionGroup)
                targetList -= target;
            else
                allTargetGroups += target.connectionGroup;
        }
        foreach(local start in startList)
        {
            if(nilToList(start.connectionGroup).
                                      intersect(allTargetGroups) == [])
                startList.removeElement(start);
            else
                allStartGroups += start.connectionGroup;
        }
        foreach(local target in targetList)
            if(target.connectionGroup.intersect(allStartGroups) == [])
                targetList.removeElement(target);

        /* If either startList or targetList are now empty, we fail the
         * search.
         */
        if(!startList.length || !targetList.length)
            return nil;

        /* If we made it this far, some start-room shares some
         * connection groups with some target-room.
         *
         * If we cannot find a connecting path that remains within a
         * single connection-group, we'll return a direct path, i.e.,
         * [start, target].
         *
         * But if we can find such a path, we'll return that.
         */

        /*-----------------------------------------------------------*/

        /* We've finished combing over our data, and we're ready to
         * begin the search. First we allocate the queue.
         */
        local queue = new Vector(60);

        /* For each element in the startList, we add an entry in the
         * queue. An entry in the queue consists of three elements:
         * (1) the room; (2) the path's connectionGroup so far (its
         * "group history"); and (3) the index (in the queue) of the
         * previous step in the path (the parent of the current entry).
         */
        foreach(local start in startList)
            queue.appendAll([start, start.connectionGroup, nil]);

        /* We iterate over each three-element entry in the queue. The
         * queue will be expanded below. This loop will run until the
         * queue is exhausted or we find a target.
         */
        for(local i = 1 ; i <= queue.length() ; i+=3)
        {

            /* pull the next room off the queue, and its group history.
             */
            local curRoom = queue[i];
            local curGroupHistory = queue[i+1];

            /* Iterate all the rooms connected compass-wise to curRoom,
             * and consider each for addition to the queue.
             */
            directionLoop:
            foreach (local dir in Direction.allDirections)
            {

                local conn;
                local dest; 

                /* if there's a working connector in this direction, its
                 * destination is the adjacent room to be considered.
                 */
                if ((conn = curRoom.getTravelConnector(dir, gActor))
                    && (dest = conn.getDestination(curRoom, gActor)))
                {

                    /* Find the last occurrance of dest in the queue, if
                     * there is any such occurrence.
                     */
                    local last = queue.lastIndexOf(dest);

                    /* Set a local variable "group" to represent the
                     * group-history of the path to the new room.
                     *
                     * If curGroupHistory does not interset with the
                     * room we're considering (that is, if group is an
                     * empty list), we will continue rather than adding
                     * the room to the queue.
                     */
                    local group = curGroupHistory.intersect(nilToList(
                                                 dest.connectionGroup));
                    if(!group.length())
                        continue;


                    /* If the group doesn't intersect with any of the
                     * targets' connectionGroups, this candidate fails.
                     */
                    if(group.intersect(allTargetGroups) == [])
                        continue;

                    /* We won't add this dest if it is already in the
                     * queue, and if its group history is a subset of a
                     * previous registration of the same dest's group
                     * history.
                     */
                    if(last)
                    {
                        for(local j=queue.indexOf(dest); j<=last ; j+=3)
                        {
                            if(dest == queue[j]
                               && !group.indexWhich(
                                   {x: !queue[j+1].indexOf(x)}))
                                continue directionLoop;
                        }
                    }

                    /* We've passed all our checks. Add the dest to
                     * the queue.
                     *
                     * We add the 'dest' room, its group history, and
                     * its parent's index in the queue.
                     */
                    queue.append(dest);
                    queue.append(group);
                    queue.append(i);

                    /* If we just added a target room, we
                     * are done.
                     */
                    if(targetList.indexOf(dest))
                        return calcPath(queue);
                }
            }
        }
        /* We have exhausted the queue without finding a path. Return
         * a direct path. Arbitrarily pick the first startList element,
         * and add the first element in the targetList which shares
         * some connection group with our startList item.
         */
        foreach(local target in targetList)
            if(startList[1].connectionGroup.
                               intersect(target.connectionGroup) != [])
                return [startList[1], target];

        /* If that didn't work, something is wrong with the algorithm.
         */
#ifdef DEBUG__
        "ERROR: Pathfinder should have found a path, but didn't. ";
#endif
        return nil;
    }

    /* Calculate a successful path, given a vector of rooms and parent
     * rooms. The final entry is the target room.
     *
     * The entries are in groups of three elements: the first is the
     * room, and the third is the index of the room which is previous
     * to the current room in the shortest path to the target room.
     * (The second is the group history, which is irrelevant at this
     * stage).
     */
    calcPath(pathVector)
    {

        /* curRoom is the current room, as we build the path backwards
         * from the target to the starting location. Initially, it is
         * the third-last element in the vector.
         */
        local curRoom = pathVector[pathVector.length() -2];

        /* pIdx is the index of the current room's parent. Initially,
         * it is the last element in the vector.
         */
        local pIdx = pathVector[pathVector.length()];

        /* ret is the path vector we'll return when we're finished
         * extracting the path from the pathVector.
         */
        local ret = new Vector(8, [curRoom]);

        while(pIdx)
        {
            curRoom = pathVector[pIdx];
            ret.prepend(curRoom);
            pIdx = pathVector[pIdx + 2];
        }

        /* that's it -- return the path */
        return ret;
    }
;

