/*
 * WorldFun.t
 * Copyright (c) 1999 The Children of Dana
 * Written by David Haire
 *
 * This file contains functions used by WorldClass.
 *
 */

/*
 * Print the game's introduction.
 */
intro: function
{
  "\b\b\b\b";
  "This is the introduction.  Prepare to play...\b";
}

/*
 * Flag when a restart has occurred by setting a flag in global.
 */
initrestart: function(parm)
{
  global.restarting := true;
}

/*
 * Format the current time for display purposes.
 */
showtime: function
{
  local time := gettime();
  local ret := cvtstr(time[6]) + ':' + cvtstr(time[7]) + ':';
  if (time[8] < 10) ret := ret + '0';
  return ret + cvtstr(time[8]);
}

/*
 * Format the current date for display purposes.
 */
showdate: function
{
  local day = ['Sun' 'Mon' 'Tue' 'Wed' 'Thu' 'Fri' 'Sat'];
  local month = ['Jan' 'Feb' 'Mar' 'Apr' 'May' 'Jun' 'Jul' 'Aug' 'Sep' 'Oct' 'Nov' 'Dec'];
  local time := gettime();
  return day[time[4]] + ' ' + cvtstr(time[3]) + ' ' + month[time[2]] +
     ' ' + cvtstr(time[1]);
}

/*
 * Format the score and turn count for display purposes.
 */
showscore: function(s, t)
{
  return cvtstr(s) + '/' + cvtstr(t);
}

/*
 * Put the status line in its initial state.
 */
init_statusline: function
{
  local scor := '';
  if (global.datedisp)
    scor := scor + ' ' + showdate();
  if (global.timedisp)
    scor := scor + ' ' + showtime();
  if (global.scoredisp)
    scor := scor + ' ' + showscore(0, 0);
  setscore(scor);
}

/*
 * Update the status line with the current score.
 */
score_statusline: function
{
  local scor := '';
  if (global.datedisp)
    scor := scor + ' ' + showdate();
  if (global.timedisp)
    scor := scor + ' ' + showtime();
  if (global.scoredisp)
    scor := scor + ' ' + showscore(global.score, global.turns);
  setscore(scor);
}

/*
 * Display the current score and turns. Override this function to add other
 *   features, such as a ranking system.
 */
display_score: function
{
  "In a total of "; say(global.turns);
  " turns, you have achieved a score of "; say(global.score);
  " points out of a possible "; say(global.maxscore); ".\n";
}

/*
 * Display the current score.
 *
 * If global.score_items_rooms is set on, the percentage of items seen and
 *   the percentage of rooms visited will also be displayed.
 *
 * If global.full_score is set on, a list of the completed scoring actions
 *   will also be displayed. This uses global.fullnums and global.fulllist
 *   to store the completed actions. To load these lists with the details
 *   of the scoring actions, you should replace all calls to incscore()
 *   with calls to fullAdd().
 */
score_and_rank: function
{
  if (global.basic_score) {
    display_score();
  }
  
  if (global.full_score) {
    fullScore();
  }

  if (global.score_items_rooms) {
    local o, seen, tot;
    "\bYou have seen ";
    tot := 0; seen := 0;
    for (o := firstobj(); o <> nil; o := nextobj(o)) {
      if (isclass(o, Item)) {
        tot += 1;
        if (o.isknownto(parserGetMe())) seen += 1;
      }
    }
    if (tot > 0) say(seen*100/tot);
    else "100";
    "% of the items, and ";
    tot := 0; seen := 0;
    for (o := firstobj(); o <> nil; o := nextobj(o)){
      if (isclass(o, Room) and not isclass(o, Nestedroom) and not (o = Walls or o = Ceiling)) {
        tot += 1;
        if (o.isseen) seen += 1;
      }
    }
    if (tot > 0) say(seen*100/tot);
    else "100";
    "% of the game locations.\n";
  }
}

/*
 * This is the fullAdd function. When changing the player's score, use
 *   fullAdd instead of incscore. It takes two variables. The first one is
 *   a number that is used to call incscore normally. The second is a
 *   single-quoted string that gets put into global.fulllist and is printed
 *   when fullScore is called. The string should NOT include the 'x points
 *   for ' part. For example:
 *
 *  oneRing:Item
 *    doTake(actor)={
 *      fullAdd(5,'getting the One Ring');
 *      "\nYou feel a lot of power surrounding you.";
 *      pass doTake;
 *    }
 *  ;
 *
 * Now, the fullScore function will print as one of its lines:
 *
 *  5 points for getting the One ring
 *
 */
fullAdd: function (amount, fulltext)
{
  local x := (find(global.fulllist, fulltext));
  incscore(amount);
  if (x = nil) {
    global.fulllist += fulltext;
    global.fullnums += amount;
  }
  else {
    global.fullnums[x] += amount;
  }
}

/*
 * This is the fullScore function. It prints the list of scoring actions in
 *   global.fulllist.
 */
fullScore: function()
{
  if (length(global.fulllist) = 0) {
    "\bYou haven't done much yet.\n";
  }
  else {
    local lng := length(global.fulllist), i;
                "\bYou have earned these points by doing the following:\b";
    for (i := 1; i <= lng; i++) {
      say (global.fullnums[i]);
      " points for ";
      say (global.fulllist[i]);
      "\n";
    }
  }
}

/*
 * Terminate the game and display a farewell message.
 */
terminate: function
{
  "Thanks for playing this game!";
}

/*
 * Response to an unrecognised input string.
 */
pardon: function
{
  "I beg your pardon? ";
}

/*
 * Increment the turn counter and update the status line.
 */
turncount: function(parm)
{
  incturn();
  global.turns += global.turnspertick;
  global.ticks++;
  score_statusline();
}

/*
 * Increment the score. If global.silentincscore is not set, a message will
 *   be displayed informing the player that the score has gone up or down.
 */
incscore: function(amount)
{
  if (not global.silentincscore) {
    "\b";
    if (amount > 1)
      "*** Your score just went up by <<amount>> points. ***";
    else if (amount < -1)
      "*** Your score just went down by <<-amount>> points. ***";
    else if (amount = 1)
      "*** Your score just went up by a point. ***";
    else if (amount = -1)
      "*** Your score just went down by a point. ***";
  }
  global.score += amount;
  score_statusline();
}

/*
 * This is a simple cover function for the built-in function input(). This 
 *   cover function switches to the 'TADS-Input' font when running in HTML 
 *   mode, so that input explicitly read through this function appears in 
 *   the same input font that is used for normal command-line input.
 */
inputline: function
{
  local ret, html_mode;
  html_mode := (systemInfo(__SYSINFO_SYSINFO) = true && systemInfo(__SYSINFO_HTML_MODE));
  if (html_mode) "<font face='TADS-Input'>";
  ret := input();
  if (html_mode) "</font>";
  return ret;
}

/*
 * Save the game.
 */
savegame: function
{
  local savefile;
  savefile := askfile('File to save game in', ASKFILE_PROMPT_SAVE, FILE_TYPE_SAVE);
  if (savefile = nil or savefile = '')
    "Failed. ";
  else if (save(savefile))
    "Save failed. ";
  else
    "Saved. ";
}

/*
 * Automatic save function. The player will be prompted to enter a name for
 *   the save file the first time only.
 */
autosave: function(parm)
{
  local savefile;
  if (global.turns - global.lastsave > global.turnsbetweensaves) {
    if (global.autosavefile = nil) {
      "\n";
      savefile := askfile('Filename for autosave game', ASKFILE_PROMPT_SAVE, FILE_TYPE_SAVE);
      if (savefile = nil or savefile = '')
        "Failed.";
      else if (save(savefile))
        "Save failed. ";
      else {
        "Saved.";
        global.autosavefile := savefile;
        global.lastsave := global.ticks;
      }
    "\b";
    }
    else {
      if (save(global.autosavefile))
        "\b*** Warning : Autosave operation failed ***\b";
      global.lastsave := global.ticks;
    }
  }
}

/*
 * mainRestore is called from within the restore command and when restoring
 *   a game from the command line.
 */
mainRestore: function(fname)
{
  if (restore(fname)) {
    "Restore failed. ";
    return nil;
  }
  else {
    score_statusline();
    "Restored.\b";
    parserGetMe().location.lookaround(parserGetMe(), true);
    return true;
  }
}

/*
 * Call this to end the game due to player death.
 */
die: function
{
  P();
  global.lastactor.diemessage; "\n";
  end();
}

/*
 * Call this to end the game (without explanation).
 */
end: function
{
  P();
  score_and_rank();
  P();
  "You may restore a saved game, start over, quit, or undo the current command.\n";
  for (;;) {
    local resp;
    "\nPlease enter RESTORE, RESTART, QUIT, or UNDO: >";
    resp := upper(inputline());
    switch (resp) {     
      case 'RESTORE':
        restoreVerb.soloaction(global.lastactor);
        break;
      case 'RESTART':
        restartVerb.soloaction(global.lastactor);
        restart();
        break;
      case 'QUIT':
        terminate();
        quit();
        abort;
        break;
      case 'UNDO':
        undoVerb.soloaction(global.lastactor);
        break;
    }
  }
}

/*
 * Return the maximum of a list of numbers
 */
maxinlist: function(l)
{
  local i, m;
  if (l = nil or l = [])
    return nil;
  else
    m := l[1];
  for (i := length(l); i > 1; i--)
    if (l[i] > m)
      m := l[i];
  return m;
}

/*
 * Return the minimum of a list of numbers
 */
mininlist: function(l)
{
  local i, m;
  if (l = nil or l = [])
    return nil;
  else
    m := l[1];
  for (i := length(l); i > 1; i--)
    if (l[i] < m)
      m := l[i];
  return m;
}

/*
 * Random number functions.  Proper use of these is important for
 *   regression testing with scripts.
 *
 * rnd(n) returns a random number between 1 and n (inclusive).
 * rndchance(n) returns true if a random n% probability is satisfied.
 *
 * When global.nondeterministic is set to nil, these function behave
 *   differently:
 *   rnd(n) always returns 1, while rndchance(n) always returns true.
 */
rnd: function(n)
{
  if (global.nondeterministic)
    return (((rand(16*n)-1)/16)+1);
  else
    return 1;
}
rndchance: function(n)
{
  if (global.nondeterministic)
    return rand(100) <= n ? true : nil;
  else
    return true;
}

/*
 * Paragraph indentation
 */
I: function
{
  if (global.indent) "\t";
}

/*
 * Paragraph separator
 */
P: function
{
  if (global.paragraphspacing) "\b";
  else "\n";
}

/*
 * Output hiding, with state save.
 */
Outhide: function(on)
{
  local state, result := nil;
  if (on) {
    state := outhide(true);
    global.outstate := [state] + global.outstate;
  }
  else {
    result := outhide(car(global.outstate));
    global.outstate := cdr(global.outstate);
  }
  return result;
}

/*
 * This function returns true if the two objects are indistinguishable in
 *   contents listings. The objects are equivalent if:
 *
 * 1) Both have isequivalent set to true
 * 2) Both have the same immediate superclass
 * 3) They have identical properties lists
 * 4) Properties in prop lists report identically
 *
 * Number 4 means that two otherwise identical objects will be listed
 *   separately if, for example, one is lit and the other isn't.
 */
indistinguishable: function(a, b)
{
  local i;
  local la, lb;
  local stat;
  if (not a.isequivalent or not b.isequivalent)
    return nil;
  if (firstsc(a) <> firstsc(b))
    return nil;
  if (a.properties <> b.properties)
    return nil;
  stat := outcapture(true);
  a.listdesc;
  la := outcapture(stat);
  stat := outcapture(true);
  b.listdesc;
  lb := outcapture(stat);
  if (upper(la) <> upper(lb))
    return nil;
  return true;
}

/*
 * Print the text equivalent of the given number.
 */
saynum: function(x)
{
  local numbers = ['one' 'two' 'three' 'four' 'five' 'six' 'seven' 'eight' 'nine' 'ten' 'eleven' 'twelve' 'thirteen' 'fourteen' 'fifteen' 'sixteen' 'seventeen' 'eighteen' 'nineteen' 'twenty'];
  if (x <= 20)
    "<<numbers[x]>>";
  else
    "<<x>>";
  return '';
}

/*
 * Routines to list the contents of a container.
 *
 * Explanation of parameters to listcontentsx:
 *
 * cont   The object whose contents are to be listed.
 * actor    The actor who's looking.
 * silent   If true, list. Otherwise, just count objects that would
 *        be listed.
 * leadingspace If true, print a blank line before the first text. If no
 *        text is printed, no blank line will be printed.
 * ird    ird = "in room description". Items can specify that they
 *        don't want to be listed in room descriptions. Likewise,
 *        items can specify that they don't want their contents
 *        listed in room descriptions.
 * loctype    Location type that describes the contents. If nil, all
 *        types are considered.
 * listfixtures If true, list items for which item.isfixture is true.
 * listactors If true, list items for which item.isactor is true.
 * cansense   Which sense to use. This is a property pointer - one of
 *        the following:
 *
 *      &cansee
 *      &cansmell
 *      &canhear
 *      &cantouch
 *
 * here   The location corresponding to "here" in "There is a Cheez
 *        Kee here."
 *      This parameter is useful for dealing with a Nestedroom,
 *        where "here" is the containing Room, not the Nestedroom
 *        itself.
 */
listcontentsX: function(cont, actor, silent, leadingspace, ird, loctype, listfixtures, listactors, cansense, here)
{
  local i, j, k, len, l, clen;
  local see := [], hear := [], smell := [], touch := [];
  local ltlist;
  local o;
  local container, lt;
  local unlisted := [];
  local tolist := [], tlen;
  local indislist := [], indiscount := [], indislen, indistot;
  local olist := [];
  local listedhere := [], tot := 0;
  local Silent := silent;
  local indisfound := nil;

  /*
   * List things the player sees.
   *
   * Always see what the player can see so we get a valid list of
   *   things that are listedhere. We need this to decide later if we
   *   should say "something is ticking" or "the bomb is ticking".
   */
  if (not (cansense = nil or cansense = &cansee))
    silent := true;

  /*
   * If the actor can't see this container, explain why and exit.
   */
  caps();
  if (not actor.cansee(cont, nil, silent)) return;

  /*
   * Construct the list of location types to look for. If loctype is
   *   non-nil, verify that it works. Otherwise, try each locationtype.
   */
  if (loctype = nil)
    ltlist := actor.canseecontents(cont, true, nil);
  else
    ltlist := [loctype];

  /*
   * Obtain a list of the items that the actor can potentially see.
   */
  if (ltlist = nil)
    see := [];
  else
    see := cont.seecont(actor, ltlist);

  /*
   * Read through the list of items that the actor can potentially see,
   *   and check if the item should be listed.
   */
  for (i := 1; i <= length(see); i++) {
    indisfound := nil;
    l := see[i];
    container := l[1];
    lt := l[2];
    clen := l[3];

    /*
     * If this container doesn't want its contents listed in room
     *   descriptions, don't list.
     */
    if (ird and not container.(global.loccont[lt])(nil)) {
      unlisted += container;
      continue;
    }

    /*
     * If this is a room description we don't want to list the
     *   contents of any container whose container doesn't want
     *   things listed. So we need to keep a list of containers
     *   that don't want things listed. Since our list of objects
     *   is guaranteed to go down the containment hierarchy as we
     *   progress from 1 to n, we'll always see containers before
     *   we see their contents, so this works.
     *
     * The exception to this is that if the container is this Thing
     *   (cont), then it doesn't have to be listable for its
     *   contents to be listed.
     */
    if (ird) {
      if (find(unlisted, container) <> nil)
        continue;
      if (find(unlisted, container.location) <> nil) {
        unlisted += container;
        continue;
      }

    }

    /*
     * If this is a room description, remove any items that don't
     *   want to be listed in room descriptions.
     */
    if (ird) for (j := 4; j <= clen; j++) {
      if (not l[j].islistable(actor))
        l[j] := nil;
    }

    /*
     * Remove any items that aren't noticeable.
     */
    for (j := 4; j <= clen; j++) {
      if (l[j] <> nil)
        if (not l[j].isnoticeable(actor))
          l[j] := nil;
    }

    /*
     * Now list all the fixtures and actors, if we're listing them.
     *   If we're not listing them, just delete them.
     */
    tolist := [];
    tlen := 0;
    for (j := 4; j <= clen; j++) {
      o := l[j];
      if (o = nil)
        continue;
      if (not o.isfixture and not o.isactor)
        continue;
      l[j] := nil;
      if (not listfixtures and not listactors)
        continue;
      if (o.isactor) {
        if (not listactors)
          continue;
        if (o = actor)
          continue;
      }
      tolist += o;
      tlen++;
    }
    if (not silent and listfixtures) {
      for (j := 1; j <= tlen; j++) {
        o := tolist[j];
        if (leadingspace) {
          leadingspace := nil;
          P(); I();
        }
        if (not o.isactor) {
          "<<o.heredesc>> ";
          tot++;
          listedhere += o;
        }
      }
    }
    if (not silent and listactors) {
      for (j := 1; j <= tlen; j++) {
        o := tolist[j];
        if (leadingspace) {
          leadingspace := nil;
          P(); I();
        }
        if (o.isactor) {
          "<<o.actordesc>> ";
          tot++;
          listedhere += o;
        }
      }
    }

    /*
     * Now list everything else. We separate out
     *   indistinguishables. We move plurals to the front to make 
     *   the listing more readable.
     */
    tolist := [];
    tlen := 0;
    indislist := [];
    indiscount := [];
    indislen := 0;
    for (j := 4; j <= clen; j++) {
      o := l[j];
      if (o = nil)
        continue;
      if (o.isequivalent) {
        indislist += o;
        indiscount += 1;
        indislen++;
      }
      else {
        if (o.isplural)
          tolist := [o] + tolist;
        else
          tolist += o;
        tlen++;
      }
    }

    /*
     * Now merge indistinguishable objects.
     */
    for (j := 1; j <= indislen; j++) {
      if (indislist[j] = nil)
        continue;
      for (k := j + 1; k <= indislen; k++) {
        if (indislist[k] = nil)
          continue;
        if (indistinguishable(indislist[j], indislist[k])) {
          indiscount[j]++;
          indislist[k] := nil;
        }
      }
    }

    /*
     * Put indistinguishable objects back in tolist.
     */
    indistot := 0;
    olist := [];
    for (j := 1; j <= indislen; j++) {
      local o := indislist[j];
      if (o = nil)
        continue;
      olist += o;
      tlen++;
      indistot++;
      indiscount[indistot] := indiscount[j];
    }
    if (indistot > 0) tolist := olist + tolist;

    /*
     * Now list the items.
     */
    if (tlen = 0)
      continue;
    tot += tlen;
    listedhere += tolist;
    if (silent)
      continue;
    if (leadingspace) {
      leadingspace := nil;
      P(); I();
    }
    "\^";
    if (container.isactor)
      "<<container.subjthedesc>> <<container.has>>\ ";
    for (j := 1; j <= tlen; j++) {
      o := tolist[j];
      if (j = 1) "";
      else if (j = tlen) {
        if (tlen > 2) ",";
        " and ";
      }
      else
        ", ";
      if (j <= indistot and indiscount[j] > 1) {
        "<<saynum(indiscount[j])>> \v<<o.listpluraldesc>>";
        indisfound := true;
      }
      else
        o.listdesc;
    }
    if (container.isactor) ". ";
    else {
      if (tlen > 1 or o.isplural or indisfound)
        " are ";
      else
        " is ";
      if (find(here, container) and global.loctypes[lt] = 'in')
        " here. ";
      else
        " <<global.loctypes[lt]>> <<container.objthedesc(nil)>>. ";
    }
  }
  global.listed += listedhere;

  /*
   * If we only did the "see" code to get a list of things that the
   *   player can see, don't count these items in the total.
   */
  silent := Silent;
  if (not (cansense = nil or cansense = &canseee))
    tot := 0;

  /*
   * Now list things the player hears.
   */
  if (cansense = nil or cansense = &canhear) {

    local h, o, contained, tolist := [], tlen := 0;

    for (i := length(global.listablesoundlist); i > 0; i--) {
      o := global.listablesoundlist[i];
      contained := nil;
      if (not o.islistablesound(actor))
        continue;
      while (o.location <> nil) {
        if (ird and not o.location.(global.locconthear[o.locationtypenum])(nil))
          break;
        if (o.location = cont and (loctype = nil or o.locationtype = loctype)) {
          contained := true;
          break;
        }
        o := o.location;
      }
      if (contained) {
        o := global.listablesoundlist[i];
        if (actor.canhear(o, nil, true)) {
          tolist += o;
          tlen++;
        }
      }
    }

    if (tlen = 0) goto EndHear;
    tot += tlen;
    if (silent) goto EndHear;
    if (leadingspace) {
      leadingspace := nil;
      P(); I();
    }
    for (j := 1; j <= tlen; j++) {
      o := tolist[j];

      /*
       * Print either the name of this thing (if we've
       *   listed it) or "something ..."
       */
      if (o.alwaysname(actor) and o.isknownto(actor) or actor.cansee(o, nil, true) or find(listedhere, o) <> nil)
          "\^<<o.subjthedesc>> ";
      else {
        if (o.isplural)
          "Some things ";
        else
          "Something ";
        if (cont.alwaysname(actor) and cont.isknownto(actor) or find(listedhere, cont) <> nil) {
          if (cont.isactor)
             "<<cont.subjdesc>> <<cont.has>> ";
          else
             "<<o.locationtype>> <<cont.objthedesc(actor)>> ";
        }
      }
      "<<o.listlistendesc>>. ";
    }
EndHear:;
  }

  /*
   * Now list things the player smells.
   */
  if (cansense = nil or cansense = &cansmell) {

    local o, contained, tolist := [], tlen := 0;

    for (i := length(global.listablesmelllist); i > 0; i--) {
      o := global.listablesmelllist[i];
      contained := nil;
      if (not o.islistablesmell(actor))
        continue;
      while (o.location <> nil) {
        if (ird and not o.location.(global.loccontsmell[o.locationtypenum])(nil))
          break;
        if (o.location = cont and (loctype = nil or o.locationtype = loctype)) {
          contained := true;
          break;
        }
        o := o.location;
      }
      if (contained) {
        o := global.listablesmelllist[i];
        if (actor.cansmell(o, nil, true)) {
          tolist += o;
          tlen++;
        }
      }
    }

    if (tlen = 0) goto EndSmell;
    tot += tlen;
    if (silent) goto EndSmell;
    if (leadingspace) {
      leadingspace := nil;
      P(); I();
    }
    for (j := 1; j <= tlen; j++) {
      o := tolist[j];
          
      /*
       * Print either the name of this thing (if we've
       *   listed it) or "something ..."
       */
      if (o.alwaysname(actor) and o.isknownto(actor) or actor.cansee(o, nil, true) or find(listedhere, o) <> nil)
          "\^<<o.subjthedesc>> ";
      else {
        if (o.isplural)
          "Some things ";
        else
          "Something ";
        if (cont.alwaysname(actor) and cont.isknownto(actor) or find(listedhere, cont) <> nil) {
          if (cont.isactor)
             "<<cont.subjdesc>> <<cont.has>> ";
          else
             "<<o.locationtype>> <<cont.objthedesc(actor)>> ";
        }
      }
      "<<o.listsmelldesc>>. ";
    }
EndSmell:;
  }

  /*
   * Now list things the player feels.
   */
  if (cansense = nil or cansense = &cantouch) {

    local o, contained, tolist := [], tlen := 0;

    for (i := length(global.listabletouchlist); i > 0; i--) {
      o := global.listabletouchlist[i];
      contained := nil;
      if (not o.islistabletouch(actor))
        continue;
      while (o.location <> nil) {
        if (ird and not o.location.(global.locconttouch[o.locationtypenum])(nil))
          break;
        if (o.location = cont and (loctype = nil or o.locationtype = loctype)) {
          contained := true;
          break;
        }
        o := o.location;
      }
      if (contained) {
        o := global.listabletouchlist[i];
        if (actor.cantouch(o, nil, true)) {
          tolist += o;
          tlen++;
        }
      }
    }

    if (tlen = 0) goto EndTouch;
    tot += tlen;
    if (silent) goto EndTouch;
    if (leadingspace) {
      leadingspace := nil;
      P(); I();
    }
    for (j := 1; j <= tlen; j++) {
      o := tolist[j];
        
      /*
       * Print either the name of this thing (if we've
       *   listed it) or "something ..."
       */
      if (o.alwaysname(actor) and o.isknownto(actor) or actor.cansee(o, nil, true) or find(listedhere, o) <> nil)
          "\^<<o.subjthedesc>> ";
      else {
        if (o.isplural)
          "Some things ";
        else
          "Something ";
        if (cont.alwaysname(actor) and cont.isknownto(actor) or find(listedhere, cont) <> nil) {
          if (cont.isactor)
             "<<cont.subjdesc>> <<cont.has>> ";
          else
             "<<o.locationtype>> <<cont.objthedesc(actor)>> ";
        }
      }
      "<<o.listtouchdesc>>. ";
    }
EndTouch:;
  }

  return tot;
}

/*
 * The front end for contents listing.
 */
listcontents: function(cont, actor, silent, leadingspace, ird, loctype, listfixtures, listactors, cansense, here)
{
  local i, l, tot;

  /*
   * Nothing listed (visual) yet.
   */
  global.listed := [] + cont;

  /*
   * List contents.
   */
  tot := listcontentsX(cont, actor, silent, leadingspace, ird, loctype, listfixtures, listactors, cansense, here);

  /*
   * Make everything we listed (visual only) known. We have to do this
   *   so that things that only become visible after the player enters
   *   the room (e.g. things that are in containers that the player
   *   opens after entering a room.) are made known to the actor.
   */
  l := global.listed;
  for (i := length(l); i > 0; i--)
    l[i].makeknownto(actor);

  return tot;
}

/*
 * The following function gets called when the parser needs the player to
 *   fill in the direct or indirect object. E.g.,
 *
 *   >unlock
 *   Unlock what?
 *
 *   >unlock door
 *   With what?
 *
 * These messages are a bit snappy, but we want to avoid saying "it" since
 *   the subject may actually be plural (and we have no way of telling what
 *   the object is).
 */
parseAskobj: function(v, ...)
{
  caps();
  if (argcount = 2) {
    local p := getarg(2);
    if (p <> nil)
      p.sdesc;
    else
      "to";
  }
  else
    "<<v.sdesc>>";
  " what?";
}

/*
 * The following function is called by the parser when objects need to be 
 *   disambiguated. It is called with the string that the player typed 
 *   which is in need of disambiguation, and a list of objects that match 
 *   the string.
 */
parseDisambig: function(str, lst)
{
  local i, tot, cnt;
  "Which <<str>> do you mean, ";
  for (i := 1, cnt := length(lst); i <= cnt; ++i) {
    lst[i].thedesc;
    if (i < cnt) ", ";
    if (i+1 = cnt) "or ";
  }
  "?";
}

/*
 * The following function is called by the parser when the verb attempted 
 *   isn't accepted by the objects involved.
 */
parseError2: function(v, d, p, i)
{
  "I don't know how to <<v.sdesc>> ";
  if (d)
    "<<d.thedesc>>.";
  else
    "anything <<p ? p.sdesc : "to">> <<i.thedesc>>.";
}

/*
 * The following function is called when the parser is assuming a default 
 *   object.
 */
parseDefault: function(obj, prp)
{
  "(";
  if (prp) "<<prp.sdesc>> ";
  obj.thedesc;
  ")";
}

/*
 * The following function gets called when there's a parse error. We
 *   complain about "all" when it's not allowed, and place square brackets 
 *   ('[]') around other error messages.
 */
parseError: function(num, str)
{
  if (global.allerror) {
    global.allerror := nil;
    return 'You can\'t use \"all\" with that verb.';
  }
  if (num < 100) return '[' + str + ']';
  if (num = 110 or num = 140) return '[' + str;
  if (num = 115 or num = 143) return str + ']';
  return nil;
}

/*
 * Footnote functions.
 *
 * To use footnotes, insert <<note(x)>> into a printed string at the point 
 *   that you wish the footnote reference to appear. 'x' should be an object 
 *   with a note property that is the text string to be printed when the player 
 *   inputs 'note y'. Footnote numbers will be automatically generated by 
 *   default. To force a specific number to be assigned to a footnote, include 
 *   the property footnum in the object, and add the number to global.footavoid.
 */
printnote: function(num)
{
  local i, o, notes, len;
  notes := global.footnotelist;
  len := length(notes);
  for (i := 1; i <= len; i++) {
    o := notes[i];
    if (o.footnum = num) {
      "[\(<<num>>\)]:\ <<o.footnote>>";
      return;
    }
  }
  "No such footnote.";
}

note: function(o)
{
  if (o.footnum = nil) {
    for (;;) {
      o.footnum := global.footnumber;
      global.footnumber++;
      if (find(global.footavoid, o.footnum) = nil)
        break;
    }
  }
  "[\(<<o.footnum>>\)]";
  return '';      /* so we can use inside << ... >> */
}

/*
 * Switch the player character to a new actor. This function uses the 
 *   parserSetMe() built-in function to change the parser's internal record 
 *   of the player character object, and in addition performs some 
 *   necessary modifications to the outgoing and incoming player character 
 *   objects. We remove the vocabulary words "me" and "myself" from the 
 *   outgoing object, and add them to the incoming object.
 *
 * Note: Any actor that can be 'switched' to *must* be defined as a Player, 
 *   not just an Actor.
 */
switchPlayer: function(newPlayer)
{
  if (parserGetMe() = newPlayer)
    return;
  if (not isclass(newPlayer, Player))
    return;
  delword(parserGetMe(), &noun, 'me');
  delword(parserGetMe(), &noun, 'myself');
  parserSetMe(newPlayer);
  addword(newPlayer, &noun, 'me');
  addword(newPlayer, &noun, 'myself');
}
