////////////////////////////////////////////////////////////////////////
//  
//  ALT Library File: DeepVerb 010123
//
//  Copyright (c) 2000, 2001 Kevin Forchione. All rights reserved.
//  Based on ADV.T (c) and STD.T (c) Michael Roberts.
//
//  This file is part of the ALT replacement library for ADV.T and 
//  STD.T and requires TADS 2.5.1 or later.
//
////////////////////////////////////////////////////////////////////////

/* Copyright (c) 2000 by Kevin Forchione.  All Rights Reserved. */
/* Based on ADV.T (c) Michael Roberts. */

#ifndef _DEEP_VERB_H_
#define _DEEP_VERB_H_

#include <askfordobj.t>
#include <itemcount.t>
#include <listcont.t>
#include <listcontcont.t>
#include <gdolist.t>
#include <global.t>
#include <glocation.t>
#include <nestlistcont.t>

#pragma C+

/*
 *  DeepVerb: object
 *
 *  A "verb object" that is referenced by the parser when the player
 *  uses an associated vocabulary word.  A DeepVerb contains both
 *  the vocabulary of the verb and a description of available syntax.
 *  The verb property lists the verb vocabulary words;
 *  one word (such as 'take') or a pair (such as 'pick up')
 *  can be used.  In the latter case, the second word must be a
 *  preposition, and may move to the end of the sentence in a player's
 *  command, as in "pick it up."  The action(actor)
 *  method specifies what happens when the verb is used without any
 *  objects; its absence specifies that the verb cannot be used without
 *  an object.  The doAction specifies the root of the message
 *  names (in single quotes) sent to the direct object when the verb
 *  is used with a direct object; its absence means that a single object
 *  is not allowed.  Likewise, the ioAction(preposition)
 *  specifies the root of the message name sent to the direct and
 *  indirect objects when the verb is used with both a direct and
 *  indirect object; its absence means that this syntax is illegal.
 *  Several ioAction properties may be present:  one for each
 *  preposition that can be used with an indirect object with the verb.
 *  
 *  The validDo(actor, object, seqno) method returns true
 *  if the indicated object is valid as a direct object for this actor.
 *  The validIo(actor, object, seqno) method does likewise
 *  for indirect objects.  The seqno parameter is a "sequence
 *  number," starting with 1 for the first object tried for a given
 *  verb, 2 for the second, and so forth; this parameter is normally
 *  ignored, but can be used for some special purposes.  For example,
 *  askVerb does not distinguish between objects matching vocabulary
 *  words, and therefore accepts only the first from a set of ambiguous
 *  objects.  These methods do not normally need to be changed; by
 *  default, they return true if the object is accessible to the
 *  actor.
 *  
 *  The doDefault(actor, prep, indirectObject) and
 *  ioDefault(actor, prep) methods return a list of the
 *  default direct and indirect objects, respectively.  These lists
 *  are used for determining which objects are meant by "all" and which
 *  should be used when the player command is missing an object.  These
 *  normally return a list of all objects that are applicable to the
 *  current command.
 *  
 *  The validDoList(actor, prep, indirectObject) and
 *  validIoList(actor, prep, directObject) methods return
 *  a list of all of the objects for which validDo would be true.
 *  Remember to include moveFloating(), which moves all Floating 
 *  class objects to their correct locations.
 * 
 *  Note that the objects returned by this list will
 *  still be submitted by the parser to validDo, so it's okay for
 *  this routine to return too many objects.  In fact, this
 *  routine is entirely unnecessary; if you omit it altogether, or
 *  make it return nil, the parser will simply submit every
 *  object matching the player's vocabulary words to validDo.
 *  The reason to provide this method is that it can significantly
 *  improve parsing speed when the game has lots of objects that
 *  all have the same vocabulary word, because it cuts down on the
 *  number of times that validDo has to be called (each call
 *  to validDo is fairly time-consuming).
 */
class DeepVerb: object  
    sdesc 		= {self.sDesc;}   // used as a parser hook for Alt version.
    doSense 	= touch
    ioSense 	= touch    
    scopeSense 	= sight

/** DISAMBIGUATION & RESOLUTION **/

    ioDefault(actor, prep) = {
        return actor.contents + actor.location.contents;
    }
    validIoList(actor, prep, dobj) = (self.validDoList(actor, prep, dobj))
    validIo(actor, obj, seqno) = {
        return actor.canSensePreferred(self.ioSense, obj);
    }
    validIoSense(actor, obj, seqno) = {
        return actor.canSenseObj(self.ioSense, obj);
    }
    
    doDefault(actor, prep, io) = {
        return actor.contents + actor.location.contents;
    }
    validDoList(actor, prep, iobj) = {
        // Move Floating objects to their locations
        moveFloating(actor);
        
        // get scope list for actor
        return nil;
    }
    validDo(actor, obj, seqno) = {
        return actor.canSensePreferred(self.doSense, obj);
    }
    validDoSense(actor, obj, seqno) = {
        return actor.canSenseObj(self.doSense, obj);
    }

	/*
	 *	This method is necessary for cases where the action of a two-
	 *	object verb template is completed by the direct object. Because
	 *	the parser does not provide a pointer to these methods (e.g. 
	 *	&doUnlockWith) they must be supplied by the author in this
	 *	deepverb method, otherwise the "parent" object will be selected 
	 *	by the disambigXobj() methods and the action may not be
	 *	completed, even though a component logically provides the means
	 *	to do so.
	 */
    getDoProp( prep ) = { return nil; }

    disambigIobj(actor, prep, dobj, verProp, wordList, objList, flagList,
               numberWanted, isAmbiguous, silent) = 
    {
    	local i, j, ret, actionProp, newObjList = [];    	
    	
        global.cmdIoWords = wordList;
		
	actionProp = verbinfo( self, prep )[ 3 ];

    	for ( i = 1; i <= length( objList ); ++i ) {
    		ret = objList[ i ].disambigIobj( actor, prep, dobj, verProp,
    			actionProp, wordList, objList, flagList, numberWanted,
    			isAmbiguous, silent );

			/* if no replacement is returned use the original object */
			if ( length( ret ) == 0 )
				ret += objList[ i ];

			/* add returned objects to new object list */
			for ( j = 1; j <= length( ret ); ++j )
				if ( find( newObjList, ret[ j ] ) == nil )
					newObjList += ret[ j ];
    	}
    	
    	/*
    	 *	Store the wordList and newObjList for reference by
    	 *	preparseCmd() for handling pronouns.
    	 */
    	global.iList = [ wordList newObjList ];

    	if ( objList == newObjList ) 
    		return DISAMBIG_CONTINUE;
    	else return newObjList;
    }
    disambigDobj(actor, prep, iobj, verProp, wordList, objList, flagList,
               numberWanted, isAmbiguous, silent) = 
    {
    	local i, j, ret, actionProp, newObjList = [];
    	
        global.cmdDoWords = wordList;
		
	if ( prep )
		actionProp = self.getDoProp( prep );
	else actionProp = verbinfo( self )[ 2 ];

    	for ( i = 1; i <= length( objList ); ++i ) {
    		ret = objList[ i ].disambigDobj( actor, prep, iobj, verProp,
    			actionProp, wordList, objList, flagList, numberWanted,
    			isAmbiguous, silent );

			/* if no replacement is returned use the original object */
			if ( length( ret ) == 0 )
				ret += objList[ i ];

			/* add returned objects to new object list */
			for ( j = 1; j <= length( ret ); ++j )
				if ( find( newObjList, ret[ j ] ) == nil )
					newObjList += ret[ j ];
    	}
    	
    	/*
    	 *	Store the wordList and newObjList for reference by
    	 *	preparseCmd() for handling pronouns.
    	 */
    	global.dList = [ wordList newObjList ];

    	if ( objList == newObjList ) 
    		return DISAMBIG_CONTINUE;
    	else return newObjList;
    }

/** ACCESSIBILITY **/

    getCmdPhrase(obj, cmd) = {
        local v;
        v = getwords(obj, &adjective);
        v += getwords(obj, &noun);
        v += getwords(obj, &plural);
        return intersect(v, cmd);
    }
	cantReach(actor, doList, ioList, prep) = {		
	    local	i, list, cmdWords;
	    
		if (doList != nil) {
			list = doList;
			cmdWords = global.cmdDoWords;
			if (length(cmdWords) == 0)
			    cmdWords = self.getCmdPhrase(doList[1], 
			        global.cmdWordList_);
			if (length(cmdWords) == 0)
			    cmdWords = objwords(1);
		    if (length(cmdWords) == 0)
		        cmdWords = ['such', 'thing'];
	        global.cantSenseWords = cmdWords;
		}
		else {
			list = ioList;
			cmdWords = global.cmdIoWords;
			if (length(cmdWords) == 0)
			    cmdWords = self.getCmdPhrase(ioList[1], 
			        global.cmdWordList_);
			if (length(cmdWords) == 0)
			    cmdWords = objwords(2);
		    if (length(cmdWords) == 0)
		        cmdWords = ['that'];
	        global.cantSenseWords = cmdWords;
		}
        
        for (i = 1; i <= length(list); ++i)
        {
            if (doList)
                self.cantSenseDoRedirect(actor, list[i], prep);
            else
                self.cantSenseIoRedirect(actor, list[i], prep);
        }
	}
	cantSenseDoRedirect(vantage, target, prep) = {
	    local sense, obs;
	    
	    sense  = target.obstructedSense;
	    if (sense == nil) sense = sight;
	    obs = target.obstructor;
	    if (obs == nil) obs = gLocation();
                            
        switch(sense)
        {
            case sight:
                obs.cantSee(vantage, target);
                break;
            case sound:
                obs.cantHear(vantage, target);
                break;
            case smell:
                obs.cantSmell(vantage, target);
                break;
            case touch:
                obs.cantTouch(vantage, target);
                break;
            default:
                obs.cantSee(vantage, target);
                break;
        }	    
    }
    cantSenseIoRedirect(vantage, target, prep) = {
        self.cantSenseDoRedirect(vantage, target, prep); 
    }

/** ACTION **/

    verbAction(actor, dobj, prep, iobj) = {
        local i, o, scopeList, ret, str, strList = [];
        
        // Set Phase and Stage
        global.setCommandPhase(CP_PREACTION);
        
        // Set command variables
        gActor(actor);
        gVerb(self);
        gDobj(dobj);
        gPrep(prep);
        gIobj(iobj);
        
        // Get any outcaptured display from the action methods
        str = nestOutcapture(nil);
        
        if (str != '')
            strList += str;

        // Get any outcaptured display from the actor
        nestOutcapture(true);
        ret = nil;
        ret = actor.actorPreAction;
        str = nestOutcapture(nil);
        if (str != '')
            strList += str;
        if (ret == true) {
            strList = [str];
            goto lbl_process;
        }

        // Get any outcaptured display from the actor location
        if (actor.location) {
            nestOutcapture(true);
            ret = nil;
            ret = actor.location.locPreAction;
            str = nestOutcapture(nil);
            if (str != '')
                strList += str;
            if (ret == true) {
                strList = [str];
                goto lbl_process;
            }
        }

        // Get scopeList based on verb.scopeSense
        scopeList = actor.scope(self.scopeSense, [], nil, nil);   
        scopeList = intersect(scopeList, global.scopePreActionList);
        
        // Get any outcaptured display from scope objects
        for (i = 1; i <= length(scopeList); ++i) {
            o = scopeList[i];
            
            if (o == actor||o == actor.location||o == dobj||o == iobj)
                continue;
                
          	if (actor.isIn( o ))
          		continue;
                
            nestOutcapture(true);
            ret = nil;
            ret = o.scopePreAction;
            str = nestOutcapture(nil);
            if (str != '')
                strList += str;
            if (ret == true) {
                strList = [str];
                goto lbl_process;
            }
        }

        // Get any outcaptured display from the indirect-object
        if (iobj) {
            nestOutcapture(true);
            ret = nil;
            ret = iobj.iobjPreAction;
            str = nestOutcapture(nil);
            if (str != '')
                strList += [str];
            if (ret == true) {
                strList = [str];
                goto lbl_process;
            }
        }

        // Get any outcaptured display from the direct-object
        if (dobj) {
            nestOutcapture(true);
            ret = nil;
            ret = dobj.dobjPreAction;
            str = nestOutcapture(nil);
            if (str != '')
                strList += str;
            if (ret == true) {
                strList = [str];
                goto lbl_process;
            }
        }

    lbl_process: ;
        for (i = 1; i <= length(strList); ++i) {
            str = strList[i];
            queue.queueMessage(actor, verb, dobj, prep, iobj, 0, str);
        }
        nestOutcapture(true);
        if (ret == true) exit;
    
        // Set Phase and Stage
        global.setCommandPhase(CP_ACTION);
    }
;

askVerb: DeepVerb
    verb = 'ask'
    sDesc = "ask"
    prepDefault = aboutPrep
    ioAction(aboutPrep) = 'AskAbout'
    validIo(actor, obj, seqno) = { return (seqno == 1); }
    validIoList(actor, prep, dobj) = (nil)
    validIoSense(actor, obj, seqno) = {
        return true;
    }
;
attachVerb: DeepVerb
    verb = 'attach' 'connect'
    sDesc = "attach"
    prepDefault = toPrep
    ioAction(toPrep) = 'AttachTo'
;
attackVerb: DeepVerb
    verb = 'attack' 'kill' 'hit'
    sDesc = "attack"
    prepDefault = withPrep
    ioAction(withPrep) = 'AttackWith'
;
boardVerb: DeepVerb
    verb = 'get in' 'get into' 'board' 'get on'
    sDesc = "get on"
    doAction = 'Board'
;
breakVerb: DeepVerb
    verb = 'break' 'ruin' 'destroy'
    sDesc = "break"
    doAction = 'Break'
;
centerVerb: DeepVerb
    verb = 'center'
    sDesc = "center"
    doAction = 'Center'
;
cleanVerb: DeepVerb
    verb = 'clean'
    sDesc = "clean"
    ioAction(withPrep) = 'CleanWith'
    doAction = 'Clean'
;
climbVerb: DeepVerb
    verb = 'climb'
    sDesc = "climb"
    doAction = 'Climb'
;
climbUpVerb: DeepVerb
    verb = 'climb up'
    sDesc = "climb up"
    doAction = 'Climbup'
;

climbDownVerb: DeepVerb
    verb = 'climb down'
    sDesc = "climb down"
    doAction = 'Climbdown'
;
closeVerb: DeepVerb
    verb = 'close'
    sDesc = "close"
    doAction = 'Close'
;
detachVerb: DeepVerb
    verb = 'detach' 'disconnect'
    sDesc = "detach"
    prepDefault = fromPrep
    ioAction(fromPrep) = 'DetachFrom'
    doAction = 'Detach'
;
digVerb: DeepVerb
    verb = 'dig' 'dig in'
    sDesc = "dig in"
    prepDefault = withPrep
    ioAction(withPrep) = 'DigWith'
;
drinkVerb: DeepVerb
    verb = 'drink'
    sDesc = "drink"
    doAction = 'Drink'
;
eatVerb: DeepVerb
    verb = 'eat' 'consume'
    sDesc = "eat"
    doAction = 'Eat'
;
fastenVerb: DeepVerb
    verb = 'fasten' 'buckle' 'buckle up'
    sDesc = "fasten"
    doAction = 'Fasten'
;
flipVerb: DeepVerb
    verb = 'flip'
    sDesc = "flip"
    doAction = 'Flip'
;
followVerb: DeepVerb
    verb = 'follow'
    sDesc = "follow"
    doAction = 'Follow'
;
getOutVerb: DeepVerb
    verb = 'get out' 'get outof' 'get off' 'get offof'
    sDesc = "get out of"
    doAction = 'Unboard'
    action(actor) = { askForDobj(); }
    doDefault(actor, prep, io) = {
        if (actor.location && actor.location.location)
            return ([] + actor.location);
        else
            return [];
    }
;
giveVerb: DeepVerb
    verb = 'give' 'offer'
    sDesc = "give"
    prepDefault = toPrep
    ioAction(toPrep) = 'GiveTo'
    doDefault(actor, prep, io) = {
        return actor.contents;
    }
;
goThroughVerb: DeepVerb
    verb = 'go through' 'go thru'
    sDesc = "go through"
    doAction = 'Gothrough'
;
helloVerb: DeepVerb
    verb = 'hello' 'hi' 'greetings'
    sDesc = "hello"
    action(actor) = {
        "Nice weather we've been having.\n";
    }
;
inspectVerb: DeepVerb
    verb = 'inspect' 'examine' 'look at' 'l at' 'x'
    doSense = sight
    sDesc = "inspect"
    doAction = 'Inspect'
;
iTallVerb: DeepVerb
    verb = 'i_tall'
    sDesc = "inventory tall"
    action(actor) = {
        iVerb.useInventoryTall = true;
        iVerb.action(actor);
    }
;
iVerb: DeepVerb
    verb = 'inventory' 'i'
    sDesc = "inventory"
    useInventoryTall = nil
    action(actor) = {
        if (itemCount(actor.contents)) {
            /* use tall or wide mode, as appropriate */
            if (self.useInventoryTall) {
                /* use "tall" mode */
                "%You% %are% carrying:\n";
                nestListCont(actor, 1);
            } else {
                /* use wide mode */
                "%You% %have% "; listCont(actor); ". ";
                listContCont(actor);
            }
        } else "%You% %are% empty-handed.\n";
    }
;
iWideVerb: DeepVerb
    verb = 'i_wide'
    sDesc = "inventory wide"
    action(actor) = {
        iVerb.useInventoryTall = nil;
        iVerb.action(actor);
    }
;
jumpVerb: DeepVerb
    verb = 'jump' 'jump over' 'jump off'
    sDesc = "jump"
    doAction = 'Jump'
    action(actor) = { "Wheeee!"; }
;
knockVerb: DeepVerb
    verb = 'knock' 'knock on' 'knock at'
    sDesc = "knock"
    doAction = 'Knock'
;
lieVerb: DeepVerb
    verb = 'lie' 'lie on' 'lie in' 'lie down' 'lie downon' 'lie downin'
    sDesc = "lie on"
    doAction = 'Lieon'
;
listenVerb: DeepVerb
    verb = 'listen'
    sDesc = "listen"
    action(actor) = {
        gDobj(actor.location);
        actor.location.doListento(actor);
    }
;
listentoVerb: DeepVerb
    verb = 'listen to'
    sDesc = "listen to"
    doAction = 'Listento'
;
lockVerb: DeepVerb
    verb = 'lock'
    sDesc = "lock"
    ioAction(withPrep) = 'LockWith'
    doAction = 'Lock'
    prepDefault = withPrep
;
lookVerb: DeepVerb
    verb = 'look' 'l' 'look around' 'l around'
    sDesc = "look"
    action(actor) = {
        actor.location.lookAround(true);
    }
;
lookBehindVerb: DeepVerb
    verb = 'look behind' 'l behind'
    sDesc = "look behind"
    doAction = 'Lookbehind'
;
lookInVerb: DeepVerb
    verb = 'look in' 'look on' 'l in' 'l on'
    sDesc = "look in"
    doAction = 'Lookin'
;
lookThruVerb: DeepVerb
    verb = 'look through' 'look thru' 'l through' 'l thru'
    sDesc = "look through"
    doAction = 'Lookthru'
;
lookUnderVerb: DeepVerb
    verb = 'look under' 'look beneath' 'l under' 'l beneath'
    sDesc = "look under"
    doAction = 'Lookunder'
;
lowerVerb: DeepVerb
    verb = 'lower'
    sDesc = "lower"
    doAction = 'Lower'
;
makeVerb: DeepVerb
    verb = 'make'
    sDesc = "make"
    doAction = 'Make'
;
moveVerb: DeepVerb
    verb = 'move'
    sDesc = "move"
    ioAction(withPrep) = 'MoveWith'
    ioAction(toPrep) = 'MoveTo'
    doAction = 'Move'
;
moveEVerb: DeepVerb
    verb = 'move east' 'move e' 'push east' 'push e'
    sDesc = "move east"
    doAction = 'MoveE'
;
moveNVerb: DeepVerb
    verb = 'move north' 'move n' 'push north' 'push n'
    sDesc = "move north"
    doAction = 'MoveN'
;
moveNEVerb: DeepVerb
    verb = 'move northeast' 'move ne' 'push northeast' 'push ne'
    sDesc = "move northeast"
    doAction = 'MoveNE'
;
moveNWVerb: DeepVerb
    verb = 'move northwest' 'move nw' 'push northwest' 'push nw'
    sDesc = "move northwest"
    doAction = 'MoveNW'
;
moveSVerb: DeepVerb
    verb = 'move south' 'move s' 'push south' 'push s'
    sDesc = "move south"
    doAction = 'MoveS'
;
moveSEVerb: DeepVerb
    verb = 'move southeast' 'move se' 'push southeast' 'push se'
    sDesc = "move southeast"
    doAction = 'MoveSE'
;
moveSWVerb: DeepVerb
    verb = 'move southwest' 'move sw' 'push southwest' 'push sw'
    sDesc = "move southwest"
    doAction = 'MoveSW'
;
moveWVerb: DeepVerb
    verb = 'move west' 'move w' 'push west' 'push w'
    sDesc = "move west"
    doAction = 'MoveW'
;
openVerb: DeepVerb
    verb = 'open'
    sDesc = "open"
    doAction = 'Open'
;
payVerb: DeepVerb
    verb = 'pay'
    sDesc = "pay"
    doAction = 'Pay'
    ioAction(forPrep) = 'PayFor'
;
payforVerb: DeepVerb
    verb = 'pay for'
    sDesc = "pay for"
    doAction = 'Payfor'
;
plugVerb: DeepVerb
    verb = 'plug'
    sDesc = "plug"
    prepDefault = inPrep
    ioAction(inPrep) = 'PlugIn'
;
pokeVerb: DeepVerb
    verb = 'poke' 'jab'
    sDesc = "poke"
    doAction = 'Poke'
;
pourVerb: DeepVerb
    verb = 'pour'
    sDesc = "pour"
    doAction = 'Pour'
    ioAction( inPrep ) = 'PourIn'
    ioAction( onPrep ) = 'PourOn'
;
pullVerb: DeepVerb
    verb = 'pull'
    sDesc = "pull"
    doAction = 'Pull'
;
pushVerb: DeepVerb
    verb = 'push' 'press'
    sDesc = "push"
    doAction = 'Push'
;
putVerb: DeepVerb
    verb = 'put' 'place'
    sDesc = "put"
    prepDefault = inPrep
    ioAction(inPrep) = 'PutIn'
    ioAction(onPrep) = 'PutOn'
    doDefault(actor, prep, io) = {
        return (takeVerb.doDefault(actor, prep, io) + actor.contents);
    }
;
raiseVerb: DeepVerb
    verb = 'raise'
    sDesc = "raise"
    doAction = 'Raise'
;
readVerb: DeepVerb
    verb = 'read'
    sDesc = "read"
    doAction = 'Read'
;
removeVerb: DeepVerb
    verb = 'take off' 'doff'
    sDesc = "take off"
    doAction = 'Unwear'
    ioAction(fromPrep) = 'RemoveFrom'
;
sayVerb: DeepVerb
    verb = 'say'
    sDesc = "say"
    doAction = 'Say'
;
screwVerb: DeepVerb
    verb = 'screw'
    sDesc = "screw"
    ioAction(withPrep) = 'ScrewWith'
    doAction = 'Screw'
;
searchVerb: DeepVerb
    verb = 'search'
    sDesc = "search"
    doAction = 'Search'
;
showVerb: DeepVerb
    verb = 'show'
    sDesc = "show"
    prepDefault = toPrep
    ioAction(toPrep) = 'ShowTo'
    doDefault(actor, prep, io) = {
        return actor.contents;
    }
;
sitVerb: DeepVerb
    verb = 'sit on' 'sit in' 'sit' 'sit down' 'sit downin' 'sit downon'
    sDesc = "sit on"
    doAction = 'Siton'
;
smellVerb: DeepVerb
    doSense = smell
    verb = 'smell'
    sDesc = "smell"
    doAction = 'Smell'
    action(actor) = {
        gDobj(actor.location);
        gDoList([actor.location]);
        actor.location.doSmell(actor);
    }
;
standVerb: DeepVerb
    verb = 'stand' 'stand up' 'get up'
    sDesc = "stand"
    outhideStatus = 0
    action(actor) = {
        if (actor.location == nil || actor.location.location == nil)
            "%You're% already standing! ";
        else {
            /* 
             *   silently check verDoUnboard, to see if we can get out of
             *   whatever object we're in 
             */
            self.outhideStatus = outhide(true);
            actor.location.verDoUnboard(actor);
            if (outhide(self.outhideStatus)) {
                /* 
                 *   verification failed - run again, showing the message
                 *   this time 
                 */
                actor.location.verDoUnboard(actor);
            } else {
                /* verification succeeded - unboard */
                actor.location.doUnboard(actor);
            }
        }
    }
;
standInVerb: DeepVerb
    verb = 'stand in'
    sDesc = "stand in"
    doAction = 'Standin'
;
standOnVerb: DeepVerb
    verb = 'stand on'
    sDesc = "stand on"
    doAction = 'Standon'
;
switchVerb: DeepVerb
    verb = 'switch'
    sDesc = "switch"
    doAction = 'Switch'
;
takeVerb: DeepVerb                   // This object defines how to take things
    verb = 'take' 'pick up' 'get' 'remove'
    sDesc = "take"
    ioAction(offPrep) = 'TakeOff'
    ioAction(outPrep) = 'TakeOut'
    ioAction(fromPrep) = 'TakeOut'
    ioAction(inPrep) = 'TakeOut'
    ioAction(onPrep) = 'TakeOff'
    doAction = 'Take'
    doDefault(actor, prep, io) = {
        local ret, rem, cur, rem2, cur2, tot, i, tot2, j;
        
        ret = [];
        
        /*
         *   For "take all out/off of <iobj>", return the (non-fixed)
         *   contents of the indirect object.  Same goes for "take all in
         *   <iobj>", "take all on <iobj>", and "take all from <iobj>".
         */
        if ((prep == outPrep || prep == offPrep || prep == inPrep
             || prep == onPrep || prep == fromPrep)
            && io != nil) {
            rem = io.contents;
            i = 1;
            tot = length(rem);
            while (i <= tot) {
                cur = rem[i];
                if (!cur.isFixed && self.validDo(actor, cur, i))
                    ret += cur;
                ++i;
            }
            return ret;
        }
        
        /*
         *   In the general case, return everything that's not fixed
         *   in the actor's location, or everything inside fixed containers
         *   that isn't itself fixed.
         */
        rem = actor.location.contents;
        tot = length(rem);
        i = 1;
        while (i <= tot) {
            cur = rem[i];
            if (cur.isFixed) {
                if (((cur.isOpenable && cur.isOpen) || (!cur.isOpenable))
                && (!cur.isAnimate)) {
                    rem2 = cur.contents;
                    tot2 = length(rem2);
                    j = 1;
                    while (j <= tot2) {
                        cur2 = rem2[j];
                        if (!cur2.isFixed && !cur2.noTakeAll){
                            ret = ret + cur2;
                        }
                        j = j + 1;
                    }
                }
            } else if (!cur.noTakeAll) {
                ret = ret + cur;
            }
            
            i = i + 1;            
        }
        return ret;
    }
;
tasteVerb: DeepVerb
    verb = 'taste'
    sDesc = "taste"
    doAction = 'Taste'
;
tellVerb: DeepVerb
    verb = 'tell'
    sDesc = "tell"
    prepDefault = aboutPrep
    ioAction(aboutPrep) = 'TellAbout'
    validIo(actor, obj, seqno) = { return (seqno == 1); }
    validIoList(actor, prep, dobj) = (nil)
    ioDefault(actor, prep) = {
        if (prep == aboutPrep)
            return [];
        else
            return (actor.contents + actor.location.contents);
    }
    validIoSense(actor, obj, seqno) = {
        return true;
    }
;
throwVerb: DeepVerb
    verb = 'throw' 'toss'
    sDesc = "throw"
    prepDefault = atPrep
    ioAction(atPrep) = 'ThrowAt'
    ioAction(toPrep) = 'ThrowTo'
;
tieVerb: DeepVerb
    verb = 'tie'
    sDesc = "tie"
    prepDefault = toPrep
    ioAction(toPrep) = 'TieTo'
;
touchVerb: DeepVerb
    verb = 'touch'
    sDesc = "touch"
    doAction = 'Touch'
;
turnVerb: DeepVerb
    verb = 'turn' 'rotate' 'twist'
    sDesc = "turn"
    ioAction(toPrep) = 'TurnTo'
    ioAction(withPrep) = 'TurnWith'
    doAction = 'Turn'
;
turnOffVerb: DeepVerb
    verb = 'turn off' 'deactiv' 'switch off'
    sDesc = "turn off"
    doAction = 'Turnoff'
;
typeVerb: DeepVerb
    verb = 'type'
    sDesc = "type"
    prepDefault = onPrep
    ioAction(onPrep) = 'TypeOn'
;
unfastenVerb: DeepVerb
    verb = 'unfasten' 'unbuckle'
    sDesc = "unfasten"
    doAction = 'Unfasten'
;
unlockVerb: DeepVerb
    verb = 'unlock'
    sDesc = "unlock"
    ioAction(withPrep) = 'UnlockWith'
    doAction = 'Unlock'
    prepDefault = withPrep
;
unplugVerb: DeepVerb
    verb = 'unplug'
    sDesc = "unplug"
    ioAction(fromPrep) = 'UnplugFrom'
    doAction = 'Unplug'
;
unscrewVerb: DeepVerb
    verb = 'unscrew'
    sDesc = "unscrew"
    ioAction(withPrep) = 'UnscrewWith'
    doAction = 'Unscrew'
;
wearVerb: DeepVerb
    verb = 'wear' 'put on' 'don'
    sDesc = "wear"
    doAction = 'Wear'
;
yellVerb: DeepVerb
    verb = 'yell' 'shout' 'yell at' 'shout at'
    sDesc = "yell at"
    action(actor) = {
        "%Your% throat is a bit sore now. ";
    }
;

#pragma C-

#endif /* _DEEP_VERB_H_ */
