////////////////////////////////////////////////////////////////////////
//  
//  ALT Library File: Thing 010123
//
//  Copyright (c) 2000, 2001 Kevin Forchione. All rights reserved.
//  Based on ADV.T (c) and STD.T (c) Michael Roberts.
//
//  This file is part of the ALT replacement library for ADV.T and 
//  STD.T and requires TADS 2.5.1 or later.
//
////////////////////////////////////////////////////////////////////////

#ifndef _THING_H_
#define _THING_H_

#include <object.h>
#include <addbulk.t>
#include <addweight.t>
#include <queue.t>

#pragma C+

/*
 *  Thing: Object
 *
 *  The basic class for objects in a game.  
 *  
 *  The aDesc method displays the name of the object with an indefinite
 *  article; the default is to display "a" followed by the sDesc,
 *  but objects that need a different indefinite article (such as "an"
 *  or "some") should override this method.  Likewise, theDesc
 *  displays the name with a definite article; by default, theDesc
 *  displays "the" followed by the object's sDesc.  The sDesc
 *  simply displays the object's name ("short description") without
 *  any articles.  The lDesc is the long description, normally
 *  displayed when the object is examined by the player; by default,
 *  the lDesc displays "It looks like an ordinary sDesc."
 */
class Thing: Object
    bulk   = 1
    weight = 0

    /* shows up in room/inventory listings */
    isListed = true  
    
    /* this location has been visited by the PC */
    isSeen = nil
    
    /* this object's initial posture */
    posture = resting 
    
    /* default actor posture when boarding this object */
    boardingPosture = standing   
    
    /* default actor posture when unboarding this object */
    unboardingPosture = standing
    
    /* 
     *  default preposition used when the actor has boarded this 
     *  object.
     */
    boardingPrep = "on"
    
    /*
     *  default preoposition used when the actor is unboarding this
     *  object.
     */
    unboardingPrep = "off of"   
    
    /*** DESCRIPTION METHODS ***/

    /*
     *  sdesc, adesc, thedesc, pluraldesc, multisdesc, and prefixdesc
     *  are all used as parser hooks for their Alt equivalents.
     */
    sdesc = {self.sDesc;}
    adesc = {self.aDesc;}
    thedesc = {self.theDesc;}
    pluraldesc = {self.pluralDesc;}
    multisdesc = {self.multiSDesc;}
    prefixdesc(show, currentIndex, count, multiFlags) = {
        return self.prefixDesc(show, currentIndex, count, multiFlags);
    }
    
    aDesc = {
        "a "; self.sDesc;   // default is "a <name>"; "self" is current object
    }
    theDesc = {
        "the "; self.sDesc; // default is "the <name>"
    }
    pluralDesc = {          // default is to add "s" to the sDesc
        self.sDesc;
        if (!self.isThem)
            "s";
    }
    invDesc = ""
    
    /* ListGroup Descriptions */
    groupDesc = {self.sDesc;}
    grouppluralDesc = {self.pluralDesc;}
    
    itNomDesc = {
        /* 
         *   display "it" in the nominative case: "it" if this is a
         *   singular object, "they" otherwise 
         */
        if (self.isThem)
            "they";
        else
            "it";
    }
    itObjDesc = {
        /* 
         *   display "it" in the objective case: "it" for singular, "them"
         *   otherwise 
         */
        if (self.isThem)
            "them";
        else
            "it";
    }
    thatDesc = {
        /* display "that" if this is singular, "them" otherwise */
        if (self.isThem)
            "those";
        else
            "that";
    }
    itselfDesc = {
        /* display "itself" if this is singular, "themselves" otherwise */
        if (self.isThem)
            "themselves";
        else
            "itself";
    }
    itIsDesc = {
        /* display "it's" if this is singular, "they're" otherwise */
        if (self.isThem)
            "they're";
        else
            "it's";
    }
    doesDesc = {
        /* display "does" if singular, "do" otherwise (it does, they do) */
        if (self == parserGetMe() || self.isThem)
            "do";
        else
            "does";
    }
    doesntDesc = {
        /* display "does" if singular, "do" otherwise (it does, they do) */
        if (self == parserGetMe() || self.isThem)
            "don't";
        else
            "doesn't";
    }
    isDesc = {
        /* display "is" if this is singular, "are" otherwise */
        if (self == parserGetMe() || self.isThem)
            "are";
        else
            "is";
    }
    isntDesc = {
        /* display "isn't" if this is singular, "aren't" otherwise */
        if (self == parserGetMe() || self.isThem)
            "aren't";
        else
            "isn't";
    }
    
    multiSDesc = {self.sDesc;}
    prefixDesc(show, currentIndex, count, multiFlags) = {
        if (global.isCommandPhase(CP_EXECUTION)) {
            global.setPrefixShow(show);
            nestOutcapture(true);
        }
    }
    
    lDesc = { "_It_ look_s_ like _an_ ordinary <<self.sDesc>> to %me%. ";}
    listenDesc = {"%You% hear%s% nothing unusual from _dobj_. ";}
    readDesc = {"%You% can't read <<self.aDesc>>. ";}
    smellDesc = {"%You% smell%s% nothing unusual about 
        _dobj_. ";}
    thruDesc = {"%You% can't see much through _dobj_.\n";}
    tasteDesc = {"%You% taste%s% nothing unexpected. ";}
    touchDesc = {"Touching _dobj_ doesn't seem to have any effect. ";}

    verifyRemove(actor) = {
        /*
         *   Check with each container to make sure that the container
         *   doesn't object to the object's removal.
         */
        local loc;

        loc = self.location;
        while (loc) {
            if (loc != actor)
                loc.verGrab(self);
            loc = loc.location;
        }
    }
    
    verGrab(obj) = {}

    /*** VERIFICATION AND ACTION METHODS ***/

    /*
     *   Make it so that the player can give a command to an actor only
     *   if an actor is reachable in the normal manner.  This method
     *   returns true when 'self' can be given a command by the player. 
     */
    validActor = (self.canSenseObj(sound, parserGetMe()))

    actorAction(v, d, p, i) = {
        "You have lost your mind. ";
        exit;
    }
    
    circularMessage(io) = {
        local cont, prep;

        /* set prep to 'on' if io.location is a Surface, 'in' otherwise */
        prep = (io.location.isSurface ? 'on' : 'in');

        /* tell them about the problem */
        "%You% can't put <<theDesc>> <<prep>> <<io.theDesc>>, because <<
            io.theDesc>> <<io.isDesc>> <<
            io.location == self ? "already" : "">> <<prep>> <<
            io.location.theDesc>>";
        for (cont = io.location ; cont != self ; cont = cont.location) {
            ", which <<cont.isDesc>> <<
                cont.location == self ? "already" : "">> <<prep>> <<
                cont.location.theDesc>>";
        }
        ". ";
    }

    verDoAskAbout(actor, io) = {
        "Surely, %you% can't think _dobj_ know_s_ anything
        about it! ";
    }
    verIoAskAbout(actor) = {}
    ioAskAbout(actor, dobj) = {
        dobj.doAskAbout(actor, self);
    }

    verDoAttachTo(actor, iobj) = { "There's no obvious way to do that."; }
    verIoAttachTo(actor) = { "There's no obvious way to do that."; }

    verDoAttackWith(actor, io) = {
        "Attacking _dobj_ doesn't appear productive. ";
    }
    verIoAttackWith(actor) = {
        "It's not very effective to attack with <<self.theDesc>>. ";
    }

    verDoBreak(actor) = {}
    doBreak(actor) = {
        "You'll have to tell me how to do that.";
    }

    verDoClean(actor) = {
        "_Dobj_ look_s_ a bit cleaner now. ";
    }

    verDoCleanWith(actor, io) = {}
    doCleanWith(actor, io) = {
        "_Dobj_ look_s_ a bit cleaner now. ";
    }

    verDoDetach(actor) = { "There's no obvious way to do that."; }
    verDoDetachFrom(actor, iobj) = { "There's no obvious way to do that."; }
    verIoDetachFrom(actor) = { "There's no obvious way to do that."; }

    verDoDrink(actor) = {
        "_Dobj_ _does_n't appear appetizing. ";
    }

    verDoDrop(actor) = {
        if (!actor.isCarrying(self)) {
            "%You're% not carrying _dobj_! ";
        }
        else self.verifyRemove(actor);
    }
    doDrop(actor) = {
        actor.location.roomDrop(self);
    }

    verDoEat(actor) = {
        "_Dobj_ _does_n't appear appetizing. ";
    }

    verDoGiveTo(actor, io) = {
        if (!actor.isCarrying(self)) {
            "%You're% not carrying _dobj_. ";
        } else self.verifyRemove(actor);
    }
    doGiveTo(actor, io) = {
        self.moveInto(io);
        "Done. ";
    }

    verDoInspect(actor) = {}
    doInspect(actor) = {
        self.lDesc;
    }
    
    verDoListento(actor) = {}
    doListento(actor) = {
        self.listenDesc;
    }

    verDoLookbehind(actor) = {
        "There's nothing behind _dobj_. ";
    }

    verDoLookin(actor) = {
        "There's nothing in _dobj_. ";
    }

    verDoLookthru(actor) = {
        "%You% can't see anything through _dobj_. ";
    }

    verDoLookunder(actor) = {
        "There's nothing under _dobj_. ";
    }

    verDoMove(actor) = {
        "Moving _dobj_ doesn't reveal anything. ";
    }
    
    verDoMoveTo(actor, io) = {
        "Moving _dobj_ doesn't reveal anything. ";
    }
    verIoMoveTo(actor) = {
        "That doesn't get us anywhere. ";
    }
    
    verDoMoveWith(actor, io) = {
        "Moving _dobj_ doesn't reveal anything. ";
    }
    verIoMoveWith(actor) = {
        caps(); self.theDesc; " <<self.doesDesc>>n't seem to help. ";
    }
    
    verDoMoveN(actor) = { self.genMoveDir; }

    verDoMoveS(actor) = { self.genMoveDir; }

    verDoMoveE(actor) = { self.genMoveDir; }

    verDoMoveW(actor) = { self.genMoveDir; }

    verDoMoveNE(actor) = { self.genMoveDir; }

    verDoMoveNW(actor) = { self.genMoveDir; }

    verDoMoveSE(actor) = { self.genMoveDir; }

    verDoMoveSW(actor) = { self.genMoveDir; }

    genMoveDir = { "%You% can't seem to do that. "; }

    verDoPlugIn(actor, io) = {
        "%You% can't plug _dobj_ into anything. ";
    }
    verIoPlugIn(actor) = {
        "%You% can't plug anything into "; self.theDesc; ". ";
    }

    verDoPoke(actor) = {
        "Poking _dobj_ doesn't seem to have any effect. ";
    }

    verDoPull(actor) = {
        "Pulling _dobj_ doesn't have any effect. ";
    }

    verDoPush(actor) = {
        "Pushing _dobj_ doesn't do anything. ";
    }

    verDoPutIn(actor, io) = {
        if (io == nil)
            return;

        if (self.location == io) {
            "_Dobj_ _is_ already in <<io.theDesc>>! ";
        } else if (io == self) {
            "%You% can't put _dobj_ in <<self.itselfDesc>>! ";
        } else if (io.isIn(self))
            self.circularMessage(io);
        else
            self.verifyRemove(actor);
    }
    verIoPutIn(actor) = {
        "%You% can't put anything into <<self.theDesc>>. ";
    }
    doPutIn(actor, io) = {
        self.moveInto(io);
        "Done. ";
    }

    verDoPutOn(actor, io) = {
        if (io == nil)
            return;

        if (self.location == io) {
            "_Dobj_ _is_ already on <<io.theDesc>>! ";
        } else if (io == self) {
            "%You% can't put _dobj_ on <<self.itselfDesc>>! ";
        } else if (io.isIn(self))
            self.circularMessage(io);
        else
            self.verifyRemove(actor);
    }
    verIoPutOn(actor) = {
        "There's no good surface on <<self.theDesc>>. ";
    }
    doPutOn(actor, io) = {
        self.moveInto(io);
        "Done. ";
    }

    verDoRead(actor) = {
        "I don't know how to read _dobj_. ";
    }

    verDoRestore(actor) = {
        "Please specify the name of the game to restore in double quotes,
        for example, RESTORE \"GAME1\". ";
    }

    verDoSave(actor) = {
        "Please specify the name of the game to save in double quotes,
        for example, SAVE \"GAME1\". ";
    }

    verDoSay(actor) = {
        "You should say what you want to say in double quotes, for example,
        SAY \"HELLO\". ";
    }

    verDoScrew(actor) = { "You see no way to do that."; }

    verDoScrewWith(actor, iobj) = { "You see no way to do that."; }
    verIoScrewWith(actor) = { "You can't use _dobj_ like that."; }

    verDoScript(actor) = {
        "You should type the name of a file to write the transcript to
        in quotes, for example, SCRIPT \"LOG1\". ";
    }

    verDoSearch(actor) = {
        "%You% find%s% nothing of interest. ";
    }

    verDoShowTo(actor, io) = {}
    verIoShowTo(actor) = {
        caps(); self.theDesc; " "; self.isntDesc; " impressed. ";
    }
    doShowTo(actor, io) = {
        if (io != nil) {
            caps(); io.theDesc; " "; io.isntDesc; " impressed. ";
        }
    }
    
    verDoSmell(actor) = {}
    doSmell(actor) = {
        self.smellDesc;
    }

    verDoTake(actor) = {
        if (self.location == actor) {
            "%You% already %have% _dobj_! ";
        }
        else
            self.verifyRemove(actor);
    }
    doTake(actor) = {
        local totBulk, totWeight;

        totBulk = addBulk(actor.contents) + self.bulk;
        totWeight = addWeight(actor.contents);
        if (!actor.isCarrying(self))
            totWeight = totWeight + self.weight + addWeight(self.contents);

        if (totWeight > actor.maxWeight) {
            "%Your% load is too heavy. ";
            exitobj;
        }
        else if (totBulk > actor.maxBulk) {           
        	    "%You've% already got %your% hands full. ";
        	    exitobj;
        } else {
            self.moveInto(actor);
            "Taken. ";
        }
    }

    verDoTakeOff(actor, io) = {
        if (io != nil && !self.isIn(io)) {
            "_Dobj_ _isn't_ on <<io.theDesc>>! ";
        }
        self.verDoTake(actor);         /* ensure object can be taken at all */
    }
    verIoTakeOff(actor) = {}
    ioTakeOff(actor, dobj) = {
        dobj.doTakeOff(actor, self);
    }
    doTakeOff(actor, io) = {
        self.doTake(actor);
    }

    verDoTakeOut(actor, io) = {
        if (io != nil && !self.isIn(io)) {
            "_Dobj_ _isn't_ in <<io.theDesc>>. ";
        } else
            self.verDoTake(actor);     /* ensure object can be taken at all */
    }
    verIoTakeOut(actor) = {}
    ioTakeOut(actor, dobj) = {
        dobj.doTakeOut(actor, self);
    }
    doTakeOut(actor, io) = {
        self.doTake(actor);
    }
    
    verDoTaste(actor) = {}
    doTaste(actor) = {
        self.tasteDesc;
    }

    verDoTellAbout(actor, io) = {
        "It doesn't look as though _dobj_ _is_ interested. ";
    }
    verIoTellAbout(actor) = {}
    ioTellAbout(actor, dobj) = {
        dobj.doTellAbout(actor, self);
    }

   verDoThrow(actor) = {
        if (!actor.isCarrying(self)) {
            "%You're% not carrying _dobj_. ";
        } else self.verifyRemove(actor);
    }
    doThrow(actor) = {
        "Thrown. ";
        self.moveInto(actor.location);
    }

    verDoThrowAt(actor, io) = {
        if (!actor.isCarrying(self)) {
            "%You're% not carrying _dobj_. ";
        } else self.verifyRemove(actor);
    }
    verIoThrowAt(actor) = {
        if (actor.isCarrying(self)) {
            "%You% could at least drop _dobj_ first. ";
        }
    }
    ioThrowAt(actor, dobj) = {
        dobj.doThrowAt(actor, self);
    }
    doThrowAt(actor, io) = {
        "%You% miss%es%. ";
        self.moveInto(actor.location);
    }

    verDoThrowTo(actor, io) = {
        if (!actor.isCarrying(self)) {
            "%You're% not carrying _dobj_. ";
        } else self.verifyRemove(actor);
    }
    doThrowTo(actor, io) = {
        "%You% miss%es%. ";
        self.moveInto(actor.location);
    }
 
    verDoTouch(actor) = {}
    doTouch(actor) = {
        self.touchDesc;
    }

    verDoTurn(actor) = {
        "Turning _dobj_ doesn't have any effect. ";
    }
    
    verDoTurnoff(actor) = {
        "I don't know how to turn _dobj_ off. ";
    }

    verDoTurnon(actor) = {
        "I don't know how to turn _dobj_ on. ";
    }

    verDoTurnTo(actor, io) = {
        "Turning _dobj_ doesn't have any effect. ";
    }
    verIoTurnTo(actor) = {
        "I don't know how to do that. ";
    }

    verDoTurnWith(actor, io) = {
        "Turning _dobj_ doesn't have any effect. ";
    }

    verDoTypeOn(actor, io) = {
        "You should say what you want to type in double quotes, for
        example, TYPE \"HELLO\" ON KEYBOARD. ";
    }

    verDoUnboard(actor) = {
        if (actor.location != self) {
            "%You're% not <<self.boardingPrep>> _dobj_! ";
        } else if (self.location == nil) {
            "%You% can't leave _dobj_! ";
        }
    }
    doUnboard(actor) = {
        if (self.fastenItem) {
            "%You%'ll have to unfasten <<actor.location.fastenItem.theDesc
            >> first. ";
        } else {
            "Okay, %you're% no longer <<actor.posture.desc>> 
            <<self.boardingPrep>> <<self.thedesc>>. ";
            self.leaveRoom(actor);
            actor.moveInto(self.location);
            actor.posture = standing;
        }
    }

    verDoUnplugFrom(actor, io) = {
        if (io != nil)
            { "\^<<self.itIsDesc>> not plugged into "; io.theDesc; ". "; }
    }
    verIoUnplugFrom(actor) = {
        "It's not plugged into "; self.theDesc; ". ";
    }

    verDoUnscrew(actor) = { "You see no way to do that."; }

    verDoUnscrewWith(actor, iobj) = { "You see no way to do that."; }
    verIoUnscrewWith(actor) = { "You can't use _dobj_ like that."; }

    verDoUnwear(actor) = {
        "%You're% not wearing _dobj_! ";
    }

    verDoWear(actor) = {
        "%You% can't wear _dobj_. ";
    }
;

#pragma C-

#endif /* _THING_H_ */
