////////////////////////////////////////////////////////////////////////
//  
//  ALT Library File: Object 010123
//
//  Copyright (c) 2000, 2001 Kevin Forchione. All rights reserved.
//  Based on ADV.T (c) and STD.T (c) Michael Roberts.
//
//  This file is part of the ALT replacement library for ADV.T and 
//  STD.T and requires TADS 2.5.1 or later.
//
////////////////////////////////////////////////////////////////////////

#ifndef _OBJECT_H_
#define _OBJECT_H_

#pragma C+

/*
 *  Object: object
 *
 *  The Object class defines the base level for containment, manipulation,
 *  creation, and destruction of game objects. 
 *  
 *  The property contents
 *  is a list that specifies what is in the object; this property is
 *  automatically set up by the system after the game is compiled to
 *  contain a list of all objects that have this object as their
 *  location property.  
 *
 *  The contents property is kept
 *  consistent with the location properties of referenced objects
 *  by the moveInto method; always use moveInto rather than
 *  directly setting a location property for this reason.  
 *
 *  The isIn(object) method returns true if the
 *  object's location is the specified object or the object's
 *  location isIn(object) method is
 *  true.  Note that if isIn() is true, it doesn't
 *  necessarily mean the object is visible or reachable, because 
 *  isIn() is true if the object is merely within the location.
 *
 *  The moveInto(object) method moves the object to be inside
 *  the specified object.  To make an object disappear, move it
 *  into nil.
 */
class Object: object
    contents = []           // set up automatically by system - do not set
    
    /* this object has been moved by the PC */
    hasMoved = nil
    
    /* count of how many times this object has been moved by the PC */
    hasMovedCount = 0

    moveInto(newLoc) = {
        local loc;

        /*
         *   For the object containing me, and its container, and so forth,
         *   tell it via a grab message that I'm going away.
         */
        loc = self.location;
        while (loc) {
            loc.grab(self);
            loc = loc.location;
        }
        
        if (self.location)
            self.location.removeFromContents(self);
            
        self.location = newLoc;
        
        /*
         *  set the hasMoved attribute to true and increment the
         *  hasMovedCount.
         */
        self.hasMoved = true;
        ++self.hasMovedCount;            

        if (self.location)
            self.location.addToContents(self);
    }
    addToContents(obj) = {
        self.contents += obj;
    }
    removeFromContents(obj) = {
        self.contents -= obj;
    }
    grab(obj) = {}

    isIn(obj) = {
        /* if obj is my immediate container, I'm obviously in it */
        if (location == obj)
            return true;

        /* if I have no location, I'm obviously not in obj */
        if (location == nil)
            return nil;

        /* I'm in obj if my container is in obj */
        return location.isIn(obj);
    }
    /* 
     * A dummy parser hook. isVisible() has been superceded by the
     * scope() methods.
     */
    isVisible(vantage) = (true)
    
    material = altima
    
    /*
     *  Returns a list of locations. If object has no location then the
     *  method returns an empty list.
     */
    getLocationsList = {
        if (self.location == nil)
            return [];
        else
            return [ self.location ];
    }
    /*
     *  Returns a list of object contents. If the object has no contents
     *  then the method returns an empty list.
     */
    getContentsList = {return self.contents;}
    
    canSensePreferred(prefSense, target) = {
        local i, len, s = global.sensesList;
        
        if (prefSense) {
            if (self.canSenseObj(prefSense, target)) {
                target.sensedByPreferred = true;
                return true;
            }
            s -= prefSense;
        }
        
        len = length(s);
        for (i = 1; i <= len; i++) {
            if (self.canSenseObj(s[i], target)) {
                target.sensedByPreferred = nil;
                return true;
            }
        }
        return nil;
    }
    canSenseObj(sense, target) = {
        local path;
        path = self.scope(sense, [], target, nil);
        if (find(path, target))
            return true;
        else
            return nil;
    }
    
    isLit(vantage) = {
        local ret;
        
        /* if we're providing light return true */
        if (self.lightsOn)
            return true;
        
        /* initialise the callback */
        lightingCallback.initialize;
        
        ret = vantage.scope(sight, [], nil, lightingCallback);
        
        /* callback holds the lighting status */
        return lightingCallback.lightFound;
    }

    /*
     *  Returns a path list of all objects accessible for the given
     *  sense.
     */
    scope(sense, path, target, callbackObj) = {
        /* don't look at object if it's in the path already */
        if (find(path, self) != nil) 
            return path;
        
        /* if we find the target look no further */
        if (find(path, target) != nil)
            return path;
               
        /* Check with callback */
        if (callbackObj && !callbackObj.main(self))
            return path;
              
        /* add self to the path */
        path += self;
        
        /* work "up" location */
        path = self.scopeLocations(sense, path, target, callbackObj);
        
        /* work "down" contents */
        path = self.scopeContents(sense, path, target, callbackObj);
    
        return path;
    }
    scopeLocations(sense, path, target, callbackObj) = {
        local i, len, l;
                
        if (sense == nil 
        || self.canSenseLocations(sense, car(path) == self)) {
            l = self.getLocationsList;
            len = length(l);
            for (i = 1; i <= len; i++) {
                local j, c, len2;
                
                /*
                 *  If we find this location object already in the 
                 *  path we evaluate the next location object.
                 */
                if (find(path, l[i])) 
                    continue;
                    
                path = l[i].scope(sense, path, target, callbackObj);
                
                /*
                 *  iterate over the contents of each location in order
                 *  to process the siblings of an object. This ensures
                 *  that we cover all the objects in a given location
                 *  when we've come into a location via an object, but 
                 *  the location doesn't allow access to its contents.
                 */
                c = l[i].contents;
                len2 = length(c);
                for (j = 1; j <= len2; ++j) {
                    /*
                     *  If we find this content object already in the
                     *  path we evaluate the next content object.
                     */
                    if (find(path, c[j])) 
                        continue;
                        
                    path = c[j].scope(sense, path, target, callbackObj);
                }
            }
                
        } else if (target) {
            if (self.senseFromBelow(nil, path, target)) {
                target.obstructor = self;
                target.obstructedSense = sense;
            }
        }
        return path;
    }
    senseFromAbove(sense, path, target) = {
        if (target) {
            local i, c = [], len, tmpPath = [];
            c = self.getContentsList;
            len = length(c);
            for (i = 1; i <= len; i++)
                tmpPath = c[i].scope(nil, path, target, nil);
            if (find(tmpPath, target) != nil)
                return true;
        }
        return nil;
    }
    
    scopeContents(sense, path, target, callbackObj) = {
        local i, len, c = [];
        
        if (sense == nil 
        || self.canSenseContents(sense, car(path) == self)) {
            c = self.getContentsList;
            len = length(c);
            for (i = 1; i <= len; i++) {
                /*
                 *  If we find this content object already in the
                 *  path we evaluate the next content object.
                 */
                if (find(path, c[i]) != nil)
                    continue;
                    
                path = c[i].scope(sense, path, target, callbackObj);
            }
        } else if (target) {
            if (self.senseFromAbove(nil, path, target)) {
                target.obstructor = self;
                target.obstructedSense = sense;
            }
        }
        return path;
    }
    
    senseFromBelow(sense, path, target) = {
        if (target) {
            local i, l = [], len, tmpPath = [];
            l = self.getLocationsList;
            len = length(l);
            for (i = 1; i <= len; i++) 
                tmpPath = l[i].scope(sense, path, target, nil);
            if (find(tmpPath, target) != nil)
                return true;
        }
        return nil;
    }
    
    /*
     *  Does the object allow its location(s) to be searched? In this
     *  case we simply default to canSenseContents.
     */
    canSenseLocations(sense, isVantage) = {
        return self.canSenseContents(sense, isVantage);
    }
    
    /*
     *  Does the object allow its contents to be sensed?
     */
    canSenseContents(sense, isVantage) = {return true;}
  
    /*
     *  cantSense methods are at the object level. They are passed the
     *  actor and the word list. 
     */
    cantSee(vantage, target) = {
        local i;
        
        "%You% can't see any ";
        for (i = 1; i <= length(global.cantSenseWords); ++i)
        {
            say(global.cantSenseWords[i]);
            " ";
        }
        "here.";
    }        
    cantHear(vantage, target) = {        
        if (vantage.canSenseObj(sight, target))
            "%You% can't hear that here.";
        else self.cantSee(vantage, target);
    }        
    cantSmell(vantage, target) = {
        if (vantage.canSenseObj(sight, target))
            "%You% can't smell that from here.";
        else self.cantSee(vantage, target);
    }        
    cantTouch(vantage, target) = {
        if (vantage.canSenseObj(sight, target))
            "%You% can't touch that from here. ";
        else self.cantSee(vantage, target);
    }

    /*** COMPOSITE / COMPONENT ATTRIBUTES AND METHODS ***/

    componentParent 	= nil
    componentClassList 	= []
    componentObjList 	= []
    
    disambigIobj(actor, prep, dobj, verProp, actionProp, wordList, objList,
               flagList, numberWanted, isAmbiguous, silent) = 
    {
		return [];
    }
    
    disambigDobj(actor, prep, iobj, verProp, actionProp, wordList, objList,
               flagList, numberWanted, isAmbiguous, silent) = 
    {
		return [];
    }

    
    /*** DYNAMIC OBJECT CONSTRUCT & DESTRUCT METHODS ***/

    /* on dynamic construction, move into my contents list */
    construct = {
        self.moveInto(location);
    }

    /* on dynamic destruction, move out of contents list */
    destruct = {
        self.moveInto(nil);
    }
;

#pragma C-

#endif /* _Object_H_ */
