////////////////////////////////////////////////////////////////////////
//  
//  ALT Library File: Floating 010123
//
//  Copyright (c) 2000, 2001 Kevin Forchione. All rights reserved.
//  Based on ADV.T (c) and STD.T (c) Michael Roberts.
//
//  This file is part of the ALT replacement library for ADV.T and 
//  STD.T and requires TADS 2.5.1 or later.
//
////////////////////////////////////////////////////////////////////////

#ifndef _FLOATING_H_
#define _FLOATING_H_

#pragma C+

/*----------------------------------------------------------------------
 *  MIX-IN CLASS:   This class must always preced any non-mix-in classes
 *                  in a superclass list - this is required because a
 *                  mix-in can't override default methods in the root of
 *                  the hierarchy.
 *--------------------------------------------------------------------*/

/*
 *  Floating: object
 *
 *  Floating objects are moved to their locations at various points in
 *  the command parsing and execution phases. They are also re-evaluated
 *  when the actor moves.
 *
 *  A Floating object has only one location at any given time, but this
 *  location is dependent upon the object's foundIn property's returned
 *  list of locations with respect to the actor's location. 
 *
 *  If, as we move recursively up from the actor's location, we find
 *  that one of them matches a location returned by the foundIn property
 *  then we set the object's location to that value. Thus a Floating 
 *  object will always be somewhere within the top-level location of the
 *  actor, or have a location of nil.
 */
class Floating: object
    isFloating = true
    foundIn = []
    /*
     *  Returns a list of the object's location. If object has no 
     *  location then the method returns an empty list. This method 
     *  is used by scope() to build a path that connects all of the 
     *  object's possible locations.
     *
     *  For Floating objects we only return the "actual" location
     *  relative to the actor.
     */
    getLocationsList = {
        if (self.location)
            return [ self.location ];
        else return [];
    }
    
    /*
     *  Moves the Floating object into a location determined by
     *  getLocation(). If getLocation() returns nil then the object is
     *  moved into nil.
     */
    setLocation(actor) = {
        local newLoc, holdMoved, holdCount;
        
        /* 
         *  store moved information. We don't want to modify this when
         *  object floats.
         */
        holdMoved = self.hasMoved;
        holdCount = self.hasMovedCount;
        
        /* determine location */
        newLoc = self.getLocation(actor);
        
        /* move object to new location */
        self.moveInto(newLoc);
        
        /* restore moved information */
        self.hasMoved = holdMoved;
        self.hasMovedCount = holdCount;
    }
    /*
     *  Returns a single location that corresponds to a foundIn value
     *  contained along the actor's containment list; if no location is
     *  found returns nil. This method is used for moving the object to
     *  the contents list of one of its locations.
     */
    getLocation(actor) = {
        local loc, foundList;
                                
        foundList = self.getMethReturn(&foundIn);
        if (length(foundList) == 0) return nil;
        
        loc = actor;
        while(loc) {
            if (find(foundList, loc))
                return loc;
            loc = loc.location;
        }
        return nil;        
    }
    /*
     *  Convert self.(ptr) into a list
     */
    getMethReturn(ptr) = {
        local lst;
        
        switch(proptype(self, ptr)) {
            case DTY_OBJECT:
                lst = [ self.(ptr) ];
                break;
            case DTY_LIST:
                lst = self.(ptr);
                break;
            case DTY_CODE:
                lst = self.(ptr);
                switch(datatype(lst)) {
                    case DTY_OBJECT:
                        lst = [ lst ];
                        break;
                    case DTY_LIST:
                        break;
                    default:
                        lst = [];
                }
                break;
            default:
                lst = [];
        }
        return lst;
    }
;

#pragma C-

#endif /* _FLOATING_H_ */
