/* Copyright (c) 1999, 2000 by Kevin Forchione.  All Rights Reserved. */
/*
 *  TADS ADV.T/STD.T LIBRARY EXTENSION
 *  TIMESYS.T				
 *  version 3.1.1
 *
 *      This file is part of TimeSys:  Advanced Timekeeping System 
 *      for TADS ADV.T / STD.T.
 *
 *	    This file defines the timesys object and functions used by 
 *      timeSys for the creation of time-oriented adventure games.  
 *
 *----------------------------------------------------------------------
 *  REQUIREMENTS
 *
 *      + HTML TADS 2.3.0 or later
 *      + Requires ADV.T and STD.T
 *      + Should be #included after ADV.T and STD.T.
 *
 *----------------------------------------------------------------------
 *  IMPORTANT LIBRARY INTERFACE AND MODIFICATION
 *
 *      This module replaces the following ADV.T functions, methods
 *      and objects:
 *
 *          + preparse()
 *          + turncount()
 *          + scoreRank()
 *          + room.statusLine()
 *          + waitVerb
 *
 *      In addition it implements a system of preparse objects that can
 *      be used to thread through preparse routines.
 *
 *----------------------------------------------------------------------
 *  COPYRIGHT NOTICE
 *
 *  	You may modify and use this file in any way you want, provided that
 *		if you redistribute modified copies of this file in source form, the
 *   	copies must include the original copyright notice (including this
 *   	paragraph), and must be clearly marked as modified from the original
 *   	version.
 *
 *------------------------------------------------------------------------------
 *  REVISION HISTORY
 *
 *		01-Jan-99:	Creation.
 *
 *		20-May-99:	Replaced parseWord() function with new TADS 2.4.0 parserTokenize()
 *					Removed room.t file, simplifying the replace of statusLine()
 *					and moved htmlBanner attribute to global. Moved all logic
 *					from room.t into timesys.t
 *
 *		25-May-99:	Changed name from timeSys to timesys for consistency.
 *
 *					Rewrote the entire preparse() / parsetime() replacing the
 *					old mechanism with preparse object logic and regular
 *					expressions. Modified settimesys() and advtimesys() to
 *					incorporate the use of parsetime(). 
 *
 *					Simplified the logic of the waitingVerb class verbs, 
 *					removing waitForVerb and renaming waitUntilVerb. Replaced 
 *					ADV.T waitVerb making all usage of 'WAIT' command go through 
 *					waitingVerb. 
 *			
 *					Added timeString to timesys object for use in AGAIN processing.
 *
 *					Finally, re-organised the file to be more readable.
 *		12-Jun-99:	Added parameter to gettimesys and various xxxxDisplay()
 *					methods, so that they can format values other than timesys
 *					specific.
 *		18-Jun-99:	Added banner display to non-HTML banner when USE_HTML_STATUS
 *					has been defined (for non-HTML run-times).
 *      16-Mar-00:  Converted to #pragma C+ style. 
 *                  Fixed calculations for base and conversions to
 *                  handle leapyear correctly.
 *      25-Mar-00:  Fixed bug with day display and progression.
 */

#ifndef __TIMESYS_MODULE_
#define __TIMESYS_MODULE_

#pragma C+

/*------------------------------------------------------------------------------
 *	CONSTANTS
 *----------------------------------------------------------------------------*/

#ifndef PARSETIME_LIM_HH
#define PARSETIME_LIM_HH 24
#endif
#ifndef PARSETIME_LIM_MM
#define PARSETIME_LIM_MM 59
#endif

/* Constants for gettimesys */
#define TS_DAY 'day'
#define TS_DATE 'date'
#define TS_TIME 'time'
#define TS_ELAPSE 'elapse'

#define FMT_AMERICAN    1
#define FMT_EUROPEAN    2

/*------------------------------------------------------------------------------
 * Declaring functions used in this module
 *----------------------------------------------------------------------------*/
 
parsetime: function;
settimesys: function;
gettimesys: function;
advtimesys: function;
stopwaiting: function;

/*------------------------------------------------------------------------------
 *	Defining the compound words used by the 
 *	waitingVerb class of verbs.
 *----------------------------------------------------------------------------*/
 
compoundWord 'wait' 'until' 'waituntil';
compoundWord 'wait' 'for' 'waitfor';

/*------------------------------------------------------------------------------
 *	OBJECTS
 *----------------------------------------------------------------------------*/

/*
 * 	timesys: object
 *
 *	This is the game 'time clock' which keeps track of various
 *	date and time values, advances the time when requested
 *	and converts numeric date and time values into parm values
 *	for statusLine display.
 */
timesys: object
	/* 
	 *	TIME ATTRIBUTES AND METHODS
	 *
	 *	Once initialized settimesys() the time is maintained 
	 *	as a cyclical value that represents the minutes
	 *	past midnight from 0 to 1439.
	 */
	timestring = nil					// last value processed by parsetime()
	time = 0 							// time representation in minutes
	timerate = 1                        // time increment per turn in minutes
	setTime( t, r ) = {
	    timesys.time = t;
	    timesys.timerate = r;
	}
	timeDisplay( t, parm ) = {
		local h, m, ampm, t24, i;

		h = t / 60;
		t24 = h;
		m = t % 60;
		if ( h >= 12 )
		{
			if ( h > 12 )
			{
				h = h - 12;
			}
			ampm = ' pm';
		}
		else
		{
			ampm = ' am';
			if ( h == 0 )
			{
				h = 12;
			}
		};
		i = cvtstr( h ) + ':';
		if ( m < 10 )
		{
			i = i + '0';
		}
		
		i = i + cvtstr( m ) + ampm;
		
		if ( parm )
		{
			t24 = cvtstr( t24 ) + ':' + cvtstr( m );	// i.e. 15:00
			h = cvtstr( h );
			m = cvtstr( m );
			
			return [ self.time i t24 h m ampm ];
		}
		else
		{
			return i;
		}
	}
	/*
	 *	DAY ATTRIBUTES AND METHODS
	 *		
	 *	Once initialized with settimesys(), either directly
	 *	or through calculation, this maintains a 
	 *	cyclical value that represents the day of the week
	 *	with values from 1=Fri to 7=Thur.
	 */
    day = 2                                   		// the numeric day of the week
	dayDisplay( d, parm ) = 
	{
		local i, j;

		switch( d )
		{
			case 0:
				i = 'Sunday';
				j = 'Sun';
				break;
			case 1:
				i = 'Monday';
				j = 'Mon';
				break;
			case 2:
				i = 'Tuesday';
				j = 'Tue';
				break;
			case 3:
				i = 'Wednesday';
				j = 'Wed';
				break;
			case 4:
				i = 'Thursday';
				j = 'Thu';
				break;
			case 5:
				i = 'Friday';
				j = 'Fri';
				break;
			case 6: 
				i = 'Saturday';
				j = 'Sat';
				break;
		};
		
		if ( parm )
		{ 
			return [ self.day i j ];
		}
		else
		{
			return i;
		}
	}
	/*
	 *	DATE ATTRIBUTES AND METHODS
	 *
	 *	Once intiliazed with settimesys() the date is maintained as
	 *	a base number calculated to take into account the 
	 *	full century and year (year 2000 compliant!) as well
	 *	as leap year days.
	 */
	date = 730500
    daysInMonth = [31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31]
    setDaysInMonth(y) = {
        if (y % 4 != 0)
            self.daysInMonth[2] = 28;
        else if (y % 400 == 0)
            self.daysInMonth[2] = 29;
        else if (y % 100 == 0)
            self.daysInMonth[2] = 28;
        else self.daysInMonth[2] = 29;
    }
	setDate(y, m, d) = {
	    local i, b;
	    	    
        self.setDaysInMonth(y);

        b = (y-1)*365 + (y-1)/4 - (y-1)/100 + (y-1)/400;        
    
        for (i = 1; i <= m - 1; ++i)
            b += self.daysInMonth[i];
    
        b += d;
        
        self.date = b;
        self.day = b % 7;
    }
    getDate(...) = {
        local b = self.date, r1, r2, t=0, t1=0, y, m, d;
        
        if (argcount == 1) b = getarg(1);
    
        r1 = b/366;
        r2 = b/365;
    
        for (y = r1; y <= r2; ++y)
        {
            d = b - ((y-1)*365 + (y-1)/4 - (y-1)/100 + (y-1)/400);
        
            if (d < 1) break;
        }
        --y;
        d = b - ((y-1)*365 + (y-1)/4 - (y-1)/100 + (y-1)/400);

        self.setDaysInMonth(y);

        for (m = 1; m <= 12; ++m)
        {
            t += self.daysInMonth[m];
            if (t >= d)
                break;
            t1 = t;
        }
        d -= t1;

        return [y, m, d];
    }
    nameOfMonth = ['January', 'February', 'March', 'April',
        'May', 'June', 'July', 'August', 'September', 'October',
        'November', 'December']
	dateDisplay(b, parm) = {
	    local r, y, m, d, s;
	    
	    r = self.getDate(b);
	    y = r[1];
	    m = r[2];
	    d = r[3];
	    
	    switch(parm)
	    {
	        case FMT_AMERICAN:
	            s = self.nameOfMonths[m] + ' ' + cvtstr(d) + ' ' + cvtstr(y);
	            break;
            default:
                s = cvtstr(d) + ' ' + self.nameOfMonth[m] + ' ' + cvtstr(y);
                break;    
	    }
	    return s;
	}
	/*
	 *	ELAPSETIME ATTRIBUTES AND METHODS
	 *
	 *	This represents the total accumulated game time
	 *	in minutes. It is updated each time timesys.advance()
	 *	is called.
	 */
	elapsetime = 0
	elapsetimeDisplay( e, parm ) =
	{
		local t, d, h, i = ' ', len;
		t = e;
		d = ( t / 1440 );
		t -= d*1440;
		h = ( t / 60 );
		t -= h*60;
		if ( d > 0 )
		{
			i = cvtstr( d );
			if ( d == 1 )
			   i = i + ' day ';
			else
			   i = i + ' days ';
		}
		if ( h > 0 )
		{
			if ( d > 0 )
			{
				i = i + ' and ';
			}
			i = i + cvtstr( h );
			if ( h == 1 )
			   i = i + ' hour ';
			else
			   i = i + ' hours ';
		}
		if ( d > 0 || h > 0 )
		{
			if ( t > 0 )
			{
				i = i + ' and ';
				i = i + cvtstr( t );
				if ( t == 1 )
		   	   	   i = i + ' minute';
				else
		   	    	   i = i + ' minutes';
			}
		}
		else
		{
			i = i + cvtstr( t );
			if ( t == 1 )
		  	   i = i + ' minute';
			else
		    	   i = i + ' minutes';
		}

		len = length( i );

		if ( substr( i, len, 1 ) == ' ' )
		{
			i = substr( i, 1, len-1 );
		}

		if ( parm )
		{
			h = self.elapsetime / 60;
			d = self.elapsetime / 1440;
			
			return [ self.elapsetime i d h ];
		}
		else
		{
			return i;
		}
	}
	/*
	 *	WAITING PROCESS ATTRIBUTES
	 *
	 *	Used by the waiting process initiated by either the 
	 *	waitingVerb class or the advanctime(). 
	 */
	waiting = nil		// waiting process active (true/nil)
	waittime = nil		// time in minutes to wait
	waitquery = nil		// ask player if they wish to continue waiting
	waitqblock = nil		// block any waitquery
	/*
	 *	ADVANCE METHOD
	 *
	 *	This progresses the time, day, date, and elapsetime
	 *	and makes a call to timepasses()
 	 */
	advance( incr ) =
	{
		local b, d, e, t;

		t = self.time;
	    d = self.day;
        b = self.date;
		e = self.elapsetime;

		if ( incr == nil )
		{
			t += self.timerate;
			e += self.timerate;
		}
		else
		{
			t += incr;
			e += incr;
		}

		while( t >= 1440 ) 
		{ 
			t -= 1440;
			d++;
			if ( d > 6 )
				d = 0;
			b++;
		}
		
		self.time = t;
		self.day = d;
		self.date = b;
		self.elapsetime = e;

		self.events;
	}
	/*
	 *	EVENTS METHOD
	 *
	 *	Allows the author to affect global time-related events, such as lighting and 
	 *	weather effects. This method is called with each increment of timesys.time.
	 *	You should use replace this method in your game source using the 'modify'
	 *	statement.
	 */
	events =
	{
		return nil;
	}	                                             
;

/*
 *	timesys_ppo: preparseObj
 *
 *	Preparses the wait command. This will return a list consisting of the 
 *	reSearch() return for the command, a calculated waittime, and a field that
 *	contains the time-value for the o'clock, or a boolean which indicates
 *	whether 24-hour restriction was necessary.
 *
 *  If ppo.t is not #include'd in the source then TimeSys reverts to the
 *  use of an ordinary TADS preparse() function.
 */
#ifdef __PPO_MODULE_

timesys_ppo: preparseObj
    ppomethod( cmd ) =

#else   /* __PPO_MODULE_ */

replace preparse: function( cmd )

#endif /* __PPO_MODULE_ */

    {
        local ret, grp, waitret;
        
        cmd = lower(cmd);
        
        waitret = reSearch('(.*)(%<wait%>)(.*)',cmd);
        if (waitret == nil)
        {
        	waitret = reSearch('(.*)(%<z%>)(.*)',cmd);
        	if (waitret == nil)
        		return true;
        }
        
       	/*
       	 *	If wait has no specified time we use the default timesys 
       	 *  timerate.
       	 */
       	if (reGetGroup(3)[3] == '' || reGetGroup(3)[3] == ' ')
    	{
    		cmd = reGetGroup(1)[3] + reGetGroup(2)[3] + 
    		    ' ' + cvtstr(timesys.timerate) + reGetGroup(3)[3];
    	}
        ret = parsetime( cmd );
        if (ret)
        {
            local tmpcmd = '';
            
            timesys.waittime = ret[2];
            
            /*
             *	Restructuring the command. The reSearch() return from parsetime is
             *	the first element of ret. We can discard the rest of the list.
             *	We save ret[3], which is are returned parsed time parm, for
             *	use with the 'Again' command.
             */
            ret = car(ret);
            
            timesys.timestring = ret[3];
            
            if (ret[1] - 1)
                tmpcmd += substr(cmd, 1, ret[1]-1);
                
            if (length(cmd) - ret[1] - ret[2] + 1)
        	{	
             	local tmpcmd2;
             	
             	tmpcmd2 = substr(cmd, ret[1] + ret[2], 
                   length(cmd) - ret[1] - ret[2] + 1);

        		ret = reSearch('%w', tmpcmd2);
        		
        		if (ret)
        		{
        			"[There appear to be extra words or commands after the
        			WAIT command.]\b";
        			return nil;
        		}
        		else
        		{
        			tmpcmd += tmpcmd2;
        		}
        	}
            return tmpcmd;
        }
        else
        {
        	if (waitret)
        	{
        			"[This does not appear to be a valid time-format.]\b";
        			return nil;
        	}
        }
        return true;
    }
;

/*------------------------------------------------------------------------------
 *	FUNCTIONS
 *----------------------------------------------------------------------------*/

/*
 *	parsetime: function( cmd )
 *
 *	This function parses cmd to determine a timesys.waittime. If
 *  successful, it returns a list consisting of 3 elements:
 *      ret[1]: a sublist containing the original reSearch() return.
 *      ret[2]: the calculated waittime
 *      ret[3]: for o'clock-style formats this is the calculated time in 
 *			minutes past midnight; for hours/minutes format it contains
 *          true if 24-hour restriction was enforced, nil otherwise.
 *  If unsuccessful the function returns nil.
 */
parsetime: function( cmd )
{
	local ret, grp, specRet, restr24;
	local tot = 0, hh = 0, mm = 0, wtime = 0;
    
	cmd = lower(cmd);
	
	/*	
	 *  FORMAT: MIDNIGHT/NOON
	 *
	 *  Convert midnight to 12:00 a.m. and noon to 12:00 p.m. 
	 */
	ret = reSearch('(midnight)|(noon)', cmd );
    if (ret)
    {
    	specRet = ret;
    	grp = reGetGroup(1);
        if (reGetGroup(1)[3] == 'midnight')
        	cmd = '12:00 a.m';
        else if (reGetGroup(2)[3] == 'noon')
            cmd = '12:00 p.m';
    }    
    
    /*
     *  FORMAT: HH:MM <AM/PM> 
     *
     *  Our ultimate goal is to determine how many minutes we need
     *  to advance from the current time (timesys.time) to reach the 
     *  desired o'clock passed in the parameter. 
     */
	ret = reSearch('([0-9]?[0-9]):([0-9][0-9]) *([ap]%.?m%.?)?', 
	    cmd );
    if (ret)
    {
        local ampmInd;
        
        grp = reGetGroup(1);
        hh = cvtnum(grp[3]);
        grp = reGetGroup(2);
        mm = cvtnum(grp[3]);        
        grp = reGetGroup(3);        
        
        /*
         *  Validate the range for hours and minutes. This code 
         *  returns nil if the hours and minutes are not within
         *  the appropriate ranges either for 24-hour or 12-hour
         *  clock-time.
         */
        if (grp[2])
        {
            if (!(hh > 0 && hh < 13))
                return nil;
            else
                ampmInd = grp[3];
        }
        else
        {
            if (!(hh >= 0 && hh < 24))
                return nil;
           else if (hh > 12)
                {
                    ampmInd = 'p.m.';
                    hh -= 12;
                }
        }
        if (!(mm >= 0 && mm < 60))
        	return nil;
        
        tot = 60*hh + mm;        	
         
        /*
         *  AM/PM ADJUSTMENT.
         *
         *  At this point total represents an unadjusted total number
         *  of minutes past midnight. Adjustments need to be made for
         *  the presence or lack of an am/pm indicator.
         */
        switch( ampmInd )
        {
            case 'am':
            case 'a.m':
            case 'a.m.':
                /* adjust 12:00 a.m. - 12:59 a.m. */
                if ( tot >= 720 )
                    tot -= 720;
                break;
                
            case 'pm':
            case 'p.m':
            case 'p.m.':
                /* Adjust time for p.m. */
                if ( tot < 720 )
                    tot += 720;
                break;

   			/*
	         * 	This handles the cases where "wait until 4:00" is requested
		 	 *	and it's 2:00 pm. In this case we probably really want to 
		 	 * 	"wait until 4:00 pm", not "wait until 4:00 am" which would 
		 	 * 	happen without this adjustment.
		 	 */
            default:
			    /* 
			     *  Adjusts for tot between 12:00 pm - 12:59 pm. This
			     *  converts HH:MM to HH:MM A.M.
			     */
			    if ( timesys.time == 720 && tot >= 720 )		
				    tot -=720;
				
				/* 
				 *  Adjusts for when it's afternoon and we've asked
				 *  for a time later in the afternoon. This converts
				 *  HH:MM to HH.MM P.M.
				 */
			    if ( timesys.time > 720 
			    && (tot > timesys.time - 720 ))
				    tot += 720;
				    
				/* 
				 *  Adjusts for when its morning and we've asked
				 *  for a time in the afternoon. This converts 
				 *  HH:MM to HH:MM P.M.
				 */
			    if ( timesys.time < 720 
			    && tot <= timesys.time )
				    tot += 720;
		}
		
		/*
		 *  COMPUTE WAITTIME
		 *
		 *  At this stage it is assumed that we have derived a point
		 *  along a 24-hour clock. It remains only to determine how
		 *  many minutes we need to cycle forward from our current
		 *  time (timesys.time) in order to reach the desired time.
		 */
		wtime = tot;
		
		if ( wtime > timesys.time )
		{
			wtime -= timesys.time;
		}
		else if ( wtime < timesys.time )
		{   
			wtime += ( 1440 - timesys.time );
		}      
		else
		{
			wtime = 1440;
		}
        		
		/* handles parsing of 'midnight/noon' */
        if (specRet)
        	return [specRet wtime tot];
        else
        	return [ret wtime tot];
    }
    
    /*
     *	EDIT: If we have gotten this far, anything with a ':' is not a valid
     *	time-format.
     */
    ret = reSearch(':', cmd);
    if (ret) return nil;

    /*
     *  FORMAT: HH AM/PM
     *
     *  Restructure the format to HH:MM am/pm. This code recurses into the 
     *	parsetime() function, after formatting parm in HH:MM AM/PM. 
     */
    ret = reSearch('([0-9]?[0-9]) *([ap]%.?m%.?)', cmd);
    if (ret)
    {
        specRet = ret;
        cmd = reGetGroup(1)[3] + ':00 ' + reGetGroup(2)[3];
        ret = parsetime( cmd );
        if (ret)
        {
        	wtime = ret[2];
        	tot = ret[3];
        	return [specRet wtime tot];
        }
    }
    
    /*
     *  FORMAT HH HOUR(S) (AND) MM MINUTE(S)
     *
     *  Calculate the number of minutes to advance for the given
     *  format. 
     */
	ret = reSearch('([0-9]*[0-9]) *hours? *(and)? *([0-9]*[0-9]) *minutes?', 
	    cmd );
    if (ret)
    {
        grp = reGetGroup(1);
        hh = cvtnum(grp[3]);
        grp = reGetGroup(3);
        mm = cvtnum(grp[3]);
        
        if (hh > PARSETIME_LIM_HH || mm > PARSETIME_LIM_MM)
        	return nil;
        	
        wtime = 60*hh + mm;
        
        return [ret wtime];
    }
    
    /*
     *  FORMAT HH HOUR(S)
     *
     *  Calculate the number of minutes to advance for the given
     *  format.
     */
	ret = reSearch('([0-9]*[0-9]) *hours?', cmd );
    if (ret)
    {
        grp = reGetGroup(1);
        hh = cvtnum(grp[3]);
        
        if (hh > PARSETIME_LIM_HH)
        	return nil;
        
        wtime = 60*hh + mm;
        return [ret wtime];
    }
    
    /*
     *  FORMAT MM MINUTE(S)
     *
     *  Calculate the number of minutes to advance for the given 
     *  format.
     */
	ret = reSearch('([0-9]*[0-9]) *minutes?', cmd );
    if (ret)
    {
        grp = reGetGroup(1);
        mm = cvtnum(grp[3]);
        
        if (mm > PARSETIME_LIM_MM)
        	return nil;
        
        wtime = 60*hh + mm;
        return [ret wtime];
    }

    /*
     *  FORMAT: MM
     *
     *  Restructure the format to MM minutes. This code recurses into the 
     *	parsetime() function, after formatting parm in MM minutes. 
     */
	ret = reSearch('([0-9]*[0-9])', cmd );
    if (ret)
    {
        specRet = ret;
        cmd = reGetGroup(1)[3] + ' minutes';
        ret = parsetime( cmd );
        if (ret)
        {
        	wtime = ret[2];
        	return [specRet wtime];
        }
    }
   	return nil;
}

/* 
 *	settimesys: function( t, r, y, m, d )
 *
 *	A function to allow us to set the time for the global object. This should be called
 *	in the Init routine to set up the initial game time. The values to be passed are:
 *
 *	t = time - in minutes from 0 (midnight) or as a parm, such as '9:30 a.m.'
 *	r = rate - rate that time passes per turn in minutes
 *  y = year - in yyyy numeric format
 *  m = month - in mm numeric format
 *  d = day - in dd numeric format
 */
settimesys: function( t, r, ...)
{
	local yyyy, ddd, y, m, d;
	
	if (argcount > 2) y = getarg(3);
	if (argcount > 3) m = getarg(4);
	if (argcount > 4) d = getarg(5);

	if (datatype( t ) == DTY_SSTRING)
	{
		local ret = parsetime( t );
		
		if (ret)
			t = ret[3];
		else
			t = 0;
	}
	if ( y == nil ) y = 1999;
	if ( m == nil ) m = 1;
	if ( d == nil ) d = 1;

	timesys.setTime(t, r);
	timesys.setDate(y, m, d);
	
	// synchronise time, day, and date without advancing 								// advancing the time.
    timesys.advance( 0 );
}

/*
 *	gettimesys: function( disp, val, parm )
 *	
 *	Used to simplify the call to the various display methods
 *	available to the author.
 *
 */
gettimesys: function( disp, val, parm )
{
	local i = lower( disp );
	switch( i )
	{
		case TS_TIME:
			return timesys.timeDisplay( val, parm );
			break;
		case TS_DAY:
			return timesys.dayDisplay( val, parm );
			break;
		case TS_DATE:
			return timesys.dateDisplay( val, parm );
			break;
		case TS_ELAPSE:
			return timesys.elapsetimeDisplay( val, parm );
			break;
		default:
			return nil;
	}
}

/*
 *	advtimesys: function( incr, style )
 *	
 *	A function to advance the time through your game code
 *	and providing the option to execute the rundaemons() and
 *	runfuses() or bypass them. This function should be the last 
 *	line of code executed in a block because it will issue an 
 *	abort upon completion regardless of the style chosen.
 *
 *	If style is true then the waitingVerb.action() method is used 
 *	to advance time. This means that daemons and fuses are run, and
 *	the advance is subject to any issued stopwaiting(). 
 *
 *	if style is nil then the timesys.advance() method is called 
 *	directly. No daemons or fuses will be run and no opportunity
 *	for stopwaiting will be available. This method, however, is 
 *	much quicker than waitingVerb.action().
 */
advtimesys: function( incr, style )
{
	local t;
	if ( timesys.waiting )
	{
		return nil;
	}
	if (datatype(incr) == DTY_SSTRING)
	{
		local ret = parsetime( incr );
	
		if (ret)
			t = ret[2];
		else
			t = 0;
	}
	else
		t = incr;
		
	timesys.waitqblock = true;	// block waitquery

	if ( style )
	{
		timesys.waittime = t;
		waitingVerb.action( nil );
	}
	else
	{
		timesys.advance( t );
		timesys.waitqblock = nil;
		abort;
	}
}

/* 
 *	stopwaiting: function( parm )
 *
 *	This function allows you to halt the waiting process, and
 *	informing the player of some timely event. The parameter
 *	should be either true or nil. True will request that the waitquery
 *	question 'Do you wish to continue waiting? [Y/N]' be issued, 
 *	giving the player a choice. Passing nil will tell the waiting process
 *	to terminate without prompting the player for a response.
 */
stopwaiting: function( parm )
{
	timesys.waiting = nil;
	timesys.waitquery = parm;
}

/*------------------------------------------------------------------------------
 *	GRAMMAR
 *----------------------------------------------------------------------------*/
	
/*
 *	class waitingVerb: darkVerb
 *
 *	This class initiates the Waiting Process. Only the fuses and 
 *	daemons are executed, allowing events in the game to unfold while
 * 	waiting. It will iterate through the process until (a) it has advanced
 *	timesys through timesys.waittime minutes, (b) the game
 *	is terminated through the actions of a daemon or fuse, (c)
 *	stopwaiting() is issued through either a daemon or fuse.
 */
class waitingVerb: darkVerb
	action( actor ) = 
    {
		local t, i;
		t = timesys.waittime;
		if ( actor )
		{
			if (timesys.waittime)
				"Time passes...";
			else
				"Time doesn't pass...";
		}
		i = timesys.timerate;
		timesys.timerate = 1;
		timesys.waiting = true;
		while ( t > 0 )
		{
			t--;
			rundaemons();
			runfuses();
			if ( !timesys.waitqblock )
			{
				if ( t > 0 && timesys.waitquery )
				{
					timesys.waitquery = nil;
					"<P>Do you want to continue waiting? <Y/N> ";
					if ( yorn() )
					{ 
						timesys.waiting = true;
					}
					else
					{
						break;
					}
				}
				if ( !timesys.waiting )
				{
					break;
				}
			}
 		}
		timesys.waiting = nil;
		timesys.waitquery = nil;
		timesys.waitqblock = nil;
		timesys.timerate = i;
	}
;

/* 
 *	waitVerb: waitingVerb
 *
 *	Allows us to wait until a certain time. For instance, we can
 *	"wait until 4:00 pm" or simply "wait until 4:00". 
 *
 *	Allows us to wait for a certain time. For instance, we can
 *	"wait for 10 minutes" or "wait 10 hours". The 'for' is optional. If
 *	neither minutes or hours is specified the default is minutes. 
 *
 *	Note that "wait" without any amount of time indicated uses the
 *	timesys.timerate value.
 */
replace waitVerb: waitingVerb
	sdesc = "wait"
	verb = 'wait' 'waitfor' 'waituntil' 'z'
	action( actor ) =
	{
		if ( timesys.waittime == 0 )
		{
		    local ret = parsetime(timesys.timestring);
		    
		    timesys.waittime = ret[2];
		}
		inherited.action( actor );
		timesys.waittime = 0;
		abort;
	}
;

/*
 * 	bannerVerb: sysverb
 *
 *	This is simply used as a toggle for the sample
 *	TimeSys displays, and allows us to move back
 *	and forth between TimeSys and classic TADS
 *	banners.
 */
bannerVerb: sysverb
	sdesc = "banner"
	verb = 'banner'
	action( actor ) =
	{
		if ( global.timeStatus == 2)
		{
			global.timeStatus = nil;
			"<P><B>";
			"[ Basic TimeSys banner display ]";
			"</B><P>";
		}
		else if ( global.timeStatus == nil )
		{
			global.timeStatus = 1;
			"<P><B>";
			"[ Advanced TimeSys banner display ]";
			"</B><P>";
		}
		else if ( global.timeStatus == 1 )
		{
			global.timeStatus = 2;
			"<P><B>";
			"[ Classic TADS banner display ]";
			"</B><P>";
		}
		abort;
	}
;

/*	
 *	timeVerb: sysverb
 *
 *	The time verb produces a display for those systems running a
 *	non-HTML run-time. This allows the game to display the time
 *	whether a banner is present or not.
 */
timeVerb: sysverb
	verb = 'time'
	action( actor ) =
	{
        "\b"; say(gettimesys( TS_DAY, timesys.day, nil ));
        if (global.timeStatus == 1)
        {
        	", "; say(gettimesys( TS_DATE, timesys.date, nil ));
        }
        " "; say(gettimesys( TS_TIME, timesys.time, nil ));
        " ";
        abort;
	}
;

/*------------------------------------------------------------------------------
 *	MODIFICATIONS TO ADV.T
 *----------------------------------------------------------------------------*/

/*
 *	turncount: function( parm )
 *
 *	The replacement of TADS default turncount() allows
 *	the TADS game system to tell timesys to advance 
 *	the default timerate.
 */
replace turncount: function( parm )
{
	incturn();
	global.turnsofar = global.turnsofar + 1;
	timesys.advance( nil );
	scoreStatus( global.score, global.turnsofar );
}
;

/*
 *	scoreRank: function
 *
 *	The TimeSys scoreRank is replaced here to
 *	demonstrate the toggling of feature of bannerVerb
 *	and to allow games to be SCORED in either turns
 *	or time.
 */
replace scoreRank: function
{
	if ( global.timeStatus == nil || global.timeStatus == 1)
	{
		"In a total of "; say( gettimesys( TS_ELAPSE, timesys.elapsetime, nil ) );
		", you have achieved a score of ";
		say( global.score ); " points out of a possible ";
		say( global.maxscore ); ".\n";
	}
	else
	{
   		 "In a total of "; say(global.turnsofar);
   		 " turns, you have achieved a score of ";
    		say(global.score); " points out of a possible ";
    		say(global.maxscore); ".\n";
	}
}
;

/*
 *  room: thing
 *
 *	Replaces statusLine with a toggling status that displays the classic TADS
 *	banner, or two timesys time-oriented banners.
 */
modify room
    /* 
     *   Generate HTML formatting that provides for the TimeSys status 
     *   line appearance and allows us to toggle between a time-oriented
     *   display and the traditional status line appearance, using the 
     *   traditional status line computation mechanism.
     */
#ifdef USE_HTML_STATUS

    replace statusLine =
    {
        /*
         *  Check the system to see if this is an HTML-enabled 
         *  run-time. If so, generate an HTML-style banner; otherwise, 
         *  generate an old-style banner. 
         */
        if (systemInfo(__SYSINFO_SYSINFO) == true
        && systemInfo(__SYSINFO_HTML) == 1)
        {
            self.htmlStatusLine;
        }
        else
        {
            /*
             *  Centering the time display around column 40. This
             *  won't be perfect, but it's close enough to the 
             *  center for a comfortable display.
             */
            local i, stat, sr, tsr, len;
            stat = outcapture(true);
            self.statusRoot;
            sr = outcapture(stat);
            stat = outcapture(true);
            self.timesysStatusRoot;
            tsr = outcapture(stat);
            if (tsr)
            {
                len = 40 - length(sr) - length(tsr)/2;
            }
            say(sr);
            for (i = 1; i <= len; ++i)
                "\ ";
            say(tsr);
            self.dispParagraph;
        }
    }
    htmlStatusLine =
    {
 		"<banner id=StatusLine height=previous border>
		<body bgcolor=statusbg text=statustext><b>";

		self.statusRoot;

		"</b>";
		
		if ( global.timeStatus == nil || global.timeStatus == 1 )
		{
			"<tab align=center><i>";	
			
            self.timesysStatusRoot;
            
			"</i>";
		}

		"<tab align=right><i>";
		
		if ( global.timeStatus == nil || global.timeStatus == 1 )
		{
			"Score: ";
			say( global.score );
		}
		else
		{
			say( global.score ); "/";
			say( global.turnsofar ); 
		}

		"</i></banner>";
    }
    
#else /* USE_HTML_STATUS */

    /* use the standard HTML status line system */
    replace statusLine =
    {
        self.statusRoot;
        self.dispParagraph;
    }

#endif /* USE_HTML_STATUS */

    timesysStatusRoot =
    {
		if ( global.timeStatus == nil )
		{
			say( gettimesys( TS_DAY, timesys.day, nil ) ); " ";
			say( gettimesys( TS_TIME, timesys.time, nil ) );
		}
		else if ( global.timeStatus == 1 )
		{
			say( gettimesys( TS_DAY, timesys.day, nil ) ); ", ";
			say( gettimesys( TS_DATE, timesys.date, nil ) ); " ";
			say( gettimesys( TS_TIME, timesys.time, nil ) );
		}        
    }
;

/*------------------------------------------------------------------------------
 *	MODIFICATIONS TO STD.T
 *----------------------------------------------------------------------------*/

/*
 *	global: object
 *
 *	Modification adds timeStatus toggle for bannerVerb.
 */
modify global
	timeStatus = nil
;

#pragma C-

#endif
