/* ex:set ts=4 sw=4:
 *
 * violence.t: organized violence comes to TADS
 *
 * This module defines all the verb structures normally associated with
 * (semi-)violent activity and tries to make sure they all map to sensible
 * method calls.  Actually, its more than a bit sensitive and assumes that
 * "poke actor" is an attack.
 *
 * The module provides two new classes "victim" and "weapon".  Attacking
 * something which is not a victim will result in "didn't do it" responses.
 * Anything that looks like an attack will eventually end up in a call to...
 *
 *		weapon.attack_with(attacker, defender, severe)
 *
 * which is responsible for assessing the impact of the specified weapon
 * and will handle it accordingly.  In the default case, it will route
 * the message on to...
 *
 *		victim.attack_with(attacker, weapon, severe)
 *
 * but in the case of cursed weapons, it might do something else (see the
 * rusty knife in Dungeon)
 *
 * Hostile Syntaxes
 * ================
 *
 *	SIT ON <victim>
 *	LIE ON <victim>
 *	STAND ON <victim>
 *	DESTROY <victim>
 *	ATTACK/KILL/HIT <victim> [ WITH <weapon> ]
 *	SWING <weapon> AT <victim>
 *	PUT <weapon> IN <victim>
 *	THROW <weapon> AT <victim>
 *	TOUCH <victim> [ WITH <weapon> ]
 *	POKE <victim> [ WITH <weapon> ]
 *	PUSH <victim> [ WITH <weapon> ]
 *
 * This module is Copyright (c) 1994 Jeff Laing.  Permission to use any or all
 * of this code in other TADS games is granted provided credit is given in
 * your "CREDITS" command for this code and that you leave this copyright
 * message in the source whenever distributed.  Please make all changes in
 * a backward compatible manner (where possible) and make them available
 * for free (like I did ;-)
 */
violenceVersion : versionTag
	id="$Id: violence.t_v 1.3 1994/04/22 00:53:40 jeffl Exp jeffl $\n"
	author='Jeff Laing'
	func='organized violence'
;

/*
 * first we change the base class to handle all the defaults properly
 */
modify class thing

	// TOUCH <self> WITH <i>	-> TOUCH <self>
	verDoTouchWith(a,i) =	(self.verDoTouch(a))
	doTouchWith(a,i) =		(self.doTouch(a))

	// TOUCH <o> WITH <self>	-> TOUCH <o>
	verIoTouchWith(a) =		(self.verDoDrop(a))
	ioTouchWith(a,o) =		(o.doTouchWith(a,self))

	// POKE <self> WITH <i>		-> POKE <self>
	verDoPokeWith(a,i) =	(self.verDoPoke(a))
	doPokeWith(a,i) =		(self.doPoke(a))

	// POKE <o> WITH <self>		-> POKE <o>
	verIoPokeWith(a) =		(self.verIoTouchWith(a))
	ioPokeWith(a,o) =		(o.doPokeWith(a,self))

	// PUSH <self> WITH <i>		-> PUSH <self>
	verDoPushWith(a,i) =	(self.verDoPush(a))
	doPushWith(a,i) =		(self.doPush(a))

	// PUSH <o> WITH <self>		-> PUSH <o>
	verIoPushWith(a) =		(self.verIoTouchWith(a))
	ioPushWith(a,o) =		(o.doPush(a,self))

	// ATTACK/DESTROY <self>
	verDoAttack(a) =		"I've known strange people, but attacking
							<<self.adesc>>?"
	doSynonym('Attack')	=	'Destroy'

	// SWING <self>
	verDoSwing(a) =			(self.verDoDrop(a))
	doSwing(a) =			"Swinging <<self.thedesc>> accomplishes nothing"

	// SWING <self> AT <i>
	// SWING <o> AT <self>
	verDoSwingAt(a,i) =		(self.verDoSwing(a))
	verIoSwingAt(a) =		{}
	ioSwingAt(a,o) =		(o.doSwingAt(a,self))
;

/*
 * then we create a class which should be applied to all things that we
 * can "threaten".
 */
class victim: object

	// note that this object is a victim
	isvictim=true

	// This is the method that you should connect up the actual fight
	// mechanics to.  The 'severe' argument is used to distinguish the
	// "soft" attacks ("POKE BOB WITH SCREWDRIVER") from the obvious "hard"
	// attacks ("KILL BOB WITH SCREWDRIVER")
	attack_with( attacker, weapon, severe ) = {
	
		if (severe) {
			caps(); self.thedesc;
			" doesn't understand why %you% attacked";
		} else {
			"That action only mildly annoys ";
			self.thedesc;
		}

		". Feeling foolish, %you% desist%s% before any damage is done.\n";
	}

	// ATTACK <victim> WITH <i>
	verDoAttackWith(a,i) =	{}

	// THROW <o> AT <victim>
	verIoThrowAt(a) =	{}					// allow things to be thrown at us
	ioThrowAt(a,o) =	(o.doThrowAt(a,self))// weapon knows how to handle it

	// ATTACK <victim>
	verDoAttack(a) =	{}					// allow us to be attacked
	doAttack(a) =		"Attacking bare-handed is not sensible!"

	// DESTROY <victim>
	verDoDestroy(a) =	{}					// allow us to be destroyed
	doDestroy(a) =		"Destroying <<self.thedesc>> would not be sensible!"

	// TOUCH <victim>
	// TOUCH <victim> WITH <weapon>

	// STAND/SIT/LIE ON <victim>
	verDoStandon(a) = {	
		"I don't think \^";
		self.thedesc;
		"would enjoy that!";
	}
	doSynonym('StandOn') = 'Siton'
	doSynonym('StandOn') = 'Lieon'

	// PUT <o> IN <victim>	-> POKE <victim> WITH <o>
	verIoPutIn(a) =			(self.verDoPoke(a))
	ioPutIn(a,o) =			(o.doPokeWith(a,self))

	// POKE <self>			-> ATTACK <victim>
	verDoPoke(a) =			{}
	doPoke(a) =				(self.attack_with(a,nil,nil))

	// SWING <o> AT <self>
	verIoSwingAt(a) =		{}
	ioSwingAt(a,o) =		(o.doSwingAt(a,self))
;

/*
 * This class can be applied to any object which might be considered to be
 * a "weapon" by "victims"
 *
 * Most aggressive syntaxes can be changed to "ATTACK <actor> WITH <weapon>.
 * In these cases, we route the handling on to the weapon.attack_with(...)
 * and that routes it on to victim.attack_with(...)
 */
class weaponitem: weapon, item;			// useful macro class
class weapon: object

	// note that this object is a weapon
	isweapon=true

	// standard bottleneck for overriding (for cursed behaviour, etc)
	attack_with(a,d,severe) =	(d.attack_with(a,self,severe))

	// ATTACK/TOUCH/POKE/PUSH <o> WITH <weapon>
	verIoAttackWith(a) =		(self.verDoDrop(a))
	ioAttackWith(a,o) =			(self.attack_with(a,o,true))

	// TOUCH/POKE/PUSH <o> WITH <weapon>
	verIoTouchWith(a) =			(self.verDoDrop(a))
	ioTouchWith(a,o) =			(self.attack_with(a,o,nil))
	ioSynonym('TouchWith')	=	'PokeWith'
	ioSynonym('TouchWith')	=	'PushWith'

	// THROW <weapon> AT <i>	-> ATTACK <i> WITH <weapon> THEN DROP <weapon>
	verDoThrowAt(a,i) =			(self.verDoDrop(a))
	doThrowAt(a,i) =			{
		self.attack_with(a,i,true);
		if (self.isIn(Me)) self.moveInto(Me.location);
	}

	// SWING <self> AT <o>	-> ATTACK
	verDoSwingAt(a,i) =			(self.verDoDrop(a))
	doSwingAt(a,i) =			(self.attack_with(a,i,true))
;

/*
 * ATTACK <x>
 * ATTACK <x> WITH <weapon>
 */
modify attackVerb
	doAction='Attack'
	doDefault(a,p,i) =		(allwithprop(a,&isvictim))
	ioDefault(a,p) =		(allwithprop(a,&isweapon))
;

/*
 * "TOUCH <x>"
 * "TOUCH <x> WITH <weapon>"
 */
modify touchVerb
    //doAction = 'Touch'
	ioAction(withPrep) = 'TouchWith'
    ioDefault( a, p ) =		(allwithprop(a,&isweapon))
;

/*
 * "POKE <x>"
 * "POKE <x> WITH <y>"
 */
modify pokeVerb
	ioAction(withPrep) = 'PokeWith'
    ioDefault( a, p ) =		(allwithprop(a,&isweapon))
;

/*
 * "PUSH <x>"
 * "PUSH <x> WITH <weapon>"
 */
modify pushVerb
	ioAction(withPrep) = 'PushWith'
    ioDefault( a, p ) =		(allwithprop(a,&isweapon))
;

/*
 * "DESTROY <object>"
 */
destroyVerb: deepverb
	verb='destroy' 'break' 'ruin' 'mung'
	sdesc="destroy"
	doAction = 'Destroy'
    doDefault(a,p,i) =		(allwithprop(a,&isvictim))
;

// SWING <weapon>
// SWING <weapon> AT <victim>
swingVerb: deepverb
	verb='swing'
	action='Swing'
	sdesc="swing"
	doAction='Swing'
	ioAction(atPrep)='SwingAt'
;
