/* Copyright (c) 1988, 1990 by Michael J. Roberts.  All Rights Reserved. */
/*
   adv.t  - standard adventure definitions for TADS games
   Version 1.0
   
   This file defines the basic classes and functions used by most TADS
   adventure games.  It is generally #include'd at the start of each game.
   
   Please see the TADS Shareware License for information on using and
   copying this file.
*/

checkDoor: function;
checkReach: function;
itemcnt: function;
listcont: function;
listcontcont: function;
turncount: function;
addweight: function;
addbulk: function;
incscore: function;
darkTravel: function;
scoreRank: function;
terminate: function;
die: function;
init: function;
pardon: function;
preinit: function;
sleepDaemon: function;
eatDaemon: function;
goToSleep: function;

checkDoor: function( d, r )
{
    if ( d.isopen ) return( r );
    else
    {
	setit( d );
	caps(); d.thedesc; " is closed. ";
	return( nil );
    }
}
checkReach: function( loc, actor, v, obj )
{
    if ( obj=numObj or obj=strObj ) return;
    if ( not ( actor.isCarrying( obj ) or obj.isIn( actor.location )))
    {
	if (find( loc.reachable, obj ) <> nil ) return;
	"You can't reach "; obj.thedesc; " from "; loc.thedesc; ". ";
	exit;
    }
}
itemcnt: function( list )     // count of takeable items in list
{
    local cnt, tot, i;
    tot := length( list );
    cnt := 0;
    i := 1;
    while ( i <= tot )
    {
	if (not list[i].isfixed)
	    cnt := cnt+1;
	i := i+1;
    }
    return( cnt );
}
listcont: function( obj )     // list contents of object
{
    local i, count, tot, list, cur, disptot;
    count := 0;
    list := obj.contents;
    tot := length( list );
    disptot := itemcnt( list );
    i := 1;
    while ( i <= tot )
    {
	cur := list[i];
	if ( not cur.isfixed )
	{
	    if ( count > 0 )
	    {
		if ( count+1 < disptot )
		    ", ";
		else if (count = 1)
		    " and ";
		else
		    ", and ";
	    }
	    count := count + 1;
	    cur.adesc;		     // list this object
	    if ( cur.isworn ) " (being worn)";
	    else if ( cur.islit ) " (providing light)";
	}
	i := i + 1;
    }
}
listcontcont: function( obj )	// list contents of contents of object
{
    local list, i, tot;
    list := obj.contents;
    tot := length( list );
    i := 1;
    while ( i <= tot )
    {
	obj := list[i];
	if (itemcnt( obj.contents ))
	{
	    if ((( obj.isopenable and obj.isopen ) or
	     ( not obj.isopenable and obj.iscontainer )) and
	     not obj.isqcontainer )
	    {
		caps(); 	  // Built-in; forces cap on next word output
		obj.thedesc; " seems to contain ";
		listcont( obj );  // List contents of this object.
		". ";
	    }
	    else if ( obj.issurface and not obj.isqsurface )
	    {
		"Sitting on "; obj.thedesc;" is "; listcont( obj );
		". ";
	    }
	}
	i := i + 1;
    }
}
turncount: function( parm )
{
    incturn();
    global.turnsofar := global.turnsofar + 1;
    setscore( global.score, global.turnsofar );
}
addweight: function( l )
{
    local tot, i, c, totweight;

    tot := length( l );
    i := 1;
    totweight := 0;
    while ( i <= tot )
    {
	c := l[i];
	totweight := totweight + c.weight;
	if (length( c.contents ))
	    totweight := totweight + addweight( c.contents );
	i := i + 1;
    }
    return( totweight );
}
addbulk: function( list )
{
    local i, tot, totbulk, rem, cur;

    tot := length( list );
    i := 1;
    totbulk := 0;
    while( i <= tot )
    {
	cur := list[i];
	if ( not cur.isworn )
	    totbulk := totbulk + cur.bulk;
	i := i + 1;
    }
    return( totbulk );
}
incscore: function( amount )
{
    global.score := global.score + amount;
    setscore( global.score, global.turnsofar );
}
chairitem: fixeditem, nestedroom, surface
    reachable = []	    // list of all containers reachable from here;
			    //	normally, you can only reach carried items
			    //	from a chair, but this makes special allowances
    ischair = true	    // it is a chair by default; for beds or other
			    //	things you lie down on, make it false
    roomAction( actor, v, do, prep, io ) =
    {
	if ( do<>nil and v<>inspectVerb )
	    checkReach( self, actor, v, do );
	if ( io<>nil and v<>askVerb and v<>tellVerb )
	    checkReach( self, actor, v, io );
    }
    enterRoom( actor ) = {}
    noexit =
    {
	"You're not going anywhere until you get out of "; self.thedesc; ". ";
	return( nil );
    }
    verDoBoard( actor ) = { self.verDoSiton( actor ); }
    doBoard( actor ) = { self.doSiton( actor ); }
    verDoSiton( actor ) =
    {
	if ( actor.location = self )
	{
	    "You're already on "; self.thedesc; "! ";
	}
    }
    doSiton( actor ) =
    {
	"Okay, you're now sitting on "; self.thedesc; ". ";
	actor.travelTo( self );
    }
    verDoLieon( actor ) =
    {
	self.doVerSiton( actor );
    }
    doLieon( actor ) =
    {
	self.doSiton( actor );
    }
;
fixeditem: thing	    // An immovable object
    verDoTakeOut( actor, io ) =
    {
	self.verDoTake( actor );
    }
    verDoTakeOff( actor, io ) =
    {
	self.verDoTake( actor );
    }
    verDoTake( actor ) =
    {
	"You can't have "; self.thedesc; ". ";
    }
    verDoPutIn( actor, io ) =
    {
	"You can't put "; self.thedesc; " anywhere. ";
    }
    verDoPutOn( actor, io ) =
    {
	"You can't put "; self.thedesc; " anywhere. ";
    }
    verDoMove( actor ) =
    {
	"You can't move "; self.thedesc; ". ";
    }
    isfixed = true	    // Item can't be taken
    weight = 0		    // no actual weight
    bulk = 0
;
nestedroom: room
    islit =
    {
	if ( self.location ) return( self.location.islit );
	return( nil );
    }
    statusLine =
    {
	self.location.sdesc; ", in "; self.thedesc; "\n\t";
    }
    lookAround( verbosity ) =
    {
	self.statusLine;
	self.location.nrmLkAround( verbosity );
    }
;
surface: item
    issurface = true	    // Item can hold objects on its surface
    ldesc =
    {
	if (itemcnt( self.contents ))
	{
	    "On "; self.thedesc; " you see "; listcont( self ); ". ";
	}
	else
	{
	    "There's nothing on "; self.thedesc; ". ";
	}
    }
    verIoPutOn( actor ) = {}
    ioPutOn( actor, do ) =
    {
	do.doPutOn( actor, self );
    }
;
inspectVerb: deepverb
    verb = 'inspect' 'examine' 'look at' 'x'
    sdesc = "inspect"
    doAction = 'Inspect'
;
askVerb: deepverb
    verb = 'ask'
    sdesc = "ask"
    prepDefault = aboutPrep
    ioAction( aboutPrep ) = 'AskAbout'
    validIo( actor, obj, seqno ) = { return( seqno = 1 ); }
;
tellVerb: deepverb
    verb = 'tell'
    sdesc = "tell"
    prepDefault = aboutPrep
    ioAction( aboutPrep ) = 'TellAbout'
    validIo( actor, obj, seqno ) = { return( seqno = 1 ); }
;
thing: object		    // Basic "thing" object.  Comprises rooms,
			    // fixed objects, actors, and moveable objects.
    contents = []	    // nothing inside it yet
    verGrab( actor ) = {}   // we allow objects within to be taken out
    Grab( obj ) = {}	    // ...and doing so has no effect
    adesc =
    {
	"a "; self.sdesc;   // default is "a <name>"; "self" is current object
    }
    thedesc =
    {
	"the "; self.sdesc; // default is "the <name>"
    }
    ldesc = { "It looks like an ordinary "; self.sdesc; " to me."; }
    readdesc = { "You can't read "; self.adesc; ". "; }
    actorAction( v, d, p, i ) =
    {
	"You have lost your mind. ";
	exit;
    }
    isIn( obj ) =
    {
	local myloc;
	
	myloc := self.location;
	if ( myloc )
	{
	    if ( myloc = obj ) return( true );
	    if ( myloc.isopenable )
	    {
		if ( myloc.isopen )
		    return( myloc.isIn( obj ));
		return( nil );
	    }
	    return( myloc.isIn( obj ));
	}
	return( nil );
    }	 
    thrudesc = { "You can't see much through "; self.thedesc; ".\n"; }
    moveInto( obj ) =
    {
	local loc;
	
	/*
	 *   For the object containing me, and its container, and so forth,
	 *   tell it via a Grab message that I'm going away.
	 */
	loc := self.location;
	while ( loc )
	{
	    loc.Grab( self );
	    loc := loc.location;
	}
	
	if ( self.location )
	    self.location.contents := self.location.contents - self;
	self.location := obj;
	if ( obj ) obj.contents := obj.contents + self;
    }
    verDoSave( actor ) =
    {
	"Please specify the name of the game to save in double quotes,
	for example, SAVE \"GAME1\". ";
    }
    verDoRestore( actor ) =
    {
	"Please specify the name of the game to restore in double quotes,
	for example, SAVE \"GAME1\". ";
    }
    verDoScript( actor ) =
    {
	"You should type the name of a file to write the transcript to
	in quotes, for example, SCRIPT \"LOG1\". ";
    }
    verDoSay( actor ) =
    {
	"You should say what you want to say in double quotes, for example,
	SAY \"HELLO\". ";
    }
    verDoPush( actor ) = 
    { 
	"Pushing "; self.thedesc; " doesn't do anything. "; 
    }
    verDoWear( actor ) =
    {
	"You can't wear "; self.thedesc; ". ";
    }
    verDoTake( actor ) =
    {
	local loc;

	loc := self.location;

	if ( loc = actor )
	{
	    "You already have "; self.thedesc; "! ";
	}
	else
	{
	    /*
	     *   Check with each container to make sure that the container
	     *   doesn't object to the object's removal.
	     */
	    while ( loc )
	    {
		if ( loc <> actor ) loc.verGrab( self );
		loc := loc.location;
	    }
	}
    }
    doTake( actor ) =
    {
	local totbulk, totweight;

	totbulk := addbulk( actor.contents ) + self.bulk;
	totweight := addweight( actor.contents );
	if ( not actor.isCarrying( self ))
	    totweight := totweight + self.weight;

	if ( totweight > actor.maxweight )
	    "Your load is too heavy. ";
	else if ( totbulk > actor.maxbulk )
	    "You've already got your hands full. ";
	else
	{
	    self.moveInto( actor );
	    "Taken. ";
	}
    }
    verDoDrop( actor ) =
    {
	if ( not actor.isCarrying( self ))
	{
	    "You're not carrying "; self.thedesc; "! ";
	}
    }
    doDrop( actor ) =
    {
	local loc;

	loc := actor.location;
	while ( loc.location and not loc.isdroploc ) loc := loc.location;
	self.moveInto( loc );
	"Dropped. ";
    }
    verDoUnwear( actor ) =
    {
	"You're not wearing "; self.thedesc; "! ";
    }
    verIoPutIn( actor ) =
    {
	"I don't know how to put anything into "; self.thedesc; ". ";
    }
    verDoPutIn( actor, io ) =
    {
	if ( io = nil ) return;
	
	if ( self.location = io )
	{
	    caps(); self.thedesc; " is already in "; io.thedesc; "! ";
	}
	else if ( io = self or io.isIn( self ))
	{
	    "You can't put "; self.thedesc; " in itself! ";
	}
    }
    doPutIn( actor, io ) =
    {
	self.moveInto( io );
	"Done. ";
    }
    verIoPutOn( actor ) =
    {
	"There's no good surface on "; self.thedesc; ". ";
    }
    verDoPutOn( actor, io ) =
    {
	if ( io = nil ) return;
	
	if ( self.location = io )
	{
	    caps(); self.thedesc; " is already on "; io.thedesc; "! ";
	}
	else if ( io = self or io.isIn( self ))
	{
	    "You can't put "; self.thedesc; " on itself! ";
	}
    }
    doPutOn( actor, io ) =
    {
	self.moveInto( io );
	"Done. ";
    }
    verIoTakeOut( actor ) = {}
    ioTakeOut( actor, do ) =
    {
	do.doTakeOut( actor, self );
    }
    verDoTakeOut( actor, io ) =
    {
	if ( io <> nil and not self.isIn( io ))
	{
	    caps(); self.thedesc; " isn't in "; io.thedesc; ". ";
	}
    }
    doTakeOut( actor, io ) =
    {
	self.doTake( actor );
    }
    verIoTakeOff( actor ) = {}
    ioTakeOff( actor, do ) =
    {
	do.doTakeOff( actor, self );
    }
    verDoTakeOff( actor, io ) =
    {
	if ( io <> nil and not self.isIn( io ))
	{
	    caps(); io.thedesc; " isn't on "; self.thedesc; "! ";
	}
    }
    doTakeOff( actor, io ) =
    {
	self.doTake( actor );
    }
    verIoPlugIn( actor ) =
    {
	"You can't plug anything into "; self.thedesc; ". ";
    }
    verDoPlugIn( actor, do ) =
    {
	"You can't plug "; self.thedesc; " into anything. ";
    }
    verIoUnplugFrom( actor ) =
    {
	"It's not plugged into "; self.thedesc; ". ";
    }
    verDoUnplugFrom( actor, io ) =
    {
	if ( io <> nil ) { "It's not plugged into "; io.thedesc; ". "; }
    }
    verDoLookin( actor ) =
    {
	"There's nothing in "; self.thedesc; ". ";
    }
    verDoLookthru( actor ) =
    {
	"You can't see anything through "; self.thedesc; ". ";
    }
    verDoLookunder( actor ) =
    {
	"There's nothing under "; self.thedesc; ". ";
    }
    verDoInspect( actor ) = {}
    doInspect( actor ) =
    {
	self.ldesc;
    }
    verDoRead( actor ) =
    {
	"I don't know how to read "; self.thedesc; ". ";
    }
    verDoLookbehind( actor ) =
    {
	"There's nothing behind "; self.thedesc; ". ";
    }
    verDoTurn( actor ) =
    {
	"Turning "; self.thedesc; " doesn't have any effect. ";
    }
    verDoTurnWith( actor, io ) =
    {
	"Turning "; self.thedesc; " doesn't have any effect. ";
    }
    verDoTurnTo( actor, io ) =
    {
	"Turning "; self.thedesc; " doesn't have any effect. ";
    }
    verIoTurnTo( actor ) =
    {
	"I don't know how to do that. ";
    }
    verDoTurnon( actor ) =
    {
	"I don't know how to turn "; self.thedesc; " on. ";
    }
    verDoTurnoff( actor ) =
    {
	"I don't know how to turn "; self.thedesc; " off. ";
    }
    verIoAskAbout( actor ) = {}
    ioAskAbout( actor, do ) =
    {
	do.doAskAbout( actor, self );
    }
    verDoAskAbout( actor, io ) =
    {
	"Surely, you can't think "; self.thedesc; " knows anything
	about it! ";
    }
    verIoTellAbout( actor ) = {}
    ioTellAbout( actor, do ) =
    {
	do.doTellAbout( actor, self );
    }
    verDoTellAbout( actor, io ) =
    {
	"It doesn't look as though "; self.thedesc; " is interested. ";
    }
    verDoUnboard( actor ) =
    {
	if ( actor.location <> self )
	{
	    "You're not in "; self.thedesc; "! ";
	}
	else if ( self.location=nil )
	{
	    "You can't get out of "; self.thedesc; "! ";
	}
    }
    doUnboard( actor ) =
    {
	if ( self.fastenitem )
	{
	    "You'll have to unfasten "; actor.location.fastenitem.thedesc;
	    " first. ";
	}
	else
	{
	    "Okay, you're no longer in "; self.thedesc; ". ";
	    self.leaveRoom( actor );
	    actor.moveInto( self.location );
	}
    }
    verDoAttachWith( actor, io ) =
    {
	"Attacking "; self.thedesc; " doesn't appear productive. ";
    }
    verIoAttackWith( actor ) =
    {
	"It's not very effective to attack with "; self.thedesc; ". ";
    }
    verDoEat( actor ) =
    {
	caps(); self.thedesc; " doesn't appear appetizing. ";
    }
    verDoDrink( actor ) =
    {
	caps(); self.thedesc; " doesn't appear appetizing. ";
    }
    verDoGiveTo( actor, io ) =
    {
	if ( not actor.isCarrying( self ))
	{
	    "You're not carrying "; self.thedesc; ". ";
	}
    }
    doGiveTo( actor, io ) =
    {
	self.moveInto( io );
	"Done. ";
    }
    verDoPull( actor ) =
    {
	"Pulling "; self.thedesc; " doesn't have any effect. ";
    }
    verDoThrowAt( actor, io ) =
    {
	if ( not actor.isCarrying( self ))
	{
	    "You're not carrying "; self.thedesc; ". ";
	}
    }
    doThrowAt( actor, io ) =
    {
	"You miss. ";
	self.moveInto( actor.location );
    }
    verIoThrowAt( actor ) =
    {
	if ( actor.isCarrying( self ))
	{
	    "You could at least drop "; self.thedesc; " first. ";
	}
    }
    ioThrowAt( actor, do ) =
    {
	do.doThrowAt( actor, self );
    }
    verDoThrowTo( actor, io ) =
    {
	if ( not actor.isCarrying( self ))
	{
	    "You're not carrying "; self.thedesc; ". ";
	}
    }
    doThrowTo( actor, io ) =
    {
	"You miss. ";
	self.moveInto( actor.location );
    }
    verDoThrow( actor ) =
    {
	if ( not actor.isCarrying( self ))
	{
	    "You're not carrying "; self.thedesc; ". ";
	}
    }
    doThrow( actor ) =
    {
	"Thrown. ";
	self.moveInto( actor.location );
    }
    verDoShowTo( actor, io ) =
    {
    }
    doShowTo( actor, io ) =
    {
	if ( io <> nil ) { caps(); io.thedesc; " isn't impressed. "; }
    }
    verIoShowTo( actor ) =
    {
	caps(); self.thedesc; " isn't impressed. ";
    }
    verDoClean( actor ) =
    {
	caps(); self.thedesc; " looks a bit cleaner now. ";
    }
    verDoCleanWith( actor, io ) = {}
    doCleanWith( actor, io ) =
    {
	caps(); self.thedesc; " looks a bit cleaner now. ";
    }
    verDoMove( actor ) =
    {
	"Moving "; self.thedesc; " doesn't reveal anything. ";
    }
    verDoMoveTo( actor, io ) =
    {
	"Moving "; self.thedesc; " doesn't reveal anything. ";
    }
    verIoMoveTo( actor ) =
    {
	"That doesn't get us anywhere. ";
    }
    verDoMoveWith( actor, do ) =
    {
	"Moving "; self.thedesc; " doesn't reveal anything. ";
    }
    verIoMoveWith( actor ) =
    {
	caps(); self.thedesc; " doesn't seem to help. ";
    }
    verDoTypeOn( actor, do ) =
    {
	"You should say what you want to type in double quotes, for
	example, TYPE \"HELLO\" ON KEYBOARD. ";
    }
    verDoTouch( actor ) =
    {
	"Touching "; self.thedesc; " doesn't seem to have any effect. ";
    }
    verDoPoke( actor ) =
    {
	"Poking "; self.thedesc; " doesn't seem to have any effect. ";
    }
    genMoveDir = { "You can't seem to do that. "; }
    verDoMoveN( actor ) = { self.genMoveDir; }
    verDoMoveS( actor ) = { self.genMoveDir; }
    verDoMoveE( actor ) = { self.genMoveDir; }
    verDoMoveW( actor ) = { self.genMoveDir; }
    verDoMoveNE( actor ) = { self.genMoveDir; }
    verDoMoveNW( actor ) = { self.genMoveDir; }
    verDoMoveSE( actor ) = { self.genMoveDir; }
    verDoMoveSW( actor ) = { self.genMoveDir; }
    verDoSearch( actor ) =
    {
	"You find nothing of interest. ";
    }
;
room: thing		    // An enterable area
    roomCheck( v ) =	    // true if verb is OK in room, first pass
    {			    // (generally, only disallow commands when dark)
	return( true );     // everything is OK in a lit room
    }
    islit = true	    // rooms are lit unless otherwise specified
    isseen = nil	    // room has not been seen yet
    enterRoom( actor ) =    // sent to room as actor is entering it
    {
	self.lookAround(( not self.isseen ) or global.verbose );
	if ( self.islit ) self.isseen := true;
    }
    roomAction( a, v, d, p, i ) =
    {
	if ( self.location ) self.location.roomAction( a, v, d, p, i );
    }
    /*
     *	 Whenever an actor leaves this room, we run through the leaveList.
     *	 This is a list of objects that have registered themselves with us
     *	 via addLeaveList().  For each object in the leaveList, we send
     *	 a "leaving" message, with the actor as the parameter.	It should
     *	 return true if it wants to be removed from the leaveList, nil
     *	 if it wants to stay.
     */
    leaveList = []
    addLeaveList( obj ) =
    {
	self.leaveList := self.leaveList + obj;
    }
    leaveRoom( actor ) =
    {
	local tmplist, thisobj, i, tot;
	
	tmplist := self.leaveList;
	tot := length( tmplist );
	i := 1;
	while ( i <= tot )
	{
	    thisobj := tmplist[i];
	    if ( thisobj.leaving( actor ))
		self.leaveList := self.leaveList - thisobj;
	    i := i + 1;
	}
    }
    /*
     *	 lookAround describes the room.  If verbosity is true, the full
     *	 description is given, otherwise an abbreviated description (without
     *	 the room's ldesc) is displayed.
     */
    nrmLkAround( verbosity ) =	    // lookAround without location status
    {
	local l, cur, i, tot;

	if ( verbosity )
	{
	    "\n\t"; self.ldesc;
	}
	"\n\t";
	if (itemcnt( self.contents ))
	{
	    "You see "; listcont( self ); " here. ";
	}
	listcontcont( self ); "\n";

	l := self.contents;
	tot := length( l );
	i := 1;
	while ( i <= tot )
	{
	    cur := l[i];
	    if ( cur.isactor )
	    {
		if ( cur <> Me )
		{
		    "\n\t";
		    cur.actorDesc;
		}
	    }
	    i := i + 1;
	}
    }
    statusLine =
    {
	self.sdesc; "\n\t";
    }
    lookAround( verbosity ) =
    {
	self.statusLine;
	self.nrmLkAround( verbosity );
    }
    north = { return( self.noexit ); }
    south = { return( self.noexit ); }
    east  = { return( self.noexit ); }
    west  = { return( self.noexit ); }
    up	  = { return( self.noexit ); }
    down  = { return( self.noexit ); }
    ne	  = { return( self.noexit ); }
    nw	  = { return( self.noexit ); }
    se	  = { return( self.noexit ); }
    sw	  = { return( self.noexit ); }
    in	  = { return( self.noexit ); }
    out   = { return( self.noexit ); }
    noexit = { "You can't go that way. "; return( nil ); }
;
item: thing		    // A standard takeable object
    weight = 0		    // Default weight of an object
    bulk = 1		    // Default bulk of an object
;
deepverb: object		// A deep-structure verb.
    validDo( actor, obj, seqno ) =
    {
	return( obj.isIn( actor.location.location ? actor.location.location :
	 actor.location ));
    }
    validIo( actor, obj, seqno ) =
    {
	return( obj.isIn( actor.location.location ? actor.location.location :
	 actor.location ));
    }
    doDefault( actor, prep, io ) =
    {
	return( actor.contents + actor.location.contents );
    }
    ioDefault( actor, prep ) =
    {
	return( actor.contents + actor.location.contents );
    }
;
aboutPrep: Prep
    preposition = 'about'
    sdesc = "about"
;
Prep: object
;
Actor: fixeditem	    // A character in the game
    weight = 10 	    // actors are pretty heavy
    bulk = 10		    // and pretty bulky
    maxweight = 50	    // Weight that can be carried at once
    maxcnt = 20 	    // Number of objects that can be carried at once
    totecnt = 0 	    // No objects, so count is zero
    toteweight = 0	    // and weight is zero as well
    isactor = true	    // flag that this is an actor
    actorAction( v, d, p, i ) =
    {
	caps(); self.thedesc; " doesn't appear interested. ";
	exit;
    }
    isCarrying( obj ) = { return( obj.isIn( self )); }
    actorDesc =
    {
	caps(); self.adesc; " is here. ";
    }
    verGrab( item ) =
    {
	caps(); self.thedesc; " is carrying "; item.thedesc;
	" and won't let you have it. ";
    }
    verDoFollow( actor ) =
    {
	"But "; self.thedesc; " is right here! ";
    }
    moveInto( obj ) =
    {
	if ( self.myfollower ) self.myfollower.moveInto( self.location );
	pass moveInto;
    }
;
class follower: Actor
    sdesc = { self.myactor.sdesc; }
    isfollower = true
    ldesc = { caps(); self.thedesc; " is no longer here. "; }
    actorAction( v, d, p, i ) = { self.ldesc; }
    actorDesc = {}
    myactor = nil   // set to the Actor to be followed
    verDoFollow( actor ) = {}
    doFollow( actor ) =
    {
	actor.travelTo( self.myactor.location );
    }
;
followVerb: deepverb
    sdesc = "follow"
    verb = 'follow'
    doAction = 'Follow'
;
class buttonitem: fixeditem
    noun = 'button'
    plural = 'buttons'
    verDoPush( actor ) = {}
;
class clothingItem: item
    checkDrop =
    {
	if ( self.isworn )
	{
	    "(Taking off "; self.thedesc; " first)\n";
	}
    }
    doDrop( actor ) =
    {
	self.checkDrop;
	pass doDrop;
    }
    doPutIn( actor, io ) =
    {
	self.checkDrop;
	pass doPutIn;
    }
    doPutOn( actor, io ) =
    {
	self.checkDrop;
	pass doPutOn;
    }
    doGiveTo( actor, io ) =
    {
	self.checkDrop;
	pass doGiveTo;
    }
    doThrowAt( actor, io ) =
    {
	self.checkDrop;
	pass doThrowAt;
    }
    doThrowTo( actor, io ) =
    {
	self.checkDrop;
	pass doThrowTo;
    }
    doThrow( actor ) =
    {
	self.checkDrop;
	pass doThrow;
    }
    moveInto( obj ) =
    {
	/*
	 *   Catch any other movements with moveInto; this won't stop the
	 *   movement from happening, but it will prevent any anamolous
	 *   consequences caused by the object moving but still being worn.
	 */
	self.isworn := nil;
	pass moveInto;
    }
    verDoWear( actor ) =
    {
	if ( self.isworn )
	{
	    "You're already wearing "; self.thedesc; "! ";
	}
	else if ( not actor.isCarrying( self ))
	{
	    "You don't have "; self.thedesc; ". ";
	}
    }
    doWear( actor ) =
    {
	"Okay, you're now wearing "; self.thedesc; ". ";
	self.isworn := true;
    }
    verDoUnwear( actor ) =
    {
	if ( not self.isworn )
	{
	    "You're not wearing "; self.thedesc; ". ";
	}
    }
    doUnwear( actor ) =
    {
	"Okay, you're no longer wearing "; self.thedesc; ". ";
	self.isworn := nil;
    }
;
doorway: fixeditem
    isdoor = true	    // Item can be opened and closed
    verDoOpen( actor ) =
    {
	if ( self.isopen ) "It's already open. ";
    }
    doOpen( actor ) =
    {
	"Opened. ";
	self.isopen := true;
	if ( self.otherside ) self.otherside.isopen := true;
    }
    verDoClose( actor ) =
    {
	if ( not self.isopen ) "It's already closed. ";
    }
    doClose( actor ) =
    {
	"Closed. ";
	self.isopen := nil;
	if ( self.otherside ) self.otherside.isopen := nil;
    }
    ldesc =
    {
	"It's "; if ( self.isopen ) "open. "; else "closed. ";
    }
;
vehicle: item, nestedroom
    reachable = []
    isvehicle = true
    verDoEnter( actor ) = { self.verDoBoard( actor ); }
    doEnter( actor ) = { self.doBoard( actor ); }
    verDoBoard( actor ) =
    {
	if ( actor.location = self )
	{
	    "You're already in "; self.thedesc; "! ";
	}
    }
    doBoard( actor ) =
    {
	"Okay, you're now in "; self.thedesc; ". ";
	actor.moveInto( self );
    }
    noexit =
    {
	"You're not going anywhere until you get out of "; self.thedesc; ". ";
	return( nil );
    }
    out = ( self.location )
;
digVerb: deepverb
    verb = 'dig' 'dig in'
    sdesc = "dig in"
    prepDefault = withPrep
    ioAction( withPrep ) = 'DigWith'
;
withPrep: Prep
    preposition = 'with'
    sdesc = "with"
;
uVerb: travelVerb
    action( actor ) = { actor.travelTo( self.travelDir( actor )); }
    verb = 'u' 'up' 'go up'
    travelDir( actor ) = { return( actor.location.up ); }
;
class travelVerb: deepverb
    isTravelVerb = true
;
jumpVerb: deepverb
    verb = 'jump' 'jump over' 'jump off'
    doAction = 'Jump'
    action = { "Wheeee!"; }
;
dVerb: travelVerb
    action( actor ) = { actor.travelTo( self.travelDir( actor )); }
    verb = 'd' 'down' 'go down'
    travelDir( actor ) = { return( actor.location.down ); }
;
container: item
    maxbulk = 10	    // maximum bulk the container can contain
    isopen = true	    // in fact, it can't be closed at all
    iscontainer = true	    // Item can contain other items
    ldesc =
    {
	if (itemcnt( self.contents ))
	{
	    "In "; self.thedesc; " you see "; listcont( self ); ". ";
	}
	else
	{
	    "There's nothing in "; self.thedesc; ". ";
	}
    }
    verIoPutIn( actor ) =
    {
    }
    ioPutIn( actor, do ) =
    {
	if (addbulk( self.contents ) + do.bulk > self.maxbulk )
	{
	    "You can't fit that in "; self.thedesc; ". ";
	}
	else
	{
	    do.doPutIn( actor, self );
	}
    }
    verDoLookin( actor ) = {}
    doLookin( actor ) =
    {
	self.ldesc;
    }
;
eVerb: travelVerb
    action( actor ) = { actor.travelTo( self.travelDir( actor )); }
    verb = 'e' 'east' 'go east'
    travelDir( actor ) = { return( actor.location.east ); }
;
sVerb: travelVerb
    action( actor ) = { actor.travelTo( self.travelDir( actor )); }
    verb = 's' 'south' 'go south'
    travelDir( actor ) = { return( actor.location.south ); }
;
openable: container
    isopenable = true
    ldesc =
    {
	if ( self.isopen ) pass ldesc;
	else
	{
	    caps(); self.thedesc; " is closed. ";
	}
    }
    isopen = true
    verDoOpen( actor ) =
    {
	if ( self.isopen )
	{
	    caps(); self.thedesc; " is already open! ";
	}
    }
    doOpen( actor ) =
    {
	if (itemcnt( self.contents ))
	{
	    "Opening "; self.thedesc; " reveals "; listcont( self ); ". ";
	}
	else "Opened. ";
	self.isopen := true;
    }
    verDoClose( actor ) =
    {
	if ( not self.isopen )
	{
	    caps(); self.thedesc; " is already closed! ";
	}
    }
    doClose( actor ) =
    {
	"Closed. ";
	self.isopen := nil;
    }
    verIoPutIn( actor ) =
    {
	if ( not self.isopen )
	{
	    caps(); self.thedesc; " is closed. ";
	}
    }
    verDoLookin( actor ) =
    {
	if ( not self.isopen ) "It's closed. ";
    }
;
decoration: fixeditem	    // An immovable object for decoration only
;
nVerb: travelVerb
    action( actor ) = { actor.travelTo( self.travelDir( actor )); }
    verb = 'n' 'north' 'go north'
    travelDir( actor ) = { return( actor.location.north ); }
;
wVerb: travelVerb
    action( actor ) = { actor.travelTo( self.travelDir( actor )); }
    verb = 'w' 'west' 'go west'
    travelDir( actor ) = { return( actor.location.west ); }
;
neVerb: travelVerb
    action( actor ) = { actor.travelTo( self.travelDir( actor )); }
    verb = 'ne' 'northeast' 'go ne' 'go northeast'
    travelDir( actor ) = { return( actor.location.ne ); }
;
nwVerb: travelVerb
    action( actor ) = { actor.travelTo( self.travelDir( actor )); }
    verb = 'nw' 'northwest' 'go nw' 'go northwest'
    travelDir( actor ) = { return( actor.location.nw ); }
;
seVerb: travelVerb
    action( actor ) = { actor.travelTo( self.travelDir( actor )); }
    verb = 'se' 'southeast' 'go se' 'go southeast'
    travelDir( actor ) = { return( actor.location.se ); }
;
swVerb: travelVerb
    action( actor ) = { actor.travelTo( self.travelDir( actor )); }
    verb = 'sw' 'southwest' 'go sw' 'go southwest'
    travelDir( actor ) = { return( actor.location.sw ); }
;
inVerb: travelVerb
    action( actor ) = { actor.travelTo( self.travelDir( actor )); }
    verb = 'in' 'go in' 'enter'
    sdesc = "enter"
    doAction = 'Enter'
    travelDir( actor ) = { return( actor.location.in ); }
;
outVerb: travelVerb
    action( actor ) = { actor.travelTo( self.travelDir( actor )); }
    verb = 'out' 'go out' 'exit' 'leave'
    travelDir( actor ) = { return( actor.location.out ); }
;
pushVerb: deepverb
    verb = 'push' 'press'
    sdesc = "push"
    doAction = 'Push'
;
attachVerb: deepverb
    verb = 'attach' 'connect'
    sdesc = "attach"
    prepDefault = toPrep
    ioAction( toPrep ) = 'AttachTo'
;
toPrep: Prep
    preposition = 'to'
    sdesc = "to"
;
wearVerb: deepverb
    verb = 'wear' 'put on'
    sdesc = "wear"
    doAction = 'Wear'
;
dropVerb: deepverb
    verb = 'drop' 'put down'
    sdesc = "drop"
    ioAction( onPrep ) = 'PutOn'
    doAction = 'Drop'
    doDefault( actor, prep, io ) =
    {
	return( actor.contents );
    }
;
onPrep: Prep
    preposition = 'on' 'onto' 'downon' 'upon'
    sdesc = "on"
;
removeVerb: deepverb
    verb = 'take off'
    sdesc = "take off"
    doAction = 'Unwear'
    ioAction( fromPrep ) = 'RemoveFrom'
;
openVerb: deepverb
    verb = 'open'
    sdesc = "open"
    doAction = 'Open'
;
closeVerb: deepverb
    verb = 'close'
    sdesc = "close"
    doAction = 'Close'
;
putVerb: deepverb
    verb = 'put' 'place'
    sdesc = "put"
    prepDefault = inPrep
    ioAction( inPrep ) = 'PutIn'
    ioAction( onPrep ) = 'PutOn'
    doDefault( actor, prep, io ) =
    {
	return( takeVerb.doDefault( actor, prep, io ) + actor.contents );
    }
;
inPrep: Prep
    preposition = 'in' 'into' 'downin'
    sdesc = "in"
;
takeVerb: deepverb		     // This object defines how to take things
    verb = 'take' 'pick up' 'get' 'remove'
    sdesc = "take"
    ioAction( offPrep ) = 'TakeOff'
    ioAction( outPrep ) = 'TakeOut'
    ioAction( fromPrep ) = 'TakeOut'
    ioAction( inPrep ) = 'TakeOut'
    ioAction( onPrep ) = 'TakeOff'
    doAction = 'Take'
    doDefault( actor, prep, io ) =
    {
	local ret, rem, cur, rem2, cur2, tot, i, tot2, j;
	
	ret := [];
	
	/*
	 *   For "take all out/off of <iobj>", return the (non-fixed)
	 *   contents of the indirect object.	Same goes for "take all in
	 *   <iobj>", "take all on <iobj>", and "take all from <iobj>".
	 */
	if (( prep=outPrep or prep=offPrep or prep=inPrep or prep=onPrep
	 or prep=fromPrep ) and io<>nil )
	{
	    rem := io.contents;
	    i := 1;
	    tot := length( rem );
	    while ( i <= tot )
	    {
		 cur := rem[i];
		 if ( not cur.isfixed ) ret := ret + cur;
		i := i + 1;
	    }
	    return( ret );
	}

	/*
	 *   In the general case, return everything that's not fixed
	 *   in the actor's location, or everything inside fixed containers
	 *   that isn't itself fixed.
	 */
	rem := actor.location.contents;
	tot := length( rem );
	i := 1;
	while ( i <= tot )
	{
	    cur := rem[i];
	    if ( cur.isfixed )
	    {
		if ((( cur.isopenable and cur.isopen ) or
		  ( not cur.isopenable )) and ( not cur.isactor ))
		{
		    rem2 := cur.contents;
		    tot2 := length( rem2 );
		    j := 1;
		    while ( j <= tot2 )
		    {
			  cur2 := rem2[j];
			if ( not cur2.isfixed and not cur2.notakeall )
			{
			    ret := ret + cur2;
			}
			j := j + 1;
		    }
		}
	    }
	    else if ( not cur.notakeall )
	    {
		 ret := ret + cur;
	    }

	    i := i + 1;	    
	}
	return( ret );
    }
;
offPrep: Prep
    preposition = 'off' 'offof' 'offof'
    sdesc = "off"
;
outPrep: Prep
    preposition = 'out' 'outof'
    sdesc = "out"
;
plugVerb: deepverb
    verb = 'plug'
    sdesc = "plug"
    prepDefault = inPrep
    ioAction( inPrep ) = 'PlugIn'
;
fromPrep: Prep
    preposition = 'from'
    sdesc = "from"
;
qcontainer: container
    isqcontainer = true
;
lockable: openable
    doOpen( actor ) =
    {
	if ( self.islocked )
	{
	    "It's locked. ";
	}
	else pass doOpen;
    }
    verDoLock( actor ) =
    {
	if ( self.islocked )
	{
	    "It's already locked! ";
	}
    }
    doLock( actor ) =
    {
	if ( self.isopen )
	{
	    "You'll have to close "; self.thedesc; " first. ";
	}
	else
	{
	    "Locked. ";
	    self.islocked := true;
	}
    }
    verDoUnlock( actor ) =
    {
	if ( not self.islocked ) "It's not locked! ";
    }
    doUnlock( actor ) =
    {
	"Unlocked. ";
	self.islocked := nil;
    }
    verDoLockWith( actor, io ) =
    {
	if ( self.islocked ) "It's already locked. ";
    }
    verDoUnlockWith( actor, io ) =
    {
	if ( not self.islocked ) "It's not locked! ";
    }
;
lookInVerb: deepverb
    verb = 'look in' 'look on'
    sdesc = "look in"
    doAction = 'Lookin'
;
screwVerb: deepverb
    verb = 'screw'
    sdesc = "screw"
    ioAction( withPrep ) = 'ScrewWith'
    doAction = 'Screw'
;
unscrewVerb: deepverb
    verb = 'unscrew'
    sdesc = "unscrew"
    ioAction( withPrep ) = 'UnscrewWith'
    doAction = 'Unscrew'
;
turnVerb: deepverb
    verb = 'turn' 'rotate' 'twist'
    sdesc = "turn"
    ioAction( toPrep ) = 'TurnTo'
    ioAction( withPrep ) = 'TurnWith'
    doAction = 'Turn'
;
articles: object
    article = 'the' 'a' 'an'
;
betweenPrep: Prep
    preposition = 'between' 'inbetween'
    sdesc = "between"
;
overPrep: Prep
    preposition = 'over'
    sdesc = "over"
;
atPrep: Prep
    preposition = 'at'
    sdesc = "at"
;
aroundPrep: Prep
    preposition = 'around'
    sdesc = "around"
;
thruPrep: Prep
    preposition = 'through' 'thru'
    sdesc = "through"
;
dirPrep: Prep
    preposition = 'north' 'south' 'east' 'west' 'up' 'down' 'northeast' 'ne'
		  'northwest' 'nw' 'southeast' 'se' 'southwest' 'sw'
    sdesc = "north"	    // Shouldn't ever need this, but just in case
;
switchVerb: deepverb
    verb = 'switch'
    sdesc = "switch"
    doAction = 'Switch'
;
flipVerb: deepverb
    verb = 'flip'
    sdesc = "flip"
    doAction = 'Flip'
;
turnOnVerb: deepverb
    verb = 'activate' 'turn on' 'switch on'
    sdesc = "turn on"
    doAction = 'Turnon'
;
turnOffVerb: deepverb
    verb = 'turn off' 'deactiv' 'switch off'
    sdesc = "turn off"
    doAction = 'Turnoff'
;
lookVerb: deepverb
    verb = 'look' 'l' 'look around'
    action( actor ) =
    {
	actor.location.lookAround( true );
    }
;
darkroom: room		    // An enterable area which might be dark
    islit =		    // true ONLY if something is lighting the room
    {
	local rem, cur, tot, i;
	
	rem := global.lamplist;
	tot := length( rem );
	i := 1;
	while ( i <= tot )
	{
	    cur := rem[i];
	    if ( cur.isIn( self ) and cur.islit ) return( true );
	    i := i + 1;
	}
	return( nil );
    }
    roomAction( actor, v, do, prep, io ) =
    {
	if ( not self.islit and not v.isTravelVerb and not v.issysverb )
	{
	    "You can't see a thing. ";
	    exit;
	}
	else pass roomAction;
    }
    statusLine =
    {
	if ( self.islit ) pass statusLine;
	else "In the dark.";
    }
    lookAround( verbosity ) =
    {
	if ( self.islit ) pass lookAround;
	else "It's pitch black. ";
    }
    noexit =
    {
	if ( self.islit ) pass noexit;
	else
	{
	    darkTravel();
	    return( nil );
	}
    }
    roomCheck( v ) =
    {
	if ( self.islit or v.issysverb or v.isTravelVerb ) return( true );
	else
	{
	    "It's pitch black.\n";
	    return( nil );
	}
    }
;
sitVerb: deepverb
    verb = 'sit on' 'sit in' 'sit' 'sit down' 'sit downin' 'sit downon'
    sdesc = "sit on"
    doAction = 'Siton'
;
lieVerb: deepverb
    verb = 'lie' 'lie on' 'lie in' 'lie down' 'lie downon' 'lie downin'
    sdesc = "lie on"
    doAction = 'Lieon'
;
theFloor: beditem
    noun = 'floor' 'ground'
    sdesc = "ground"
    adesc = "the ground"
    location =
    {
	if ( Me.location = self )
	    return( self.sitloc );
	else
	    return( Me.location );
    }
    locationOK = true	      // suppress warning about location being a method
    doSiton( actor ) =
    {
	"Okay, you're now sitting on "; self.thedesc; ". ";
	self.sitloc := actor.location;
	actor.moveInto( self );
    }
    doLieon( actor ) =
    {
	self.doSiton( actor );
    }
    ioPutOn( actor, do ) =
    {
	do.doDrop( actor );
    }
    ioPutIn( actor, do ) =
    {
	do.doDrop( actor );
    }
;
beditem: chairitem
    ischair = nil
    isbed = true
    sdesc = "bed"
;
getOutVerb: deepverb
    verb = 'get out' 'get outof' 'get off' 'get offof'
    sdesc = "get out of"
    doAction = 'Unboard'
    doDefault( actor, prep, io ) =
    {
	if ( actor.location and actor.location.location )
	    return( [] + actor.location );
	else return( [] );
    }
;
boardVerb: deepverb
    verb = 'get in' 'get into' 'board'
    sdesc = "get on"
    doAction = 'Board'
;
againVerb: deepverb	    // Required verb:  repeats last command.  No
			    // action routines are necessary; this one's
			    // handled internally by the parser.
    verb = 'again' 'g'
;
waitVerb: deepverb
    verb = 'wait' 'z'
    action( actor ) =
    {
	"Time passes...\n";
    }
;
iVerb: deepverb
    verb = 'inventory' 'i'
    action( actor ) =
    {
	if (length( actor.contents ))
	{
	    "You have "; listcont( actor ); ". ";
	    listcontcont( actor );
	}
	else
	    "You are empty-handed.\n";
    }
;
lookThruVerb: deepverb
    verb = 'look through' 'look thru'
    sdesc = "look through"
    doAction = 'Lookthru'
;
sysverb: deepverb
    issysverb = true
;
quitVerb: sysverb
    verb = 'quit'
    action( actor ) =
    {
	local yesno;

	scoreRank();
	"\bDo you really want to quit? (YES or NO) > ";
	yesno := yorn();
	"\b";
	if ( yesno = 1 )
	{
	    terminate();    // allow user good-bye message
	    quit();
	}
	else
	{
	    "Okay. ";
	}
	abort;
    }
;
verboseVerb: sysverb
    verb = 'verbose'
    action( actor ) =
    {
	"Okay, now in VERBOSE mode.\n";
	global.verbose := true;
	Me.location.lookAround( true );
	abort;
    }
;
terseVerb: sysverb
    verb = 'brief' 'terse'
    action( actor ) =
    {
	"Okay, now in TERSE mode.\n";
	global.verbose := nil;
	abort;
    }
;
scoreVerb: sysverb
    verb = 'score' 'status'
    action( actor ) =
    {
	scoreRank();
	abort;
    }
;
saveVerb: sysverb
    verb = 'save'
    sdesc = "save"
    doAction = 'Save'
    action( actor ) =
    {
	local savefile;
	
	savefile := askfile( 'File to save game in' );
	if ( savefile = nil or savefile = '' )
	    "Failed. ";
	else if (save( savefile ))
	    "Saved failed. ";
	else
	    "Saved. ";
	abort;
    }
;
restoreVerb: sysverb
    verb = 'restore'
    sdesc = "save"
    doAction = 'Restore'
    action( actor ) =
    {
	local savefile;
	
	savefile := askfile( 'File to restore game from' );
	if ( savefile = nil or savefile = '' )
	    "Failed. ";
	else if (restore( savefile ))
	    "Restore failed. ";
	else
	{
	    setscore( global.score, global.turnsofar );
	    "Restored. ";
	}
	abort;
    }
;
scriptVerb: deepverb
    verb = 'script'
    doAction = 'Script'
    action( actor ) =
    {
	local scriptfile;
	
	scriptfile := askfile( 'File to write transcript to' );
	if ( scriptfile = nil or scriptfile = '' )
	    "Failed. ";
	else
	{
	    logging( scriptfile );
	    "Writing script file. ";
	}
	abort;
    }
;
unscriptVerb: sysverb
    verb = 'unscript'
    action( actor ) =
    {
	logging( nil );
	"Script closed.\n";
	abort;
    }
;
restartVerb: sysverb
    verb = 'restart'
    action( actor ) =
    {
	local yesno;
	while ( true )
	{
	    "Are you sure you want to start over? (YES or NO) > ";
	    yesno := yorn();
	    if ( yesno = 1 )
	    {
		"\n";
		setscore( 0, 0 );
		restart();
		abort;
	    }
	    else if ( yesno = 0 )
	    {
		"\nOkay.\n";
		abort;
	    }
	}
    }
;
attackVerb: deepverb
    verb = 'attack' 'kill' 'hit'
    sdesc = "attack"
    prepDefault = withPrep
    ioAction( withPrep ) = 'AttackWith'
;
climbVerb: deepverb
    verb = 'climb'
    sdesc = "climb"
    doAction = 'Climb'
;
eatVerb: deepverb
    verb = 'eat' 'consume'
    sdesc = "eat"
    doAction = 'Eat'
;
drinkVerb: deepverb
    verb = 'drink'
    sdesc = "drink"
    doAction = 'Drink'
;
giveVerb: deepverb
    verb = 'give' 'offer'
    sdesc = "give"
    prepDefault = toPrep
    ioAction( toPrep ) = 'GiveTo'
;
pullVerb: deepverb
    verb = 'pull'
    sdesc = "pull"
    doAction = 'Pull'
;
readVerb: deepverb
    verb = 'read'
    sdesc = "read"
    doAction = 'Read'
;
throwVerb: deepverb
    verb = 'throw' 'toss'
    sdesc = "throw"
    prepDefault = atPrep
    ioAction( atPrep ) = 'ThrowAt'
    ioAction( toPrep ) = 'ThrowTo'
;
standOnVerb: deepverb
    verb = 'stand on'
    sdesc = "stand on"
    doAction = 'Standon'
;
standVerb: deepverb
    verb = 'stand' 'stand up' 'get up'
    sdesc = "stand"
    action( actor ) =
    {
	if ( actor.location=nil )
	    "You're already standing! ";
	else if ( actor.location.location=nil )
	    "You're already standing! ";
	else
	{
	    actor.location.doUnboard( actor );
	}
    }
;
helloVerb: deepverb
    verb = 'hello' 'hi' 'greetings'
    action( actor ) =
    {
	"Nice weather we've been having.\n";
    }
;
versionVerb: sysverb
    verb = 'version'
    action( actor ) =
    {
	version.sdesc;
	abort;
    }
;
showVerb: deepverb
    verb = 'show'
    sdesc = "show"
    prepDefault = toPrep
    ioAction( toPrep ) = 'ShowTo'
;
cleanVerb: deepverb
    verb = 'clean'
    sdesc = "clean"
    ioAction( withPrep ) = 'CleanWith'
    doAction = 'Clean'
;
sayVerb: deepverb
    verb = 'say'
    sdesc = "say"
    doAction = 'Say'
;
yellVerb: deepverb
    verb = 'yell' 'shout' 'yell at' 'shout at'
    action( actor ) =
    {
	"Your throat is a bit sore now. ";
    }
;
moveVerb: deepverb
    verb = 'move'
    sdesc = "move"
    ioAction( withPrep ) = 'MoveWith'
    ioAction( toPrep ) = 'MoveTo'
    doAction = 'Move'
;
fastenVerb: deepverb
    verb = 'fasten' 'buckle' 'buckle up'
    sdesc = "fasten"
    doAction = 'Fasten'
;
unfastenVerb: deepverb
    verb = 'unfasten' 'unbuckle'
    sdesc = "unfasten"
    doAction = 'Unfasten'
;
unplugVerb: deepverb
    verb = 'unplug'
    sdesc = "unplug"
    ioAction( fromPrep ) = 'UnplugFrom'
    doAction = 'Unplug'
;
prepUnder: Prep
    preposition = 'under' 'beneath'
    sdesc = "under"
;
lookUnderVerb: deepverb
    verb = 'look under' 'look beneath'
    sdesc = "look under"
    doAction = 'Lookunder'
;
behindPrep: Prep
    preposition = 'behind'
    sdesc = "behind"
;
lookBehindVerb: deepverb
    verb = 'look behind'
    sdesc = "look behind"
    doAction = 'Lookbehind'
;
typeVerb: deepverb
    verb = 'type'
    sdesc = "type"
    prepDefault = onPrep
    ioAction( onPrep ) = 'TypeOn'
;
lockVerb: deepverb
    verb = 'lock'
    sdesc = "lock"
    ioAction( withPrep ) = 'LockWith'
    doAction = 'Lock'
    prepDefault = withPrep
;
unlockVerb: deepverb
    verb = 'unlock'
    sdesc = "unlock"
    ioAction( withPrep ) = 'UnlockWith'
    doAction = 'Unlock'
    prepDefault = withPrep
;
readable: item
    verDoRead( actor ) =
    {
    }
    doRead( actor ) =
    {
	self.readdesc;
    }
    readdesc =
    {
	self.ldesc;
    }
;
detachVerb: deepverb
    verb = 'detach' 'disconnect'
    prepDefault = fromPrep
    ioAction( fromPrep ) = 'DetachFrom'
    doAction = 'Detach'
    sdesc = "detach"
;
sleepVerb: deepverb
    action( actor ) =
    {
	if ( actor.cantSleep )
	    "You are much too anxious worrying about your continued
	    survival to fall asleep now. ";
	else if ( global.awakeTime+1 < global.sleepTime )
	    "You're not tired. ";
	else if ( not ( actor.location.isbed or actor.location.ischair ))
	    "I don't know about you, but I can never sleep
	    standing up. You should find a nice comfortable
	    bed somewhere. ";
	else
	{
	    "You quickly drift off into dreamland...\b";
	    goToSleep();
	}
    }
    verb = 'sleep'
;
fooditem: item
    verDoEat( actor ) = {}
    doEat( actor ) =
    {
	"That was delicious! ";
	global.lastMealTime := 0;
	self.moveInto( nil );
    }
;
pokeVerb: deepverb
    verb = 'poke' 'jab'
    sdesc = "poke"
    doAction = 'Poke'
;
touchVerb: deepverb
    verb = 'touch'
    sdesc = "touch"
    doAction = 'Touch'
;
moveNVerb: deepverb
    verb = 'move north' 'move n' 'push north' 'push n'
    sdesc = "move north"
    doAction = 'MoveN'
;
moveSVerb: deepverb
    verb = 'move south' 'move s' 'push south' 'push s'
    sdesc = "move south"
    doAction = 'MoveS'
;
moveEVerb: deepverb
    verb = 'move east' 'move e' 'push east' 'push e'
    sdesc = "move east"
    doAction = 'MoveE'
;
moveWVerb: deepverb
    verb = 'move west' 'move w' 'push west' 'push w'
    sdesc = "move west"
    doAction = 'MoveW'
;
moveNEVerb: deepverb
    verb = 'move northeast' 'move ne' 'push northeast' 'push ne'
    sdesc = "move northeast"
    doAction = 'MoveNE'
;
moveNWVerb: deepverb
    verb = 'move northwest' 'move nw' 'push northwest' 'push nw'
    sdesc = "move west"
    doAction = 'MoveNW'
;
moveSEVerb: deepverb
    verb = 'move southeast' 'move se' 'push southeast' 'push se'
    sdesc = "move southeast"
    doAction = 'MoveSE'
;
moveSWVerb: deepverb
    verb = 'move southwest' 'move sw' 'push southwest' 'push sw'
    sdesc = "move southwest"
    doAction = 'MoveSW'
;
centerVerb: deepverb
    verb = 'center'
    sdesc = "center"
    doAction = 'Center'
;
searchVerb: deepverb
    verb = 'search'
    sdesc = "search"
    doAction = 'Search'
;
class basicMe: Actor
    roomCheck( v ) = { return( self.location.roomCheck( v )); }
    noun = 'me' 'i' 'myself'
    sdesc = "you"
    thedesc = "yourself"
    adesc = "yourself"
    maxweight = 10
    maxbulk = 10
    verDoFollow( actor ) =
    {
	if ( actor = self ) "You can't follow yourself! ";
    }
    actorAction( verb, dobj, prep, iobj ) = 
    {
    }
    travelTo( room ) =
    {
	if ( room )
	{
	    if ( not ( self.location.islit or room.islit ))
	    {
		 darkTravel();
	    }
	    else
	    {
		if ( self.location ) self.location.leaveRoom( self );
		self.location := room;
		room.enterRoom( self );
	    }
	}
    }
    moveInto( room ) =
    {
	self.location := room;
    }
;
dialItem: fixeditem
    maxsetting = 10 // it has settings from 1 to this number
    setting = 1     // the current setting
    ldesc =
    {
	caps(); self.thedesc; " can be turned to settings
	numbered from 1 to << self.maxsetting >>. It's
	currently set to << self.setting >>. ";
    }
    verDoTurn( actor ) = {}
    doTurn( actor ) =
    {
	askio( toPrep );
    }
    verDoTurnTo( actor, io ) = {}
    doTurnTo( actor, io ) =
    {
	if ( io = numObj )
	{
	    if ( numObj.value < 1 or numObj.value > self.maxsetting )
	    {
		"There's no such setting! ";
	    }
	    else if ( numObj.value <> self.setting )
	    {
		self.setting := numObj.value;
		"Okay, it's now turned to "; say( self.setting ); ". ";
	    }
	    else
	    {
		"It's already set to "; say( self.setting ); "! ";
	    }
	}
	else
	{
	    "I don't know how to turn "; self.thedesc;
	    " to that. ";
	}
    }
;
keyedLockable: lockable
    mykey = nil     // set 'mykey' to the key which locks/unlocks me
    doLock( actor ) =
    {
	askio( withPrep );
    }
    doUnlock( actor ) =
    {
	askio( withPrep );
    }
    doLockWith( actor, io ) =
    {
	if ( self.isopen )
	{
	    "You can't lock << self.thedesc >> when it's open. ";
	}
	else if ( io = self.mykey )
	{
	    "Locked. ";
	    self.islocked := true;
	}
	else "It doesn't fit the lock. ";
    }
    doUnlockWith( actor, io ) =
    {
	if ( io = self.mykey )
	{
	    "Unlocked. ";
	    self.islocked := nil;
	}
	else "It doesn't fit the lock. ";
    }
;
keyItem: item
    verIoUnlockWith( actor ) = {}
    ioUnlockWith( actor, do ) =
    {
	do.doUnlockWith( actor, self );
    }
    verIoLockWith( actor ) = {}
    ioLockWith( actor, do ) =
    {
	do.doLockWith( actor, self );
    }
;
switchItem: fixeditem
    verDoSwitch( actor ) = {}
    doSwitch( actor ) =
    {
	self.isActive := not self.isActive;
	"Okay, "; self.thedesc; " is now switched ";
	if ( self.isActive ) "on"; else "off";
	". ";
    }
    verDoTurnon( actor ) =
    {
	if ( self.isActive ) "It's already turned on! ";
    }
    doTurnon( actor ) =
    {
	self.isActive := true;
	"Okay, it's now turned on. ";
    }
    verDoTurnoff( actor ) =
    {
	if ( not self.isActive ) "It's already turned off! ";
    }
    doTurnoff( actor ) =
    {
	self.isActive := nil;
	"Okay, it's now turned off. ";
    }
;

basicStrObj: object		 // when a string is used in a player command,
    value = ''		    //	this is set to its value
    sdesc = "that"
    adesc = "that"
    thedesc = "that"
    verDoTypeOn( actor, io ) = {}
    doTypeOn( actor, io ) = { "\"Tap, tap, tap, tap...\" "; }
    verDoSave( actor ) = {}
    doSave( actor ) =
    {
	if (save( self.value ))
	    "Save failed. ";
	else
	    "Saved. ";
	abort;
    }
    verDoRestore( actor ) = {}
    doRestore( actor ) =
    {
	if (restore( self.value ))
	    "Restore failed. ";
	else
	{
	    "Restored. ";
	    setscore( global.score, global.turnsofar );
	}
	abort;
    }
    verDoScript( actor ) = {}
    doScript( actor ) =
    {
	logging( self.value );
	"Writing script file. ";
	abort;
    }
    verDoSay( actor ) = {}
    doSay( actor ) =
    {
	"Okay, \""; say( self.value ); "\".";
    }
;

basicNumObj: object		 // when a number is used in a player command,
    value = 0		    //	this is set to its value
    sdesc = "that"
    adesc = "that"
    thedesc = "that"
    verDoTypeOn( actor, io ) = {}
    doTypeOn( actor, io ) = { "\"Tap, tap, tap, tap...\" "; }
    verIoTurnTo( actor ) = {}
    ioTurnTo( actor, do ) = { do.doTurnTo( actor, self ); }
;
