/* -------------------------- */
/*   Spectrum Screen module   */
/* -------------------------- */

/* ss_lowres.c
 *
 * Spectrum Screen module for 256x192 screen with 8x8 color characters
 *
  
 *
 * Copyright (C) 1998 Jose Luis Cebrian Pague
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; see the file COPYING.  If not, write to
 * the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 * 
 */

#include <stdlib.h>		/* malloc, realloc */
#include <string.h>		/* memmove, memset */
#include <assert.h>

#include "ss.h"

/* --- Declarations and module struct --- */

void ss_lowres_init 	() ;
void ss_lowres_clear	() ;
void ss_lowres_move	(int x, int y) ;
void ss_lowres_plot	(int mode) ;
void ss_lowres_rmove	(int incx, int incy) ;
void ss_lowres_line	(int incx, int incy, int mode) ;
void ss_lowres_fill	(int incx, int incy, char pattern[8]) ;
void ss_lowres_attbl	(int col, int row, int w, int h) ;
void ss_lowres_putch	(int col, int row, int set, unsigned char c, int mode) ;
void ss_lowres_scroll   (int row) ;
void ss_lowres_render   () ;

struct SS_MODULE ss_lowres_mod =
{
    ss_lowres_init,
    ss_lowres_clear,
    ss_lowres_move,
    ss_lowres_plot,
    ss_lowres_rmove,
    ss_lowres_line,
    ss_lowres_fill,
    ss_lowres_attbl,
    ss_lowres_putch,
    ss_lowres_scroll,
    ss_lowres_render
} ;

/* --- Module data --- */

/* Pixel distribution emulates somewhate the Spectrum video memory.
 * ss_lowres_pixels is a 1 bpp 256x192 framebuffer, while ss_lowres_attrib
 * contains the color values for every 8x8 character in screen. Note that
 * the Y direction in ss_lowres_pixels is down-to-up, with pixels[0][0]
 * being the lower left corner and pixels[31][191] the upper right. */

static unsigned char ss_lowres_pixels [192] [32] ;
static unsigned char ss_lowres_attrib [24] [32] ;
static unsigned char ss_lowres_x, ss_lowres_y ;

/* --- Module initialization and screen clearing --- */

void ss_lowres_init ()
{
    ss_lowres_clear () ;
}
 
void ss_lowres_clear ()
{
    memset (ss_lowres_pixels, 0, sizeof(ss_lowres_pixels)) ;
    memset (ss_lowres_attrib, ss_attrib, sizeof(ss_lowres_attrib)) ;

    ss_lowres_x = 0 ;
    ss_lowres_y = 0 ;
}

/* --- Move functions (they do range checking) --- */

void ss_lowres_move (int x, int y)
{
    ss_lowres_x = MAX (0, MIN (x, 255)) ;
    ss_lowres_y = MAX (0, MIN (y, 191)) ;
}

void ss_lowres_rmove (int incx, int incy)
{
    ss_lowres_x = MAX (0, MIN (ss_lowres_x + incx, 255)) ;
    ss_lowres_y = MAX (0, MIN (ss_lowres_y + incy, 191)) ;
}

/* --- Plot function --- */

void ss_lowres_plot (int mode)
{
    unsigned char *ptr ;
    unsigned char mask ;

    /* Calculate pixel offset and mask */

    ptr = &ss_lowres_pixels[ss_lowres_y][ss_lowres_x >> 3] ;
    mask = (0x80 >> (ss_lowres_x & 0x07)) ;
    
    switch (mode)
      {
        case SS_MODE_OVER:		/* Erase (over) */
	    *ptr ^= mask ;
	    break ;

	case SS_MODE_DRAW:		/* Draw */
	    *ptr |= mask ;
	    break ;

	case SS_MODE_INVERSE:		/* Inverse */
	    *ptr &= ~mask ;
	    break ;
      }

    /* Changes the attributes */

    ptr = &ss_lowres_attrib[(ss_lowres_y) >> 3][ss_lowres_x >> 3] ;
    *ptr = (*ptr & ss_attribmask) | (ss_attrib) ;
}

/* --- Line drawing --- */

/* FIXME: Drop this ugly mess and use a properly done line drawing routine.
 * At least it is compatible with the Spectrum ROM. */

void ss_lowres_line (int incx, int incy, int mode)
{
    int stepx, stepy ;
    int signx, signy ;
    int max, inc ;
    int cur, pas ;

    int destx = ss_lowres_x + incx ;
    int desty = ss_lowres_y + incy ;
    
    /* Set up direction constants */

    assert (destx <= 255 && destx >= 0) ;
    assert (desty <= 255 && desty >= 0) ;

    if (incx < 0)
	signx = -1, incx = -incx ;
    else
	signx = 1 ;
    
    if (incy < 0)
	signy = -1, incy = -incy ;
    else
	signy = 1 ;
    
    /* Decide upon direction (hor or vert)  */

    if (incy >= incx)
      {
	inc = incx ;
	max = incy ;
	stepx = 0 ;
	stepy = signy ;
      }
    else
      {
	inc = incy ;
	max = incx ;
	stepx = signx ;
	stepy = 0 ;
      }
    
    cur = max / 2 ;
    pas = 0 ;

    /* The whole thing */

    while (pas++ < max)
      {
	cur += inc ;
	if (cur >= max)
	  {
	    ss_lowres_y += signy ;
	    ss_lowres_x += signx ;
	    cur -= max ;
	  }
	else
	  {
	    ss_lowres_y += stepy ;
	    ss_lowres_x += stepx ;
	  }
	ss_lowres_plot (mode) ;		/* I cannot believe I wrote this */
      }
    
    assert (ss_lowres_x == destx) ;
    assert (ss_lowres_y == desty) ;
}


/* --- Flood and pattern fill --- */

/* It somewhat resembles the PAW algorithm, but has some problems when
 * painting shades in a zone that already has been partially painted or
 * VERY complex zones with many isolated pixels around.
 *
 * TODO: Check for to many calls to avoid infinite looping */

/* The hfill function fills current horiz line and then searches
 * in a direction (up, down or both) for every pixel painted
 * for blank spaces, then calls itself to fill them */

static int ss_lowres_hfill (char pattern[8], int direction)
{
    unsigned char *ptr, *attribptr ;
    unsigned char *beginptr ;
    unsigned char pixel, beginpixel, patdata ;
    int beginx, beginy, endx, nextx ;
    int backupx, backupy ;

    /* The hfill function preserves current screen position */
    backupx = ss_lowres_x ;
    backupy = ss_lowres_y ;

    /* Calculate pixel offset */
    ptr = &ss_lowres_pixels[ss_lowres_y][ss_lowres_x >> 3] ;
    attribptr = &ss_lowres_attrib[(ss_lowres_y) >> 3][ss_lowres_x >> 3] ;
    pixel = (0x80 >> (ss_lowres_x & 0x07)) ;

    if (direction == 0 && ss_lowres_y < 191)
    {
	    /* If upper pixel is filled, only fill down.
	       This is a bug in the PAW fill routine */
	    if (ptr[32] & pixel)
		    direction = 2 ;
    }
    if (*ptr & pixel)
	    return ss_lowres_x+1 ;
    
    assert (ptr == &ss_lowres_pixels[ss_lowres_y][ss_lowres_x >> 3]) ;

    /* Checks for left wall or screen limit */
    while (!(*ptr & pixel))
      {
	if (ss_lowres_x == 0) break ;

	/* Moves 1 pixel left */
	if (!(ss_lowres_x & 0x07))
	  {
	    attribptr-- ;
	    pixel = 0x01 ;
	    ptr-- ;
	  }
	else
	    pixel <<= 1 ;

	ss_lowres_x-- ;
      }

    assert (ptr == &ss_lowres_pixels[ss_lowres_y][ss_lowres_x >> 3]) ;

    /* If wall reached, moves 1 pixel right */
    if ((*ptr & pixel) && ss_lowres_x < 255)
      {
	ss_lowres_x++ ;
	if (!(ss_lowres_x & 0x07))
	  {
	    attribptr++ ;
	    ptr++ ;
	    pixel = 0x80 ;
	  }
	else
	    pixel >>= 1 ;
      }

    /* Start filling until right wall reached */
    beginx = ss_lowres_x ;
    beginy = ss_lowres_y ;
    beginptr = ptr ;
    beginpixel = pixel ;
    patdata = pattern[ss_lowres_y & 0x07] ;

    assert (ptr == &ss_lowres_pixels[ss_lowres_y][ss_lowres_x >> 3]) ;

    /* Paint attribute of the character at the first pixel */
    *attribptr = (*attribptr & ss_attribmask) | (ss_attrib) ;
    endx = -1 ;
    while (!(*ptr & pixel))
      {
	/* Draws the pixel */
	*ptr &= ~pixel ;
	*ptr |= (patdata & pixel) ;

	/* Moves 1 pixel right */
	if (ss_lowres_x == 255)
	{
	    endx = 255 ;
	    break ;
	}
	ss_lowres_x++ ;
	if (!(ss_lowres_x & 0x07))
	  {
	    pixel = 0x80 ;
	    ptr++ ;
	    attribptr++ ;
	    /* Paints next attribute, put only if the fill will continue */
	    if (!(*ptr & pixel))
		*attribptr = (*attribptr & ss_attribmask) | (ss_attrib) ;
	  }
	else
	    pixel >>= 1 ;
      }
    if (endx == -1) endx = ss_lowres_x-1 ;

    /* Now search lower or upper line for blank spaces to continue
     * filling. The recursive call returns the x coordinate of the
     * right wall of the line we are searching. */

    /* Direction 0 : Fill down, then up */
    /* Direction 1 : Fill up only */
    /* Direction 2 : Fill down only */
    
    do
      {
	ss_lowres_x = beginx ;
	ss_lowres_y = beginy ;
	ptr = beginptr ;
	pixel = beginpixel ;
	assert (ptr == &ss_lowres_pixels[ss_lowres_y][ss_lowres_x >> 3]) ;

	if (direction == 1)
	  {
	    if (ss_lowres_y == 191)
		break ;
	    ptr += 32 ;
	    ss_lowres_y ++ ;
	  }
	else
	  {
	    if (ss_lowres_y == 0)
		break ;
	    ss_lowres_y -- ;
	    ptr -= 32 ;
	  }

	while (ss_lowres_x <= endx)
	  {

	    if (!(*ptr & pixel))
	      {
		/* A recursive call won't do direction 0 */
		assert (ptr == &ss_lowres_pixels[ss_lowres_y][ss_lowres_x >> 3]) ;
		nextx = ss_lowres_hfill (pattern, direction == 1 ? 1:2) ;

		if (nextx > 255) nextx = 255 ;
		assert (nextx >= ss_lowres_x) ;

		/* FIXME: Calculate next position without loop */
		while (ss_lowres_x < nextx)
		  {
		    ss_lowres_x++ ;
		    if (!(ss_lowres_x & 0x07))
		      {
			ptr++ ;
			pixel = 0x80 ;
		      }
		    else
			pixel >>= 1 ;
		  }
	      }
	    /* Move right */
	    if (ss_lowres_x == 255) break ;
	    ss_lowres_x++ ;
	    if (!(ss_lowres_x & 0x07))
	      {
		ptr++ ;
		pixel = 0x80 ;
	      }
	    else
		pixel >>= 1 ;

	  }
      }
    while (!direction++) ;

    /* The hfill function preserves current screen position */
    ss_lowres_x = backupx ;
    ss_lowres_y = backupy ;

    return endx + 1 ;
}

/* The fill function is really only a neat stub */
void ss_lowres_fill (int incx, int incy, char pattern[8])
{
    int backupx = ss_lowres_x ;
    int backupy = ss_lowres_y ;

    ss_lowres_rmove (incx, incy) ;
    ss_lowres_hfill (pattern, 0) ;
    ss_lowres_move (backupx, backupy) ;
}

/* --- Block attribute fill --- */

void ss_lowres_attbl (int col, int row, int w, int h)
{
    unsigned char *attribptr ;
    int x, y ;

    attribptr = &ss_lowres_attrib[row][col] ;
    for (y = 0 ; y < h ; y++, attribptr += 32-w)
	for (x = 0 ; x < w ; x++, attribptr++)
	    *attribptr = (*attribptr & ss_attribmask) | (ss_attrib) ;
}

/* --- Character print --- */

void ss_lowres_putch (int col, int row, int set, unsigned char c, int mode)
{
    unsigned char *ptr, *attribptr, *chardata ;
    int n ;

    attribptr = &ss_lowres_attrib[row][col] ;
    *attribptr = (*attribptr & ss_attribmask) | (ss_attrib) ;

    if (c >= 129 && c <= 143)
	chardata = ss_sys_udg[c-129] ;
    else if (c >= 144 && c <= 162)
	chardata = ss_udg[c-144] ;
    else
	chardata = ss_charset[set][c-32] ;
    
    ptr = &ss_lowres_pixels[8*(24-row) - 1][col] ;
    if (mode == SS_MODE_INVERSE)
      {
	for (n = 0 ; n < 8 ; n++, ptr -= 32, chardata++)
	    *ptr ^= *chardata ;
      }
    else
      {
	for (n = 0 ; n < 8 ; n++, ptr -= 32, chardata++)
	    *ptr = *chardata ;
      }
}

/* --- Scrolling --- */

void ss_lowres_scroll (int row)
{
    if (row < 23)
      {
	memmove (ss_lowres_pixels[8],
		 ss_lowres_pixels[0], 32*8*(23-row)) ;
	memmove (ss_lowres_attrib[1],
		 ss_lowres_attrib[0], 32*(23-row)) ;
      }
    memset (ss_lowres_pixels[0], 0, 32*8) ;
    memset (ss_lowres_attrib[0], ss_attrib, 32) ;
}

/* --- Render --- */

void ss_lowres_render ()
{
    int x, y, size ;
    unsigned char pixel, *ptr, color ;
    
    /* Stores buffer dimensions */

    ss_render_buffer_width = 256 ;
    ss_render_buffer_height = 192 ;

    switch (ss_render_mode)
      {
	case SS_RENDER_24BPP:
	    size = 256*192*3 ;
	    break ;

	case SS_RENDER_8BPP:
	    size = 256*192 ;
	    break ;
      }

    /* Allocates framebuffer if needed */

    if (ss_render_buffer_size < size)
      {
	ss_render_buffer_size = 0 ;

	if (ss_render_buffer != 0)
	    ss_render_buffer = (unsigned char *) 
		realloc (ss_render_buffer, size) ;
	else
	    ss_render_buffer = (unsigned char *) malloc (size) ;

	if (!ss_render_buffer)
	    return ;

	ss_render_buffer_size = size ;
      }

    /* Rendering: it is assumed that the ss_render_palette struct
     * already has at least the 16 spectrum colors */

    switch (ss_render_mode)
      {
	case SS_RENDER_24BPP:

	    ptr = ss_render_buffer ;
	    
	    for (y = 191 ; y >= 0 ; y--)
	      {
		for (x = 0 ; x < 32 ; x++)
		  {
		    for (pixel = 0x80 ; pixel >= 0x01 ; pixel >>= 1)
		      {
			color = ss_lowres_attrib[y >> 3][x] ;
			if (ss_lowres_pixels[y][x] & pixel)
			    color &= 0x0F ;
			else
			    color = (color & 0x08) | ((color & 0x70) >> 4) ;
			*ptr++ = ss_render_palette[color].r ;
			*ptr++ = ss_render_palette[color].g ;
			*ptr++ = ss_render_palette[color].b ;
		      }
		  }
	      }
	    break ;

	case SS_RENDER_8BPP:

	    ptr = ss_render_buffer ;
	    
	    for (y = 191 ; y >= 0 ; y--)
	      {
		for (x = 0 ; x < 32 ; x++)
		  {
		    for (pixel = 0x80 ; pixel >= 0x01 ; pixel >>= 1)
		      {
			color = ss_lowres_attrib[y >> 3][x] ;
			if (ss_lowres_pixels[y][x] & pixel)
			    color &= 0x0F ;
			else
			    color = (color & 0x08) | ((color & 0x70) >> 4) ;
			*ptr++ = color ;
		      }
		  }
	      }
	    break ;
      }
}
