/*
	HESOUND.C

	Music (MOD, S3M, XM format modules, MIDI) and sound
	(RIFF WAVE sample) playing routines for djgpp:

		hugo_playmusic
		hugo_stopmusic
		hugo_playsample
		hugo_stopsample

	for the Hugo Engine

	Copyright (c) 1995-2006 by Kent Tessman

	Both hugo_playmusic() and hugo_playsample() are called with the
	file hot--i.e., opened and positioned to the start of the resource.
	It must be closed before returning.

	NOTE:  The version of the MikMod library used here is 2.10.
*/

#include "heheader.h"
#include "mikmod.h"

#if defined (ALLEGRO)
#define line line_unused
#include "allegro.h"
#undef line
#endif


int hugo_playmusic(FILE *infile, long reslength, char loop_flag);
void hugo_musicvolume(int vol);
void hugo_stopmusic(void);
int hugo_playsample(FILE *infile, long reslength, char loop_flag);
void hugo_samplevolume(int vol);
void hugo_stopsample(void);

int InitPlayer(void);
void tickhandler(void);
void WaitForMusic(void);
void SuspendAudio(void);
void ResumeAudio(void);

void delay(unsigned int milliseconds);	/* C library */


char no_audio = false;
UNIMOD *modfile;
char music_is_playing = false;
int music_volume = 100;
MSAMPLE *sample;
char sample_is_playing = false;
int sample_volume = 100;
time_t music_last_started = 0;		/* music restart timing */

/* For suspension of audio: */
struct
{
	char filename[MAXPATH];
	char resname[MAXPATH];
	char loop_flag;
	char was_playing;
} resume_sample;
int suspended_mp_volume;
char audio_suspended = false;

extern int forbid;			/* from MikMod mplayer.c */

#define MAXCHANNELS 32			/* for MikMod */


/* InitPlayer */

int InitPlayer(void)
{
	ML_RegisterLoader(&load_mod);	/* module loaders */
	ML_RegisterLoader(&load_s3m);
	ML_RegisterLoader(&load_xm);

//	MD_RegisterDriver(&drv_nos);	/* soundcard drivers */
	MD_RegisterDriver(&drv_ss);
	MD_RegisterDriver(&drv_sb);
	MD_RegisterDriver(&drv_gus);

	MD_RegisterPlayer(tickhandler);

	install_timer();		/* Allegro function call */
	install_int(MD_Update, 10);

	md_mixfreq = 44100;			/* standard mixing frequency */
	md_mode = DMODE_16BITS|DMODE_STEREO;	/* standard mixing mode */
	md_dmabufsize = 2000;			/* standard DMA buffer size */
	md_device = 0;				/* standard device: autodetect */

	if (!MD_Init())			/* initialize driver */
		return false;

	/* Start the player here and keep it running globally */
	md_numchn = MAXCHANNELS;	/* +1 for a sample */
	MD_PlayStart();

	return true;
}


/* tickhandler() is used to manage MikMod's MOD-playing loop */

void tickhandler(void)
{
	if (no_audio) return;
	
	/* If we don't do this check, MikMod will crash us if there's a
	   sample but no module playing
	*/
	if (!music_is_playing) return;

	MP_HandleTick();    	/* play 1 tick of the module */
	MD_SetBPM(mp_bpm);
}


/* hugo_playmusic

	Returns false if it fails because of an ERROR.
*/

int hugo_playmusic(FILE *f, long reslength, char loop_flag)
{
	if (no_audio)
	{
		fclose(f);
		return true;	/* not an error */
	}

	if (music_is_playing)
	{
		hugo_stopmusic();
	}
	
	/* Set the rewind/SEEK_SET reference used by MikMod to the
	   start of the resource
	*/
	_mm_setiobase(ftell(f));

	modfile = ML_LoadFP(f);
	fclose(f);

	if (modfile)
	{
		/* Be sure to wait since the last mod was started to prevent
		   overlapping crashes
		*/
		WaitForMusic();

		MP_Init(modfile);

		/* Force the module to fit into MAXCHANNELS (which shouldn't
		   really be much of a constraint), since we started the
		   driver globally and want to keep it running--i.e., we
		   won't be calling MD_PlayStart() with a new md_numchn
		*/
// Or should modfile->numchn be constrained instead?
		md_numchn = modfile->numchn + 1;	/* +1 for a sample */
		/* If we're using all 32 channels for the module, any sample
		   that wants to get played in the future during this song
		   may be out of luck
		*/
		if (md_numchn > MAXCHANNELS)
			modfile->numchn = md_numchn = MAXCHANNELS;
		else
			modfile->numchn = md_numchn - 1;

		if (loop_flag)
		{
			forbid = 1;		/* suspend interrupts */
			modfile->reppos = 1;
			mp_loop = 1;
			forbid = 0;		/* restore interrupts */
		}
		else mp_loop = 0;

		mp_volume = music_volume;	/* 0 - 100 */
		if (audio_suspended) mp_volume = 0;

		music_is_playing = true;
		
		return true;
	}
#if defined (DEBUGGER)
	else
		DebugMessageBox("Sound Library Error", myerr);
#endif
	return false;
}


/* WaitForMusic

	When switching to/from or stopping/starting music, it usually pays to
	wait a couple of seconds for the buffer to clear out.
*/

void WaitForMusic(void)
{
	time_t t;

	if (music_is_playing)
	{
		do
		{
			time(&t);
			t = t - music_last_started;
		}
		while ((music_last_started) && t < 3);
		time(&music_last_started);
	}
}


/* hugo_musicvolume */

void hugo_musicvolume(int vol)
{
	music_volume = vol;
}


/* hugo_stopmusic */

void hugo_stopmusic(void)
{
	if (no_audio) return;

	if (music_is_playing)
	{
		music_is_playing = false;
		modfile->numpos = 0;
		
		delay(20);		/* wait for the player to stop */
		ML_Free(modfile);
	}
}


/* hugo_playsample

	Returns false if it fails because of an ERROR.
*/

int hugo_playsample(FILE *f, long reslength, char loop_flag)
{
	int channel;
	
	if (no_audio)
	{
		fclose(f);
		return true;	/* not an error */
	}

	if (sample_is_playing) hugo_stopsample();

	sample = MW_LoadWavFP(f);
	fclose(f);

	if (sample)
	{
		/* for ResumeSample() */
		strcpy(resume_sample.filename, loaded_filename);
		strcpy(resume_sample.resname, loaded_resname);
		resume_sample.loop_flag = loop_flag;
		
		/* Always play a sample on the last/highest channel */
		channel = md_numchn - 1;

		MD_VoiceSetVolume(channel, 
			(!audio_suspended)?(sample_volume*64/100):0);
		MD_VoiceSetPanning(channel, 128);
		MD_VoiceSetFrequency(channel, sample->c2spd);

		if (loop_flag)
		{
			sample->flags|=SF_LOOP;
			MD_VoicePlay(channel, sample->handle, 0,
				sample->length, 0, sample->length, sample->flags);
		}
		else
		{
			sample->flags&=~SF_LOOP;
			MD_VoicePlay(channel, sample->handle, 0,
				sample->length, 0, 0, sample->flags);
		}

		sample_is_playing = true;
		
		return true;
	}
#if defined (DEBUGGER)
	else
		DebugMessageBox("Sound Library Error", myerr);
#endif
	return false;
}


/* hugo_samplevolume */

void hugo_samplevolume(int vol)
{
	sample_volume = vol;
}


/* hugo_stopsample */

void hugo_stopsample(void)
{
	if (no_audio) return;

	if (sample_is_playing)
	{
		sample_is_playing = false;
		sample->length = 0;
		sample->flags&=~SF_LOOP;
		delay(10);		/* wait for the sample to stop */
		MW_FreeWav(sample);
		
		resume_sample.was_playing = false;
	}
}


void SuspendAudio(void)
{
	if (audio_suspended || no_audio) return;

	/* suspend sample */
	if (sample_is_playing)
	{
		resume_sample.was_playing = true;
		hugo_stopsample();
	}
	else
		resume_sample.was_playing = false;
		
	/* suspend music */
	suspended_mp_volume = mp_volume;
	mp_volume = 0;
	
	audio_suspended = true;
}


void ResumeAudio(void)
{
	if (!audio_suspended || no_audio) return;

	audio_suspended = false;
	
	/* resume sample */
	if (resume_sample.was_playing)
	{
		if (!FindResource(resume_sample.filename, resume_sample.resname))
			return;
		hugo_playsample(resource_file, 0, resume_sample.loop_flag);
	}

	/* resume music */
	mp_volume = suspended_mp_volume;
}
