/* gtschan.c: Sound channel objects
        for GlkDOS, curses.h/allegro implementation of the Glk API
    Designed by L. Ross Raszewski <lraszewski@justice.loyola.edu>
    
    based upon GlkTerm by Andrew Plotkin <erkyrath@netcom.com>
    http://www.eblong.com/zarf/glk/index.html
*/
#ifndef NO_SOUND
#include "gtpref.h"
#include "gtoption.h"
#include <stdio.h>
#include "glk.h"
#include "glkdos.h"
#include "gi_blorb.h"
#include <allegro.h>
#include "cscr.h"
#include <jgmod.h>
#include "alaiff.h"
void gtblorb_describe_error(giblorb_err_t error);
#define audio_AIFF 1
#define audio_MIDI 2
#define audio_MOD  3
union sound_data
{
    SAMPLE *spl;
    MIDI *midi;
    JGMOD *mod;
};
int typeAudio(void *data)
{
  unsigned char *ptr=(unsigned char *)data;

  if (memcmp(ptr,"FORM",4)==0)
   return audio_AIFF;
#ifdef USE_MIDI
  else if (memcmp(ptr,"MThd",4)==0)
   return audio_MIDI;
#endif
  else if (memcmp(ptr+1080,"M.K.",4)==0)
   return audio_MOD;
  else if (memcmp(ptr+1080,"M!K!",4)==0)
   return audio_MOD;
  else return audio_MOD;
}

#define sound_Allegro 1
#define sound_mod 2
int sounddev=0;

int gi_sound_check()
{
 return 1;
}
static schanid_t gli_schanlist = NULL; /* linked list of all schans */

#ifdef GLK_MODULE_SOUND

schanid_t glk_schannel_create(glui32 rock)
{
 schanid_t new;
 if (!(new=(schanid_t)malloc(sizeof(struct glk_schannel_struct))))
  return NULL;
 new->data=(union sound_data *)malloc(sizeof(union sound_data));
 new->rock=rock;
 new->volume=0x10000;
 new->sound=0;
 new->loop=0;
 new->prev=gli_schanlist;
 new->type=0;
 new->next=NULL;
 if (gli_schanlist) gli_schanlist->next=new;
 gli_schanlist=new;
    if (gli_register_obj)
        new->disprock = (*gli_register_obj)(new, gidisp_Class_Schannel);

 return new;
}
void glk_schannel_destroy(schanid_t chan)
{
 if (!chan) return;
 if (chan == gli_schanlist) gli_schanlist=chan->prev;
 if (chan->prev)
  chan->prev->next=chan->next;
 if (chan->next)
  chan->next->prev=chan->prev;

 switch(chan->type)
  {
   case audio_AIFF: destroy_sample(chan->data->spl); chan->data->spl=NULL; break;
   case audio_MOD: destroy_mod(chan->data->mod); chan->data->mod=NULL; break;
  }
 free(chan->data);
 free(chan);

}

schanid_t glk_schannel_iterate(schanid_t chan, glui32 *rockptr)
{
  schanid_t next;
  if (!chan) next=gli_schanlist;
  else next=chan->prev;
  if (rockptr)
   *rockptr = next ? next->rock : 0;

  return next;
}

glui32 glk_schannel_get_rock(schanid_t chan)
{
 if (chan) return chan->rock;
  gli_strict_warning("schannel_get_rock: invalid id.");
  return 0;
}

glui32 glk_schannel_play_ext(schanid_t chan, glui32 snd, glui32 repeats,
    glui32 notify)
{
  giblorb_err_t error;
  giblorb_result_t res;
  if (!glk_preferences[pref_sound]) return 0;
  if (!chan) { gli_strict_warning("schannel_play: bad schannel"); return 0; }
  switch(chan->type)
  {
   case audio_AIFF: destroy_sample(chan->data->spl); chan->data->spl=NULL; break;
   case audio_MOD: destroy_mod(chan->data->mod); chan->data->mod=NULL; break;
  }
  if (!repeats) return 1;
  error=giblorb_load_resource(giblorb_get_resource_map(),giblorb_method_Memory,&res,giblorb_ID_Snd,snd);
  if (error) { gtblorb_describe_error(error); return 0; }
   if (!sounddev)
   {
     if (cscr_version() != CSCR_ALLEGRO)
     {
      install_allegro(SYSTEM_AUTODETECT,&errno,atexit);
      install_timer();
     }
     if (install_sound(DIGI_AUTODETECT,MIDI_AUTODETECT, NULL))
      { gli_strict_warning("Can't install sound"); }
   sounddev=sound_Allegro;
   }
  chan->loop=repeats;
  chan->notify=notify;
  switch(typeAudio(res.data.ptr))
  {
   case audio_AIFF:
    chan->data->spl=AIFFtoSAMPLE((unsigned char *)res.data.ptr);
    chan->type=audio_AIFF;
    
    if (play_sample(chan->data->spl, (chan->volume)/256,128,1000,(chan->loop==-1))<0)
    {
     destroy_sample(chan->data->spl); chan->data->spl=NULL;
     return 0;
    }
    break;
   case audio_MOD:
  if (sounddev!=sound_mod)
   {
        install_mod(8);
        sounddev=sound_mod;
   }
   { char *fn=tmpnam(NULL); FILE *temp;
     temp=fopen(fn,"wb");
     fwrite(res.data.ptr,1,res.length,temp);
     fclose(temp);
     chan->data->mod=load_mod(fn);
     unlink(fn);
   }
    chan->type=audio_MOD;
    set_mod_volume(chan->volume/256);
    play_mod(chan->data->mod,chan->loop==-1);
    break;

  }
  return 1;
}

glui32 glk_schannel_play(schanid_t chan, glui32 snd)
{
return glk_schannel_play_ext(chan,snd,1,0);
}

void glk_schannel_stop(schanid_t chan)
{
  if (!chan) gli_strict_warning("schannel_stop: invalid id.");
  else
  switch(chan->type)
  {
   case audio_AIFF: destroy_sample(chan->data->spl); chan->data->spl=NULL; break;
   case audio_MOD: destroy_mod(chan->data->mod); chan->data->mod=NULL; break;
  }
}

void glk_schannel_set_volume(schanid_t chan, glui32 vol)
{
  if (!chan) gli_strict_warning("schannel_set_volume: invalid id.");
  chan->volume=vol;
  switch(chan->type)
  {
   case audio_AIFF: adjust_sample(chan->data->spl,(chan->volume)/256,128,1000,(chan->loop==-1));
    break;
   case audio_MOD: set_mod_volume((chan->volume)/256); break;
  }
  
}

void glk_sound_load_hint(glui32 snd, glui32 flag)
{
}

#define VIRTUAL_VOICES  256


typedef struct          /* a virtual (as seen by the user) soundcard voice */
{
   SAMPLE *sample;      /* which sample are we playing? (NULL = free) */
   int num;             /* physical voice number (-1 = been killed off) */
   int autokill;        /* set to free the voice when the sample finishes */
   long time;           /* when we were started (for voice allocation) */
   int priority;        /* how important are we? */
} VOICE;

extern VOICE _voice[VIRTUAL_VOICES];

int is_sample_playing(SAMPLE *spl)
{
   int c;

   for (c=0; c<VIRTUAL_VOICES; c++)
      if (voice_check(c)==spl)
        return 1;
 return 0;
}
int gt_get_sound_events()
{
  schanid_t chan=NULL;
  int event=0; int done;
  while((chan=glk_schannel_iterate(chan,NULL)))
   switch(chan->type)
   {
    case audio_MOD: done=!is_mod_playing();
                if (done) {
                     if (chan->loop>1)
                     {
                      chan->loop--;
                    play_mod(chan->data->mod,0);
                     }
                    else {
                          chan->type=0;
                          destroy_mod(chan->data->mod); chan->data->mod=NULL;
                          if (chan->notify && !event)
                          { event=1;
                            gli_event_store(evtype_SoundNotify,NULL,chan->rock,chan->notify);
                          }
                         }
                      }  
                break;

    case audio_AIFF: done=!is_sample_playing(chan->data->spl);
                if (done) {
                     if (chan->loop>1)
                     {
                      chan->loop--;
                      play_sample(chan->data->spl,
                                (chan->volume)/256,128,1000,0);
                     }
                    else {
                          chan->type=0;
                          destroy_sample(chan->data->spl); chan->data->spl=NULL; break;
                          if (chan->notify && !event)
                          { event=1;
                            gli_event_store(evtype_SoundNotify,NULL,chan->rock,chan->notify);
                          }
                         }}
                break;
    }
  return event;
}
#endif /* GLK_MODULE_SOUND */
#else
/* gtschan.c: Sound channel objects
        for GlkTerm, curses.h implementation of the Glk API.
    Designed by Andrew Plotkin <erkyrath@eblong.com>
    http://www.eblong.com/zarf/glk/index.html
*/

#include "gtoption.h"
#include <stdio.h>
#include "glk.h"
#include "glkdos.h"

/* The whole sound-channel situation is very simple for us;
   we don't support it. */
int gi_sound_check() { return 0; }
#ifdef GLK_MODULE_SOUND

schanid_t glk_schannel_create(glui32 rock)
{
  return NULL;
}

void glk_schannel_destroy(schanid_t chan)
{
}

schanid_t glk_schannel_iterate(schanid_t chan, glui32 *rockptr)
{
  if (rockptr)
    *rockptr = 0;
  return NULL;
}

glui32 glk_schannel_get_rock(schanid_t chan)
{
  gli_strict_warning("schannel_get_rock: invalid id.");
  return 0;
}

glui32 glk_schannel_play(schanid_t chan, glui32 snd)
{
  gli_strict_warning("schannel_play: invalid id.");
  return 0;
}

glui32 glk_schannel_play_ext(schanid_t chan, glui32 snd, glui32 repeats,
    glui32 notify)
{
  gli_strict_warning("schannel_play_ext: invalid id.");
  return 0;
}

void glk_schannel_stop(schanid_t chan)
{
  gli_strict_warning("schannel_stop: invalid id.");
}

void glk_schannel_set_volume(schanid_t chan, glui32 vol)
{
  gli_strict_warning("schannel_set_volume: invalid id.");
}

void glk_sound_load_hint(glui32 snd, glui32 flag)
{
  gli_strict_warning("schannel_sound_load_hint: invalid id.");
}

#endif /* GLK_MODULE_SOUND */
#endif
