/***************************************************************************/
/* zcpatch.c version 2.3                                                   */
/*                                                                         */
/* Applies a patchfile produced by zcdiff to a source gamefile to produce  */
/* a target gamefile.                                                      */
/*                                                                         */
/* This is done by XORing the patchfile with the source gamefile,          */
/* hence may be seen as "decryption", complementary to zcdiff being        */
/* "encryption".                                                           */
/*                                                                         */
/*                                                                         */
/* Copyright (c) 1997/98 by Paul Gilbert (paulfgilbert AT gmail.com)       */
/* Modifications Copyright (c) 2000/2001 Rodney Hester and Nils Barth      */
/* and 2009 by Mike Ciul and Nils Barth                                    */
/*                                                                         */
/*                                                                         */
/* This program is free software: you can redistribute it and/or modify    */
/* it under the terms of the GNU General Public License as published by    */
/* the Free Software Foundation, either version 3 of the License, or       */
/* (at your option) any later version.                                     */
/*                                                                         */
/* This program is distributed in the hope that it will be useful,         */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of          */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the           */
/* GNU General Public License for more details.                            */
/*                                                                         */
/* You should have received a copy of the GNU General Public License       */
/* along with this program.  If not, see <http://www.gnu.org/licenses/>.   */
/*                                                                         */
/***************************************************************************/

#define VERSION "2.3"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <limits.h>

/* Compiler check that unsigned long ints are at least 4 bytes */
#if (ULONG_MAX < 4294967295)
#error "Code requires that unsigned long ints be at least 32 bits"
#endif

typedef struct {
  unsigned int zcode_version;
  unsigned int release;
  unsigned char version[6];
} FILE_DATA;

char *patchname, *cryptname, *outname;
FILE *patchfile, *cryptfile, *outfile;
FILE_DATA outfile_data, cryptfile_data;
char game[33];
unsigned long int cryptfilesize;

void open_files(void);
void verify_patchfile(void);
void verify_cryptfile(void);
void retrieve_details(FILE_DATA *data);
void get_details(void);
void verify_decryptfile(void);
void extract_gamename(int argv, char *argc[]);
void fprint_hex_digit(FILE *stream, int d);
void fprint_version(FILE *s, FILE_DATA *d);
void decrypt_file(void);
void close_files(void);


/* open_files
 * Opens up the input file, output file, and decryption file for access.
 * Exits the program if any of the openings fail.
 */

void open_files()
{
 /* Open up the input file for access */
  if ((patchfile = fopen(patchname, "rb")) == NULL)
  {
    /* Input patch file could not be opened, exit with error */
    fprintf(stderr, "Could not open input patch file %s\n", patchname);
    exit(1);
  }

  /* Open up the decryption file for access */
  if ((cryptfile = fopen(cryptname, "rb")) == NULL)
  {
    /* decryption file opening failed, exit with error */
    fclose(patchfile);
    fprintf(stderr, "Could not open source file %s.\n", cryptname);
    exit(1);
  }

  /* Open up the output file for writing */
  if ((outfile = fopen(outname, "wb")) == NULL)
  {
    /* Output file opening failed, exit with error */
    fclose(patchfile); fclose(cryptfile);
    fprintf(stderr, "Could not open output file %s.\n", outname);
    exit(1);
  }
}

/* verify_patchfile
 * Verifies that the patch file is a valid patch file. When it returns, the
 * file pointer points after the header "PFG" to the start of the version
 * details
 */
void verify_patchfile()
{
  int c;

  /* Test the patch file to ensure that it's valid (starts with "PFG") */
  c = fgetc(patchfile);
  if (c == 'P')
  {
    c = fgetc(patchfile);
    if (c == 'F')
    {
      c = fgetc(patchfile);
      if (c == 'G')
      {
        /* Valid "PFG" header, so return */
        return;
      }
    }
  }

  /* Invalid header for patchfile, so close files and exit */
  close_files();
  fprintf(stderr, "'%s' is not a valid Infocom patchfile.\n\n", patchname);
  exit(1);
}

/* verify_cryptfile
 * Verifies that the decryption file is a valid Infocom gamefile
 */

void verify_cryptfile()
{
  int c;

  /* Test the decryption file to see if it's a valid Infocom gamefile */
  c = fgetc(cryptfile);
  /* If the value is not in the range of 1 to 8, then not an Infocom game */
  if ((c < 1) || (c > 8))
  {
    fprintf(stderr, "source file '%s' is not a valid gamefile.\n\n", cryptname);
    close_files();
    exit(1);
  }

  /* Reset back to the start of the decryption file */
  fseek(cryptfile, 0, SEEK_SET);
}

/* get_version_details
 * Reads the game name and version details of both the required decryption
 * gamefile and the output gamefile version from the patch file. Uses
 * retrieve_details to retrieve the data
 */

void retrieve_details(FILE_DATA *data)
{
  int c, ctr;

  /* Get in the zcode version number */
  data->zcode_version = fgetc(patchfile);

  /* Get release number */
  c = fgetc(patchfile);            /* Get in HI byte of release */
  data->release = c*256;           /* Move into release number  */
  c = fgetc(patchfile);            /* Get in LO byte of release */
  data->release += c;              /* Add into release value    */

  /* Get in version */
  for (ctr=0; ctr<6; ++ctr)
    data->version[ctr] = (char) fgetc(patchfile);
}

void get_details()
{
  int ctr;

  /* Read in the game name */
  for (ctr=0; ctr<32; ctr++) game[ctr] = (char) fgetc(patchfile);

  /* Read in the version details for output file and required decrypt file */
  retrieve_details(&outfile_data);
  retrieve_details(&cryptfile_data);
}

/* verify_decryptfile
 * Once the required gamefile version data has been filled out, this method
 * is called. It reads in from the supplied decryption file to ensure that
 * it matches the required decryption gamefile version data
 */

void verify_decryptfile()
{
  unsigned int c, ctr;


  /* Test that zcode version matches */
  fseek(cryptfile, 0, SEEK_SET);
  c = fgetc(cryptfile);
  if (c != cryptfile_data.zcode_version) goto Error;

  /* Test that release number matches */
  fseek(cryptfile, 2, SEEK_SET);
  c = fgetc(cryptfile);
  if (c != (cryptfile_data.release / 256)) goto Error;
  c = fgetc(cryptfile);
  if (c != (cryptfile_data.release % 256)) goto Error;

  /* Test that version string matches */
  fseek(cryptfile, 0x12, SEEK_SET);
  for (ctr=0; ctr<6; ctr++)
  {
    c = fgetc(cryptfile);
    if (c != cryptfile_data.version[ctr]) goto Error;
  }

  /* Correct gamefile is present, so extract file size */
  fseek(cryptfile, 0x1A, SEEK_SET);
  cryptfilesize = 256 * ((unsigned int) fgetc(cryptfile));
  cryptfilesize += (unsigned int) fgetc(cryptfile);

  /* Multiply compressed filesize by correct multiplier */
  switch (cryptfile_data.zcode_version)
  {
    case 1: case 2: case 3:
      cryptfilesize *= 2; break;
    case 4: case 5:
      cryptfilesize *= 4; break;
    case 6: case 7: case 8:
      cryptfilesize *= 8; break;
  }
  if (cryptfilesize == 0)
  {
    /* Issue a warning and use the entire file */
    fprintf(stderr, "Warning: Source file '%s' does not have an embedded file size, ", outname);
    fprintf(stderr, "so the entire file\n");
    fprintf(stderr, "Will be used. You should make sure the gamefile does not have ");
    fprintf(stderr, "any padding, or an\n");
    fprintf(stderr, "inaccurate output file may be generated.\n\n");

    /* Work out the physical file size */
    fseek(cryptfile, 0, SEEK_END);
    cryptfilesize = ftell(cryptfile);
  }

  /* Reset file pointer back to start of file */
  fseek(cryptfile, 0, SEEK_SET);
  return;

  /* Error jump point */
Error:
  close_files();
  fprintf(stderr, "Output file '%s' does not match the specification in the header of\n", outname);
  fprintf(stderr, "patch file '%s'. Make sure source file '%s'\n", patchname, cryptname);
  fprintf(stderr, "is actually version ");
  fprint_version(stderr, &cryptfile_data);
  fprintf(stderr, " of \"%s\".\n", game);
  
  exit(1);
}

/* fprint_version
 * prints out the version given a data structure
 * Format: release/version [vzcode-version]
 */

void fprint_hex_digit(FILE *stream, int d)
{
  char digits[16] = "0123456789ABCDEF";
  fprintf(stream, "%c", digits[d]);
}

void fprint_version(FILE *stream, FILE_DATA *d)
{
  int ctr;
  int printable;

  fprintf(stream, "%d/", d->release);

  /* Check to make sure all 6 version digits are printable. If not, */
  /* print all six in hexadecimal format                            */
  printable = 1;
  for (ctr=0; (ctr<6) && (printable); ctr++)
  {
    if ((printable) && ((d->version[ctr] < ' ') || (d->version[ctr] > '\127')))
      printable = 0;
  }
  /* Print digits depending on resulting printable flag */
  if (printable)
  {
    for (ctr=0; ctr<6; ctr++) fprintf(stream, "%c", d->version[ctr]);
  }
  else
  {
    fprintf(stream, "$");
    for (ctr=0; ctr<6; ctr++)
    {
      fprint_hex_digit(stream, ((unsigned char) d->version[ctr]) / 16);
      fprint_hex_digit(stream, ((unsigned char) d->version[ctr]) % 16);
    }
  }

  fprintf(stream, " [v%d]", d->zcode_version);
}

/* decrypt_file
 * Performs the decryption of the input file and writing to the output file
 */

void decrypt_file()
{
  unsigned long int filepos;
  unsigned int c1, c2;

  /* Print out the details of what is being encrypted */
  printf("Patching game \"%s\" ", game);
  fprint_version(stdout, &outfile_data);
  printf("\nusing ");
  fprint_version(stdout, &cryptfile_data);
  printf(" as source.\n\n");


  /* Main loop for encrypting the file */
  filepos = 0;
  for (;;)
  {
    /* Read a byte in from the patch file  */
    if ((c1 = fgetc(patchfile)) == (unsigned int) -1)
    {
      /* Reached end of patch file, so gamefile is completely decrypted */
      return;
    }
    if ((c2 = fgetc(cryptfile)) == (unsigned int) -1)
    {
      /* decryption file EOF reached. This should only happen if the */
      /* decryption gamefile specified has no file length specifier  */
      filepos = 0;
      fseek(cryptfile, 0, SEEK_SET);
      c2 = fgetc(cryptfile);
    }

    /* Decrypt the byte via XOR and write it out to the output file */
    fputc((char) (c1 ^ c2), outfile);
    ++filepos;

    /* If the decryption file position moves past that specified in the */
    /* header, reset back to the file start. This ensures that files    */
    /* with padding bytes can still be used for decryption              */
    if (filepos >= cryptfilesize)
    {
      filepos = 0;
      fseek(cryptfile, 0, SEEK_SET);
    }
  }
}

/* close_files
 * Closes all three of the files
 */

void close_files()
{
  fclose(patchfile);
  if (outfile) {
    fclose(outfile);
  }
  if (cryptfile) {
    fclose(cryptfile);
  }
}

/*-------------------------------------------------------------------------*/

int main(int argc, char *argv[])
{
  /* Print out the name of the program */
  printf("zcpatch version " VERSION "\n\n");

  /* First check to make sure two parameters (the input file and */
  /* decryption file) have been entered.                         */
  if ((argc < 2) || ((argc == 2) &&
      ((strcmp(argv[1], "?")==0) || (strcmp(argv[1], "/?")==0))))
  {
    printf("Applies a patchfile produced by zcdiff to an Infocom gamefile.\n\n");
    printf("zcpatch patch.pat source.dat output.dat\n\n");
    printf("  patch.pat   Patchfile to apply\n");
    printf("  input.dat   Existing gamefile to patch\n");
    printf("  output.dat  File name for patched gamefile to produce\n\n");
    printf("Alternative use:\n\n");
    printf("zcpatch patch.pat\n\n");
    printf("    Prints versions details of the patch:\n");
		printf("    which version it requires, and which version it produces.\n");
    exit(1);
  }

  /* If a single file provided, simply print the details of it */
  if (argc == 2)
  {
    patchname = argv[1];

    /* Open patch file */
    if ((patchfile = fopen(patchname, "rb")) == NULL)
    {
      fprintf(stderr, "Could not open input patch file\n");
      exit(1);
    }
    cryptfile = NULL;
    outfile = NULL;

    /* Verify that the patch file is valid */
    verify_patchfile();
    /* Get the version information from the patch file */
    get_details();

    /* Print out the details */
    printf("Game: %s\n", game);
    printf("Produces version: "); fprint_version(stdout, &outfile_data); printf("\n");
    printf("Requires version: "); fprint_version(stdout, &cryptfile_data); printf("\n\n");
    exit(0);
  }

  patchname = argv[1];
  cryptname = argv[2];
  outname = argv[3];

  /* Open up the files */
  open_files();

  /* Make sure the files are valid  */
  verify_patchfile();
  verify_cryptfile();

  /* Extract the input file and game version data */
  get_details();

  /* Verify that the decryption file is the one required */
  verify_decryptfile();

  /* Decrypt the file */
  decrypt_file();

  /* Close the files */
  close_files();

  printf("Patch applied successfully.\n");
  return 0;
}
