
/* This file is part of "MidpSSH".
 * Copyright (c) 2005 Karl von Randow.
 * Modified 2008 by William H. Johnson for ZaxMidlet
 * Contains the ZUserinterface implementation 
 * 
 * --LICENSE NOTICE--
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 * --LICENSE NOTICE--
 *
 */
package terminal;

import java.io.InputStream;
import java.util.Enumeration;
import java.util.Vector;

import javax.microedition.lcdui.*;

import terminal.Settings;
//import app.session.Session;
import zmachine.ZUserInterface;
import zmachine.ui.*;
 

/**
 * Class that acts as terminal. It can basicly draw input from emulation (see
 * variable "buffer"), execute and store actions defined by user.
 */

public class Terminal 
	extends Canvas 
	implements ZUserInterface, CommandListener {
	
    private static final int MODE_DISCONNECTED = 0;
    
    private static final int MODE_CONNECTED = 1;

//#ifndef nocursororscroll
    private static final int MODE_CURSOR = 2;

    private static final int MODE_SCROLL = 3;
//#endif

    private int[] cursorpos = new int[2];
    private int[] fontsize = new int[2];
    private String curr_cmd = "";

    private int previous_window = 0;
    private int[] cursor_row = new int[5];
    private int[] cursor_col = new int[5];
    
    private int upper_size = 0;
    private int status_size = 0;
    private int new_char=0;
    private int read_char=0;
    
	// The midlet the canvas is working for
	private ZaxMidlet midlet;
	private Vector inputQueue;
	private int  Last_style =0;
	private FileDialog filedialog;    
	private int reading_line = 0;

	private boolean buffermode = true;
	
//#ifndef notyping
    private static final int MODE_TYPING = 4;
//#endif
    
    private static int commandPriority = 1;

	private Command exit = new Command("Exit", Command.EXIT, 3);

 
    private static final Command textTypeCommand = new Command("Type", Command.OK, commandPriority++);
    
    // Have this separate back command as a Command.ITEM so that it will show first in the menu on
    // the phone, so that you know you're in typing mode
    private static final Command backMainCommand = new Command( "Back", Command.ITEM, commandPriority++ );
    
    private static final Command textInputCommand = new Command( "Input", Command.ITEM, commandPriority++ );
	private Command quitstory = new Command("Quit Story", Command.ITEM, commandPriority++ );

    
    
    private static final Command textEnterCommand = new Command("Enter", Command.ITEM, commandPriority++);
    private static final Command typeCommand = new Command( "Type", Command.ITEM, commandPriority++ );
   
//#ifndef nomacros
    private static final Command macrosCommand = new Command( "Macros", Command.ITEM, commandPriority++ );
//#endif
    
    private static final Command tabCommand = new Command( "TAB", Command.ITEM, commandPriority++ );

    private static final Command spaceCommand = new Command( "SPACE", Command.ITEM, commandPriority++ );

    private static final Command enterCommand = new Command( "ENTER", Command.ITEM, commandPriority++ );

    private static final Command escCommand = new Command( "ESC", Command.ITEM, commandPriority++ );

    //private static final Command backspaceCommand = new Command( "BACKSPACE", Command.ITEM, commandPriority++ );

    private static final Command ctrlCommand = new Command( "CTRL", Command.ITEM, commandPriority++ );

    private static final Command altCommand = new Command( "ALT", Command.ITEM, commandPriority++ );
    
    private static final Command shiftCommand = new Command( "SHIFT", Command.ITEM, commandPriority++ );

//#ifndef nospecialmenu
    private static final Command specialCommand = new Command( "Special", Command.ITEM, commandPriority++ );
//#endif
    
//#ifndef nocursororscroll
    private static final Command cursorCommand = new Command( "Cursor", Command.ITEM, commandPriority++ );

    private static final Command scrollCommand = new Command( "Scroll", Command.ITEM, commandPriority++ );
//#endif
    
    private static final Command backCommand = new Command( "Back", Command.BACK, commandPriority++ );
    
    //#ifndef noinstructions
    private static final Command showBindingsCommand = new Command( "Show Key Bindings", Command.ITEM, commandPriority++ );
    //#endif
    
    //private static final Command settingsCommand = new Command( "Settings", Command.ITEM, commandPriority++ );
    
    private static final Command disconnectCommand = new Command( "Disconnect", Command.ITEM, commandPriority++ );
    
    private static final Command closeCommand = new Command( "Exit", Command.STOP, commandPriority++ );

    private static final Command[] commandsDisconnected = new Command[] {
            closeCommand
    };
    
    private static final Command[] commandsConnected = new Command[] {
        textInputCommand,
//#ifndef notyping
        typeCommand,
//#endif
//#ifndef nomacros
        macrosCommand,
//#endif
        tabCommand,
        spaceCommand,
        enterCommand,
        escCommand, 
        //backspaceCommand,
        ctrlCommand,
        altCommand,
        shiftCommand,
//#ifndef nospecialmenu     
        specialCommand,
//#endif
//#ifndef nocursororscroll
        cursorCommand, scrollCommand,
//#endif
        //#ifndef noinstructions
        showBindingsCommand,
        //#endif
        disconnectCommand
    };

//#ifndef nocursororscroll
    private static final Command[] commandsCursor = new Command[] {
        backCommand
    };
//#endif

//#ifndef notyping
    private static final Command[] commandsTyping = new Command[] {
        backMainCommand,
        backCommand,
        textInputCommand,
//#ifndef nomacros
        macrosCommand,
//#endif
        tabCommand,
        spaceCommand,
        enterCommand,
        escCommand, 
        //backspaceCommand,
        ctrlCommand,
        altCommand,
        shiftCommand,
//#ifndef nospecialmenu
        specialCommand,
//#endif
//#ifndef nocursororscroll
        cursorCommand, scrollCommand,
//#endif
        disconnectCommand
    };
//#endif
    
    private static int [] bindingKeys = new int[] {
            Canvas.KEY_NUM1, Canvas.KEY_NUM2, Canvas.KEY_NUM3,
            Canvas.KEY_NUM4, Canvas.KEY_NUM5, Canvas.KEY_NUM6,
            Canvas.KEY_NUM7, Canvas.KEY_NUM8, Canvas.KEY_NUM9,
            Canvas.KEY_STAR, Canvas.KEY_NUM0, Canvas.KEY_POUND
    };

//    private Session session;

    private TextBox inputDialog;
    
//#ifndef nospecialmenu 
//    private SpecialMenu menuSpecialKeys;
//#endif
    
    private TextBox controlKeyDialog, altKeyDialog, shiftKeyDialog;

    private Command[] currentCommands;

    private int mode;

    private Settings settings;
    /**
     * @param buffer
     */
    public Terminal( VT320 buffer, ZaxMidlet midlet, Settings settings) {
        this.buffer = buffer;
        this.settings = settings;
        this.midlet = midlet;
        buffer.setDisplay( this );
            fgcolor = color[7];
            bgcolor = color[0];

        
		inputQueue = new Vector();
		buffer.setCursorPosition(0, 1);
		buffer.setTopMargin(1);

		rotated = settings.ROT_NORMAL;
        initFont();
        setFullScreenMode( true );

        top = 0;
        left = 0;
        changeMode( MODE_DISCONNECTED );

        setCommandListener( this );
        bgcolor = settings.bgcolor;
//            /* If specified fgcolor is white then use default fgcolor, which is our off white */
            if (settings.fgcolor != 0xffffff) {
                fgcolor = settings.fgcolor;
            }
            
        sizeChanged();
        // Assume that the top line is be used as a status bar.
        // I think this will break "seastalker" which uses splitwindow AND the status bar,but oh well.
        //splitScreen(1);
        setTextStyle(0); 
        buffer.setCursorPosition(0, 1);
        
        
        
        addCommand(quitstory);
        addCommand(textInputCommand);
        addCommand(exit);

        
        
        
    }

//#ifdef midp2
    /* (non-Javadoc)
     * @see javax.microedition.lcdui.Displayable#sizeChanged(int, int)
     */
    protected void sizeChanged(int w, int h) {
        super.sizeChanged(w, h);
        sizeChanged();
    }
//#endif
    
    protected void sizeChanged() {
        width = getWidth();
        height = getHeight();
        if ( rotated != settings.ROT_NORMAL ) {
            width = getHeight();
            height = getWidth();
        }
        cols = width / fontWidth;
        rows = height / fontHeight ;
        backingStore = Image.createImage( width, height );
        
        int virtualCols = cols;
        int virtualRows = rows;
        
        if ( settings.terminalCols != 0 ) {
            virtualCols = settings.terminalCols;
        }
        if ( settings.terminalRows != 0 ) {
            virtualRows = settings.terminalRows;
        }
        
        //System.out.println( "ROWS " + virtualRows + " COLS " + virtualCols );
        
        buffer.setScreenSize( virtualCols, virtualRows );
    }
    
    public void connected() {
        changeMode( MODE_CONNECTED );
    }
    
    public void disconnected() {
        changeMode( MODE_DISCONNECTED );
    }

    protected void changeMode( int mode ) {
        this.mode = mode;

        switch ( mode ) {
            case MODE_DISCONNECTED:
                changeCurrentCommands( commandsDisconnected );
                break;
            case MODE_CONNECTED:
                changeCurrentCommands( commandsConnected );
                break;
//#ifndef nocursororscroll
            case MODE_CURSOR:
            case MODE_SCROLL:
                changeCurrentCommands( commandsCursor );
                break;
//#endif
//#ifndef notyping
            case MODE_TYPING:
                changeCurrentCommands( commandsTyping );
                break;
//#endif
        }
    }

    protected void changeCurrentCommands( Command[] commands ) {
//        if ( currentCommands != null ) {
//           for ( int i = 0; i < currentCommands.length; i++ ) {
//                removeCommand( currentCommands[i] );
//            }
//        }
//
//        for ( int i = 0; i < commands.length; i++ ) {
//            addCommand( commands[i] );
//        }
//
//        this.currentCommands = commands;
    }

    /*
     * (non-Javadoc)
     * 
     * @see gui.Activatable#activate()
     */
    public void activate() {
    	midlet.getDisplay().setCurrent(this);
//		try{
//		// Kludge to give the terminal time to get ready - Random crashes without this...
//		 Thread.sleep(700);
 //       } catch (Exception ex){}            
    
    }
    	
    	
 //   	try{
 //           Thread.sleep(500);  // Give it some time to activate 
 //       	} catch (Exception ex){}
		//      MainMenu.setDisplay( this );
 //   }
    
 //   public void activate( Activatable back ) {
 //       activate();
//    }

    public void commandAction( Command command, Displayable displayable ) {
    	if (displayable == inputDialog) {
    		if (command != backCommand) {
	            commandBuffer.setLength( 0 );
	            commandBuffer.append(inputDialog.getString());
	    		inputDialog.setString( "" );
	            activate();
	            if ( command == textEnterCommand ) {
	                commandBuffer.append( '\n' );
	                buffer.putStringUpper(commandBuffer.toString());
	                int i;
	                String tmp_cmd = "";
		            for (i=0;i<commandBuffer.length();i++)
					if (((commandBuffer.charAt(i) == '\n')  || (commandBuffer.charAt(i)=='\r')) && tmp_cmd != "") // Enter
					{
						synchronized (inputQueue) {
							inputQueue.addElement(tmp_cmd.substring(0,tmp_cmd.length()));
							inputQueue.notifyAll();
						}
						tmp_cmd = "";
					}
					else
						tmp_cmd = tmp_cmd + commandBuffer.charAt(i);
	            }
	            else // Type Command
	            {
		            new Thread(new Runnable() {
		                public void run() {
		                    int i;
	    	        		try{
	    	                    Thread.sleep(100);  // Kludge 
	    	                	} catch (Exception ex){}
		    	            for (i=0;i<commandBuffer.length();i++)
		    	            {
		    	        		try{
		    	                    Thread.sleep(5);  // Kludge 
		    	                	} catch (Exception ex){}
		    	             	keyReleasedTyping( (int) commandBuffer.charAt(i) );
		    	            }
		                }
		            }).start();
	            }	
    		}
    		else
	            activate();
    			
            
    	
    	
    	}
    	else if (displayable == controlKeyDialog) {
    		handleModifierDialog(command, controlKeyDialog, VT320.KEY_CONTROL);
    	}
    	else if (displayable == altKeyDialog) {
    		handleModifierDialog(command, altKeyDialog, VT320.KEY_ALT);
    	}
    	else if (displayable == shiftKeyDialog) {
    		handleModifierDialog(command, shiftKeyDialog, VT320.KEY_SHIFT);
    	}
    	else if (displayable != this) { 
    		/* A message form or something, just come back to this screen */
    		activate();
    	}
    	else if ( command == disconnectCommand || command == closeCommand ) {
            doDisconnect();
        }
//#ifndef nomacros
        else if ( command == macrosCommand ) {
//            MainMenu.doMacros(this);
        }
//#endif
        else if ( command == tabCommand ) {
            buffer.keyTyped( 0, '\t', 0 );
        }
        else if ( command == spaceCommand ) {
            buffer.keyTyped( 0, ' ', 0 );
        }
        else if ( command == enterCommand ) {
            buffer.keyTyped( 0, '\n', 0 );
        }
        else if ( command == escCommand ) {
            buffer.keyTyped( 0, (char) 27, 0 );
        }
        /*else if ( command == backspaceCommand ) {
            buffer.keyPressed( VT320.VK_BACK_SPACE, 0 );
        }*/
//#ifndef nocursororscroll
        else if ( command == cursorCommand ) {
            doCursor();
        }
        else if ( command == scrollCommand ) {
            doScroll();
        }
//#endif
//#ifndef notyping
        else if ( command == typeCommand ) {
            doTyping();
        }
//#endif
        else if ( command == textInputCommand ) {
            doTextInput(null);
        }
    	//#ifndef nospecialmenu
        else if ( command == specialCommand ) {
//            if ( menuSpecialKeys == null ) {
//                menuSpecialKeys = new SpecialMenu();
//            }
//            menuSpecialKeys.activate( this );
        }
//#endif        
        else if ( command == backCommand  || command == backMainCommand ) {
            changeMode( MODE_CONNECTED );
        }
    	//#ifndef noinstructions
        else if ( command == showBindingsCommand ) {
            doShowBindings();
        }
    	//#endif
        else if (command == exit) {
        	midlet.notifyPaused();
        } 
        else if (command == quitstory) {
        	midlet.notifyDestroyed();
        } 
    	    
    
    }
	
	private StringBuffer commandBuffer = new StringBuffer();
    
    private void handleModifierDialog(Command command, TextBox box, int modifier) {
    	if (command != backCommand) {
    		String str = box.getString();
	    	for ( int i = 0; i < str.length(); i++ ) {
//				session.typeChar( str.charAt( i ), modifier );
			}
	    	box.setString("");
    	}
    	activate();
    }

    protected void keyPressed( int keycode ) { 
    	keyPressedTyping( keycode );

    }

    protected void keyReleased( int keycode ) {
     	keyReleasedTyping( keycode );
    	
    }

//#ifndef nocursororscroll
    protected void keyRepeated( int keycode ) {
        switch ( mode ) {
            case MODE_CURSOR:
                keyPressedCursor( keycode );
                break;
            case MODE_SCROLL:
                keyPressedScroll( keycode );
                break;
        }
    }
//#endif
    
    protected boolean handleGameAction( int keycode ) {
        int gameAction = getGameAction( keycode );
        
        if ( gameAction != 0 ) {
            switch ( gameAction ) {
            case Canvas.UP:
                buffer.keyPressed( VT320.VK_UP, VT320.KEY_ACTION );
                return true;
            case Canvas.DOWN:
                buffer.keyPressed( VT320.VK_DOWN, VT320.KEY_ACTION );
                return true;
            case Canvas.LEFT:
                buffer.keyPressed( VT320.VK_LEFT, VT320.KEY_ACTION );
                return true;
            case Canvas.RIGHT:
                buffer.keyPressed( VT320.VK_RIGHT, VT320.KEY_ACTION );
                return true;
            }
        }
        return false;
    }

 
 
    private static final int KEY_BACKSPACE = -8; // Keycode for clear on sony
    
//#ifndef notyping
    private static final int KEY_SHIFT = 137; // Keycode for shift on blackberry
    
    private boolean typingShift;
    
    protected void keyPressedTyping( int keycode ) {
   
    	
    	
    	if ( keycode == KEY_SHIFT ) {
            typingShift = true;
        }
        
        // If a game action is used, allow it to operate the cursor even when not in cursor mode
        // But need to make sure it's not a character that we might accept for typing
        if ( keycode == 8 || keycode == KEY_BACKSPACE || keycode == 10 || keycode == 13 ||
                keycode == KEY_SHIFT || ( keycode >= 32 && keycode < 128 ) )
        {
            // NOOP in keyPressedTyping
        }
        else {
            if ( handleGameAction( keycode ) ) return;
        }
    }
    
    protected void keyReleasedTyping( int keycode ) {
    	/* Debug typing */
    	//buffer.putString("KEY" + keycode + " ");
    	if ((keycode < 8) || ((keycode > 8) && (keycode < 10)) ||((keycode > 10) && (keycode < 13)) || ((keycode > 13) && (keycode < 27)) || ((keycode > 27) && (keycode < 32))|| ((keycode > 126) && (keycode < 127)) || ((keycode > 127) && (keycode < 129)) || (keycode > 132)) 
    		return;  // Unrecognized Character.
    	
    	
    	new_char = keycode;
//    	if (read_char == 0 && buffer.keypressed ==1)  // not in readchar mode
		if(reading_line == 1)
    	{
	    	if ( keycode == 8 || keycode == KEY_BACKSPACE || keycode == 127  ) {
	        	            // Backspace
				if(curr_cmd.length() > 0)
					buffer.keyPressed( VT320.VK_BACK_SPACE, 0 );
	        }
	        else if ( keycode == 10 || keycode == 13 ) {
	            //buffer.keyTyped( keycode, (char) keycode, 0 );
	            buffer.keyTyped( 0, '\n', 0 );
	        }
	        else if ( keycode == KEY_SHIFT ) {
	            typingShift = false;
	        }
	        else if ( keycode > 0 && keycode < 32) {
	            buffer.keyTyped( keycode, (char)keycode, 0);
	        }
	        else if ( keycode >= 32 && keycode < 128 ) {
	            char c = (char) keycode;
	            if ( typingShift ) {
	                c = shiftChar( c );
	            }
	            
	            // Don't pass through the keycode, as we don't want the terminal to do any keycode mapping
	            // we just care about the char
	            buffer.keyTyped( 0, c, 0 );
	        }
	 
	    
	   		if  ( keycode == 8 || keycode == KEY_BACKSPACE|| keycode == 127 )  // delete
			{
				if(curr_cmd.length() > 0)
					curr_cmd = curr_cmd.substring(0,curr_cmd.length()-1);
			}
			else
			{
				if ((keycode == 10)  || (keycode==13))  // Enter
				{
					synchronized (inputQueue) {
						inputQueue.addElement(curr_cmd.substring(0,curr_cmd.length()));
						inputQueue.notifyAll();
					}
					curr_cmd = "";
				}
				else
					curr_cmd = curr_cmd + (char)keycode;
					
			}
    
    	}    

    	buffer.keypressed = 1;

    	
    }
    
    private char shiftChar( char c ) {
        if ( c >= 'a' && c <= 'z' ) {
            return (char) ( c - 'a' + 'A' );
        }
        else {
            switch ( c ) {
                case '0': return ')';
                case '1': return '!';
                case '2': return '@';
                case '3': return '#';
                case '4': return '$';
                case '5': return '%';
                case '6': return '^';
                case '7': return '&';
                case '8': return '*';
                case '9': return '(';
                default: return c;
            }
        }
    }
//#endif
    
//#ifndef nocursororscroll
    private int gameKeysToNumeric( int keycode ) {
        // Convert game actions to keys
        int gameAction = getGameAction( keycode );
        switch ( gameAction ) {
            case Canvas.UP:
                keycode = Canvas.KEY_NUM2;
                break;
            case Canvas.DOWN:
                keycode = Canvas.KEY_NUM8;
                break;
            case Canvas.LEFT:
                keycode = Canvas.KEY_NUM4;
                break;
            case Canvas.RIGHT:
                keycode = Canvas.KEY_NUM6;
                break;
        }
        return keycode;
    }

    protected void keyPressedCursor( int keycode ) {
        keycode = gameKeysToNumeric( keycode );
        
        switch ( keycode ) {
            case Canvas.KEY_NUM2:
                buffer.keyPressed( VT320.VK_UP, VT320.KEY_ACTION );
                break;
            case Canvas.KEY_NUM8:
            case Canvas.KEY_NUM0:
                buffer.keyPressed( VT320.VK_DOWN, VT320.KEY_ACTION );
                break;
            case Canvas.KEY_NUM4:
                buffer.keyPressed( VT320.VK_LEFT, VT320.KEY_ACTION );
                break;
            case Canvas.KEY_NUM6:
                buffer.keyPressed( VT320.VK_RIGHT, VT320.KEY_ACTION );
                break;
            case Canvas.KEY_NUM1:
                keyPressedCursor( Canvas.UP );
                keyPressedCursor( Canvas.LEFT );
                break;
            case Canvas.KEY_NUM3:
                keyPressedCursor( Canvas.UP );
                keyPressedCursor( Canvas.RIGHT );
                break;
            case Canvas.KEY_NUM7:
            case Canvas.KEY_STAR:
                keyPressedCursor( Canvas.DOWN );
                keyPressedCursor( Canvas.LEFT );
                break;
            case Canvas.KEY_NUM9:
            case Canvas.KEY_POUND:
                keyPressedCursor( Canvas.DOWN );
                keyPressedCursor( Canvas.RIGHT );
                break;
        }
    }

    protected void keyPressedScroll( int keycode ) {
        keycode = gameKeysToNumeric( keycode );
        
        switch ( keycode ) {
            case Canvas.KEY_NUM2:
                if ( top > 0 ) {
                    top--;
                }
                redraw();
                break;
            case Canvas.KEY_NUM8:
            case Canvas.KEY_NUM0:
                if ( top + rows < buffer.height ) {
                    top++;
                }
                redraw();
                break;
            case Canvas.KEY_NUM4:
                if ( left > 0 ) {
                    left--;
                }
                redraw();
                break;
            case Canvas.KEY_NUM6:
                if ( left + cols < buffer.width ) {
                    left++;
                }
                redraw();
                break;
            case Canvas.KEY_NUM1:
                keyPressedScroll( Canvas.UP );
            keyPressedScroll( Canvas.LEFT );
                break;
            case Canvas.KEY_NUM3:
                keyPressedScroll( Canvas.UP );
            keyPressedScroll( Canvas.RIGHT );
                break;
            case Canvas.KEY_NUM7:
            case Canvas.KEY_STAR:
                keyPressedScroll( Canvas.DOWN );
                keyPressedScroll( Canvas.LEFT );
                break;
            case Canvas.KEY_NUM9:
            case Canvas.KEY_POUND:
                keyPressedScroll( Canvas.DOWN );
                keyPressedScroll( Canvas.RIGHT );
                break;
        }
    }
//#endif
    
    private void doDisconnect() {
//        session.disconnect();
//        session.goMainMenu();
    }
    

    


 
//#ifndef nocursororscroll
    public void doCursor() {
        changeMode( MODE_CURSOR );
    }

    public void doScroll() {
        changeMode( MODE_SCROLL );
    }
//#endif
    
//#ifndef notyping
    public void doTyping() {
        changeMode( MODE_TYPING );
    }
//#endif
    
    //#ifndef noinstructions
    private void doShowBindings() {
        StringBuffer str = new StringBuffer();
        
        if ( currentCommands != null ) {
            for ( int i = 0; i < bindingKeys.length && i < currentCommands.length; i++ ) {
                int keycode = bindingKeys[i];
                Command comm = currentCommands[i];
                String keyName = getKeyName( keycode );
                str.append( keyName );
                str.append( ": " );
                str.append( comm.getLabel() );
                str.append( "\n" );
            }
        }
        
//        MainMenu.showMessage("Key Bindings", str.toString(), this);
    }
    //#endif
    
    public void doTextInput(String text) {
        if ( inputDialog == null ) {
            inputDialog = new TextBox( "Input", "", 255, TextField.ANY );
            inputDialog.addCommand( textEnterCommand );
            inputDialog.addCommand( typeCommand );
            //inputDialog.addCommand( tabCommand );
            inputDialog.addCommand( backCommand );

    		//#ifdef midp2
    		if (!Settings.predictiveText) {
    			inputDialog.setConstraints(TextField.ANY | TextField.NON_PREDICTIVE);
    		}
    		//#endif
    		
    		inputDialog.setCommandListener(this);
        }
        if (text != null) {
        	inputDialog.setString(text);
        }
        midlet.getDisplay().setCurrent(inputDialog);
        //midlet.setDisplay(inputDialog);
    }
    
    
    
	/** the VDU buffer */
	protected VT320 buffer;

	/** first top and left character in buffer, that is displayed */
	protected int top, left;
	
	protected int width, height;
	
	private int fontWidth, fontHeight;
	
	protected int rotated;

	/** display size in characters */
	public int rows, cols;

	private Image backingStore = null;


	/** A list of colors used for representation of the display */
    
    private static final int color[] = {
            // black, red, green, yellow
            0x000000, 0xcc0000, 0x00cc00, 0xcccc00,
            // blue, magenta, cyan, white
            0x0000cc, 0xcc00cc, 0x00cccc, 0xcccccc 
    };
    

	public int fgcolor = 0x000000;

	public int bgcolor = 0xffffff;
//	public int bgcolor = color[6];
    
    
    private static final int boldcolor[] = {
            // black, red, green, yellow
            0x333333, 0xff0000, 0x00ff00, 0xffff00,
            // blue, magenta, cyan, white
            0x0000ff, 0xff00ff, 0x00ffff, 0xffffff
    };
    
    private static final int lowcolor[] = {
            // black, red, green, yellow
            0x000000, 0x990000, 0x009900, 0x999900,
            // blue, magenta, cyan, white
            0x000099, 0x990099, 0x009999, 0x999999 
    };

    //#ifndef nopaintsync
	private Object paintMutex = new Object();
	//#endif
	
	protected void paint( Graphics g ) {
		
		// Erase display
		g.setColor( bgcolor );
		g.fillRect( 0, 0, getWidth(), getHeight() );

		// Draw terminal image
//#ifndef nopaintsync
		synchronized ( paintMutex ) {
//#endif
			// Redraw backing store if necessary
			redrawBackingStore();
			
			/*
			 * Note the y coord is offset by 1 because without the offset it sometimes fails to
			 * draw on my SonyEricsson K700i. This is seen in width - 1 on the two rotated image widths
             * and the y coord of 1 in drawImage.
			 */
			Image image = backingStore;

			g.drawImage( image, 0, 1, Graphics.TOP | Graphics.LEFT );
//#ifndef nopaintsync
		}
//#endif
	}

	private boolean invalid = true;

	public void redraw() {
//#ifndef nopaintsync
	    synchronized ( paintMutex ) {
//#endif
	        invalid = true;
	        repaint();
//#ifndef nopaintsync
	    }
//#endif
	}

	protected void redrawBackingStore() {
		// Only redraw if we've been marked as invalid by a call to redraw
		// The idea is that if multiple calls to redraw occur before the call to
		// paint then we save
		// time not redrawing our backingStore each time
		if ( invalid ) {
			//long st = System.currentTimeMillis();
			
			Graphics g = backingStore.getGraphics();
			g.setColor( bgcolor );
			g.fillRect( 0, 0, width, height );

			for ( int l = top; l < buffer.height && l < ( top + rows ); l++ ) {
				if ( !buffer.update[0] && !buffer.update[l + 1] ) {
					continue;
				}
				buffer.update[l + 1] = false;
				for ( int c = left; c < buffer.width && c < ( left + cols ); c++ ) {
					int addr = 0;
					int currAttr = buffer.charAttributes[buffer.windowBase + l][c];

					int fg = fgcolor;
					int bg = bgcolor;
					
					
//					if (MainMenu.useColors) {
	                    int fgcolorindex = ( ( currAttr & VT320.COLOR_FG ) >> 4 ) - 1;
						if ( fgcolorindex >= 0 && fgcolorindex < 8 ) {
	                        /* Colour index 8 is invalid, 9 means use default */
	                        if ( (currAttr & VT320.BOLD) != 0) {
	                            fg = boldcolor[fgcolorindex];
	                        }
	                        else if (( currAttr & VT320.LOW ) != 0) {
	                            fg = lowcolor[fgcolorindex];
	                        }
	                        else {
	                            fg = color[fgcolorindex];
	                        }
							}
	                    int bgcolorindex = ( ( currAttr & VT320.COLOR_BG ) >> 8 ) - 1;
						if ( bgcolorindex >= 0 && bgcolorindex < 8) {
	                        /* Colour index 8 is invalid, 9 means use default */
	                        bg = color[bgcolorindex];
						}
						if ( ( currAttr & VT320.INVERT ) != 0 ) {
							bg = fgcolor;
							fg = bgcolor;
//							int swapc = bg;
//							bg = fg;
//							fg = swapc;
						}
//					}

					// determine the maximum of characters we can print in one
					// go
					while ( ( c + addr < buffer.width )
							&& ( ( buffer.charArray[buffer.windowBase + l][c + addr] < ' ' ) || ( buffer.charAttributes[buffer.windowBase
									+ l][c + addr] == currAttr ) ) ) {
						if ( buffer.charArray[buffer.windowBase + l][c + addr] < ' ' ) {
							buffer.charArray[buffer.windowBase + l][c + addr] = ' ';
							buffer.charAttributes[buffer.windowBase + l][c + addr] = 0;
							continue;
						}
						addr++;
					}

					// clear the part of the screen we want to change (fill
					// rectangle)
					g.setColor( bg );

					g.fillRect( ( c - left ) * fontWidth, ( l - top ) * fontHeight, addr * fontWidth, fontHeight );

					g.setColor( fg );

					// draw the characters
					//System.out.println("fg,bg:" +fg + ","+bg);
					//System.out.println("l,c,addr:" +l + ","+c+","+addr);

					drawChars( g, fg, bg, buffer.charArray[buffer.windowBase + l], c, addr, ( c - left ) * fontWidth,
							( l - top ) * fontHeight );

					c += addr - 1;
				}
			}

			// draw cursor
			if ( buffer.showcursor
					&& ( buffer.screenBase + buffer.cursorY >= buffer.windowBase && buffer.screenBase + buffer.cursorY < buffer.windowBase
							+ buffer.height ) ) {
				g.setColor( fgcolor );
				g.fillRect( ( buffer.cursorX - left ) * fontWidth,
						( buffer.cursorY - top + buffer.screenBase - buffer.windowBase ) * fontHeight, fontWidth,
						fontHeight );
			}

			invalid = false;
			//System.out.println("REDRAW " + (System.currentTimeMillis() - st));
		}
	}
	
//#ifndef nofonts
	private LCDFont lcdfont;
	

//#endif
	
	private void initFont() {
		fontMode = settings.fontMode;
		System.out.println("fontMode: " + fontMode);

//#ifdef nofonts
        initInternalFont();
//#else
        //#ifdef midp2
        String lcdFontFile = null;
        //#endif
	    switch ( fontMode ) {
        case Settings.FONT_NORMAL:
	        initInternalFont();
            break;
//        case settings.FONT_DEVICE:
//	        initSystemFont( Font.SIZE_SMALL );
 //           break;
            //#ifdef midp2
        case 1:
        	lcdFontFile = "/font3x6lcd.png";
        	break;
        case 2:
        	lcdFontFile = "/font4x6lcd.png";
        	break;
        case 3:
        	lcdFontFile = "/font4x7lcd.png";
        	break;
        case 4:
        	lcdFontFile = "/font5x9lcd.png";
        	break;
        case 5:
        	lcdFontFile = "/font8x13lcd.png";
        	break;
        	//#endif
	    }
		//System.out.println("lcdfontfile: " + lcdFontFile);

	    
	    //#ifdef midp2
	    if (lcdFontFile != null) {
	    	boolean BGR = settings.lcdFontMode != 0;
	    	if (rotated != settings.ROT_NORMAL) {
	    		lcdFontFile = "/font4x6lcdV.png";
	    		if (rotated == settings.ROT_270) {
	    			BGR = !BGR;
	    		}
	    	}
	    	lcdfont = new LCDFont(lcdFontFile, BGR);
        	fontWidth = lcdfont.fontWidth;
        	fontHeight = lcdfont.fontHeight;
        }
	    //#endif
//#endif
	}

	private void initInternalFont() {
	    fontWidth = 4;
	    fontHeight = 6;
	    fontData = new int[128][];
	    
	    try {
			InputStream in = getClass().getResourceAsStream( FONT_RESOURCE );
			for ( int i = 33; i < 128; i++ ) {
				int b = in.read();
				int l = ( b & 3 ) + 2; // length could be 1,2,3 or 4; this is
				// len+1
				fontData[i] = new int[l]; // one more for template
				fontData[i][0] = ( b >> 2 ) - 32; // draw this template
				//        System.out.println("--- ascii "+i +"---" );
				//        System.out.println("header "+b );
				//        System.out.println("len "+(l-1) );
				//        System.out.println("template "+ data[i][0] );
				for ( int j = 1; j < l; j++ ) {
				    fontData[i][j] = in.read();
					//          System.out.println("data["+j+"]" + data[i][j] );
				}
			}
			in.close();
		}
		catch ( Exception e ) {
			//e.printStackTrace();
		}
	}
	
	private void initSystemFont( int size ) {
	    font = Font.getFont( Font.FACE_MONOSPACE, Font.STYLE_PLAIN, size );
		fontHeight = font.getHeight();
		fontWidth = font.charWidth( 'W' );
	}
	
	protected void drawChars( Graphics g, int fg, int bg, char[] chars, int offset, int length, int x, int y ) {
//#ifndef nofonts
		fontMode = settings.fontMode;
		if ( fontMode == settings.FONT_NORMAL ) {
//#endif
	        for ( int i = offset; i < offset + length; i++ ) {
				drawChar( g, chars[i], x, y );
				x += fontWidth;
			}
//#ifndef nofonts
	    }
//	    else if (fontMode == settings.FONT_DEVICE) {
//	        g.setFont( font );
//			for ( int i = offset; i < offset + length; i++ ) {
//				g.drawChar( chars[i], x, y, Graphics.TOP|Graphics.LEFT);
//				x += fontWidth;
//			}
//	    }
	    //#ifdef midp2
	    else {
	    	/* Change colour */
	    	//if (fg != prevfg || bg != prevbg) {
	    		lcdfont.setColor(fg, bg);
	    	//	prevfg = fg;
	    	//	prevbg = bg;
	    	//}
	    	/* Draw chars */
	    	for ( int i = offset; i < offset + length; i++ ) {
				lcdfont.drawChar( g, chars[i], x, y );
				x += fontWidth;
			}
	    }
	    //#endif
//#endif
	}

	private void drawChar( Graphics g, char c, int x, int y ) {
		if ( c >= fontData.length || fontData[c] == null )
			return;
		for ( int j = 1; j < fontData[c].length; j++ ) {
			int x1 = fontData[c][j] & 3;
			int y1 = ( fontData[c][j] & 12 ) >> 2;
			int x2 = ( fontData[c][j] & 48 ) >> 4;
			int y2 = ( fontData[c][j] & 192 ) >> 6;

			if ( x1 == 3 ) {
				x1 = y1;
				y1 = 4;
			}

			if ( x2 == 3 ) {
				x2 = y2;
				y2 = 4;
			}

			//      System.out.println( "char " + c + " x1=" + x1 + " y1="+ (4-y1) +"
			// x2=" + x2 + " y2="+ (4-y2));

			g.drawLine( x + x1, y + y1, x + x2, y + y2 );
		}
		if ( fontData[c][0] != 0 )
			drawChar( g, (char) ( c + fontData[c][0] ), x, y ); // draw template
	}
	
	private int fontMode = settings.fontMode;
	
	private Font font;
	
	private int[][] fontData;
	
	private static final String FONT_RESOURCE = "/font";
	
	
	
	
	
// ****************** ZMACHINE INTERFACE ***************
	public boolean defaultFontProportional() {
		return false;
	}

	public void eraseLine(int s) {
		System.out.println("EraseLine");
		buffer.deleteArea( buffer.getCursorColumn(), s, cols, 1 );
		redraw();
	}

	public void eraseWindow(int window) {
		System.out.println("EraseWindow");
		
		if (window==0) //Lower
		{
			buffer.rows_counted = 0;
			buffer.deleteArea( 0, 0, cols, rows );
			upper_size = 0;
			status_size = 0;
			buffer.setCursorPosition(0, 1);
			buffer.setTopMargin(1);
		
		}
		else // Upper
		{
			if(upper_size >0)
				buffer.deleteArea( 0, 0, cols, upper_size+status_size );	
		}
	}

	public void fatal(String errmsg) {
		Alert alert = 
			new Alert("Fatal error", errmsg, null, AlertType.ERROR);
		alert.setTimeout(Alert.FOREVER);
		quit();
	}

	public int[] getCursorPosition() {
			cursorpos[0] = buffer.getCursorColumn()+1;
			cursorpos[1] = buffer.getCursorRow()+1;
		return cursorpos;
	}


	public int getDefaultBackground() {
		return 9; // White
	}

	public int getDefaultForeground() {
		return 2;  // Black
	}

	public String getFilename(
		String title,
		String suggested,
		boolean saveFlag) 
	{
		filedialog = new FileDialog(midlet,settings);
		filedialog.running =1;
		if(saveFlag)
			filedialog.OpenBrowser(1);
		else
			filedialog.OpenBrowser(2);
		
		while(filedialog.running == 1)
		{	        
		try{
            Thread.sleep(50);
        	} catch (Exception ex){}
		}
		System.out.println("filename: " + filedialog.selected_file);
		return filedialog.selected_file;
	}

	public int[] getFontSize() {
		return new int[] { fontWidth, fontHeight };
	}

	public int[] getScreenCharacters() {
		System.out.println("getScreenCharacters");
		return new int[] { cols, rows };
	}

	public int[] getScreenUnits() {
		return new int[] { fontWidth*cols, fontHeight*rows };
	}

	public int[] getWindowSize(int window) {
		if(window==0) //lower
 			return new int[] { cols, rows - (upper_size+status_size) };
		else
 			return new int[] { cols, upper_size };
		}

	public boolean hasBoldface() {
		return false;
	}

	public boolean hasColors() {
		return false;
	}

	public boolean hasFixedWidth() {
		return true;
	}

	public boolean hasItalic() {
		return false;
	}

	public boolean hasStatusLine() {
		return true;
	}

	public boolean hasUpperWindow() {
		return true;
	}

	public void initialize(int ver) {
		//splitScreen(1);
	}

	public void quit() {
		midlet.notifyDestroyed();
	}

	public int readChar(int time) {
		// Todo   Fix non-zero getchar time
		buffer.rows_counted = 0;  // reset rows counted
		int t=0;
		System.out.println("readchar: time = " + time);
		new_char = 0;
		read_char = 1;
		while(new_char==0){
	        
			try{
	            Thread.sleep(100);
	        } catch (Exception ex){
	            
	        }
	        //System.out.println(".");
	        t++;
	        if((t>time)  && (time != 0))// timeout
	        	return 0;
			
		}
		read_char = 0;
		
		if (new_char == 10) // carriage return
			new_char = 13;

		if ((new_char == KEY_BACKSPACE )||( new_char == 127 ) )  // delete
			new_char = 8;
		
		
		return new_char;
	}

    // This method returns true if timed input is supported.
    public boolean hasTimedInput()
    {
    	return true;
    }

	public void restart() {
		eraseWindow(0);
	}

	public void scrollWindow(int lines) {
		System.out.println("scrollWindow not implemented");
	}

	public void setColor(int fg, int bg) {
		System.out.println("setColor not implemented");
	}

	public void setCurrentWindow(int window) {
		// Track window cursors
		cursor_row[previous_window] = buffer.getCursorRow();
		cursor_col[previous_window] = buffer.getCursorColumn();
		buffer.setCursorPosition(cursor_col[window], cursor_row[window]);
		previous_window = window;
		// TODO Fix this...
		if(window==1)  // Upper window always gets cursor set to top
			buffer.setCursorPosition(0, status_size);
		
		System.out.println("setCurrentWindow: " + window);
	}

	public void setCursorPosition(int x, int y) {
		System.out.println("setCursorPosition:" +x + ","+y);
		buffer.setCursorPosition(x-1, y-1);
		redraw();
	}

	public void setFont(int font) {
		System.out.println("setFont not implemented: " + font);
	}

	public void setTerminatingCharacters(Vector chars) {
		System.out.print("setTerminatingCharacters: ");
		Enumeration enumm = chars.elements();
		while (enumm.hasMoreElements()) {
			System.out.print(enumm.nextElement() + " ");
		}
		System.out.println();
	}

	/* Text can be printed in five different styles 
	 *	(modelled on the VT100 design of terminal). 
	 *	These are: Roman (the default), Bold, Italic, 
	 *	Reverse Video (usually printed with foreground and 
	 *	background colours reversed) and Fixed Pitch.  */
	
	public void setTextStyle(int style) {
		Last_style = style;
		System.out.println("setTextStyle: " + style);
		if(style ==0) // Should be Roman, but only fixed is available.
			buffer.setAttributes(0);
		if(style ==1)
			buffer.setAttributes(4);
		if(style ==2)
			buffer.setAttributes(2);
		if(style ==3)
			buffer.setAttributes(1);
		if(style ==4)  // Fixed Pitch, which is currently our only style.
			buffer.setAttributes(0);
		redraw();
		}

	public void showStatusBar(String s, int a, int b, boolean flag) {
		buffer.rows_counted = 0;  // If I am writing the upper area, I am done writing to lower.

		//System.out.println("StatusBar:"+s +"," +a +","+b+ "flag=" + flag);
		status_size = 1;
		buffer.setTopMargin(upper_size + status_size);
		redraw();
		
		String stmp;
		int tmp_style = Last_style;
		int tmp_col = buffer.getCursorColumn();
		int tmp_row = buffer.getCursorRow();
		buffer.setCursorPosition(0,0);
		setTextStyle(1);
		
		if (flag) // Is a time game
		{
			if (a==0 || a==24)
				stmp = " 12:";
			else 
				{
				if (a<13)
					stmp = " "+a+":";
				else 
					stmp = " "+(a-12)+":";
				}	
					
			if (b<10)
				stmp = stmp+"0" +b;
			
			else
				stmp = stmp +b;
		
			if (a<12)
				stmp = stmp +" AM";
			else
				stmp = stmp +" PM";
				
		
		}
		else
		{
			stmp = " " +a +"/" +b;
		}

		String stmp2 = "";
		int len = s.length() + stmp.length();
		int i;
		if (len > cols )
		{
			if (s.length() > cols)
				buffer.putStringUpper( s.substring(0,cols-4)+"..."); 
			else
				{
				buffer.putStringUpper(s); 
				stmp2 ="";
				for (i=0;i<cols;i++)
					stmp2 = stmp2+" ";
				}
			buffer.putStringUpper(stmp2);
		}
		else
		{
			buffer.putStringUpper(s); 
			stmp2 ="";
			
			for (i=len;i<cols;i++)
				stmp2 = stmp2+" ";
			buffer.putStringUpper(stmp2 +stmp);
		}
		setTextStyle(tmp_style);
		buffer.setCursorPosition(tmp_col,tmp_row);
		redraw();
	}

	// Set the Buffer Mode - Handle this at UI level.
	public void setBufferMode(boolean buffermode)
	{
		this.buffermode = buffermode;
	}
	
	
	public void showString(String s) {
		
		if (previous_window == 0)  // This is poorly named, means this is currently window 0.
		{
			// No Word Wrap.
			if (( buffermode == false) || ((buffer.getCursorColumn() + s.length()) < cols)) {
				buffer.putString(s);
			} else {  // buffer (word-wrap)
				StringTokenizer st = new StringTokenizer(s, "\t ", true);
				String str = new String();
				while (st.hasMoreTokens()) {
					String word = st.nextToken();
					if ((buffer.getCursorColumn() + str.length() + word.length()) > cols) {
						buffer.putString(str + "\n");
						str = "";
					}
					str += word;
					if (word.indexOf("\n") != -1)  // CR in string
					{
						buffer.putString(str);
						str = "";	
					}
				}
				buffer.putString(str);
			}
		}
		else  // Upper window
			{
			buffer.putStringUpper(s);  // Never Print More
			}
		redraw();
		}
	
	public void splitScreen(int lines) {
		upper_size = lines;
		buffer.deleteArea( 0, status_size, cols, upper_size );	
		if (buffer.getCursorRow()<= upper_size + status_size)
			buffer.setCursorPosition(0, upper_size + status_size);
			buffer.setTopMargin(upper_size + status_size);
			redraw();

		System.out.println("splitScreen: " + lines);
	}
	
   public int readLine(StringBuffer sbuf, int time) {
		reading_line = 1;
	    System.out.println("readline: time = " + time);
		buffer.rows_counted = 0;
		try {
			synchronized (inputQueue) {
				if (inputQueue.size() > 0) {
					//System.out.println("Command: time = " + inputQueue.firstElement());
					sbuf.append((String) inputQueue.firstElement());
					inputQueue.removeElementAt(0);
				} else {
					try {
						inputQueue.wait(time * 100);
					} catch (InterruptedException e) {}
					
					if (inputQueue.size() > 0) {
						//System.out.println("Command: " + inputQueue.firstElement());

						sbuf.append((String) inputQueue.firstElement());
						inputQueue.removeElementAt(0);
						//System.out.println("return 1");
						reading_line = 0;
						return 1;	
					}
				}
			}
			
		} catch (Throwable t) { t.printStackTrace(); }
		//System.out.println("return -1");
		reading_line = 0;
		return -1;
	}
	

	
	
	

	
	
	
	
}