/*
 * files.c
 *
 * Transscription, recording and playback
 *
 */

#include "frotz.h"
#include "S5api.h"

#ifndef SEEK_SET
#define SEEK_SET 0
#define SEEK_CUR 1
#define SEEK_END 2
#endif

extern void set_more_prompts (struct sg *g, short);

extern short is_terminator (struct sg *g, short);

extern short read_yes_or_no (struct sg *g, const char *);

/*
 * script_open
 *
 * Open the transscript file. 'AMFV' makes this more complicated as it
 * turns transscription on/off several times to exclude some text from
 * the transscription file. This wasn't a problem for the original V4
 * interpreters which always sent transscription to the printer, but it
 * means a problem to modern interpreters that offer to open a new file
 * every time transscription is turned on. Our solution is to append to
 * the old transscription file in V1 to V4, and to ask for a new file
 * name in V5+.
 *
 */

void script_open (struct sg *g)
{
    char new_name[MAX_FILE_NAME + 1];

    g->h_flags &= ~SCRIPTING_FLAG;

    if (g->h_version >= V5 || !g->script_valid) {

	if (!os_read_file_name (g, new_name, g->script_name, FILE_SCRIPT))
	    goto done;

	Srvstrcpy (g->script_name, new_name);

    }

    /* Opening in "at" mode doesn't work for script_erase_input... */

    if ((g->sfp = SrvOpenRWText (g, g->script_name)) != NULL || (g->sfp = SrvOpenRWText(g, g->script_name)) != NULL) {

	SrvSeek (g, g->sfp, 0, SEEK_END);

	g->h_flags |= SCRIPTING_FLAG;

	g->script_valid = TRUE;
	g->ostream_script = TRUE;

	g->script_width = 0;

    } else print_string (g, "Cannot open file\n");

done:

    SET_WORD (H_FLAGS, g->h_flags)

}/* script_open */

/*
 * script_close
 *
 * Stop transscription.
 *
 */

void script_close (struct sg *g)
{

    g->h_flags &= ~SCRIPTING_FLAG;
    SET_WORD (H_FLAGS, g->h_flags)

    SrvClose (g, g->sfp); g->ostream_script = FALSE;

}/* script_close */

/*
 * script_new_line
 *
 * Write a newline to the transscript file.
 *
 */

void script_new_line (struct sg *g)
{

    if(SrvPutc (g, '\n', g->sfp) == 0)
	  script_close (g);

    g->script_width = 0;

}/* script_new_line */

/*
 * script_char
 *
 * Write a single character to the transscript file.
 *
 */

void script_char (struct sg *g, zchar c)
{

    if (c == ZC_INDENT && g->script_width != 0)
	c = ' ';

    if (c == ZC_INDENT)
	{ script_char (g, ' '); script_char (g, ' '); script_char (g,' '); return; }
    if (c == ZC_GAP)
	{ script_char (g, ' '); script_char (g, ' '); return; }

    SrvPutc (g, c, g->sfp); g->script_width++;

}/* script_char */

/*
 * script_word
 *
 * Write a string to the transscript file.
 *
 */

void script_word (struct sg *g, const zchar *s)
{
    short width;
    short i;

    if (*s == ZC_INDENT && g->script_width != 0)
	script_char (g, *s++);

    for (i = 0, width = 0; s[i] != 0; i++)

	if (s[i] == ZC_NEW_STYLE || s[i] == ZC_NEW_FONT)
	    i++;
	else if (s[i] == ZC_GAP)
	    width += 3;
	else if (s[i] == ZC_INDENT)
	    width += 2;
	else
	    width += 1;

    if (g->option_script_cols != 0 && g->script_width + width > g->option_script_cols) {

	if (*s == ' ' || *s == ZC_INDENT || *s == ZC_GAP)
	    s++;

	script_new_line (g);

    }

    for (i = 0; s[i] != 0; i++)

	if (s[i] == ZC_NEW_FONT || s[i] == ZC_NEW_STYLE)
	    i++;
	else
	    script_char (g, s[i]);

}/* script_word */

/*
 * script_write_input
 *
 * Send an input line to the transscript file.
 *
 */

void script_write_input (struct sg *g,const zchar *buf, zchar key)
{
    short width;
    short i;

    for (i = 0, width = 0; buf[i] != 0; i++)
	width++;

    if (g->option_script_cols != 0 && g->script_width + width > g->option_script_cols)
	script_new_line (g);

    for (i = 0; buf[i] != 0; i++)
	script_char (g, buf[i]);

    if (key == ZC_RETURN)
	script_new_line (g);

}/* script_write_input */

/*
 * script_erase_input
 *
 * Remove an input line from the transscript file.
 *
 */

void script_erase_input (struct sg *g,const zchar *buf)
{
    short width;
    short i;

    for (i = 0, width = 0; buf[i] != 0; i++)
	width++;

    SrvSeek (g,g->sfp, -width, SEEK_CUR); g->script_width -= width;

}/* script_erase_input */

/*
 * script_mssg_on
 *
 * Start sending a "debugging" message to the transscript file.
 *
 */

void script_mssg_on (struct sg *g)
{

    if (g->script_width != 0)
	script_new_line (g);

    script_char (g, ZC_INDENT);

}/* script_mssg_on */

/*
 * script_mssg_off
 *
 * Stop writing a "debugging" message.
 *
 */

void script_mssg_off (struct sg *g)
{

    script_new_line (g);

}/* script_mssg_off */

/*
 * record_open
 *
 * Open a file to record the player's input.
 *
 */

void record_open (struct sg *g)
{
    char new_name[MAX_FILE_NAME + 1];

    if (os_read_file_name (g, new_name, g->command_name, FILE_RECORD)) {

	Srvstrcpy (g->command_name, new_name);

	if ((g->rfp = SrvOpenWText(g, new_name)) != NULL)
	    g->ostream_record = TRUE;
	else
	    print_string (g, "Cannot open file\n");

    }

}/* record_open */

/*
 * record_close
 *
 * Stop recording the player's input.
 *
 */

void record_close (struct sg *g)
{
    SrvSetFileSize(g, g->rfp, -1);
    SrvClose (g, g->rfp); g->ostream_record = FALSE;

}/* record_close */

/*
 * record_code
 *
 * Helper function for record_char.
 *
 */

void record_code (struct sg *g, short c, short force_encoding)
{

    if (force_encoding || c == '[' || c < 0x20 || c > 0x7e) {

	short i;

	SrvPutc (g, '[', g->rfp);

	for (i = 10000; i != 0; i /= 10)
	    if (c >= i || i == 1)
		SrvPutc (g, '0' + (c / i) % 10, g->rfp);

	SrvPutc (g, ']', g->rfp);

    } else SrvPutc (g, c, g->rfp);

}/* record_code */

/*
 * record_char
 *
 * Write a character to the command file.
 *
 */

void record_char (struct sg *g, short c)
{

    if (c != ZC_RETURN)
        {
        switch(c)
		  {
		  case 18: case 16:
          case 19: case 21:
          case 14: case 24:
          case 4:  return;
		  default:break;
		  }

	    record_code (g, translate_to_zscii (g, c), FALSE);

	    if (c == ZC_SINGLE_CLICK || c == ZC_DOUBLE_CLICK)
		    {
			record_code (g, g->mouse_x, TRUE);
			record_code (g, g->mouse_y, TRUE);
			}
        }
}/* record_char */

/*
 * record_write_key
 *
 * Copy a keystroke to the command file.
 *
 */

void record_write_key (struct sg *g, short key)
{

    record_char (g, key);

    if (SrvPutc (g, '\n', g->rfp) == 0)
	  record_close (g);

}/* record_write_key */

/*
 * record_write_input
 *
 * Copy a line of input to a command file.
 *
 */

void record_write_input (struct sg *g, const zchar *buf, zchar key)
{
    zchar c;

    while ((c = *buf++) != 0)
	record_char (g, c);

    record_char (g, key);

    if (SrvPutc (g, '\n', g->rfp) == 0)
	  record_close (g);

}/* record_write_input */

/*
 * replay_open
 *
 * Open a file of commands for playback.
 *
 */

void replay_open (struct sg *g)
{
    char new_name[MAX_FILE_NAME + 1];

    if (os_read_file_name (g, new_name, g->command_name, FILE_PLAYBACK)) {

	Srvstrcpy (g->command_name, new_name);

	if ((g->pfp = SrvOpenRText(g, new_name)) != NULL) {

	    set_more_prompts (g, read_yes_or_no (g,"Do you want MORE prompts"));

	    g->istream_replay = TRUE;

	} else print_string (g, "Cannot open file\n");

    }

}/* replay_open */

/*
 * replay_close
 *
 * Stop playback of commands.
 *
 */

void replay_close (struct sg *g)
{

    set_more_prompts (g, TRUE);

    SrvClose (g, g->pfp); g->istream_replay = FALSE;

}/* replay_close */

/*
 * replay_code
 *
 * Helper function for replay_key and replay_line.
 *
 */

short replay_code (struct sg *g)
{
    short c;

    if ((c = SrvGetc (g, g->pfp)) == '[') {

	short c2;

	c = 0;

	while ((c2 = SrvGetc (g, g->pfp)) != 0 && c2 >= '0' && c2 <= '9')
	    c = 10 * c + c2 - '0';

	return (c2 == ']') ? c : 0;

    } else return c;

}/* replay_code */

/*
 * replay_char
 *
 * Read a character from the command file.
 *
 */

zchar replay_char (struct sg *g)
{
    short c;

    if ((c = replay_code (g)) != 0) { // != EOF

	if (c != '\n')

	    if (c < 1000) {

		c = translate_from_zscii (g, (unsigned char)c);

		if (c == ZC_SINGLE_CLICK || c == ZC_DOUBLE_CLICK) {
		    g->mouse_x = replay_code (g);
		    g->mouse_y = replay_code (g);
		}

		return (unsigned char)c;

	    }

	SrvSeek(g, g->pfp, -1, SEEK_CUR); // Back from \n -1

	return ZC_RETURN;

    } else return ZC_BAD;

}/* replay_char */

/*
 * replay_read_key
 *
 * Read a keystroke from a command file.
 *
 */

zchar replay_read_key (struct sg *g)
{
    zchar key;

    key = replay_char (g);

    if (SrvGetc (g,g->pfp) != '\n') {

	replay_close (g);
	return ZC_BAD;

    } else return key;

}/* replay_read_key */

/*
 * replay_read_input
 *
 * Read a line of input from a command file.
 *
 */

zchar replay_read_input (struct sg *g,zchar *buf)
{
    zchar c;

    for (;;) {

	c = replay_char (g);

	if (c == ZC_BAD || is_terminator (g, c) || c == 0x7f)
	    break;

	*buf++ = c;

    }

    *buf = 0;

    if (SrvGetc (g,g->pfp) != '\n') {

	replay_close (g);
	return ZC_BAD;

    } else return c;

}/* replay_read_input */
