/*
 * Polyadventure
 *
 * A remake of the various versions of the classic Adventure game by Don
 * Woods and Willie Crowther, based on their sources.  Currently, the 350,
 * 550, and 551-point versions are implemented.  See the file "ccr-help.t"
 * for more information.
 *
 * Please document all changes in the history so we know who did what.
 *
 * This source code is copylefted under the terms of the GNU Public
 * License.  Essentially, this means that you are free to do whatever
 * you wish with this source code, provided you do not charge any
 * money for it or for any derivative works.
 *
 *
 * Contributors (see history.t for current e-mail addresses)
 *
 *      dmb     In real life:   David M. Baggett
 *
 *      djp     In real life:   David J. Picton
 *
 *      bjs     In real life:   Bennett J. Standeven
 *
 * Modification History (this file)
 *
 * CCR
 * ===
 *
 *  1-Jan-93    dmb     rec.arts.int-fiction BETA release (source only)
 *                      For beta testing only -- not for general
 *                      distribution.
 * 20-Apr-93    dmb     Added a few verbs for regression testing.
 *
 * 14-Apr-99    djp     Initial release of Adventure 551 (1.01)
 *
 * AD551
 * =====
 *
 * 23-Apr-99    djp     New release (1.10) of Adventure 551.
 *                      Changes since 1.01:
 *                      * Lamplife verb changed to use new Brass_lantern
 *                        method.
 *
 * 30-Apr-99    djp     New release (1.11) of Adventure 551.
 *                      Changes since 1.10:
 *                      * Defaults for wave command changed to exclude
 *                        fixed and non-removable items.
 *                      * Added 'swim' verb.
 *                      * Changed vocab for 'rub'.
 *
 * 17-May-99    djp     New release (1.20)
 *
 * 20-May-99    djp     New release (1.21)
 *
 * 15-Jul-99    djp     New release (2.00)
 *                      Changes in this release:
 *                      * Added 'middle' verb.
 *                      * Changed doAction for 'climb up' and 'climb down'
 *                        to 'Climbup' and 'Climbdown' (for consistency with
 *                        the normal naming convention)
 *                      * Changed the actions for elvish magic words so
 *                        they can't be used until the player has seen them.
 *                      * Added synonym 'opening' for 'passage' verb, for
 *                        consistency with 'passage' as an object.
 *                      * Changed the 'purloin' and 'gonear' debugging
 *                        verbs to accept the syntax 'verb object number <n>'
 *                        For example, 'gonear gate number 3' will go to
 *                        the third 'gate' object in the game.
 *
 * 3-Aug-99     djp     Bugfix release - Version 2.01
 * 9-Nov-99     djp     New release - Version 2.10
 *                      Changes in this release
 *                      * Incorporated some of Bennett Standeven's modified
 *                        code for easier integration of new versions (e.g.
 *                        the magicWord and magicTravelVerb classes).
 *
 * 17-Feb-00    djp     New release - Version 2.20
 *                        Changes in this version
 *                      * Further generalization of version handling
 *                      * Object selection for purloin and gonear now
 *                        uses disambigDobj.
 *                      * Purloin and gonear now reject objects with the
 *                        deleted property set to true.
 *                      * Changed version-restart verbs to suppress the
 *                        prompt if issued very early in the game.  Defined
 *                        a class for these verbs.
 *                      * Changed gonear verb to remove the player
 *                        from a nested room before issuing travelTo.
 *
 * POLYADV
 * =======
 *
 * 24-Aug-99    bjs     Added the verb "ride", introduced class for magic
 *                      words.
 *
 *          djp+bjs     Incorporated ad551 mods up to 2.20.  Enhanced gonear
 *                      to handle objects in Elsewhere (player is sent to
 *                      the appropriate catacomb.)
 *
 * 3-Mar-00     djp     Initial beta release - Version 1.00
 *                      * changed the disambigDobj function to ignore objects
 *                        for which the verification property is undefined.
 *                      * Fixed 'douse x with y' to use disambigDobjFirst for
 *                        correct mapping to 'pour y on x'
 *                      * Similarly changed 'feed x to y' to use
 *                        disambigDobjFirst, to allow it to be mapped
 *                        (in ccr-npc.t) to 'feed y with x'.
 *                      * Removed redundant emerald-dropping code from
 *                        ploverVerb.
 *                      * Implemented the count verb properly (for most
 *                        items, anyway :-) )
 *                      * Updated the scoretotal verb.
 *
 * 4-Apr-00     djp     Version 1.01: bugfix release
 *                      Changes in this version:
 *                      * Fixed an incorrect definition for 'pole north'.
 *                      * Implemented code to wave 'at' an object.  Waving
 *                        the rod will work as usual provided that the
 *                        indirect object is relevant to the magic being
 *                        attempted.  Weapons can be waved at any enemy,
 *                        indicating that hand-to-hand combat is to be
 *                        attempted.  In extended games, typing (say)
 *                        'wave axe' will cause a suitable indirect object to
 *                        be found or prompted for, using the askio command.
 *                      * added a noresurrect verb.
 *                      * Changed purloin to reject fixed items in the
 *                        verification method.
 *                      * Changed smellVerb and rubVerb to use action methods
 *                        rather than verification methods.
 *                      * Change poleverb to print an error if there's no
 *                        pole.
 *                      * Changed 'pole north' etc to work sensibly if used
 *                        without an object (the boat is assumed).
 *                      * Changed certain verbs to the darkVerb class.
 *
 * 4-May-00     djp     Version 1.02: bugfix release
 *
 * 18-Sep-00    djp     Version 2.00: New version with 701-point game
 *                      Changes in this version:
 *                      * Changed gonearVerb not to be a travel verb (it
 *                        cannot be used without an object)
 *                      * Changed the wave verb to draw attention to the
 *                        rusty appearance of the sword when seen in the
 *                        mirror.
 *                      * Added the new property omegaps701_order for the 
 *                        701-point game. 
 *                      * Widened the phugggVerb check for water to include
 *                        all streamItem objects, sea water, and water in the
 *                        cask.
 *
 * 8-Jun-01     djp     Version 2.05: bugfix release
 *
 * 17-Aug-01    djp     Version 2.06: bugfix release with e-mail address update
 *                      update.
 *                      * Added missing sdesc method to 'click' verb
 *
 * 22-Nov-01    djp     Version 2.08: bugfix release
 *                      * Implemented new 'leave' verb
 *
 * 13-Jan-03    djp     Version 2.09 rev C: bugfixes and code tidy-up
 *                      * Eliminated self.yanking property.
 *                      * Changed gonear to use the new optional travelTo
 *                        argument.
 *
 *
 * 12-Aug-03    bjs     Version 2.11: Added 580-point version, with new magic 
 *                      words.
 *                      * added lift x with y (for floor with cups).
 *
 * 23-Aug-03    djp     Version 2:12 
 *                      * Added a suite of verbs for flying the rug, similar
 *                        to those for poling the boat.
 *                    
 * 23-Jul-04    djp     Version 3.00.  Addition of a new game mode (701+).
 *                      * Addition of transindection-related code
 *                      * Addition of code relating to the Wumpi and
 *                        the teleportation-following properties of the gold
 *                        rings.
 *
 * 15-Jan-05    djp     Version 3.01: bugfix release.
 *                      * The modified action methods for direction verbs, e.g.
 *                        nVerb, are now defined using replace (which causes
 *                        control to be transferred to the travelVerb
 *                        class.  Without the replace, control passes back
 *                        up to the original nVerb method!
 *
 * 12-Apr-05    djp     Version 3.10: Bugfixes and enhancements
 *                      * Adjustments to magicTravelVerb coding to take into 
 *                        account PC knowledge after a 'throne mishap'
 *                        (after sitting on the throne without a crown)
 *                      * Removed incorrect comment about the Wumpus not 
 *                        wearing his ring while asleep.
 *                      * The response to Elvish magic words is simply
 *                        "nothing happens" in non-game551 modes.  In
 *                        game551 mode there is a new explanation that
 *                        humans can work Elvish magic only in locations
 *                        where elves have used the same magic.
 * 16-Jul-06    djp     Version 3.20: Bugfixes and extensions
 *                      * Changed the 'omegapsical' properties of magic words
 *                        (see comments in ccr-endg.t)
 * 20-Oct-07    djp     Version 3.20 Rev. C
 *                      * Added transitive verb methods and extra vocab for 
 *                        outVerb, uVerb and dVerb. These customizations were
 *                        in polyadv_208, but went missing when the source
 *                        files were reorganized in polyadv_209.
 *                      * Corrected the code for issuing general magic
 *                        words in the presence of the Wumpi.  (They should
 *                        repeat the word in their sleep, but they didn't
 *                        because of incorrect conditions).
 * 21-Nov-10    djp     Version 3.21
 *                      * Upper Transindection Rooms can now define 
 *                        additional Transindection destinations which are
 *                        not available with any pendants.  If 
 *                        has_ana_extradot is set to true, the property 
 *                        ana_extradot will define the 'ana' destination when 
 *                        the room is active.  Similar properties are defined 
 *                        for 'kata'. 
 */

/*
 * This file defines new verbs, including the myriad new travel verbs.
 */

/*
 * system verbs - now defined as sysverb - DJP.
 */
spaceVerb: sysverb
    sdesc = "space"
    verb = 'space'
    action(actor) = {
        if (global.doublespace) {
            "Now single-spacing text.";
            global.doublespace := nil;
        }
        else {
            "Now double-spacing text.";
            global.doublespace := true;
        }

        abort;    /* doesn't count as a turn */
    }
;
indentVerb: sysverb
    sdesc = "indent"
    verb = 'indent'
    action(actor) = {
        if (global.indent) {
            "Paragraph indentation now off.";
            global.indent := nil;
        }
        else {
            "Paragraph indentation now on.";
            global.indent := true;
        }

        abort;    /* doesn't count as a turn */
    }
;
deterministicVerb: sysverb
    verb = 'deterministic' 'norandom' 'norandomize'
    action(actor) = {
        if(global.randomized) { // DJP - check if it's too late.
        "You're too late.  Once I've randomized, I can't go back to
        deterministic random numbers. ";
        abort;
        }
        else {
        global.nondeterministic := nil;
        abort;
        }
    }
;
nodwarfVerb: sysverb
    verb = 'nodwarf' 'nodwarves'
    action(actor) = {
        if(global.nodwarves) { // DJP - check if we've already done it
            "You've already issued the nodwarves command. ";
            abort;
        }
        Dwarves.loclist := [];
        Pirates.loclist := [];
        "\b Dwarves and pirates will be disabled for the rest of the
        game, at the cost of 5 points. \b";
        "Do you indeed want to disable the dwarves and pirate? >";
        if(yorn()) {
            if(not treasure_chest.spotted) {
            treasure_chest.moveInto(Dead_End_13);
            }
            global.nodwarves := true;
            "OK - dwarves and pirates disabled!";
            incscore(-5);
        }
        else "OK - dwarves and pirates still enabled. ";
        abort;
    }
;

noreincarnateVerb: sysverb
verb = 'noreincarnate' 'noresurrect'
    action(actor) = {
        global.no_allow_resurrect := true;
        "OK - resurrections will not be allowed in this game.";
        abort;
    }
;

// DJP - for polyadv, we delete 'newgame' which is now ambiguous, and
// we define a class for version-restart verbs.     We also turn off the
// prompt if the command comes within the first five commands of the game.
class versionrestart: sysverb
    points = 350
    vnumber = 0
    restartGame(actor) = {
        local yesno,silent := nil;
        while ( true ) {
            if(global.turnsofar > 5) {
            if(self.vnumber = 11) {
                "Are you sure you want to restart the game (in the
                701+ points mode)? (YES or NO) > ";
            }
            else {
                "Are you sure you want to restart the game (in the
                <<self.points>>-point mode)? (YES or NO) > ";
            }
            yesno := yorn();
            }
            else {
                if(self.vnumber = 11) {
                    "Restarting the game in the 701+ points mode (with bonus
                    points for finding extensions).\n ";
                }
                else {
                    "Restarting the game in the <<self.points>>-point 
                    mode.\n ";
                }
                yesno := 1;
                silent := true;
            }
            if ( yesno = 1 ) {
                "\n";
                scoreStatus(0, 0);
                // DJP - pass game version information.
                global.initRestartParam := [self.vnumber global.randomized
                silent];
                restart(initRestart, global.initRestartParam);
                break;
            }
            else if ( yesno = 0 ) {
                "\nOkay.\n";
                break;
            }
        }
    }
    action( actor ) = {
        self.restartGame(actor);
        abort;
    }
;

restart350Verb: versionrestart
    verb = 'oldgame' 'game350'
    points = 350
    vnumber = 0
;

restart551Verb: versionrestart
    verb = 'game551'
    points = 551
    vnumber = 1
;

restart550Verb: versionrestart
    verb = 'game550'
    points = 550
    vnumber = 2
;

restart701Verb: versionrestart
    verb = 'game701'
    points = 701
    vnumber = 15
;

restart701pVerb: versionrestart
    verb = 'game701P'
    points = 701
    vnumber = 11
;

restart580Verb: versionrestart
    verb = 'game580'
    points = 580
    vnumber = 7
;

noviceVerb: sysverb
    verb = 'novice'
    action (actor) = {
        if(global.turnsofar > 0) {
            "This verb is only valid in the first turn of the game.";
            abort;
        }
        "By default, your lamp will
        last for ";say(brass_lantern.turnsleft);" turns, but in novice
        mode it will last for 1000 turns.  However, this will cost you ";
        say(-1 * global.novicepoints); " points.\n
        Do you want a novice mode game? >";

        if (yorn()) {
            global.novicemode := true;
            incscore(global.novicepoints);
            brass_lantern.turnsleft := 1000;
        }
        abort;
    }
;
healthVerb: sysverb
    verb = 'health' 'diagnose'
    healthmess = [
    'You are in perfect health.'
    'You are fit as a fiddle.'
    'You are a bit off top form, but nothing to worry about.'
    'You are weaker than usual.  Better avoid fights.'
    'You really ought to take a break.  You\'re in tough shape.'
    'You are on the edge of collapse.  Lots of sun and fresh air will
    speed your recovery.'
    'Your strength is nearly gone.    Only a miracle can save you now.'
    ]
    action (actor) = {
        local i;
        P(); I();
        "Your health rating is ", say(actor.health)," out of 100.";
        P(); I();
        if(actor.health >= 95) {
            if (rand(100) <= 50) i := 2;
            else i := 1;
        }
        else i := 3 + (100 - actor.health) / 20;
        say(self.healthmess[i]);
    }
;

/* Note that magic words which don't exist in a given version are currently
 * programmed to say "Nothing happens," as opposed to, say, "I don't know
 * that word." */
class MagicWord: darkVerb
    endsaid = nil  // These properties are used in the cylindrical room.
    tused = -2
    omegapsical_order = nil    // To future extenders: leave these nil if
    omegaps701_order = nil     // you aren't altering the version in question.
    omegaps580_order = nil     // The unlabeled omegapsical_order
                               // refers to the 550-point version.
;

class magicTravelVerb: travelVerb, MagicWord    // DJP
    action( actor ) = {
        local travelsave, currentsave, toploc := toplocation(actor);
        local Wumpimove, newtoploc; 

        // prevent the possibility of a Transindection movement straight
        // after a security alert
        if (global.triggered_alert) {
             self.alert_message(actor);
             return;
        }

        // The Wumpus is wearing his ring (but there aren't any
        // magic words which would transport him while sleeping)
        if (Wumpus.isVisible(actor) and Wumpus.isasleep) {
            if (self = clickVerb)
                "You see the Wumpus click his heels in his sleep.\n";
            else
                "You hear the Wumpus repeat the magic word in his sleep.\n";
        }

        // modified to remove all the agonizing if the player has been through
        // it before, then sat on the throne without a crown.

        if ((Wumpi.isVisible(actor) or Wumpi_remnant.isVisible(actor)) and 
        (Wumpi.phase < 4)) {
            Wumpimove := true;
            if(gold_ring.seenspecial and not self.hesitated
            and not Green_Tight_Crack_2.isseen) {
                "You hesitate.  When you spied on an elf with the crystal
                ball, his teleportation spell took the sapphire with him.
                You feel sure that this has something to do with the gold
                ring which you found on the Wumpus.  These Wumpi are also
                wearing gold rings, so you ask yourself what will happen
                if you teleport yourself from here ... ";

                "But then you question whether the Wumpi would be silly enough
                to wear rings which could transport them while sleeping.  In
                any case, you'll need to use magic to get out of here without
                waking the Wumpi, so ";
                if (self = clickVerb)
                    "you go ahead ... ";
                else
                    "you say the word ... ";
                P();
                self.hesitated := true;
            }
            else if(gold_ring.deducedmagic and not self.hesitated
            and not Green_Tight_Crack_2.isseen) {
                "You hesitate, remembering how the Wumpus managed to follow
                you when you tried to escape using the slippers.  You
                suspected that this had something to do with his ring.  These
                Wumpi 
                are also wearing gold rings, which look remarkably similar
                to the one you took from the Wumpus ... ";
                P();
                "But then you question whether the Wumpi would be silly enough
                to wear rings which would transport them while sleeping.
                In any case, you'll need to use magic to
                get out of here without waking the Wumpi, so ";

                if (self = clickVerb)
                    "you go ahead ... ";
                else
                    "you say the word ... ";
                P();
                self.hesitated := true;
            }

            if(toploc = Green_Large_Circular_Room) {
                if(not (pendant2.obtained or (self = kataVerb))) {
                    local proportion := 'most';
                    if(Wumpi_remnant.isVisible(actor)) 
                        proportion := 'about half';
                    if(self = clickVerb)
                        "You see <<proportion>> of the Wumpi click their heels
                        in their sleep. \n";
                    else
                        "You hear <<proportion>> of the Wumpi repeat the magic
                        word in their sleep. \n";
                }
            }
            else {
                if(self = clickVerb)
                    "All of the Wumpi click their heels in their sleep.\n";
                else
                    "All of the Wumpi repeat the magic word in their sleep.\n";
            }
        }

        actor.nextRoute := 10; // indicating that it was magic

        travelsave := global.travelActor;
        currentsave := global.currentActor;

        global.travelActor := actor;  // actor doing the travelling
        global.currentActor := actor; // reference actor for location method
                                      // evaluation
        actor.travelTo(self.travelDir(actor));

        global.travelActor := travelsave;
        global.currentActor := currentsave;

        actor.nextRoute := 0; // return to default in case the travel method
                              // changed it.

        newtoploc := toplocation(actor);

        // Deal with the Wumpi ...
        if(Wumpimove) {
            if((toploc = Green_Large_Circular_Room) and
            (newtoploc = Blue_Large_Circular_Room))
                Wumpi.firstmove;
            // This situation is catered for but shouldn't be possible.
            if((toploc = Blue_Large_Circular_Room) and
            (newtoploc = Green_Large_Circular_Room))
                Wumpi.moveback;
            if((toploc = Blue_Large_Circular_Room) and
            (newtoploc = Large_Circular_Room))
                Wumpi.secondmove;
        }
    }
    alert_message(actor) = "Nothing happens. "
;

/*
 * Magic verbs for debugging only - disallowed when game is not compiled
 * for debugging.
 */

class wizardVerb: sysverb
;

randomizeVerb: wizardVerb
    verb = 'randomize'
    action(actor) = {
        "\nRandomizing ... \n";
        randomize();
        abort;
    }
;
randVerb: wizardVerb
    verb = 'rand'
    sdesc = "rand"
    doAction = 'Rand'
;
modify numObj
    verDoRand(actor) = {}
    doRand(actor) = {
        if(self.value <= 0) {
            "\nInvalid argument - must be 1 or greater\n";
        }
        else {
            "\nrand(";say(self.value);")= ";say(rand(self.value));"\n";
        }
        abort;
    }
;
numPrep: Prep
    preposition = 'number'
    sdesc = 'number'
;

/* Purloin and Gonear have been modified to accept the syntax
   <verb> <object> <number> <n>
   For example, you can type
   gonear gate number 3
   to go to the 3rd gate in the game.  (To be precise, the third object
   which (a) matches the noun phrase used and (b) is present in the
   game version.)
   The disambigDobj routine is now used.  This will work only for TADS
   version 2.4.0 or later.
*/
class numberedVerb: wizardVerb
    disambigDobj(actor,prep,iobj,verprop,wordlist,objlist,flaglist,
    numberWanted,isAmbiguous, silent) = {
        local i,j,k,o,l,selection,lastsel,hidestatus,
        outlist := [DISAMBIG_DONE];
        if (not (iobj=numObj)) selection := 1;
        else selection := numObj.value;
        l := length(objlist); j := 0; k := 0;
        // j is a counter of eligible objects (which must all have a
        // verprop defined)
        // k counts the number of objects to be returned
        lastsel := selection + numberWanted -1;
        for (i := 1; i <=l; i++) {
            o := objlist[i];
            /*
             *   skip objects which lack the verification method.
             */
            if (not defined(o, verprop))continue;
            j++;
            if (j > lastsel) break;
            if(j >= selection) {
                k++;
                outlist += objlist[i];
            }
        }
        if (k = 0) {
            if (not silent) {
                "\n";caps(); self.sdesc; ": No objects found.\n";
            }
            return [DISAMBIG_ERROR];
        }
        else return outlist;
    }
    validDo( actor, obj, seqno ) = {return true;}
    validDoList(actor, prep, dobj) = (nil)
;

purloinVerb: numberedVerb
    verb = 'purloin'
    sdesc = "purloin"
    doAction = 'Purloin'
    ioAction(numPrep) = 'PurloinNum'
;

modify numObj
    verIoPurloinNum(actor) = {}
    ioPurloinNum(actor,dobj) = {
        dobj.doPurloin(actor);
    }
;

gonearVerb: numberedVerb
    verb = 'gonear'
    sdesc = "gonear"
    doAction = 'Gonear'
    ioAction(numPrep) = 'GonearNum'
;

// Select the loclist element to use for the next gonear verb
locselVerb: wizardVerb
    verb = 'locsel'
    sdesc = "locsel"
    doAction = 'Locsel'
    value = 1
;

modify numObj

    verIoGonearNum(actor) = {}
    ioGonearNum(actor,dobj) = {
        dobj.doGonear(actor);
    }

    verDoLocsel(actor) = {}
    doLocsel(actor) = {
        "Selecting loclist element: <<self.value>> for gonear. ";
        locselVerb.value := self.value;
    }
;

// SUMMON and BANISH use the numberedVerb class for the special disambigDobj
// method, although we don't actually need to allow numbers to be used with
// them.

summonVerb: numberedVerb
    verb = 'summon'
    sdesc = "summon"
    doAction = 'Summon'
;

banishVerb: numberedVerb
    verb = 'banish'
    doAction = 'Banish'
    sdesc = "banish"
;

lamplifeVerb: wizardVerb
    verb = 'lamplife'
    sdesc = "set the lamp life to"
    doAction = 'Lamplife'
    action(actor) = {
        "The lamp life is "; say(brass_lantern.turnsleft); " turns. ";
        abort;
    }
;
dwarfVerb: wizardVerb
    sdesc = "num-dwarves"
    verb = 'num-dwarves'
    doAction = 'Dwarf'
;
pirateVerb: wizardVerb
    sdesc = "num-pirates"
    verb = 'num-pirates'
    doAction = 'Pirate'
;
modify numObj
    verDoLamplife(actor) = {}
    doLamplife(actor) = {
        brass_lantern.setlife(self.value);
        "Lamp life is now "; say(self.value); " turns.";
    }
    verDoDwarf(actor) = {}
    doDwarf(actor) = {
        global.dwarves := self.value;
        "OK - the number of dwarves will now be <<self.value>>. ";
    }
    verDoPirate(actor) = {}
    doPirate(actor) = {
        global.pirates := self.value;
        "OK - the number of dwarves will now be <<self.value>>. ";
    }
;

burden: wizardVerb // test addbulk, addweight and addmass subroutines.
        // The answers from addweight and addmass will differ if
        // items are worn.
    verb = 'burden'
    action(actor) = {
        "\n Total bulk: \n"; say(addbulk(actor.contents));
        "\n Total weight: \n"; say(addweight(actor.contents));
        "\n Total mass: \n"; say(addmass(actor.contents));
        abort;
    }
;

scoretotal: wizardVerb // Check total scores
    verb = 'scoretotal'
    action(actor) = {
        local i,l,o,n,treascore := 0, objscore := 0, tot := 0;
        local combtreasures := global.treasurelist + global.bonustreasures;
        local combpointobjs := global.pointobjlist + global.bonuspointobjs;
        "\nInitial score: ";say(global.startscore);"\n";
        tot += global.startscore;
        "\nFar-in Points: ";say(global.farinpoints);"\n";
        tot += global.farinpoints;
        l := length(combtreasures);
        for (i := 1; i <= l; i++) {
            o := combtreasures[i];
            treascore += o.takepoints + o.depositpoints;
        }
        "\nTreasure Points: ";say(treascore);"\n";
        tot += treascore;
        l := length(combpointobjs);
        for (i := 1; i <= l; i++) {
            o := combpointobjs[i];
            objscore += o.takepoints + o.depositpoints;
        }
        "\nObject deposit Points: ";say(objscore);"\n";
        tot += objscore;
        if(global.extenpoints) {
            for (o := firstobj(room); o ; o := nextobj(o, room)) {
                if (not o.isbonus) continue;
                "\nPoints for entering <<o.sdesc>>: ";
                say (global.extenpoints);"\n";
                tot += global.extenpoints;
            }
        }

        if(global.closurepoints) {
            "\nClosure points (endgame timer): ";
            say(global.closurepoints); "\n";
            tot += global.closurepoints;
        }
        if(global.closingpoints) {
            "\nClosing points: "; say(global.closingpoints); "\n";
            tot += global.closingpoints;
        }
        if(global.endpoints) {
            "\nEndgame points: "; say(global.endpoints); "\n";
            tot += global.endpoints;
        }
        if(global.winpoints) {
            "\nWin points: "; say(global.winpoints); "\n";
            tot += global.winpoints;
        }
        if(global.escapepoints) {
            "\nEscape points: "; say(global.escapepoints); "\n";
            tot += global.escapepoints;
        }
        if(global.finalepoints) {
            "\nFinale points: "; say(global.finalepoints); "\n";
            tot += global.finalepoints;
        }
        P();
            "\nTotal score is "; say(tot); "\n";
            
            "\nDeclared total is: ";say(global.maxscore); "\n";
        abort;
    }
;
closecave: wizardVerb // short cut to the endgame
    verb = 'closecave'
    sdesc = "closecave"
    action(actor) = {
        global.treasures := 0;
        global.closingtime := 5;
        global.bonustime := 5;
        "\nStarting countdown for cave closure in 5 moves ...\n";
        "\nCave will fully close after another 5 moves ...\n";
        "\nN.B. you must be well inside the cave and not at Y2.\n";
        abort;
    }
;
closecave30: wizardVerb // short cut to the endgame
    verb = 'closecave30'
    sdesc = "closecave30"
    action(actor) = {
        global.treasures := 0;
        global.closingtime := 5;
        global.bonustime := 30;
        "\nStarting countdown for cave closure in 5 moves ...\n";
        "\nCave will fully close after another 30 moves ...\n";
        "\nN.B. you must be well inside the cave and not at Y2.\n";
        abort;
    }
;
pirloc: wizardVerb // print pirate and dwarf locations
    verb = 'pirloc'
    sdesc = "pirloc"
    action(actor) = {
        local i,l,r;
        l := length(Dwarves.loclist);
        P(); "Dwarf locations:\n";
        for (i := 1; i <= l; i++) {
            r := Dwarves.loclist[i];
            r.sdesc;"\n";
        }
        l := length(Pirates.loclist);
        P(); "Pirate location:\n";
        for (i := 1; i <= l; i++) {
            r := Pirates.loclist[i];
            r.sdesc;"\n";
        }
    }
;
/*
 * Special and/or magic verbs
 */

yankVerb: deepverb // DJP, for cloak, sword
    verb = 'yank'
    sdesc = "yank"
    doAction = 'Yank'
    ioAction(fromPrep) = 'YankOut'
;

replaceVerb: darkVerb // DJP, for batteries and telephone receivers
    verb = 'replace'
    sdesc = "replace"
    doAction = 'Replace'
;

changeVerb: darkVerb // DJP, for batteries
    verb = 'change'
    sdesc = "change"
    doAction = 'Change'
;

liftVerb: deepverb // DJP, for telephone receivers, defaults to 'take'
    verb = 'lift'
    sdesc = "lift"
    doAction = 'Lift'
    ioAction(withPrep) = 'TakeWith'
;

compoundWord 'hang' 'up' 'hangup'; // DJP, for telephones
hangupVerb: deepverb
    verb = 'hangup'
    sdesc = "hang up"
    doAction = 'Hangup'
;
answerVerb: deepverb // DJP, for telephones
    verb = 'answer'
    sdesc = "answer"
    doAction = 'Answer'
;
dialVerb: deepverb // DJP, for telephones
    verb = 'dial'
    sdesc = "dial"
    doAction = 'Dial'
    ioAction(onPrep) = 'DialOn'
;
modify numObj
    verDoDialOn(actor,io) = {}
;

cutVerb: deepverb //DJP
    verb = 'cut' 'slash' 'cut through' 'cut down'
    sdesc = "cut"
    ioAction(withPrep) = 'CutWith'
    prepDefault = withPrep
;
swimVerb: deepverb //DJP
    verb = 'swim' 'swim in'
    sdesc = 'swim'
    action(actor) = {
        "You don't know how. ";
    }
    doAction = 'Swim'
;
yesVerb: darkVerb  // BJS: made darkVerb
    //
    // This is a hack to allow the following:
    //
    // >kill dragon
    // With what, your bare hands?
    // >yes
    //
    sdesc = "answer yes to"
    verb = 'yes' 'uh-huh' 'uhuh' 'yea' 'yeah' 'yup' 'sure' 'yep'

    action(actor) = {
        // kludge to prevent questions being answered after
        // reincarnation
        if (self.ignore = global.turnsofar) {
            self.defaultmess;
            return;
        }
        //
        // If we asked "with your bear hands?" last turn,
        // do special stuff.
        //
        if (Dragon.rhetoricalturn = global.turnsofar - 1)
            Dragon.kill;
        else if (Bear.rhetoricalturn = global.turnsofar - 1)
            Bear.nicetry;
        // DJP - changed to attack with hands because this does work
        // in a 550-point game.
        else if (Dwarves.rhetoricalturn = global.turnsofar - 1)
            Dwarves.doAttackWith(actor, Hands);
        else if (Waterfall.rhetoricalturn = global.turnsofar - 1)
            Waterfall.enter;
        else if (Ogre.rhetoricalturn = global.turnsofar - 1)
            Ogre.nicetry;
        else if (Slime.rhetoricalturn = global.turnsofar - 1)
            Slime.doTouch(actor);
        else if (Wumpi.rhetoricalturn = global.turnsofar - 1)
            Wumpi.disaster;
        else
            self.defaultmess;
    }
    defaultmess = "You're sounding awfully positive!";
;

noVerb: darkVerb  // DJP
    //
    // This is a hack to allow the following:
    //
    // >kill dragon
    // With what, your bare hands?
    // >no
    //
    sdesc = "answer no to"
    verb = 'no'

    action(actor) = {
        // kludge to prevent questions being answered after
        // reincarnation
        if (self.ignore = global.turnsofar){
            self.defaultmess;
            return;
        }
        //
        // If we asked "with your bear hands?" last turn,
        // do special stuff.
        //
        if (Dragon.rhetoricalturn = global.turnsofar - 1)
            self.okay;
        else if (Bear.rhetoricalturn = global.turnsofar - 1)
            self.okay;
        else if (Dwarves.rhetoricalturn = global.turnsofar - 1)
            self.okay;
        else if (Waterfall.rhetoricalturn = global.turnsofar - 1)
            self.okay;
        else if (Ogre.rhetoricalturn = global.turnsofar - 1)
            self.okay;
        else if (Slime.rhetoricalturn = global.turnsofar - 1)
            "Okay.    Perhaps it wasn't such a bright idea
            anyway. ";
        else
            self.defaultmess;
    }
    okay = "Okay. "
    defaultmess = "You're sounding awfully negative!"
;
hidebehindVerb: deepverb
    sdesc = "go behind"
    verb = 'hide behind' 'go behind' 'get behind'
    doAction= 'Hidebehind'
;
hideunderVerb: deepverb
    sdesc = "go under"
    verb = 'hide under' 'go under' 'get under'
    doAction= 'Hideunder'
;

class VaultKeyVerb: MagicWord
    wordnum = nil
    action(actor) = { // Will always print "Nothing happens" in 350-pt mode
        if(actor.isIn(Vault) or actor.isIn(Peelgrunt)) self.SafeOpen;
        else return actor.location.nothinghappens;
    }
    SafeOpen() = {
        if(In_Safe.isopen or In_Safe.is_fused) "Nothing happens.";
        else {    if(self.wordnum = In_Safe.password) In_Safe.opens;
            else In_Safe.melts;
        }
    }
;
blerbiVerb: VaultKeyVerb
    omegapsical_order = 16
    omegaps580_order = 18
    omegaps701_order = 20
    omegaps701p_order = 22
    wordnum = 1
    sdesc = "blerbi"
    verb = 'blerbi'
;
zortonVerb: VaultKeyVerb
    omegapsical_order = 1
    omegaps701_order = 1
    omegaps580_order = 1
    omegaps701p_order = 1
    wordnum = 2
    sdesc = "zorton"
    verb = 'zorton'
;
klaetuVerb: VaultKeyVerb
    omegapsical_order = 11
    omegaps580_order = 13
    omegaps701_order = 14
    omegaps701p_order = 15
    wordnum = 3
    sdesc = "klaetu"
    verb = 'klaetu'
;
knerlVerb: VaultKeyVerb
    omegapsical_order = 10
    omegaps580_order = 12
    omegaps701_order = 13
    omegaps701p_order = 14
    wordnum = 4
    sdesc = "knerl"
    verb = 'knerl'
;
snoezeVerb: VaultKeyVerb
    omegapsical_order = 4
    omegaps580_order = 5
    omegaps701_order = 4
    omegaps701p_order = 4
    wordnum = 5
    sdesc = "snoeze"
    verb = 'snoeze'
;

feeVerb: MagicWord
    omegapsical_order = 15
    omegaps580_order = 17
    omegaps701_order = 18
    omegaps701p_order = 20
    said = nil

    sdesc = "fee"
    verb = 'fee'
    action(actor) = {
        if (self.said)
            self.fail;
        else {
            "Ok!";
            fieVerb.tcount := global.turnsofar + 1;
            foeVerb.tcount := global.turnsofar + 2;
            fooVerb.tcount := global.turnsofar + 3;
            self.said := true;
            fieVerb.said := nil;
            foeVerb.said := nil;
        }
    }

    fail = {
        if (self.said) {
            "What's the matter, can't you read?  Now
            you'd best start over. ";
        }
        else {
            "Nothing happens.";
        }

        self.reset;
    }

    reset = {
        fieVerb.tcount := -1;
        foeVerb.tcount := -1;
        fooVerb.tcount := -1;

        self.said := nil;
        fieVerb.said := nil;
        foeVerb.said := nil;
    }
;
fieVerb: MagicWord
    omegapsical_order = 14
    omegaps580_order = 16
    omegaps701_order = 17
    omegaps701p_order = 19
    said = nil
    tcount = -1
    sdesc = "fie"
    verb = 'fie'
    action(actor) = {
        if (self.tcount = global.turnsofar) {
            self.said := true;
            "Ok!";
        }
        else
            feeVerb.fail;
    }
;
foeVerb: MagicWord
    omegapsical_order = 13
    omegaps580_order = 15
    omegaps701_order = 16
    omegaps701p_order = 18
    said = nil
    tcount = -1
    sdesc = "foe"
    verb = 'foe'
    action(actor) = {
        if (not fieVerb.said)
            feeVerb.fail;
        else if (self.tcount = global.turnsofar) {
            self.said := true;
            "Ok!";
        }
        else
            feeVerb.fail;
    }
;
fooVerb: MagicWord // move eggs back to their original location.  (N.B. the
                   // original 550-point game disallows 'foo' if the eggs
                   // have been lost.  This feature has not been
                   // implemented in the TADS port.)
    omegapsical_order = 12
    omegaps580_order = 14
    omegaps701_order = 15
    omegaps701p_order = 17
    tcount = -1
    sdesc = "foo"
    verb = 'foo'
    action(actor) = {
        if (not foeVerb.said)
            feeVerb.fail;
        else if (self.tcount = global.turnsofar) {
            if (golden_eggs.isIn(In_Giant_Room))
                "Nothing happens.";
            else {
                if (golden_eggs.isIn(actor.location))
                    "The nest of golden eggs has
                    vanished!";
                else
                    "Done!";

                if (golden_eggs.isIn(Troll_Treasure)) {
                    Troll.ispaid := nil;
                    Troll.isduped := true;
                }
                golden_eggs.moveInto(In_Giant_Room);

                if (golden_eggs.isIn(actor.location)) {
                    P(); I();
                    "A large nest full of golden
                    eggs suddenly appears out of
                    nowhere!";
                }
            }

            feeVerb.reset;
        }
        else
            feeVerb.fail;
    }
;
fumVerb: MagicWord         // Not really a magic word but we don't punish  
    sdesc = "fum"          // the user for uttering it in the Cylindrical Room
    verb = 'fum'
    action(actor) = {
        feeVerb.fail;
    }
;

nosideVerb: MagicWord
    omegapsical_order = 8
    omegaps580_order = 10
    omegaps701_order = 11
    omegaps701p_order = 12
    said = nil

    sdesc = "noside"
    verb = 'noside'
    action(actor) = {
        if (self.said or not global.game550)
            self.fail;
        else {
            "Ok!";
            samohtVerb.tcount := global.turnsofar + 1;
            self.said := true;
        }
    }

    fail = {
        "Nothing happens.";
        self.reset;
    }

    reset = {
        samohtVerb.tcount := -1;
        self.said := nil;
    }
;
samohtVerb: MagicWord
    omegapsical_order = 5
    omegaps580_order = 6
    omegaps701_order = 5
    omegaps701p_order = 5
    tcount = -1
    sdesc = "samoht"
    verb = 'samoht'
    action(actor) = {
        /* No way that nosideVerb.said could be true in 350-point
         * mode. */
        if (nosideVerb.said and (self.tcount = global.turnsofar)) {
            brass_lantern.magic_recharge;
            nosideVerb.reset;
        }
        else nosideVerb.fail;
    }
;

compoundWord 'noside' 'samoht' 'noside-samoht';
// DJP - added a one-line version of noside samoht, because this is what
// the original 550-point game seems to want (whereas the 660-point game
// requires that the words be typed separately).
nosidesamohtVerb: MagicWord
    sdesc = "noside samoht"
    verb = 'noside-samoht'
    action(actor) = {
        if (not global.game550) {
            nosideVerb.fail;
        }
        else {
            brass_lantern.magic_recharge;
            nosideVerb.reset;
        }
    }
;

thgirwVerb: MagicWord
    omegaps580_order = 4
    said = nil

    sdesc = "thgirw"
    verb = 'thgirw'
    action(actor) = {
        if (self.said or not global.game580)
            self.fail;
        else {
            "Ok!";
            rubliwVerb.tcount := global.turnsofar + 1;
            self.said := true;
        }
    }

    fail = {
        "Nothing happens.";
        self.reset;
    }

    reset = {
        rubliwVerb.tcount := -1;
        self.said := nil;
    }
;
rubliwVerb: MagicWord
    omegaps580_order = 7
    tcount = -1
    sdesc = "rubliw"
    verb = 'rubliw'
    action(actor) = {
        if (thgirwVerb.said and (self.tcount = global.turnsofar)) {
            self.activate_rug(actor);
            thgirwVerb.reset;
        }
        else thgirwVerb.fail;
    }
    activate_rug(actor) = {
        local topl;
        topl := toplocation(actor);
        if ((topl = E_Of_Rift || topl = W_Of_Rift)
            && persian_rug.isIn(topl)) 
        {
            if (persian_rug.isActive) 
            {
                "The persian rug gently settles to the ground. ";
                persian_rug.isActive := nil;
            }
            else if (persian_rug.isIn(actor)) "The rug ruffles in <<actor.fmtYour>> hands
                for a moment, and then subsides. ";
            else
            {
                "The persian rug levitates itself off the ground! ";
                persian_rug.isActive := true;
            }
        }
        else thgirwVerb.fail;
    }
;

compoundWord 'thgirw' 'rubliw' 'thgirw-rubliw';
// BJS - added a one-line version of rubliw thgirw, for consistency.
thgirwrubliwVerb: MagicWord
    sdesc = "thgirw rubliw"
    verb = 'thgirw-rubliw'
    action(actor) = {
        if (not global.game580) {
            thgirwVerb.fail;
        }
        else {
            rubliwVerb.activate_rug(actor);
            thgirwVerb.reset;
        }
    }
;

melenkurionVerb: MagicWord
    omegapsical_order = 9
    omegaps580_order = 11
    omegaps701_order = 12
    omegaps701p_order = 13
    sdesc = "melenkurion"
    verb = 'melenkurion'
    action(actor) = {
        if(actor.location<>By_Figure or Rock_Wall.has_crumbled)
            return actor.location.nothinghappens;
            // always the case in 350-point mode.
        else Rock_Wall.has_crumbled := true;
            "Rock silently crumbles off of the wall in
             front of you, revealing dark passages leading
             northwest, north, and northeast.\n";
    }
;

phugggVerb: MagicWord
    omegapsical_order = 7
    omegaps580_order = 9
    omegaps701_order = 9
    omegaps701p_order = 9
    isused = nil // The djinn will only mention it if it hasn't been used.
    sdesc = "phuggg"
    verb = 'phuggg'
    action(actor) = {
        local toploc := toplocation(actor);
        // Don't have djinn mention the word in the future.
        self.isused := true;
        if(not global.game550) return actor.location.nothinghappens;
        if(isclass(actor.location, Outside))
            return actor.location.nothinghappens; // Won't work outside.
        if(Streamitem.classfind(toploc) or
            Sea_Water.isIn(toploc) or
            water_in_the_bottle.isIn(toploc) or
            water_in_the_flask.isIn(toploc) or
            water_in_the_cask.isIn(toploc)) { // Uh, oh!
            if(rand(100) < 86) return actor.location.nothinghappens;
            else if(rand(100) < 96) {
                P(); ">splurch!<"; P();
                "Oh, no!  You've turned yourself into
                a jellyfish, and fallen to the ground
                and been splattered far and wide!
                Well, that certainly wasn't very
                smart!!!  You were warned not to use
                that word near water!\n"; die();
            }
            else { P(); "The ground begins to shudder ominously,
                and the very cave walls around you begin
                to creak and groan!  A sulphurious stench
                fills the air!"; P();

                "With an incredible lurch, the ground
                begins to dance and ripple as though
                it were liquid!  You are thrown off of
                your feet and tossed violently up and
                down!     The cave walls begin to crumble
                and split from the stress!"; P();

                "There is a terrible ROAR of rending
                rock!!     The cave ceiling splits, and
                rocks plunge down and smash your
                lower body to a gooey paste!!"; P();

                "There is a violent blast in the
                distance!  Steam and smoke drift
                into view through the rents in the
                walls, and furiously-bubbling red-hot
                lava flows in and surrounds you.
                The cave ceiling disintegrates in
                an incredible orgy of grinding
                destruction, and the cave walls fall
                and are pounded into fine dust."; P();

                "You are lying, badly mangled, on a
                small rock island in a sea of molten
                lava.    Above you, the sky is faintly
                visible through a thick pall of smoke
                and steam.  A short distance to the
                north, the remains of a well-house
                are sinking slowly into the bubbling
                ooze."; P();

                "There is a distant, sourceless screech
                of incredible anguish!     With a sharp
                >poof< and a small puff of orange smoke,
                a bent and bearded elf appears.  He
                is dressed in working clothes, and
                has a name-tag marked \"Ralph\" on
                his shirt.  \"You blithering idiot!\"
                he storms.  \"You were warned quite
                clearly not to use that word near
                water!!  I hadn't gotten all of the
                bugs out of it yet, and now your
                incredible incompetence has totally
                destroyed Colossal Cave!!  Do you
                have the faintest scintilla of an
                iota of an understanding of how much
                work I'm going to have to do to get
                the cave rebuilt?!?  I'll have to go
                all the way to Peking for another
                dragon, and I'll have to convince the
                Goblin's Union to send me another team
                of gooseberry goblins; I'll have to
                sub-contract the building of the volcano
                out to the local totrugs, and worst
                of all I'll have to go through eight
                months of paperwork and red tape to
                file a new Environmental Impact
                statement!!  All because you couldn't
                follow directions, you purblind and
                meatbrained moron!  I'm rescinding all
                of your game points and throwing you
                out!  Out!   OUT!   GET OUT!$!%#&'@%!!%%!";
                quit(); abort; }
        }
        else {
            local shp; // Did anything happen?
            if (axe.isIn(actor.location)) { // delete axe.
                "Your axe glows bright orange and fades into
                nothingness.\n"; axe.moveInto(nil); shp := true;
            } // Note that as the code currently stands, the next
              // dwarf to encounter the player after this will throw
              // another axe!
            if (singing_sword.isIn(actor.location)) { // delete it.
                "Your singing sword jumps into the air, chants several
                bars of the \"Volga Boatman\", shoots off several
                fitful blue sparks, and disintegrates.\n";
                singing_sword.moveInto(nil); shp := true;
            }
            // Added the elfin sword for the 701-point game.
            if (sword.isIn(actor.location)) { // delete it.
                "Your gleaming sword jumps into the air, then disintegrates
                before your very eyes.   The fragments rust away to
                nothing. ";
                singing_sword.moveInto(nil); shp := true;
            }
            {  // Zap dwarves, if any.
                local numd,i;
                // Find number of dwarves present and store in numd.
                numd := Dwarves.numberhere(actor);

                if(numd) { // Kill off the dwarf(s) and/or the player.
                    shp := true;
                    if(rand(10) < 8) { // it worked: kill dwarves.
                        switch(rand(3)) {
                            case 1: "A clear, liquid chime
                            sounds in midair.";
                            if (numd > 1) "     A long green
                                tentacle covered with
                                sucker disks reaches out
                                from nowhere, grabs the
                                dwarves, and pulls them
                                back to wherever it came
                                from.";
                            else "    A large, four-clawed
                                hand reaches out of the
                                ground, grabs the dwarf,
                                and pulls it down into
                                nothingness.";
                            break;
                            case 2: "There is a sharp sizzling
                            sound.";
                            if(numd = 1) "    The dwarf explodes
                                into flame and vanishes.";
                            else "    The dwarves are engulfed
                                in a wave of fire that
                                appears from nowhere, and
                                are completely incinerated;
                                the flames then vanish into
                                nothingness again.";
                            break;
                            case 3: "There is a sharp whistling
                            sound from nowhere.";
                            if(numd > 1) "    The dwarves stiffen,
                                shudder, and melt down into a
                                large puddle of soggy goo that
                                quickly soaks into the ground
                                and vanishes.";
                            else "    The dwarf shudders and
                                turns into a moth, which then
                                flies away.";
                            break;
                        }
                        // Remove each of the dwarves.
                        for (i:=0; i<numd; i++)
                            Dwarves.loclist -= Dwarves.location;
                        "\n";
                    }
                    else { // It failed: kill player.
                    switch(rand(3)) {
                        case 1: "A clear, liquid chime
                        sounds in midair.";
                        if (numd > 1) "     A large and
                            very toothy mouth appears
                            in midair and chomps
                            ferociously.  The dwarves
                            manage to evade it, but
                            it bites you in half.";
                        else "    A large, four-clawed
                            foot appears in midair
                            and stomps violently
                            downward, missing the
                            dwarf but thoroughly
                            squashing you.";
                        break;
                        case 2: "There is a sharp sizzling
                        sound.";
                        if(numd = 1) "    A ball of fire
                            roars out of nowhere, misses
                            the dwarf, bounces off of a
                            wall, and incinerates you.";
                        else "    A ball of fire appears
                            from nowhere, bounces off
                            the ground, and explodes
                            violently, incinerating
                            both you and the dwarves.";
                        break;
                        case 3: "There is a sharp crackling
                        sound from the air above you.";
                        if(numd > 1) "    The dwarves stiffen,
                            fall to the ground, and melt
                            into a large puddle of soggy
                            goo.  The goo twitches a few
                            times and then flows at you
                            with incredible speed;  it
                            attacks and strangles you
                            with little difficulty.";
                        else "    The dwarf shudders and
                            turns into a sabre-toothed
                            tiger, which attacks and kills
                            you in short order.";
                        break;
                    }
                    "\n";
                    // Remove each of the dwarves.
                    for (i:=0; i<numd; i++)
                        Dwarves.loclist -= Dwarves.location;
                        die();
                    }
                }
            if (not shp) return actor.location.nothinghappens;
            }
        }
    }
;

compoundWord 'open' 'sesame' 'opensesame';
compoundWord 'abra' 'cadabra' 'abracadabra';
compoundWord 'hocus' 'pocus' 'hocuspocus';
compoundWord 'hokus' 'pokus' 'hokuspokus';
compoundWord 'foo' 'bar' 'foobar';

losingmagicVerb: MagicWord     // Not really a magic word.
    sdesc = "foobar"
    verb = 'sesame' 'open-sesame' 'opensesame' 'abracadabra'
        'shazam' 'shazzam' 'hocus' 'pocus' 'hokus' 'pokus'
        'hocuspocus' 'hocus-pocus' 'hokuspokus' 'hokus-pokus'
        'foobar'
    action(actor) = {
        "Good try, but that is an old worn-out magic word.";
    }
;

listenVerb: darkVerb // DJP
    verb = 'listen' 'listen to'
    sdesc = "listen to"
    action(actor) = {
        local toploc := toplocation(actor);
        toploc.listendesc;
    }
    doAction = 'Listento'
;


/*
 * new general-purpose verbs.
 */
helpVerb: darkVerb
    sdesc = "help"
    verb = 'help' 'info' 'information'
    action(actor) = { help(); }
;

// implement wave <item> at <object>

waveVerb: darkVerb
    sdesc = "wave"
    verb = 'wave' 'shake'
    doAction = 'Wave'
    ioAction(atPrep) = [ disambigDobjFirst ] 'WaveAt'
    ioDefault( actor, prep ) = {
        // find dobj using the parserGetObj function
        local dobj := parserGetObj(PO_DOBJ),list := [];
        // just in case .. (but dobj should be available here because
        // we've specified disambigDobjFirst
        if (dobj = nil) pass ioDefault;
        // if the direct object is a weapon, use the throw targets (plus
        // the shadowy figure, if present.)
        else if(isclass(dobj,weapon)) {
            list := throwVerb.ioDefault(actor,atPrep);
            if (actor.isIn(Window.location)) list += ShadowyFigure;
            return list;
        }
        else pass ioDefault;
    }
;

rideVerb: deepverb
    sdesc = "ride"
    verb = 'ride'
    doAction = 'Ride'
;

smellVerb: darkVerb
    sdesc = "smell"
    verb = 'smell' 'sniff' 'waft'
    doAction = 'Smell'
;

rubVerb: deepverb
    sdesc = "rub"
    verb = 'rub' 'caress' 'fondle' 'pat' 'pet' 'hug' 'cuddle' 'squeeze'
    doAction = 'Rub'
;

countVerb: deepverb // DJP - enhanced to actually work sensibly
                    // (in most cases).
    verb = 'count'
    sdesc = "count"
    doAction = 'Count'
    disambigDobj(actor,prep,iobj,verprop,wordlist,objlist,flaglist,
    numberWanted,isAmbiguous, silent) = {
       local objsel := objlist[1];
       self.value := length(objlist);
       return [DISAMBIG_DONE, objsel];
    }
    validDo( actor, obj, seqno ) =
    {
        return( obj.isVisible( actor ));
    }
;

tieVerb: deepverb
    sdesc = "tie"
    verb = 'tie' 'knot'
    doAction = 'Tie'
;

untieVerb: deepverb
    sdesc = "untie"
    verb = 'untie' 'unknot'
    doAction = 'Untie'
;

pourVerb: deepverb
    sdesc = "pour"
    verb = 'pour' 'dump'
    prepDefault = onPrep
    ioAction(onPrep) = 'PourOn'
;
douseVerb: deepverb
    sdesc = "douse"
    verb = 'douse' 'drench'
    prepDefault = withPrep
    // DJP - added disambigDobjFirst
    ioAction(withPrep) = [ disambigDobjFirst ] 'DouseWith'
;

oilVerb: deepverb
    sdesc = "oil"
    verb = 'oil' 'grease' 'lubricate'
    doAction = 'Oil'
;

waterVerb: deepverb
    sdesc = "water"
    verb = 'water'
    doAction = 'Water'
;

kickVerb: deepverb
    sdesc = "kick"
    verb = 'kick' 'knee'
    doAction = 'Kick'
;

singVerb: darkVerb
    sdesc = "sing"
    verb = 'sing' 'whistle'
    doAction = 'Sing'
    action(actor) = { "You don't sound half bad. But don't quit your day
    job."; }
;

useVerb: deepverb
    sdesc = "use"
    verb = 'use' 'utilize' 'employ'
    doAction = 'Use'
;

sweepVerb: deepverb
    sdesc = "dust"    // this sdesc is a clue!
    verb = 'brush' 'sweep' 'dust'
    ioAction( withPrep ) = 'SweepWith'
    doAction = 'Sweep'
;

lightVerb: deepverb, darkVerb
/* Changed to work in the dark so we can turn on the lamp. */
    sdesc = "light"
    verb = 'light'
    doAction = 'Light'
;

pickVerb: deepverb
    sdesc = "pick"
    verb = 'pick'
    doAction = 'Pick'
;

wakeVerb: deepverb
    sdesc = "wake"
    verb = 'wake' 'awaken' 'wake up' 'disturb'
    doAction = 'Wake'
;

blastVerb: deepverb
    sdesc = "blast"
    verb = 'blast' 'blast with'
    action(actor) = {
        if (actor.isIn(At_Ne_End) or actor.isIn(At_Sw_End))
            black_mark_rod.doBlastWith(actor);
        else
            "Blasting requires dynamite.";
    }
    doAction = 'BlastWith'
;

feedVerb: deepverb
    sdesc = "feed"
    verb = 'feed' 'stuff' 'fatten'
    doAction = 'Feed'
    ioAction( withPrep ) = 'FeedWith'
    // DJP - added disambigDobjFirst
    ioAction( toPrep ) = [ disambigDobjFirst ] 'FeedTo'
    prepDefault = withPrep
;

fillVerb: deepverb
    sdesc = "fill"
    verb = 'fill'
    doAction = 'Fill'
    ioAction (withPrep) = 'FillWith'
;

emptyVerb: darkVerb
    sdesc = "empty"
    verb = 'empty'
    doAction = 'Empty'
;

/* blow, play (for lyre, horn) */

blowVerb: darkVerb
    sdesc = "blow"
    verb = 'blow'
    doAction = 'Blow'

;

playVerb: darkVerb
    sdesc = "play"
    verb = 'play'
    doAction = 'Play'

;

// DJP -
// New verbs 'climb up' and 'climb down'.  As simple verbs these are usually
// equivalent to 'up' and 'down' unless the room redefines the climbup and
// climbdown methods (which just call the up and down methods by default).

climbdownVerb: deepverb, travelVerb // DJP
    verb = 'climb down'
    travelDir( actor ) = { return(actor.location.climbdown); }
    sdesc = "climb down"
    doAction = 'Climbdown'
;
climbupVerb: deepverb, travelVerb // DJP
    verb = 'climb up'
    travelDir( actor ) = { return(actor.location.climbup); }
    sdesc = "climb up"
    doAction = 'Climbup'
;

/*
 * From here until the end of the file we define the new direction verbs.
 * Not very exciting reading.  Note that we've made the travel action method
 * more complex, so verbs now inherit their action method from the class.
 */

modify travelVerb
    action( actor ) = {
        local travelsave, currentsave;
        actor.nextRoute := 0; // in case it wasn't reset to its default

        travelsave := global.travelActor;
        currentsave := global.currentActor;

        global.travelActor := actor;  // actor doing the travelling
        global.currentActor := actor; // reference actor for location method
                                      // evaluation

        // travelDir is defined for each individual verb, and is
        // normally a method like this:
        // { return(actor.location.<prop>); }
        // where <prop> is the associated travel property, which should
        // evaluate to a room, a door, or nil (if no travel is to take place).

        actor.travelTo(self.travelDir(actor));

        global.travelActor := travelsave;
        global.currentActor := currentsave;

        actor.nextRoute := 0; // return to default in case the travel method
                              // changed it.
    }
;

modify eVerb replace action(actor) = {pass action;};
modify sVerb replace action(actor) = {pass action;};
modify nVerb replace action(actor) = {pass action;};
modify wVerb replace action(actor) = {pass action;};
modify neVerb replace action(actor) = {pass action;};
modify nwVerb replace action(actor) = {pass action;};
modify seVerb replace action(actor) = {pass action;};
modify swVerb replace action(actor) = {pass action;};
modify inVerb replace action(actor) = {pass action;};
replace outVerb: travelVerb
    sdesc = "go out"
    verb = 'out' 'go out' 'exit' // 'leave' removed
    travelDir(actor) = { return actor.location.out; }
    doAction = 'Unboard' // DJP
;
leaveVerb: travelVerb // separated 
    verb = 'leave'
    sdesc = "leave"
    // Same as 'out' or 'exit' if used intransitively
    travelDir( actor ) = { return( actor.location.out ); }
    // Transitive verbs now use a special method
    doAction = 'Leave' // DJP
;
modify dVerb 
    verb = 'd' 'down' 'go down' 'descend' // DJP added descend
    replace action(actor) = {pass action;}
    doAction = 'Climbdown' // DJP
;
modify uVerb 
    verb = 'u' 'up' 'go up' 'ascend' // DJP added ascend
    replace action(actor) = {pass action;}
    doAction = 'Climbup' // DJP
;
     
/*
 * NB the Back verb is now implemented using the room 'back' property.
 */

backVerb: travelVerb
    verb = 'back' 'return' 'retreat'
    action( actor ) = {
        local travelsave, currentsave;
        actor.nextRoute := actor.travelRoute; // use same route code
                          // as previous move unless
                          // overridden explicitly
        travelsave := global.travelActor;
        currentsave := global.currentActor;

        global.travelActor := actor;  // actor doing the travelling
        global.currentActor := actor; // reference actor for location method
                                      // evaluation
        actor.travelTo(self.travelDir(actor));

        global.travelActor := travelsave;
        global.currentActor := currentsave;

        actor.nextRoute := 0; // return to default in case the travel method
                              // changed it.
    }
    travelDir(actor) = {
        return(actor.location.back);
    }
;

replace climbVerb: travelVerb // DMB: made this into a travelVerb
    verb = 'climb'
    travelDir( actor ) = { return(actor.location.climb); }

    sdesc = "climb"
    doAction = 'Climb'
;

replace jumpVerb: travelVerb         // this is normally defined in adv.t
    verb = 'jump' 'leap' 'jump over' 'leap over'
    sdesc = "jump"
    travelDir(actor) = {
        return(actor.location.jump); // DJP -- corrected
    }
    doAction = 'Jump'                // DJP -- reinstated
;

roadVerb: travelVerb
    verb = 'road' 'hill'
    travelDir(actor) = {
        return(actor.location.road);
    }
;

upstreamVerb: travelVerb
    verb = 'upstream'
    travelDir(actor) = {
        return(actor.location.upstream);
    }
;

downstreamVerb: travelVerb
    verb = 'downstream'
    travelDir(actor) = {
        return(actor.location.downstream);
    }
;

forestVerb: travelVerb
    verb = 'forest'
    travelDir(actor) = {
        return(actor.location.forest);
    }
;

forwardsVerb: travelVerb
    verb = 'forwards' 'continue' 'onward'
    travelDir(actor) = {
        return(actor.location.forwards);
    }
;

valleyVerb: travelVerb
    verb = 'valley'
    travelDir(actor) = {
        return(actor.location.valley);
    }
;

stairsVerb: travelVerb
    verb = 'stairs'
    travelDir(actor) = {
        return(actor.location.stairs);
    }
;

buildingVerb: travelVerb
    verb = 'building' 'house'
    travelDir(actor) = {
        return(actor.location.building);
    }
;

gullyVerb: travelVerb
    verb = 'gully'
    travelDir(actor) = {
        return(actor.location.gully);
    }
;

streamVerb: travelVerb
    verb = 'stream'
    travelDir(actor) = {
        return(actor.location.stream);
    }
;

rockVerb: travelVerb
    verb = 'rock'
    travelDir(actor) = {
        return(actor.location.rock);
    }
;

bedVerb: travelVerb
    verb = 'bed'
    travelDir(actor) = {
        return(actor.location.bed);
    }
;

crawlVerb: travelVerb
    verb = 'crawl'
    travelDir(actor) = {
        return(actor.location.crawl);
    }
;

cobbleVerb: travelVerb
    verb = 'cobble'
    travelDir(actor) = {
        return(actor.location.cobble);
    }
;

surfaceVerb: travelVerb
    verb = 'surface'
    travelDir(actor) = {
        return(actor.location.tosurface);
    }
;

dark_Verb: travelVerb
    verb = 'dark'
    travelDir(actor) = {
        return(actor.location.dark);
    }
;

passageVerb: travelVerb
    verb = 'passage' 'tunnel' 'opening'
    travelDir(actor) = {
        return(actor.location.passage);
    }
;

lowVerb: travelVerb
    verb = 'low'
    travelDir(actor) = {
        return(actor.location.low);
    }
;

canyonVerb: travelVerb
    verb = 'canyon'
    travelDir(actor) = {
        return(actor.location.canyon);
    }
;

awkwardVerb: travelVerb
    verb = 'awkward'
    travelDir(actor) = {
        return(actor.location.awkward);
    }
;

giantVerb: travelVerb
    verb = 'giant'
    travelDir(actor) = {
        return(actor.location.giant);
    }
;

viewVerb: travelVerb
    verb = 'view'
    travelDir(actor) = {
        return(actor.location.view);
    }
;

pitVerb: travelVerb
    verb = 'pit'
    travelDir(actor) = {
        return(actor.location.pit);
    }
;

outdoorsVerb: travelVerb
    verb = 'outdoors'
    travelDir(actor) = {
        return(actor.location.outdoors);
    }
;

crackVerb: travelVerb
    verb = 'crack'
    travelDir(actor) = {
        return(actor.location.crack);
    }
;

stepsVerb: travelVerb
    verb = 'steps'
    travelDir(actor) = {
        return(actor.location.steps);
    }
;

domeVerb: travelVerb
    verb = 'dome'
    travelDir(actor) = {
        return(actor.location.dome);
    }
;

leftVerb: travelVerb
    verb = 'left'
    travelDir(actor) = {
        return(actor.location.left);
    }
;

rightVerb: travelVerb
    verb = 'right'
    travelDir(actor) = {
        return(actor.location.right);
    }
;

replace centerVerb: object // zapped
;

middleVerb: travelVerb
    verb = 'middle' 'center' 'centre'
    travelDir(actor) = {
        return(actor.location.middle);
    }
;

hallVerb: travelVerb
    verb = 'hall'
    travelDir(actor) = {
        return(actor.location.hall);
    }
;

barrenVerb: travelVerb
    verb = 'barren'
    travelDir(actor) = {
        return(actor.location.barren);
    }
;

overVerb: travelVerb
    verb = 'over'
    travelDir(actor) = {
        return(actor.location.over);
    }
;

acrossVerb: travelVerb
    verb = 'across'
    travelDir(actor) = {
        return(actor.location.across);
    }
;

debrisVerb: travelVerb
    verb = 'debris'
    travelDir(actor) = {
        return(actor.location.debris);
    }
;

holeVerb: travelVerb
    verb = 'hole'
    travelDir(actor) = {
        return(actor.location.hole);
    }
;

wallVerb: travelVerb
    verb = 'wall'
    travelDir(actor) = {
        return(actor.location.wall);
    }
;

brokenVerb: travelVerb
    verb = 'broken'
    travelDir(actor) = {
        return(actor.location.broken);
    }
;

y2Verb: travelVerb
    verb = 'y2'
    travelDir(actor) = {
        return(actor.location.y2);
    }
;

pantryVerb: travelVerb
    verb = 'pantry'
    travelDir(actor) = {
        return(actor.location.pantry);
    }
;

floorVerb: travelVerb
    verb = 'floor'
    travelDir(actor) = {
        return(actor.location.floor);
    }
;

roomVerb: travelVerb
    verb = 'room'
    travelDir(actor) = {
        return(actor.location.toroom);
    }
;

slitVerb: travelVerb
    verb = 'slit'
    travelDir(actor) = {
        return(actor.location.slit);
    }
;

slabVerb: travelVerb
    verb = 'slab' 'slabroom'
    travelDir(actor) = {
        return(actor.location.slab);
    }
;

xyzzyVerb: magicTravelVerb
    omegapsical_order = 2
    omegaps580_order = 2
    omegaps701_order = 2
    omegaps701p_order = 2
    verb = 'xyzzy'
    travelDir(actor) = {
        return(actor.location.xyzzy);
    }
;

depressionVerb: travelVerb
    verb = 'depression' 'grate'    // DMB: added 'grate'
    travelDir(actor) = {
        return(actor.location.depression);
    }
;

entranceVerb: travelVerb
    verb = 'entrance'
    travelDir(actor) = {
        return(actor.location.entrance);
    }
;

/* All locations in the 350-point version have thurb properties of nil,
 *    so it should always produce "Nothing happens." in that mode. */
thurbVerb: magicTravelVerb
    omegapsical_order = 3
    omegaps580_order = 3
    omegaps701_order = 3
    omegaps701p_order = 3
    verb = 'thurb'
    travelDir(actor) = {
        return(actor.location.thurb);
    }
;

plughVerb: magicTravelVerb
    omegapsical_order = 6
    omegaps580_order = 8
    omegaps701_order = 8
    omegaps701p_order = 8
    verb = 'plugh'
    travelDir(actor) = {
        return(actor.location.plugh);
    }
;

secretVerb: travelVerb
    verb = 'secret'
    travelDir(actor) = {
        return(actor.location.secret);
    }
;

caveVerb: travelVerb
    verb = 'cave'
    travelDir(actor) = {
        return(actor.location.cave);
    }
;

crossVerb: travelVerb
    verb = 'cross'
    sdesc = "cross"
    travelDir(actor) = {
        return(actor.location.cross);
    }
    doAction = 'Cross'
;

bedquiltVerb: travelVerb
    verb = 'bedquilt'
    travelDir(actor) = {
        return(actor.location.bedquilt);
    }
;

ploverVerb: magicTravelVerb // Note that the original 550-point, 660-point,
                // and 580-point versions of Adventure don't treat this
                // as a magic verb.     For this reason, its use
                // in the Cylindrical Room has been made
                // optional.

    omegapsical_order = -7
    omegaps580_order = -9
    omegaps701_order = -9
    omegaps701p_order = -9
    verb = 'plover'
    // Removed redundant emerald-dropping code.
    travelDir(actor) = {
        return(actor.location.plover);
    }
;

orientalVerb: travelVerb
    verb = 'oriental'

    travelDir(actor) = {
        return(actor.location.oriental);
    }
;

cavernVerb: travelVerb
    verb = 'cavern'
    travelDir(actor) = {
        return(actor.location.cavern);
    }
;

shellVerb: travelVerb
    verb = 'shell'
    travelDir(actor) = {
        return(actor.location.shell);
    }
;

reservoirVerb: travelVerb
    verb = 'reservoir'
    travelDir(actor) = {
        return(actor.location.reservoir);
    }
;

mainVerb: travelVerb
    verb = 'main' 'office'
    travelDir(actor) = {
        return(actor.location.main);
    }
;

forkVerb: travelVerb
    verb = 'fork'
    travelDir(actor) = {
        return(actor.location.fork);
    }
;

chimneyVerb: travelVerb
    verb = 'chimney'
    travelDir(actor) = {
        return(actor.location.chimney);
    }
;

slideVerb: travelVerb
    verb = 'slide'
    travelDir(actor) = {
        return(actor.location.slide);
    }
;

poolVerb: travelVerb
    verb = 'pool' 'whirlpool'
    travelDir(actor) = {
        return(actor.location.pool);
    }
;

knollVerb: travelVerb
    verb = 'knoll'
    travelDir(actor) = {
        return(actor.location.knoll);
    }
;

phuceVerb: magicTravelVerb
    verb = 'phuce'
    omegaps701_order = 10
    omegaps701p_order = 10
    action(actor) = {
        if (not global.newgame)
            "Nothing happens. ";
        else if (not Knoll.seenit) {
            "Nothing happens.  If that's an attempt at Elvish magic, 
            it won't work at all until you've heard how to pronounce the word
            correctly. ";
        }
        else pass action;
    }
    travelDir(actor) = {
        return(actor.location.phuce);
    }
;

ledgeVerb: travelVerb
    verb = 'ledge' 'shelf'
    travelDir(actor) = {
        return(actor.location.ledge);
    }
;

thunderVerb: travelVerb
    verb = 'thunder' 'thunderhole'
    travelDir(actor) = {
        return(actor.location.thunder);
    }
;

clickVerb: magicTravelVerb
    verb = 'click'
    sdesc = "click"
    omegaps701_order = 19
    omegaps701p_order = 21
    action( actor ) = {
        if (not (slippers.location = actor) or not slippers.isworn) {
        "%You% click%s% %your% heels, but nothing happens. ";
        return;
        }
        else pass action;
    }

    travelDir(actor) = {
        slippers.clicked := true;
        return(actor.location.click);
    }
    doAction = 'Click'
;

iceVerb: travelVerb
    verb = 'ice'
    travelDir(actor) = {
        return(actor.location.ice);
    }
;

bridgeVerb: travelVerb
    verb = 'bridge'
    travelDir(actor) = {
        return(actor.location.bridge);
    }
;

altarVerb: travelVerb
    verb = 'altar'
    travelDir(actor) = {
        return(actor.location.altar);
    }
;

grottoVerb: travelVerb
    verb = 'grotto'
    travelDir(actor) = {
        return(actor.location.grotto);
    }
;

prayVerb: magicTravelVerb
    verb = 'pray' 'prayer'
    omegaps701_order = 7
    omegaps701p_order = 7
    travelDir(actor) = {
        return(actor.location.pray);
    }
;

smichelVerb: magicTravelVerb
    verb = 'saint-michel'
    omegaps701_order = 6
    omegaps701p_order = 6
    action(actor) = {
        if (not global.newgame) {
            "Nothing happens. ";
        }
        else if (not Rise_Over_Bay.seenit) {
            "Nothing happens.  If that's an attempt at Elvish magic, 
            it won't work at all until you've heard the correct intonation
            for the words. ";
        }
        else pass action;
    }

    travelDir(actor) = {
        return(actor.location.smichel);
    }
;

gateVerb: travelVerb
    verb = 'gate'
    travelDir(actor) = {
        return(actor.location.gate);
    }
;

reflectVerb: magicTravelVerb   // This really belongs in the 440-point and
                               // 660-point versions, so we don't require it
                               // in the cylindrical room.
    verb = 'reflect'
    travelDir(actor) = {
        return(actor.location.reflect);
    }
;

balconyVerb: travelVerb
    verb = 'balcony'
    travelDir(actor) = {
        return(actor.location.balcony);
    }
;

gorgeVerb: travelVerb
    verb = 'gorge'
    travelDir(actor) = {
        return(actor.location.gorge);
    }
;

lairVerb: travelVerb
    verb = 'lair'
    travelDir(actor) = {
        return(actor.location.lair);
    }
;

fourierVerb: travelVerb
    verb = 'fourier'
    travelDir(actor) = {
        return(actor.location.fourier);
    }
;

jonahVerb: travelVerb
    verb = 'jonah'
    travelDir(actor) = {
        return(actor.location.jonah);
    }
;

pentagramVerb: travelVerb
    verb = 'pentagram'
    travelDir(actor) = {
        return(actor.location.pentagram);
    }
;

nondescriptVerb: travelVerb
    verb = 'nondescript'
    travelDir(actor) = {
        return(actor.location.nondescript);
    }
;

tubeVerb: travelVerb
    verb = 'tube'
    travelDir(actor) = {
        return(actor.location.tube);
    }
;

peelgruntVerb: travelVerb
    verb = 'peelgrunt'
    travelDir(actor) = {
        return(actor.location.peelgrunt);
    }
;

safeVerb: travelVerb
    verb = 'safe'
    travelDir(actor) = {
        return(actor.location.safe);
    }
;

goldenVerb: travelVerb
    verb = 'golden'
    travelDir(actor) = {
        return(actor.location.golden);
    }
;

arabesqueVerb: travelVerb
    verb = 'arabesque'
    travelDir(actor) = {
        return(actor.location.arabesque);
    }
;

translucentVerb: travelVerb
    verb = 'translucent'
    travelDir(actor) = {
        return(actor.location.translucent);
    }
;

anaVerb: magicTravelVerb
    omegaps701p_order = 23
    verb = 'ana'
    travelDir(actor) = {
        local toploc := toplocation(actor);
        // check that the gold pendant is worn
        if (not toploc.isdotroom and not (pendant.location = actor and
        pendant.isworn)) {
            if(global.fourd) return(actor.location.nothinghappens);
            else return(actor.location.nofourd(self));
        }
        // travel in a Zarkalon room is only possible if the actor is wearing
        // a Zarkalonized pendant.
        if (toploc.Zarkalonroom) {
            local o := firstobj(pendantItem),zpend := nil;
            while (o) {
               if((o.location = actor) and o.isworn and o.Zarkalonized) 
                    zpend := true;
               o := nextobj(o,pendantItem);
            }
            if(not zpend) return actor.location.nothinghappens;            
        }
        // upper transindection chambers work as if the player has all
        // the pendants. In addition, if the property has_ana_extradot is set
        // to true, the property ana_extradot will be used as the 'ana'
        // property for the room.
        if (toploc.isdotroom) {
            self.setfourd;
            if (toploc.has_ana_extradot)
                return(actor.location.ana_extradot);
            else
                return(actor.location.ana3);
        }     
        // check for any other pendants worn in conjunction with the
        // gold pendant.
        else if ((pendant3.location = actor) and pendant3.isworn) {
            self.setfourd;
            return(actor.location.ana3);
        }
        else if ((pendant2.location = actor) and pendant2.isworn) {
            self.setfourd;
            return(actor.location.ana2);
        }
        else {
            self.setfourd;
            return(actor.location.ana);
        }
    }
    setfourd = {
        local actor := getActor(&travelActor);
        if ((not global.fourd) and actor.isIn(Upper_Trans_Room)) {
            "The air around you seems to shimmer, and the room around you 
            appears to melt and blur for a moment, before solidifying again.  
            At first you think that nothing has really happened - but then
            you notice that the floor has changed.  And the west passage.
            Somehow you \(know\) what has happened - you've traveled in
            Transindection! ";
        }
        global.fourd := true;
        P();
   } 
   alert_message(actor) = {
       if (toplocation(actor).isdotroom) {
           "A hollow voice says, \"This room is out of use due
           to a cave security alert at Red level.  Please try again later.\" ";
       }
       else if ((pendant.location = actor) and pendant.isworn) {
           "A voice whispers in your ear: \"Security error 30:  Control Room
           reports a cave security alert at this Level.  Please try again
           later.\" ";
       }
       else pass alert_message;
   }
;

kataVerb: magicTravelVerb
    omegaps701p_order = 16
    verb = 'kata'
    travelDir(actor) = {
        local toploc := toplocation(actor);
        // check that the gold pendant is worn
        if (not toploc.isdotroom and not (pendant.location = actor and
        pendant.isworn)) {
            if(global.fourd) return(actor.location.nothinghappens);
            else return(actor.location.nofourd(self));
        }
        // travel in a Zarkalon room is only possible if the actor is wearing
        // a Zarkalonized pendant.
        if (toploc.Zarkalonroom) {
            local o := firstobj(pendantItem),zpend := nil;
            while (o) {
               if((o.location = actor) and o.isworn and o.Zarkalonized) 
                    zpend := true;
               o := nextobj(o,pendantItem);
            }
            if(not zpend) return actor.location.nothinghappens;            
        }
        // upper transindection chambers work as if the player has all
        // the pendants. In addition, if the property has_kata_extradot is set
        // to true, the property kata_extradot will be used as the 'kata'
        // property for the room.
        if (toploc.isdotroom) {
            self.setfourd;
            if (toploc.has_kata_extradot)
                return(actor.location.kata_extradot);
            else
                return(actor.location.kata3);
        }
        // check for any other pendants worn in conjunction with the
        // gold pendant.
        else if ((pendant3.location = actor) and pendant3.isworn) {
            return(actor.location.kata3);
        }
        else if ((pendant2.location = actor) and pendant2.isworn) {
            return(actor.location.kata2);
        }
        else {
            return(actor.location.kata);
        }
    }
    alert_message(actor) = {anaVerb.alert_message(actor);}
;

osalVerb: MagicWord // not really...
    verb='osal'
    sdesc="osal"
    action(actor) = {
        "Nothing happens, and you have the distinct feeling that you're
        missing something here ... ";
    }
;
phrosalVerb: MagicWord
    omegaps701p_order = 11
    verb='phrosal'
    sdesc="phrosal"
    action(actor) = {
        local o, zpend := nil, toploc := toplocation(actor);
        // No effect if we're not playing the 701+ point game, or the player
        // is inside a machine chamber ...
        if(not global.game701p or toploc.isMachineChamber) {
             "Nothing happens. ";
             return;
        }
        // No effect unless at least one pendant is carried
        o := firstobj(pendantItem),zpend := nil;
        while (o) {
            if(o.isInside(actor)) {
                zpend := true;
            }
            o := nextobj(o,pendantItem);
        }
        if(not zpend) {
             "Nothing happens. ";
             return;
        }

        // Otherwise, the chamber is opened or closed
        else if (not global.OpenedMachineChamber) {
             Machine_Chamber_Class.chamberopen(actor);
             self.invoked := true;
        }
        else
             Machine_Chamber_Class.chamberclose(actor);
        // If we're not close to the machine chamber, hint that
        // something has happened.
        if (not (toploc.isUpperTransRoom or toploc.isOctagonalRoom))
             "Nothing obvious happens. ";
    }
;   

/* verb for flying the rug */
flyVerb: travelVerb
    verb = 'fly'
    sdesc = "fly"
    action(actor) = {
        /* Apply standard checks for flying the rug */
        if (not persian_rug.flying_check)
            return;
        "Just tell me which direction you want to fly! ";
    }
    doAction = 'Fly'
;

class flyingVerb: travelVerb
    action(actor) = {
        local outhideStatus;
        /* Apply standard checks for flying the rug */
        if (not persian_rug.flying_check)
            return;

        /*
         *   silently check the verification method.
         */
        outhideStatus := outhide(true);
        persian_rug.(self.verprop)(actor);
        if (outhide(outhideStatus)) {
            /*
             *   verification failed - run again, showing the message
             *   this time
             */
            persian_rug.(self.verprop)(actor);
        }
        else {
            /* verification succeeded - do the action */
            persian_rug.(self.doprop)(actor);
        }
    }
;

flynVerb: flyingVerb
    verb = 'fly north' 'fly n'
    sdesc = "fly north"
    doAction = 'FlyN'
    verprop = &verDoFlyN // for 'fly north'
    doprop = &doFlyN
;
flysVerb: flyingVerb
    verb = 'fly south' 'fly s'
    sdesc = "fly south"
    doAction = 'FlyS'
    verprop = &verDoFlyS
    doprop = &doFlyS
;
flyeVerb: flyingVerb
    verb = 'fly east' 'fly e'
    sdesc = "fly east"
    doAction = 'FlyE'
    verprop = &verDoFlyE
    doprop = &doFlyE
;
flywVerb: flyingVerb
    verb = 'fly west' 'fly w'
    sdesc = "fly west"
    doAction = 'FlyW'
    verprop = &verDoFlyW
    doprop = &doFlyW
;
flyneVerb: flyingVerb
    verb = 'fly ne' 'fly northeast'
    sdesc = "fly ne"
    doAction = 'FlyNE'
    verprop = &verDoFlyNE
    doprop = &doFlyNE
;
flynwVerb: flyingVerb
    verb = 'fly nw' 'fly northwest'
    sdesc = "fly nw"
    doAction = 'FlyNW'
    verprop = &verDoFlyNW
    doprop = &doFlyNW
;
flyseVerb: flyingVerb
    verb = 'fly se' 'fly southeast'
    sdesc = "fly se"
    doAction = 'FlySE'
    verprop = &verDoFlySE
    doprop = &doFlySE
;
flyswVerb: flyingVerb
    verb = 'fly sw' 'fly southwest'
    sdesc = "fly sw"
    doAction = 'FlySW'
    verprop = &verDoFlySW
    doprop = &doFlySW
;

/* verbs for boats */
launchVerb: deepverb
    verb = 'launch'
    sdesc = "launch"
    doAction = 'Launch'

;
rowVerb: travelVerb
    verb = 'row'
    sdesc = "row"
    travelDir(actor) = {
        return(actor.location.row);
    }
    doAction = 'Row'
;

poleVerb: deepverb
    verb = 'pole' 'punt'
    sdesc = "pole"
    doAction = 'Pole'
    action(actor) = {
        if(not pole.isVisible(actor)) {
            "I can't see a pole here. ";
        }
        else askdo;
    }
;
class polingVerb: travelVerb
    action(actor) = {
        local outhideStatus;
        /* Check that we're in the boat. */
        if(not pole.isVisible(actor)) {
            "I can't see a pole here. ";
        }
        else if(not actor.isIn(Boat)) {
            "That might make some sense if %you% were in a boat, but %you%
            %are%n't. ";
        }
        else {
            /*
             *   silently check the verification method.
             */
            outhideStatus := outhide(true);
            Boat.(self.verprop)(actor);
            if (outhide(outhideStatus)) {
                /*
                 *   verification failed - run again, showing the message
                 *   this time
                 */
                Boat.(self.verprop)(actor);
            }
            else {
                /* verification succeeded - do the action */
                Boat.(self.doprop)(actor);
            }
        }
    }
;
polenVerb: polingVerb
    verb = 'pole north' 'punt north' 'pole n' 'punt n'
    sdesc = "pole north"
    doAction = 'PoleN'
    verprop = &verDoPoleN // for 'pole north'
    doprop = &doPoleN
;
polesVerb: polingVerb
    verb = 'pole south' 'punt south' 'pole s' 'punt s'
    sdesc = "pole south"
    doAction = 'PoleS'
    verprop = &verDoPoleS
    doprop = &doPoleS
;
poleeVerb: polingVerb
    verb = 'pole east' 'punt east' 'pole e' 'punt e'
    sdesc = "pole east"
    doAction = 'PoleE'
    verprop = &verDoPoleE
    doprop = &doPoleE
;
polewVerb: polingVerb
    verb = 'pole west' 'punt west' 'pole w' 'punt w'
    sdesc = "pole west"
    doAction = 'PoleW'
    verprop = &verDoPoleW
    doprop = &doPoleW
;
poleneVerb: polingVerb
    verb = 'pole ne' 'punt ne' 'pole northeast' 'punt northeast'
    sdesc = "pole ne"
    doAction = 'PoleNE'
    verprop = &verDoPoleNE
    doprop = &doPoleNE
;
polenwVerb: polingVerb
    verb = 'pole nw' 'punt nw' 'pole northwest' 'punt northwest'
    sdesc = "pole nw"
    doAction = 'PoleNW'
    verprop = &verDoPoleNW
    doprop = &doPoleNW
;
poleseVerb: polingVerb
    verb = 'pole se' 'punt se' 'pole southeast' 'punt southeast'
    sdesc = "pole se"
    doAction = 'PoleSE'
    verprop = &verDoPoleSE
    doprop = &doPoleSE
;
poleswVerb: polingVerb
    verb = 'pole sw' 'punt sw' 'pole southwest' 'punt southwest'
    sdesc = "pole sw"
    doAction = 'PoleSW'
    verprop = &verDoPoleSW
    doprop = &doPoleSW
;

// DJP - This is now a travel verb, but is hard to use.  Firstly, it's an
// Elvish magic word and can therefore only be used if the player has seen
// an elf use it.  Secondly, it requires a bracelet which is only available at
// White level in the 701+ point game.
phleeceVerb: magicTravelVerb
    verb = 'phleece'
    // we give this verb an 'omegapsical' order only in the 701+ point
    // version.
    omegaps701p_order = -12 //optional after 11
    action(actor) = {
        if(not Outer_Courtyard.seenit) {
            "Nothing happens.  If that's an attempt at Elvish magic, 
            it won't work at all until you've heard how to pronounce the word
            correctly. ";
        }
        else if((copper_bracelet.location = actor) and 
        copper_bracelet.isworn) {
            pass action;
        }
        else if(toplocation(actor) = Outer_Courtyard) {
            "Nothing happens.  It looks to me as if this word works
            only if you're wearing a special bracelet!
            I don't believe for a moment that the elves would
            be careless enough to leave one lying around";
            if(gold_ring.seenspecial) {
                ", or to let you follow them using the gold ring you found
                on the Wumpus - they're wise to that trick. ";
            }
            else ". ";
            "You'd be well advised to concentrate on exploring the 
            garden. ";
        }
        else {
            "Nothing happens. ";
        }
    }
    travelDir(actor) = {
        return(actor.location.phleece);
    }

;

/* verb for the bear */
stayVerb: deepverb
    verb = 'stay' 'stop'
    sdesc = "stay"
    action(actor) = {
        if(actor = Me)
            "This command should be issued to someone else,
            e.g. \"dog, stay\".";
        else actor.stay;
    }
;

/* verb for 'span over' */
spanVerb: deepverb
    verb = 'span'
    sdesc = "span"
    ioAction(overPrep) = 'SpanOver'
    ioAction(acrossPrep) = 'SpanOver'
;

