/*
 * Polyadventure
 *
 * A remake of the various versions of the classic Adventure game by Don
 * Woods and Willie Crowther, based on their sources.  Currently, the 350,
 * 550, and 551-point versions are implemented.  See the file "ccr-help.t"
 * for more information.
 *
 * Please document all changes in the history so we know who did what.
 *
 * This source code is copylefted under the terms of the GNU Public
 * License.  Essentially, this means that you are free to do whatever
 * you wish with this source code, provided you do not charge any
 * money for it or for any derivative works.
 *
 *
 * Contributors (see history.t for current e-mail addresses)
 *
 *      dmb     In real life:   David M. Baggett
 *
 *      djp     In real life:   David J. Picton
 *
 *      bjs     In real life:   Bennett J. Standeven
 *
 *
 * Modification History
 *
 * CCR
 * ===
 *
 * 1-Jan-93     dmb     rec.arts.int-fiction BETA release (source only)
 *                      For beta testing only -- not for general
 *                      distribution.
 *
 * 29-Apr-96    djp     Supplied some suggested enhancements/bugfixes
 *                      to dmb.  These were incorporated in Version 2.0.
 *
 * AD551
 * =====
 *
 * 24-Mar-99    djp     Preview version (1.0) of Adventure 551
 *
 * 14-Apr-99    djp     Initial release of Adventure 551 (1.01)
 *                      Changes since version 1.0:
 *                      * When adventurer dies, move 'phuced' objects back
 *                      * to normal-size rooms.
 *                      * Added global.treasures_to_find list (for
 *                        debugging)
 *                      * Me.actorAction method now allows 'purloin axe'
 *                        in non-debug-mode games if the nodwarves command
 *                        has been used (and the axe hasn't already been
 *                        obtained).
 *
 * 23-Apr-99    djp     New release (1.10) of Adventure 551.
 *                      Changes since version 1.01:
 *                      * Added mention of HELP command in introductory
 *                        text.
 *                      * When game terminates, issue the usual prompt for
 *                        RESTORE, UNDO etc.
 *
 * 17-May-99    djp     New release (1.20)
 *                      Changes since version 1.11
 *                      * Fixed scoreRank bug preventing the number of
 *                        points to the next category from being correctly
 *                        reported.
 *                      * Fixed a bug which caused Dial.combination to be
 *                        set incorrectly after a 'sitting on the throne'
 *                        restart.
 *                      * Restored more variables after sitting on the throne.
 *                      * Improved the efficiency of the scoring code.
 *                      * Changes to the preinit and init code which handles
 *                        objects with a different location/loclist in
 *                        different game versions.  For floatingdecoration
 *                        objects a new 'oldloclist' property will now set a
 *                        different loclist for 350-point games.  Changed
 *                        locations are now in global.changeloc; objects
 *                        with changed loclists are listed in
 *                        global.moveloclist, and the loclists themselves
 *                        are in global.changeloclist.
 *                      * Changed preinit and init to handle floatingdecoration
 *                        items more efficiently.  These are now placed in
 *                        room contents lists if the variable
 *                        global.floatcontents is set.
 *                      * Also changed preinit to implement a makecontents
 *                        class for floating items.  Note that this class
 *                        should only be used when the following condition
 *                        won't change during the game, or with the game
 *                        version:
 *                        when Me.location = room,  <item>.location = <room>
 *                        Here <item> is the floating item of interest, and
 *                        <room> is any location with class room including
 *                        nested rooms.
 *                      * Implemented a 'global.noall' list with all the
 *                        objects we want ALL to ignore, like your hands.
 *                        (Needed because these objects are now put into
 *                        contents lists.)
 *
 * 15-Jul-99    djp     New release (2.00)
 *                      Changes in this release
 *                      * Added new 'protection' method to Me object
 *                      * Changed the dwarf code to delay the startup of
 *                        dwarves and pirates, which can now be initiated
 *                        whenever the player enters a 'far in' room.  However,
 *                        the startup is suppressed on the first
 *                        'global.NPCwaitmoves' occasions, and then occurs
 *                        with a probability of 'global.NPCprob'.
 *                      * In a 'special' restart, additionally restore
 *                        Knoll.seenit and Rise_Over_Bay.seenit.
 *                      * Changed the handling of the oldlocation property.
 *                        If set explicitly to nil, this now causes an
 *                        object to start with a nil location in the old
 *                        game without being flagged as 'deleted'.
 *                      * Change to actorAction to support the 'number'
 *                        operand of purloin and gonear (see comments in
 *                        ccr-verb.t for a full explanation.)
 *
 * 3-Aug-99     djp     Bugfix release - Version 2.01
 *
 *              djp     Bugfix release - Version 2.02
 *                      * Changed the coding which warns about leaving on the
 *                        lamp.  It now looks for a special nolampwarn
 *                        property set by lit rooms outside the cave for
 *                        which the warning is to be suppressed.  The
 *                        rooms now include Top_Of_Steps and its associated
 *                        rooms.
 *                      * Fixed version.sdesc to indicate the game mode.
 *
 * 9-Nov-99     djp     New release - Version 2.10
 *                        Changes in this version
 *                      * Incorporated Bennett Standeven's new code for
 *                        generalized version handling.
 *                        Note that vnumber = 2 is for the 550-point game
 *                        (this is not included in Adventure 551, but will
 *                        be included in Bennett Standeven's forthcoming game.)
 *
 * 17-Feb-00    djp     New release - Version 2.20
 *                        Changes in this version
 *                      * Further generalization of version handling.  There
 *                        are now gamevvv properties (as defined in
 *                        global.gameflaglist) which indicate the version(s)
 *                        in which an object or room is present.  If none
 *                        are set the object/room will exist in all versions.
 *
 *                        The locationvvv and loclistvvv properties similarly
 *                        indicate version-dependent locations and loclists.
 *                        Note that explicitly setting a locationvvv property
 *                        to nil will change the initial object location to
 *                        be nil when the relevant version is played.
 *                        Leaving locationvvv undefined has a different
 *                        effect; it causes the location variable to be
 *                        left unchanged.
 *
 *                      * Fixed opening messages to avoid stating the
 *                        game version twice.
 *
 * POLYADV
 * =======
 *
 * 24-Aug-99    bjs     Pre-release version 0.00
 *                      Added 550-point extensions, upgraded version handling.
 *
 *          djp+bjs     Incorporated ad551 mods up to 2.20
 *
 * 3-Mar-00     djp     Initial beta release - Version 1.00
 *                      * Changed the resurrection code in die() to
 *                        run all daemons, disable the YES command, then
 *                        issue an abort to prevent further commands or
 *                        code from being processed.
 *                      * Fixed RESTART response to end-of-game prompt,
 *                        added GAME350, GAME550 and GAME551.
 *                      * Fixed broken code for version-dependent loclists.
 *                      * Added a global.maxresurrect variable to control
 *                        the maximum number of resurrections.
 *                      * Check_for_closing is now called from global.gendaemon
 *                        to give an immediate response when the last
 *                        treasure is taken.
 *
 * 4-Apr-00     djp     Version 1.01: bugfix release
 *                      Changes in this version:
 *                      * Changed resurrection code to close the safe in the
 *                        550-point game.
 *                      * Implemented the noresurrect verb (mainly for game
 *                        testing.)
 *                      * The property Me.ringlist (containing a list of
 *                        objects providing protection against the dwarves'
 *                        knives) is now built at the preinit stage.
 *                      * Suppressed the start-up of the pirate if the
 *                        chest has been purloined.
 *
 * 4-May-00     djp     Version 1.02: bugfix release
 *
 * 15-May-00    djp     Version 1.03: bugfix release
 *
 * 18-Sep-00    djp     Version 2.00: New version with 701-point game
 *                      Changes in this version
 *                      * Added verglob object for 701-point game
 *                      * Various minor changes for 701-point mode e.g.
 *                        set global.numactwords701 in preinit, set vnumber
 *                        in specialrestart.
 *
 * 8-Jun-01     djp     Version 2.05: bugfix release
 *                      Changes in this version
 *                      * Corrected the code which reverts to the 350-point
 *                        mode when the verglob object isn't found.
 *
 * 17-Aug-01    djp     Version 2.06: bugfix release with e-mail address update
 *                      Changes in this version
 *                      * (701-point mode) Corrected global.closingmess to 
 *                        cater for attempts to unlock entrances after
 *                        escaping the Cylindrical Room.
 *
 * 22-Aug-01    djp     Version 2.07: bugfix release
 *                      Changes in this version
 *                      * Streamlined the initial messages somewhat.
 * 
 * 22-Nov-01    djp     Version 2.08: bugfix release
 *                      Changes in this version
 *                      * Standardized quitpoints to -4 for all versions.
 *                      * Standardized sacklist to [sack,treasure_chest]
 *                        for the 551-point and 701-point games.
 *                      * Renamed oldglob as verglob.  (The old name was
 *                        a hangover from ad551 and isn't appropriate now.)
 *
 *
 * 13-Jan-03    djp     Version 2.09 rev C: bugfixes and code tidy-up
 *                      Changes in this version
 *                      * Changed Me.ldesc to do a diagnose in appropriate
 *                        situations.                       
 *                      * When cave is closing, Me.travelTo doesn't check the 
 *                        isoutside property of doors, only rooms.
 *                      * When trying to escape from the cave, don't
 *                        change global.bonustime after full closure.
 *                      * New DEFVER define added, allowing compilation with
 *                        a different default version. 
 *                      * Fixes to global.gendaemon.  The check for the
 *                        removal of a deposited object is now suppressed 
 *                        by a new flag, global.dont_rescind.  When an 
 *                        object has been duplicated, care is taken to ensure
 *                        that points aren't rescinded if at least one of
 *                        the equivalent objects has been correctly deposited.
 *                      * Added a new flag, global.extend_moveInto to turn
 *                        on extra moveInto code for the item class.  The
 *                        extensions are turned on at the end of init, and
 *                        are temporarily turned off during the 550/701-point
 *                        endgame (when items are removed from the surface
 *                        areas).
 *                      * Moved all the player character customization to the
 *                        basicMe class (see advmods.t)
 *                      * Modified darkTravel() so that magic words won't
 *                        cause the player to fall into a pit.
 *                      * Modified gendaemon to allow the player to be the
 *                        target location of a scoring object.
 *                      * Changed the handling of version-dependent loclists,
 *                        which are now ignored if set to nil or equal to the
 *                        loclist.
 *
 * 31-May-03    djp     Version 2.10: minor bugfixes and enhancements
 *
 * 12-Aug-03    bjs     Version 2.11: added 580-point version.
 *
 * 23-Aug-03    djp     Version 2:12 
 *                      * Modifications to the die() routine:
 *                        Reset noAutoOpen properties of external doors after
 *                        a Blob attack.
 *                        Check for an in-between state (the summoning of
 *                        the Blob must be unnotified if it has been
 *                        notified, but hasn't yet occurred).
 *                        Two new global variables, which are set during a
 *                        security alert, must now be reset.
 *
 * 23-Jul-04    djp     Version 3.00.  Addition of a new game mode (701+).
 *
 * 12-Apr-05    djp     Version 3.10: Bugfixes and enhancements
 *                      * Updated specialrestart to restore additional
 *                        game state in the game701+ mode. 
 *                      * Added CircularRoomLighting to global.noall
 * 7-May-05     djp     Version 3.10 Rev. A
 *                      * If the Wumpus saw that the Octagonal Room rocks
 *                        had been moved, he puts them back!
 * 16-May-05    djp     Version 3.10 Rev. B
 *                      * Fixed a bug preventing "Kata" from being used after
 *                        the Wumpi had removed their rings (phase=5).
 * 26-May-05    djp     Version 3.10 Rev. C
 *                      * Restore location of Blue1 and Keyswitch in
 *                        specialrestart.
 *                      * The specialrestart function now consistently uses
 *                        moveInto to restore location properties.
 *
 * 9-Aug-05     djp     Version 3.12: Enhancements and bugfixes
 *                      * Use the new Wumpi.nodeath() method to determine
 *                        whether the lamp life is to be frozen and deaths
 *                        (from dark travel) are to be prevented.  (Deaths
 *                        from dropping the vial or throwing the vial at
 *                        oneself are also suppressed). After a special restart
 *                        ('throne mishap') we now restore global.verbose and
 *                        relevant properties of access shaft grilles.
 *                      * Kludge to global.tadsversion (set to 2.5.9 if
 *                        version is 2.5.10 or later) so that the test for old
 *                        versions (global.tadsversion < 2.5.6) gives the
 *                        correct result.  ('New' versions can set 
 *                        global.travelActor, global.verbActor and
 *                        global.currentActor to parserGetObj(PO_ACTOR) within
 *                        the preparseCmd function.  'Old' versions have to
 *                        set these variables by other means, due to a bug in
 *                        preparseCmd.)
 * 20-Oct-07    djp     Version 3.20 Rev. C
 *                      * Corrected the coding for setting Blue1.location in
 *                        specialrestart - in some cases the blueberries were
 *                        not restored when the poster had been read.     
 *
 * 21-Nov-10    djp     Version 3.21
 *                      * In the 551 point game and its derivatives, allow
 *                        food to be referred to as 'watercress sandwiches'
 *                        (The extra vocabulary is added in the copy methods
 *                        of the relevant 'verglob' objects)
 *                      * Added code to die() to put the Wumpi to bed if 
 *                        the player dies during the chase, unless they
 *                        are asleep at Blue level.  (Note that the
 *                        game blocks most causes of death).   
 *
 */             

/*
 * This file includes the typical standard definitions.
 */

/* Additional functions to be called by setfuse etc */
makend: function;
check_for_endgame: function;
check_for_closing: function;

/*
 *  mainRestore: function
 *
 *  Restore a saved game.  Returns true on success, nil on failure.  If
 *  the game is restored successfully, we'll update the status line and
 *  show the full description of the current location.  The restore command
 *  uses this function; this can also be used by initRestore (which must be
 *  separately defined by the game, since it's not defined in this file).
 */
/*
 * Later versions of TADS call this instead of init when -r is used on
 * the command line
 */
initRestore: function(fname) // DJP
{
    // Run init without printing any text
    global.initsilent := true;
    init();
    // Tell the player what's going on
    "Restoring game from saved file "; say(fname); ". ";
    P();
    mainRestore(fname);
}

verge: function(...) {
    local text;
    if (argcount > 0) text := getarg(1);
    else text := systemInfo(__SYSINFO_VERSION);
    if (text >= '2.1.5') "<<text>> ge '2.1.5'\n";
    else "not <<text>> ge '2.1.5'\n";
}

verlt: function(...) {
    local text;
    if (argcount > 0) text := getarg(1);
    else text := systemInfo(__SYSINFO_VERSION);
    if (text <= '2.1.5') "<<text>> lt '2.1.5'\n";
    else "not <<text>> lt '2.1.5'\n";
}

/*
 * This is the program's entry point.
 */
init: function
{
    local copied,o,i,l,loc,p,q,tversion;

// In debug mode, call the preinit coding here (see comments in preinit)
#ifdef __DEBUG
    preinit_0();
#endif

    // store the interpreter version
    // for now, we use a horrible kludge to make 2.5.10 or later look like
    // 2.5.9
    tversion := systemInfo(__SYSINFO_VERSION);
    switch (tversion) {
        case '2.5.10':
        case '2.5.11':
        case '2.5.12':
        case '2.5.13':
        case '2.5.14':
        global.tadsversion := '2.5.9';
        break;
        default:
        global.tadsversion := tversion;
        break;
    }
    // adjust global variables for the game version (and do any
    // extra setup needed.)
    for (o:=firstobj(verglob); o; o:=nextobj(o, verglob)) {
        if(global.vnumber = o.vnumber) {
            o.copy;
            copied := true;
            break;
        }
    }

    if (not copied) {
        "Internal error - no verglob object found for version
        number <<global.vnumber>>. Reverting to 350-point version.";
        global.vnumber := 0;
        glob0.copy;
        copied := true;
    }

    // kludge for closingtime to avoid changing walkthroughs
    global.closingtime++;

    // save initial score for later use
    global.startscore := global.score;

    // update status line
    scoreStatus( global.score, global.turnsofar );

    // move player -> 1st location
    parserGetMe().location := At_End_Of_Road;

    // print initial text if appropriate
    if (not global.restarting and not global.initsilent) {
        clearscreen();
        "\b";

    // if(global.tadsversion < '2.5.6') " tadsversion less than 2.5.6\n";
    // else "tadsversion 2.5.6 or later\n";

        "Somewhere nearby is Colossal Cave, where others have found
        fortunes in treasure and gold, though it is rumored that some
        who enter are never seen again.  Magic is said to work in the
        cave.  I will be your eyes and hands.  Direct me by typing
        simple commands in natural English.   Commands of one or two
        words, like \"west\" or \"take ingot\", may still be used, but
        I now understand more complex sentences like 
        \"put the large keys in suitcase\" or \"attack giant with
        long sword\".\b";

        "(To scroll the screen after a 'more' prompt, press the space
        bar.)\b";

        "This TADS port provides several versions of the Colossal Cave
        adventure in one game file!  The alternatives are:\b";

        "* The original 350-point game by Crowther and Woods, ported
           by David Baggett.\b";

        "* The 550-point extended version by David Platt, ported by
           Bennett Standeven.\b";
           
        "* The 580-point extended version by Mike Goetz, ported by
           Bennett Standeven.\b";

        "* The 551-point version by David Long and an anonymous author,
           ported by David Picton.\b";

        "* A 701-point version by David Picton, combining the 550-point and  
           551-point versions.\b";

        "* A 701+ point version by David Picton, based on the 701-point
           version but with new extensions.  The maximum score is for you
           to discover! \b";
    }
    if (not global.initsilent) {
        "\bType \"commands\" for details of special verbs specific to
        Polyadv, including the commands to select the game version and
        display help information.\b ";
    }
    // Remove any objects and rooms which belong only to
    // this version of the game, and move any which have different
    // locations.
    //
    // Only objects with simple locations, and floatingdecoration objects,
    // are moved into nil.  In other cases the location method may
    // need to be hand-coded to ensure that the object can only be seen
    // in the appropriate game versions.

    l := length(global.removelist);
    for (i := 1; i <= l; i++) {
        o := global.removelist[i];
        p := proptype(o,&location);
        if(p = 2) {
            // "\nmoving ";o.sdesc;" into nil\n";
            o.moveInto(nil);
        }
        if(isclass(o,floatingdecoration)) o.loclist := [];

        // Objects and rooms which are not in this version are all flagged
        // as deleted, and game routines can check for this (including
        // the place and move routines for NPC's, which are never moved into
        // deleted rooms.)

        //"\nflagging room as deleted: ";o.sdesc;"\n";
        o.deleted := true;
    }
    // Objects in the movelist are assumed to have simple locations.
    l := length(global.movelist);
    for (i := 1; i <= l; i++) {
        o := global.movelist[i];
        // "\nmoving ";o.sdesc;" into ";global.changeloc[i].sdesc;"\n";
        o.moveInto(global.changeloc[i]);
        //no longer needed in init, due to global.extend_moveInto flag
        /* o.moved := nil; */
    }
    // Objects in the moveloclist have a version-dependent loclist.
    l := length(global.moveloclist);
    for (i := 1; i <= l; i++) {
        o := global.moveloclist[i];
        // "\nchanging loclist for ";o.sdesc;"\n";
        o.loclist :=(global.changeloclist[i]);
    }

    if(global.oldgame) {
        brass_lantern.turnsleft := 350;
    }
    else {
        brass_lantern.turnsleft := 650;
    }
    "\b";

    // Handle a special restart
    if(global.specialstart) {
        // special restart
        parserGetMe().location := In_Hall_Of_Mists;
        silent_incscore(global.farinpoints);
        if (global.novicemode) silent_incscore(global.novicepoints);
        if (global.nodwarves) silent_incscore(-5);
        parserGetMe().awardedpointsforgettingfarin := true;
        /* Give me the lamp and turn it on */
        brass_lantern.moveInto(parserGetMe());
        brass_lantern.ison := true;
        notify(brass_lantern, &wearbatteries, 0);
        /* move certain objects into the rainbow room */
        little_bird.moveInto(Rainbow_Room);
        little_bird.moved := nil;
        black_rod.moveInto(Rainbow_Room);
        black_rod.moved := nil;
        slippers.moveInto(Rainbow_Room);
        slippers.moved := nil;
    }

    //
    // Randomize after the first turn.  (This allows us to make a verb
    // that prevents randomization for regression testing.)
    //
    setfuse(makend, 1, nil);

    setdaemon(turncount, nil);              // start turn counter daemon
    // check_for_closing is now called from global.gendaemon
    setdaemon(check_for_endgame, nil);      // start endgame daemon

    //
    // Initialize non-player characters (dwarves, pirate) if we're
    // in debug mode.  Otherwise do it from preinit.
    //
    if (global.debug)
            initNPC();

    //
    // Check room connections for integrity, if not restarted.
    //
    if (not global.restarted and not global.specialstart)
    check_connections();

    // If global.floatcontents is set, add version-dependent
    // floatingdecoration loclists to contents lists.
    if(global.floatcontents) {
            o := firstobj(floatingdecoration);
            while (o <> nil) {
                if(not o.versionloc or o.loclist = nil) {
                    o := nextobj(o,floatingdecoration);
                    continue;
                }
                l := length(o.loclist);
                for (i := 1; i <= l; i++) {
                    loc := o.loclist[i];
                    loc.contents += o;
                }
                o := nextobj(o,floatingdecoration);
            }
    }
    //

    // show player where he is
    if (global.restarting or (not global.initsilent)) {
        version.sdesc;      // display the game's name and version number
    }
    if (global.specialstart or global.restarting or
    (not global.initsilent)) {
        parserGetMe().location.lookAround(true);
    }
    
    // turn on moveInto extensions
    global.extend_moveInto := true;

}

makend: function(parm)
{
    local randcom,i,j,done,ok;
    if (global.nondeterministic) {
        randomize();
        // DJP - record the fact that we've randomized.  This
        // enables us to fix a bug with the restore command (and
        // -r command option).
        global.randomized := true;
    }
    else {
        "\b*** This will be a deterministic game. ***";
    }

/* General purpose daemon */

    notify(global, &gendaemon, 0);

/* Player's health */

    notify(parserGetMe(), &healthdaemon, 0);
}

dwarfstart: function(parm)
{
    // Place all the NPC's.  Now initiated from Me.travelTo.
    Dwarves.place;
    notify(Dwarves, &move, 0);

    if (not treasure_chest.spotted) {
        Pirates.place;
        notify(Pirates, &move, 0);
    }

    global.NPCstarted := true;
}

// special restart mode for 551- and 701-point games
// Note that this is not a normal restart.  
specialrestart: function(parm) {
    global.specialstart := true;
    global.initsilent := true; // suppress introductory text
    global.vnumber := parm[1];  // now set from parm list
    global.novicemode := parm[2];
    global.turnsofar := parm[3];
    global.randomized := parm[4];
    global.nodwarves := parm[5];
    global.verbose := parm[6];
    // Restore variables to do with the safe.  If the player knows
    // it's there, flag it as being hidden behind the poster.
    if(parm[7] != Safe.location) Safe.moveInto(parm[7]);
    if(Safe.location) Safe.hidden := true;
    SafeCombination.isseen := parm[8];
    Dial.combination_set := parm[9];
    Safe.hasopened := parm[10];
    Dial.combination := parm[11];
    // Restore variables to do with the player's memory.  For example,
    // if the throne room has been seen, the player won't crawl around
    // in little passages when trying to find it.
    Throne_Room.isseen := parm[12];
    River_Styx_E.isseen := parm[13];
    In_Swiss_Cheese_Room.foundorient := parm[14];
    River_Styx_E.isseen := parm[15];
    Pantry.isseen := parm[16];
    Pantry.islookedin := parm[17];
    // If these variables are set to true, the player already knows the
    // correct pronunciation of the relevant Elvish magic words
    Knoll.seenit := parm[18];            /* PHUCE */
    Rise_Over_Bay.seenit := parm[19];    /* SAINT-MICHEL */
    Outer_Courtyard.seenit := parm[20];  /* PHLEECE */
    if(parm[21] != Blue1.location) Blue1.moveInto(parm[21]);
    if(global.game701p) {
        // Do we know that the east rocks can be moved?
        Rockfalls.movable := parm[22];
        // Have we seen the Green-level Large Circular Room?
        Green_Large_Circular_Room.isseen := parm[23];
        // Have we completed the Wumpi chase scene?
        Green_Tight_Crack_2.isseen := parm[24];
        // Have we seen the Orange Level Office in the crystal ball?
        Orange_Office.seenit := parm[25];
        // Have we seen the hidden door at the top of the blocks?
        // (N.B. this has an important effect on gameplay)
        Top_Of_Blocks.seendoor := parm[26];
        if (parm[27] != Access_Shaft_Door.location) 
            Access_Shaft_Door.moveInto(parm[27]);
        // Access shafts (affecting messages about despondency)
        Red_Access_East.isseen := parm[28];
        Blue_Access_East.isseen := parm[29];
        // Do we know how to open the desk?
        UnderDesk.isseen := parm[30];            
        // Have we seen the Control Room in the crystal ball?
        Control_Room.seenit := parm[31];
        // Have we read the manual (re topaz-through-hole puzzle)
        manual.isread := parm[32];
        // Have we misunderstood the scrawled note?
        manual.misunderstood := parm[33];
        // Have we seen 'span' in the relevant sense?
        DisplayBoards.spanseen := parm[34];
        // Have we found the location of the tower staircase door?
        // (if so, we stop at this location when descending or ascending
        // the staircase at Blue level.)
        Green_Zarkalon_Doorsite.isseen := parm[35];
        // Keyswitch behind bookcase in Wumpus living room
        if (parm[36] != Keyswitch.location) Keyswitch.moveInto(parm[36]);
        Red_Access_Shaft_Grille.isopened := parm[37];
        Blue_Access_Shaft_Grille.isfoundlocked := parm[38];
        Red_Access_Shaft_Grille.isfoundlocked := parm[39];
    }
}

/*
 * The following function is run at compile time to build lists and
 * do other compute-intensive things ahead of time.  (But note: if you
 * compile the program for debugging, preinit runs at run time and is 
 * re-run when you restart.)
 */
preinit: function
{
    // If the game is compiled in debug mode, defer the execution of preinit
    // coding to the 'init' stage.  This is to allow the code to examine the
    // value of global.vnumber (for the purposes of an efficiency saving).  
    // In a restarted game the version information is set in initRestart, 
    // which runs after preinit and before init.
#ifdef __DEBUG
    return;
#else
    preinit_0();
#endif
}
preinit_0: function
{

    local i,j,o,l,loc,vnumsav,gameprop,locprop,loclprop;
    //
    // Construct list of light sources
    //
    global.lamplist := [];
    o := firstobj(lightsource);
    while (o <> nil) {
            global.lamplist := global.lamplist + o;
            o := nextobj(o, lightsource);
    }
    initSearch();

    global.numactwords := 0;
    for(o:=firstobj(MagicWord); o; o:=nextobj(o, MagicWord)) {
        if(o.omegapsical_order) 
            if(o.omegapsical_order > 0)global.numactwords++;
    }
    global.numactwords580 := 0;
    for(o:=firstobj(MagicWord); o; o:=nextobj(o, MagicWord)) {
        if(o.omegaps580_order) 
            if(o.omegaps580_order > 0)global.numactwords580++;
    }
    global.numactwords701 := 0;
    for(o:=firstobj(MagicWord); o; o:=nextobj(o, MagicWord)) {
        if(o.omegaps701_order) 
            if(o.omegaps701_order > 0)global.numactwords701++;
    }
    global.numactwords701p := 0;
    for(o:=firstobj(MagicWord); o; o:=nextobj(o, MagicWord)) {
        if(o.omegaps701p_order) 
            if(o.omegaps701p_order > 0)global.numactwords701p++;
    }
    // Find number of active magic words for cylindrical puzzle.

    /* Set an 'allversions' property on all
       rooms/objects for which no gamevvv properties are true
     */
    for (i:=firstobj(thing); i; i:=nextobj(i, thing)) {
        i.allversions := true;
        for (j := 0; j <= global.vmax; j++) {
            gameprop := global.gameflaglist[j+1];
            if(i.(gameprop) = true) i.allversions := nil;
        }
    }

    /* Loop over all defined versions */
    vnumsav := global.vnumber;
    for (i:=firstobj(verglob); i; i:=nextobj(i, verglob)) {
        local condit;
// In debug mode, the initialization is done at runtime, so we save time by
// generating data only for the version of interest.
#ifdef __DEBUG
        if (i.vnumber != vnumsav) continue;
#endif
        /* Number of coin sets */
        i.coinsets := 0;
        /* Temporarily set global.vnumber to the appropriate value so
        we see the correct values of object scores */
        global.vnumber := i.vnumber;
        //
        // Count treasures in each version to get initial number
        // of treasures left to find.
        //
        i.treasures := 0;
        /* version-specific property */
        gameprop := global.gameflaglist[i.vnumber + 1];
        locprop := global.locproplist[i.vnumber + 1];
        loclprop := global.loclproplist[i.vnumber + 1];
        /* count the treasures */
        o := firstobj(CCR_treasure_item);

        while (o <> nil) {
            condit := (o.allversions or o.(gameprop));
            if(condit) {
                if (not o.bonustreasure) {
                    i.treasurelist := i.treasurelist + o;
                    i.treasures := i.treasures + 1;
                } 
                else
                    i.bonustreasures += o;
            }
            o := nextobj(o, CCR_treasure_item);
        }
        /* count coins (whether valuable or not) and all non-treasure 
           objects which score points */
        o := firstobj(item);
        while (o <> nil) {
            condit := (o.allversions or o.(gameprop));
            if(condit) {
                if(isclass(o,coinitem)) i.coinsets++;
            }
            if( ((o.takepoints <> nil) or (o.depositpoints <> nil))
            and (not isclass(o,CCR_treasure_item)) and condit) {
                if(not o.bonustreasure) {
                    i.pointobjlist := i.pointobjlist + o;
                    i.pointobjs := i.pointobjs + 1;
                }
                else
                    i.bonuspointobjs += o;
            }
            o := nextobj(o, item);
        }

        // List of treasures still to find (for debugging)
        i.treasures_to_find := i.treasurelist;
        i.allpointlist := i.treasurelist + i.pointobjlist;
        i.allpoints := i.treasures + i.pointobjs;
        // save the number of treasures (which is reduced by one each
        // time a treasure is taken)
        i.origtreasures := i.treasures;

        /* list objects to be removed from each game version. */
        i.removelist := [];
        o := firstobj(thing);
        while (o <> nil) {
            condit := (o.allversions or o.(gameprop));
            if(not condit) {
                i.removelist += o;
            }
            o := nextobj(o,thing);
        }

        /* list objects with different locations in each version. */
        i.movelist := [];
        o := firstobj(thing);
        while (o <> nil) {
            condit := defined(o,locprop) and (o.(locprop) <> o.location);
            if(condit) {
                i.movelist += o;
                i.changeloc += o.(locprop);
            }
            o := nextobj(o,thing);
        }

        /* list floatingdecoration objects with a version-dependent
           loclist */
        o := firstobj(floatingdecoration);
        while (o <> nil) {
            condit := ((o.(loclprop) <> nil) and (o.(loclprop) <> o.loclist));
            if(condit) {
                i.moveloclist += o;
                // Note the syntax to add a list as a single element.
                i.changeloclist += [o.(loclprop)];
                // Flag to indicate that the initial loclist is
                // version-dependent (contents lists will be filled in
                // at the init stage)
                o.versionloc := true;
            }
            // objects which exist only in some versions are also
            // version-dependent
            if (not o.allversions) o.versionloc := true;
            o := nextobj(o,floatingdecoration);
        }
    }
    /* Reset global.vnumber to its default value */
    global.vnumber := vnumsav;

    // If global.floatcontents is set, add version-independent
    // floatingdecoration loclists to contents lists.
    // Similarly add makecontents objects to contents lists.
    if(global.floatcontents) {
            o := firstobj(floatingdecoration);
            while (o <> nil) {
                if(o.versionloc or o.loclist = nil) {
                    o := nextobj(o,floatingdecoration);
                    continue;
                }
                l := length(o.loclist);
                for (i := 1; i <= l; i++) {
                    loc := o.loclist[i];
                    loc.contents += o;
                }
                o := nextobj(o,floatingdecoration);
            }
            o := firstobj(makecontents);
            while (o <> nil) {
                loc := firstobj(room);
                while (loc <> nil) {
                    parserGetMe().location := loc;
                    if(o.location = loc)loc.contents += o;
                    loc := nextobj(loc,room);
                }
                o := nextobj(o,makecontents);
            }
    }

    //
    // Initialize non-player characters (dwarves, pirate) from
    // here if we're not in debug mode.  Otherwise do it in init.
    //
    if (not global.debug)
            initNPC();
    // Initialize list of dwarf-protection items
    // (named Me.ringlist for historical reasons)
    for (o := firstobj(item); o; o := nextobj(o, item)) {
            if (datatype(o.protection) = 1)
                    basicMe.ringlist += o;
    }
}

die: function // DJP - changed to run end-of-turn daemons and abort.
              // NOTE TO PROGRAMMERS: This means that any code which follows
              // die() will not be executed.
              // DJP - also changed to allow the number of resurrections to
              // vary with the game version.
{
    local   resurrect;

    P(); I();

    // DMB: "reincarnate" in the text below should really
    // be "resurrect," since presumably you're coming back
    // as the same person you were before.  I have left the
    // original text as is, however, for the sake of purity.
    resurrect := nil;       // assume no resurrection

    if (global.closed) {
        "It looks as though you're dead.  Well, seeing as how
        it's so close to closing time anyway, I think we'll
        just call it a day.\n";

        call_it_a_day();
    }
    else if (global.no_allow_resurrect) {
        incscore(global.deathpoints,'for getting killed');
        call_it_a_day();
        return;
    }
    // we always allow one resurrection ...
    else if (global.deaths = 0) {
        "Oh dear, you seem to have gotten yourself killed.  I
        might be able to help you out, but I've never really
        done this before.  Do you want me to try to
        reincarnate you?\b>";

        if (yorn()) {
            "\bAll right.  But don't blame me if something
            goes wr...... \b \ \ \ \ \ \ \ \ \ \ \ \ \ \
            \ \ \ \ \ \ --- POOF!! --- \bYou are engulfed
            in a cloud of orange smoke.  Coughing and
            gasping, you emerge from the smoke and
            find that you're....";

            resurrect := true;
        }
        else
            "\bVery well.";
    }
    // extra message issued in 550-point game ...
    else if (global.deaths = 1 and global.maxresurrect >= 3) {
        "Tsk, tsk - you did it again!  Remember - you're only human,
        and you don't have as many lives as a cat!  (at least, I don't
        think so...) That's twice you've ended up dead - want to try
        for three?\b>";

        if (yorn()) {
            "\bOkay, now where did I put my orange
            smoke?....  >POOF!<\bEverything disappears in
            a dense cloud of orange smoke.";

            resurrect := true;
        }
        else
            "\bProbably a wise choice.";
    }

    // message for subsequent resurrections
    else if (global.deaths < global.maxresurrect) {
        "You clumsy oaf, you've done it again!  I don't know
        how long I can keep this up.  Do you want me to try
        reincarnating you again?\b>";

        if (yorn()) {
            "\bOkay, now where did I put my orange
            smoke?....  >POOF!<\bEverything disappears in
            a dense cloud of orange smoke.";

            resurrect := true;
        }
        else
            "\bProbably a wise choice.";
    }
    // resurrections used up,
    else {
        "Now you've really done it!  I'm out of orange smoke!
        You don't expect me to do a decent reincarnation
        without any orange smoke, do you?\b>";

        if (yorn()) {
            "\bOkay, if you're so smart, do it yourself!
            I'm leaving!";
        }
        else
            "\bI thought not!";
    }

    global.deaths := global.deaths + 1;
    incscore(global.deathpoints,'for getting killed');

    if (resurrect) { 
        local i,o,l,pcount := 0,savecont := parserGetMe().contents;
        local doorlist := [Grate Small_door_1 Small_door_2 Iron_door_1 
        Iron_door_2]; 
        //
        // Resurrection:
        //
        // 1) Drop all player's items where he was killed, except for the
        //    pendants in the 701+ point game.  The broken pendant is excluded
        //    but the player gets to keep all other pendants, even if they
        //    have been de-activated.  (They can be re-activated using any
        //    of the pendant-charging machines).
        // 2) Reverse the effect of the 'phuce' spell on objects.
        // 3) Move lamp outside of building and turn it off.
        // 4) Banish all chasers (done here because player might
        //    be killed for another reason while being chased).
        //    Replace the rocks if the Wumpus was chasing.
        // 5) Reset health to 100.
        // 6) Close the safe (550-point game)
        // 7) Kill off the control room fume daemon
        // 8) Return the 701+ point elevator back to the lower level and 
        //    close the doors
        // 9) Move the player into the building.
        // 10) If there was a security alert, unlock the 580-point elevator
        //    and reset the noAutoOpen property of doors.
        // 11) Run all daemons.
        // 12) Increment the turn counter and disable the YES command
        // 13) Issue an abort to terminate any command processing.
        // 14) Move the Wumpi if the phase is >1
        l := length(savecont);
        for (i := 1; i <= l; i++) {
            o := savecont[i];
            if (not o.isworn or not isclass(o,pendantItem) or 
            (o = broken_pendant))
                o.moveInto(parserGetMe().location);
            else
                pcount++;
        }
        room_move(Ledge_By_Door,Top_Of_Steps);
        room_move(Underground_Sea,Grotto_West);
        room_move(Dense_Jungle, Knoll);
        room_move(Vast_Chamber, Throne_Room_East);
        if(not brass_lantern.destroyed) {
            brass_lantern.turnoff;
            brass_lantern.moveInto(At_End_Of_Road);
        }
        if(Wumpus.ischasing and Wumpus.upset and 
        not Trans_Room_Door.isunlocked) {
            Wumpus.upset := nil;
            Trans_Room_Door.moveInto(nil);
            Octagonal_Room.east := Noexit;
            Octagonal_Room.NPCexits -= Trans_Room_Door;
            Rockfalls.ismoved := nil;
        }            
        for(o:=firstobj(Chaser); o<>nil; o:=nextobj(o, Chaser))
            // DJP: banish chasers only if they're actually chasing.
            if (o.ischasing) o.banish;
        // Check for an in-between state, when the blob has been notified
        // for summoning but this hasn't yet taken place.
        if(global.triggered_alert and not global.security_alert)
            unnotify(Blob,&summon);
        "\b";
        parserGetMe().health := 100;
        // close the walk-in safe
        In_Safe.isopen := nil;
        In_Safe.out := nil;
        if(Controls.fumeturns) {
             Controls.fumeturns := 0;
             unnotify(Controls, &fumedaemon);
        }
        In_Elevator.outdest := nil;
        parserGetMe().travelTo(Inside_Building);
        parserGetMe().prevloc := nil; // DJP - destroy info about previous
                           // location
        // Reset external door and 580-point elevator properties after a 
        // security alert.  Note that the doors remain locked, but it is
        // possible to unlock them.
        global.triggered_alert := nil;
        if(global.security_alert) {
            global.security_alert := nil;
            l := length(doorlist);
            for (i := 1; i <= l; i++) {
                o := doorlist[i];
                o.noAutoOpen := o.wasAutoOpen;
            }
        }
        // Sort out the state of the Wumpi and the door.
        if ((Wumpi.phase > 1) and (Wumpi.phase < 5)) {
            Wumpi.phase := 5;
        // Three objects must be moved to reset the position of the Wumpi:
            Wumpi.moveInto(Green_Large_Circular_Room); // main party
            Wumpi2.moveInto(nil);                      // advance party
            Wumpi_remnant.moveInto(nil);               // left behind
        }
        if ((Wumpi.phase >= 0) and (Wumpi.phase < 5)) {
            NoisyDoor.setIsopen(true);
            if(EndDoor.passcount = 1)
                doorwedge.moveInto(Green_Large_Circular_Room);
        }
        rundaemons();      // run all daemons
        yesVerb.ignore := global.turnsofar;  // Ignore any
                                             // pending questions.
        scoreStatus( global.score, global.turnsofar );
        if(pcount >= 1) {
            P(); I();
            "For a moment, you feel as if something is missing.  Then the
            air starts to shimmer, and you feel a strange sensation on the
            back of your neck.  You glance down, and notice that ";
            if (pcount = 1)
                "the pendant has reappeared! ";
            else  
                "the pendants have reappeared! ";
        }
        abort;             // abort any further processing.
    }
    else call_it_a_day();
}

/*
 * The player loses without any chance of reincarnation (for example,
 * if his batteries run out.)
 */
call_it_a_day: function
{
    scoreRank();
    exitprompt();
}

/*
 * The player has finished the game on a positive note.
 */
win: function
{
    scoreRank();
    exitprompt();
}

exitprompt: function // DJP - this is based on the standard std.t die()
                     // function but without the scoreRank() call,
                     // and with extra commands to select the game version.
{
    "\bYou may restore a saved game, start over, quit, or undo
    the current command.\n";
    while (true)
    {
        local resp;

        "\nPlease enter RESTORE, RESTART, GAME350, GAME550, etc,
        QUIT, or UNDO: \n>";
        resp := upper(input());
        if (resp = 'RESTORE')
        {
            resp := askfile( 'File to restore' );
            if (resp = nil)
                "Restore failed. ";
            else if (restore(resp))
                "Restore failed. ";
            else
            {
                parserGetMe().location.lookAround(true);
                scoreStatus(global.score, global.turnsofar);
                abort;
            }
        }
        else if (resp = 'RESTART') // Updated to preserve current vnumber
        {
                if(global.vnumber = 11)
                    "\nRestarting in 701+ point mode.\n";
                else
                    "\nRestarting in <<global.maxscore>>-point mode.\n";

                global.initRestartParam := [global.vnumber global.randomized
                    nil];
                restart(initRestart, global.initRestartParam);
                break;
        }
        else if (resp = 'GAME350') // Added
        {
                "\nRestarting in 350-point mode.\n";
                global.initRestartParam := [0 global.randomized
                    nil];
                restart(initRestart, global.initRestartParam);
                break;
        }
        else if (resp = 'GAME551') // Added
        {
                "\nRestarting in 551-point mode.\n";
                global.initRestartParam := [1 global.randomized
                    nil];
                restart(initRestart, global.initRestartParam);
                break;
        }
        else if (resp = 'GAME550') // Added
        {
                "\nRestarting in 550-point mode.\n";
                global.initRestartParam := [2 global.randomized
                    nil];
                restart(initRestart, global.initRestartParam);
                break;
        }
        else if (resp = 'GAME580') // Added
        {
                "\nRestarting in 580-point mode.\n";
                global.initRestartParam := [7 global.randomized
                    nil];
                restart(initRestart, global.initRestartParam);
                break;
        }
        else if (resp = 'GAME701') // Added
        {
                "\nRestarting in 701-point mode.\n";
                global.initRestartParam := [15 global.randomized
                    nil];
                restart(initRestart, global.initRestartParam);
                break;
        }
        else if (resp = 'GAME701+') // Added
        {
                "\nRestarting in 701+ point mode.\n";
                global.initRestartParam := [11 global.randomized
                    nil];
                restart(initRestart, global.initRestartParam);
                break;
        }
        else if (resp = 'QUIT')
        {
            terminate();
            quit();
            abort;
        }
        else if (resp = 'UNDO')
        {
            if (undo())
            {
                "(Undoing one command)\b";
                parserGetMe().location.lookAround(true);
                scoreStatus(global.score, global.turnsofar);
                abort;
            }
            else
                "Sorry, no undo information is available. ";
        }
    }
}

/*
 * Show score and ranking.  DJP - the messages are the same for the
 * game versions but the score boundaries are different.
 */
scoreRank: function
{
    local   pn;
    local  amateur,novice,experienced,seasoned,jmaster,masterc,masterb;
    local  mastera, tribute;


    // DJP - note that each value represents the lower bound of the
    // next score category.  In the 350-point game, this means that any
    // score up to 34 is in the amateur category; 35 and above is novice,
    // etc.
    //
    // BJS - note that a "tribute" score will be needed for the 430-point
    // version.

    switch(global.vnumber) {
    case 0:
           amateur := 35;
           novice := 100;
           experienced := 130;
           seasoned := 200;
           jmaster := 250;
           masterc := 300;
           masterb := 330;
           mastera := 350;
           tribute := 350;
           break;
    case 1:
           amateur := 72;
           novice := 130;
           experienced := 200;
           seasoned := 250;
           jmaster := 350;
           masterc := 450;
           masterb := 500;
           mastera := 551;
           tribute := 551;
           break;
    case 2:
           amateur := 20;
           novice := 130;
           experienced := 240;
           seasoned := 350;
           jmaster := 470;
           masterc := 510;
           masterb := 530;
           mastera := 550;
           tribute := 550;
           break;
    case 7: 
           amateur := 20;
           novice := 135;
           experienced := 250;
           seasoned := 365;
           jmaster := 485;
           masterc := 525;
           masterb := 545;
           mastera := 565;
           tribute := 580;
           break;
    case 15:
    case 11:
           amateur := 20;
           novice := 130;
           experienced := 350;
           seasoned := 475;
           jmaster := 601;
           masterc := 661;
           masterb := 681;
           mastera := 701 + global.extras;
           tribute := 701 + global.extras;
           break;
    }

    "You have scored "; say(global.score); " out of a possible ";
    say(global.maxscore); ", using "; say(global.turnsofar);
    if (global.turnsofar = 1)
        " turn.";
    else
        " turns.";

    "\n";

    if (global.score < amateur) {
        //
        // DMB: This originally said, "Better luck next time,"
        // but this isn't really appropriate anymore since
        // we now give the player his ranking when he types
        // \"score.\"  (In the original, you only got your
        // rank when the game was over.)
        //
        "You are obviously a rank amateur.";
        pn := amateur - global.score;
    }
    else if (global.score < novice) {
        "Your score qualifies you as a Novice Class adventurer.";
        pn := novice - global.score;
    }
    else if (global.score < experienced) {
        "You have achieved the rating: \"Experienced Adventurer\".";
        pn := experienced - global.score;
    }
    else if (global.score < seasoned) {
        "You may now consider yourself a \"Seasoned Adventurer\".";
        pn := seasoned - global.score;
    }
    else if (global.score < jmaster) {
        "You have reached \"Junior Master\" status.";
        pn := jmaster - global.score;
    }
    else if (global.score < masterc) {
        "Your score puts you in Master Adventurer Class C.";
        pn := masterc - global.score;
    }
    else if (global.score < masterb) {
        "Your score puts you in Master Adventurer Class B.";
        pn := masterb - global.score;
    }
    else if (global.score < mastera) {
        "Your score puts you in Master Adventurer Class A.";
        pn := mastera - global.score;
    }
    else {
        "All of Adventuredom gives tribute to you, Adventurer
        Grandmaster!";

        pn := 0;
    }

    "\n";

    if (pn > 0) {
        "To achieve the next higher rating, you need ";
        say(pn); " more ";

        if (pn = 1)
            "point.";
        else
            "points.";
    }
    else
        "To achieve the next higher rating would be a neat trick!";

    "\n";
}

/*
 * Print a nice message when we exit.
 */
terminate: function
{
    "\b";
    "Come back and visit the newly remodelled and extended Colossal Cave
     soon!";
}

/*
 * What we say when the user enters a blank line.
 */
pardon: function
{
    "What?";
}

/*
 *   The numObj object is used to convey a number to the game whenever
 *   the player uses a number in his command.  For example, "turn dial
 *   to 621" results in an indirect object of numObj, with its "value"
 *   property set to 621.
 */
numObj: basicNumObj  // use default definition from adv.t.  Methods are
                     // added in various places by modify commands.
;

/*
 *   strObj works like numObj, but for strings.  So, a player command of
 *     type "hello" on the keyboard
 *   will result in a direct object of strObj, with its "value" property
 *   set to the string 'hello'.
 *
 *   Note that, because a string direct object is used in the save, restore,
 *   and script commands, this object must handle those commands.
 */
strObj: basicStrObj;     // use default definition from adv.t

/*
 *   The "global" object is the dumping ground for any data items that
 *   don't fit very well into any other objects.  The properties of this
 *   object that are particularly important to the objects and functions
 *   are defined here; if you replace this object, but keep other parts
 *   of this file, be sure to include the properties defined here.
 */
global: object
#ifndef DEFVER
#define DEFVER 0
#endif  
     vnumber = DEFVER
/* Version 0 is the 350-point version, Version 1 is the 551-point version,
   Version 2 is the 550-point version, Version 7 is the 580-point version,
   Version 15 is the combined game.
   We define some properties derived from vnumber:

   newgame - 551-point game and derivatives
   oldgame - Games without 551-point extensions (350, 550-points)
   game550 - 550-point game and derivatives (e.g. 701-point game)
   game580 - 580-point game (and derivatives)
   game701 - 701-point game (combining the 550 and 551-point versions)
   game701p - 701+ point game (extended version of 701-point)
 */


     newgame = { if(find([1, 11, 15], vnumber)) return true; 
         else return nil;}
     oldgame = { if(find([0, 2, 7], vnumber)) return true; 
         else return nil; }
     game550 = { if(find([2, 7, 11, 15], vnumber)) return true; 
         else return nil; }
     game580 = { return (vnumber = 7); }
     game701 = { if(find([11, 15], vnumber)) return true;
         else return nil; }
     game701p = { return (vnumber = 11); }

    /* Maximum version number */
    /* (Note that not all versions need to actually exist - preinit
       detects actual versions by looking for objects with the
       verglob class)
    */
    vmax = 15
    /* List of valid gamevvv properties for vnumber=0,1,2 etc */
    gameflaglist = [&game350, &game551, &game550,
    &dum, &dum, &dum, &dum, &game580,
    &dum, &dum, &dum, &game701p, 
    &dum, &dum, &dum, &game701]

    /* List of version-specific location properties */
    locproplist = [&location350, &location551, &location550,
    &dum, &dum, &dum, &dum, &location580, 
    &dum, &dum, &dum, &location701p, 
    &dum, &dum, &dum, &location701]

    /* List of version-specific loclist properties */
    loclproplist = [&loclist350, &loclist551, &loclist550,
    &dum, &dum, &dum, &dum, &loclist580, 
    &dum, &dum, &dum, &loclist701p, 
    &dum, &dum, &dum, &loclist701]

    //
    // Scoring values
    // Points for treasures are kept in the treasures themselves.
    //
    specialstart = nil      // special restart option

    closure = nil       // Has the endgame timer started yet?
                // preinit

    floatcontents = true    // Add floatingdecoration objects to
                // contents lists

    floatingList = []       // filled in by initSearch (from preinit)
                            // list of floating items

    herefloatList = []      // filled in by initSearch (from preinit)
                            // list of floating items which have heredesc
                            // properties

    //
    // NPC stuff
    //
    NPCwaitmoves = 10    // Number of times the player can enter
                         // a 'far in' room before the possibility
                         // of starting the dwarves and pirate will be
                         // considered.

    NPCprob = 10         // Once NPCwaitmoves is exhausted, the dwarves
                         // and pirate may start up each time the player
                         // enters a 'far in' room.  The percentage
                         // probability is NPCprob.

    dwarves = 5          // number of dwarves wandering about the cave
                         // Was 5 in original.

    pirates = 1          // number of pirates lurking in the cave
                         // (Was 1 in original.)

    dspotfromadj = 10    // percentage chance that a dwarf will
                         // spot the player from an adjacent location
                         // and enter the player's room, instead of
                         // taking a random exit.

    dtenacity = 96       // percentage chance that a dwarf will
                         // follow the player once he's entered the
                         // player's room.  (Was 100 in original.  It might
                         // be desirable to change it to 100 in the
                         // 430-point game.)

    pspotfromadj = 50    // percentage chance that a pirate will
                         // spot the player from an adjacent location.

    //
    // Where to start dwarves and pirate(s) out.  Dwarves are placed
    // in the locations given by dwarfloc; if no locations remain when
    // we're placing a dwarf, we just put him in a random room selected
    // from NPCrooms.
    //
    // Note that the player gets the axe from the dwarf, so it's
    // fairly important to put at least one dwarf early on (but only
    // in a room that's not off limits to NPC's!)
    //
    // Ditto for pirate(s).
    //
    dwarfloc = [
        In_Hall_Of_Mists
    ]
    pirateloc = []

    dwarfattack = 75    // percentage chance that a dwarf will
                        // throw a knife at the player if he's
                        // in the same location.  (Was 100 in
                        // the original.)

    dwarfhit = 66       // percentage chance the player will
                        // hit a dwarf with his axe
                        // (Was 1 in 3 chance in the original.)

    luckyhit = 90       // percentage chance the player will hit
                        // a dwarf when he carries a lucky 4-leafed
                        // clover.

/* DJP.  Dwarf accuracy was reduced to 5, but it's been increased slightly
 * now that dwarves don't attack after you take the axe.  Something still
 * needs to be done to stop dwarves from ganging up on the player more than
 * they did in the original.
 */
    dwarfaccuracy = 7   // percentage chance that a dwarf will
                        // hit the player with a thrown knife
                        // (Was 9.5 in the original.)

    //
    // Output formatting
    //
    verbose = nil       // we are currently in TERSE mode
    indent = true       // indent paragraph beginnings
    doublespace = true  // double space the text

    //
    // Other stuff
    //
    deaths = 0          // number of times the player has died
    closingtime = 30    // close this many turns after player takes
                        // the last treasure.  (was 30 in original)
    bonustime = 30      // start endgame this many turns after
                        // the cave closes.  In the original this
                        // was 50.  In CCR it was decreased to
                        // 20, because the timer started when the
                        // last treasure was deposited in the building.
                        // In the present implementation we start
                        // the timer once the last treasure has
                        // been taken, so bonustime has been bumped
                        // back up a little.
    panictime = 15      // set timer to this many turns when player
                        // panics at closing time

    closed = nil        // is the cave closed yet?

    // message when trying to enter or exit the cave at closing time.
    closingmess = {
        local actor := getActor(&verbActor);
        "A mysterious recorded voice groans into life
        and announces, \"This ";
        if(toplocation(actor).isoutside) "entrance"; else "exit";
        " is closed.  ";

        if(global.game550) "Please report to the treasure room
        via the alternate entrance to claim your
        treasure.\"";
        else "Please leave via main office.\"";
    }


    turnsofar = 0       // no turns have transpired so far
    lamplist = []       // list of all known light providers
                        // in the game
    NPCrooms = []       // list of rooms NPC's (dwarves, pirate)
                        // can go to

    //
    // Special wizard stuff.  Don't mess with it!
    //
    nondeterministic = true // default to nondeterministic game
    debug = nil         // debugging messages on/off
    scoreinc = 0        // score increment for automatic end-of-turn
                        // score adjustment.

    // game version selection
    removelist = []     // list of objects to be removed
    movelist = []       // list of objects to be moved to different
                // locations
    changeloc = []      // list of modified locations
    moveloclist = []    // floatingdecoration objects to be given
                        // a different loclist
    changeloclist = []  // list of modified loclists

    noall = [Hands Heels Air Footsteps Walls Ceiling theFloor TheRoom
         LeftCrack RightCrack MiddleCrack Chapter1 Chapter2 Chapter3 Chapter4
         Foreboding available_keys CircularRoomLighting]
                        // objects which will be ignored by ALL. These 
                        // include objects which are always present (Hands),
                        // objects which form part of another object,
                        // and dummy objects which exist only to print messages
                        // in room descriptions.
    checklist = []      // treasures etc. whose locations should be
                        // checked.

    travelActor = Me    // actor doing the travelling, for use by travel
                        // methods.

    verbActor = nil     // actor executing the current verb.  This is set
                        // early in the parsing process, but is unset once
                        // the execution of the verb has finished.  

    currentActor = nil  // the current actor for the purpose of evaluating
                        // the location properties of multi-location objects

    extras = 0          // extra points for the extended 701-point game.

    wrongtreasloc = 0   // Number of incorrectly-deposited treasures (excluding
                        // discharged pendants held by the player).  Computed
                        // in the Cylindrical Room endgame only.

    trolltolls = 0      // Number of treasures paid to the troll and still in
                        // his treasure room.  Computed in the Cylindrical
                        // Room endgame only.
    
    vendingtreasures = 0 // Number of treasures used to buy batteries.

    spinrodlist = [black_rod, gray_rod, red_rod, green_rod, multicolored_rod]
                        // list of rods with spinning 
                        // capability (game701+ mode)
    gendaemon = {
        local i,t,l,cl,c,tloc,cloc,cloc2,ccond;
        local addlist := [];
        local Medisqual;
        At_Y2.hollowvoice;    // plugh message
        if(global.game550)     Fake_Y2.hollowvoice;  // ditto
        if(global.newgame)     Knoll.elfcurse;       // phuce message

        // do the expensive score checking only when the global.checklist
        // has at least one item.   This happens automatically whenever
        // a non-empty container or scoring object is moved, or
        // a container for liquids is moved, filled or emptied.

        if (length(self.checklist) = 0) goto set_score;

        // 'Takepoints' are applied when an object is first taken into
        // the player's possession, and 'depositpoints' are applied when
        // an object has been deposited in location 'targloc'.  Note
        // that 'takepoints' is never rescinded but 'depositpoints' is
        // deducted if an object is no longer correctly deposited.  (If an
        // object has dynamically-created copies, only one set of 
        // 'depositpoints' is awarded.  The points will be deducted only if
        // all objects are removed from their correct position)
        // If 'targloc' is a container rather than a room, 'contloc'
        // can define the location of the container.   If 'contloc'
        // is also a container, the 'outercontloc' property may be set
        // to define the location of this outer container.  Note that
        // these properties are normally ignored in a 350- or 550-point game,
        // unless the 'oldkeep' property is defined for the object.
        // Remember to set 'oldkeep' for non-treasure objects like the
        // magazines.
        //
        // 'undiscovered' is a special property for an object (the book
        // in the safe) which is in the right place to start with, but must
        // be discovered before points are awarded.  When the discovery takes
        // place, undiscovered should be set to nil and the item added to
        // global.checklist.
        //
        // 'objclass', if defined, is the class of a dynamically
        // created object.  The deposit score will be awarded if at least
        // one of the objects in class 'objclass' (or the parent object)
        // is in the desired location.
        //
        // This method requires the isInside function in ccr-thx.t (same
        // as isIn but without regard to visibility.)
        //
        // First loop: check for objects which are no longer in their
        // target locations (omitted at the end of the cylindrical room puzzle)

        self.checkpointlist := intersect(self.checklist,self.allpointlist);
        l := length(self.checkpointlist);
        if(not global.dont_rescind) {
            for (i := 1; i <= l; i++) {
                t := self.checkpointlist[i];
                // for cloned objects, set cl to the class; otherwise set
                // cl to the object itself.
                if(t.objclass) {
                    cl := t.objclass; 
                    // add any equivalent objects to be checked in the
                    // second loop
                    addlist += cl.list + cl - t;
                }
                else {
                    // add any equivalent objects to be checked in the
                    // second loop
                    cl := t;
                    addlist += t.list;
                } 
                if(global.newgame or t.oldkeep) {
                    tloc := t.targloc; // required location
                    c := tloc;     // required container
                    cloc := t.contloc; // required location of container
                    cloc2 := t.outercontloc; // reqd. loc. of outer container
                }
                else {
                    tloc := Inside_Building;
                    c := tloc;
                    cloc := nil;
                    cloc2 := nil;
                }
                // Medisqual indicates that the object is disqualified
                // from the deposit score because the player has it.  We have
                // to make an exception if the target location actually is
                // the player!
                if (tloc = parserGetMe() or cloc = parserGetMe() or 
                cloc2 = parserGetMe())
                    Medisqual := nil;
                else
                    Medisqual := t.isInside(Me);

                if (cl.awardedpointsfordepositing) {
                    if(cloc = nil)ccond := true;
                    else ccond := (c.isInside(cloc));
                    if(cloc2 <> nil)
                        if (not cloc.isInside(cloc2))ccond := nil;
                    if (Medisqual or not t.isInside(tloc) or 
                    not ccond) {
                        self.scoreinc := self.scoreinc - t.depositpoints;
                        cl.awardedpointsfordepositing := nil;
                    }
                }
            }
        }
        // Second loop: check for objects which have been taken, or
        // correctly deposited.
        self.checkpointlist += addlist;
        l := length(self.checkpointlist);
        for (i := 1; i <= l; i++) {
            t := self.checkpointlist[i];
            // for cloned objects, set cl to the class; otherwise set
            // cl to the object itself.
            if(t.objclass) cl := t.objclass; else cl := t;
            if(global.newgame or t.oldkeep) {
                tloc := t.targloc; // required location
                c := tloc;     // required container
                cloc := t.contloc; // required location of container
                cloc2 := t.outercontloc; // reqd. loc. of outer container
            }
            else {
                tloc := Inside_Building;
                c := tloc;
                cloc := nil;
                cloc2 := nil;
            }
            // Medisqual indicates that the object is disqualified
            // from the deposit score because the player has it.  We have
            // to make an exception if the target location actually is
            // the player!
            if (tloc = parserGetMe() or cloc = parserGetMe() or 
            cloc2 = parserGetMe())
                Medisqual := nil;
            else
                Medisqual := t.isInside(Me);

            if (not cl.awardedpointsfordepositing and
            not t.undiscovered) {
                // Check for 'taking' points if object is being
                // carried.
                if (t.isInside(parserGetMe()) and
                (not cl.awardedpointsfortaking) ) {
                    self.scoreinc := self.scoreinc + t.takepoints;
                    cl.awardedpointsfortaking := true;
                    // self.treasures now counts the items remaining
                    // to be taken (truer to the original Fortran
                    // versions).
                    if(isclass(t,CCR_treasure_item)) {
                        self.treasures := self.treasures - 1;
                        global.treasures_to_find -= t;
                    }
                }
                // Check for deposited object.
                if (t.isInside(tloc) and not Medisqual and
                not cl.awardedpointsfordepositing) {
                    // if contloc is nil, ignore location of container ...
                    if(cloc = nil)ccond := true;
                    // otherwise check location of the container
                    else ccond := (c.isInside(cloc));
                    // condition for second-level container
                    if(cloc2 <> nil)
                    if (not cloc.isInside(cloc2))ccond := nil;
                    if(ccond) {
                        // check that 'taking' points have been
                        // awarded (needed for book)
                        if (not cl.awardedpointsfortaking) {
                            self.scoreinc := self.scoreinc + t.takepoints;
                            cl.awardedpointsfortaking := true;
                            if(isclass(t,CCR_treasure_item)) {
                                self.treasures := self.treasures - 1;
                                global.treasures_to_find -= t;
                            }
                        }
                        // award points for depositing ...
                        self.scoreinc := self.scoreinc + t.depositpoints;
                        cl.awardedpointsfordepositing := true;
                    }
                }
            }
        }
        // adjust the score
        set_score: if(self.scoreinc <> 0)incscore(self.scoreinc);
        self.scoreinc := 0;
        self.checklist := [];
        // check for discovery of a bonus treasure
        if (global.maxhiked) {
             "\n Hmmm... You seem to have found an extra item I 
             didn't know about, so I've increased the maximum score
             to <<global.maxscore>> points!\n";
             global.maxhiked := nil;
        }
        // check for closing (now done after object scores have been
        // checked.)
        check_for_closing(nil);
    }
;

// verglob objects contain global settings for the various versions.
verglob: object

    // quantities to be filled in by preinit()
    treasures = 0           // treasures left to deposit;
                            // filled in by preinit
    origtreasures = 0       // original number of treasures; set by
                            // preinit
    treasurelist = []       // list of all treasures in the game;
                            // filled in by preinit
    bonustreasures = []     // list of all bonus treasure objects (e.g.
                            // pendants in 701+ point game)
    pointobjs = 0           // number of objects which score points
                            //
    pointobjlist = []       // objects which score points
    bonuspointobjs = []     // list of bonus non-treasure objects.

    allpoints = 0           // combined list of point-scoring
                            // objects
    allpointlist = []


    // game version selection
    removelist = []         // list of objects to be removed
    movelist = []           // list of objects to be moved to different
                            // locations
    changeloc = []          // list of modified locations
    moveloclist = []        // floatingDecoration objects to be given
                            // a different loclist
    changeloclist = []      // list of modified loclists

    sacklist = []           // containers for automatic inventory
                            // management

    // copy verglobal settings to global.  This method can be overridden
    // if extra copying or setup is to be done.
    copy = {
        global.maxresurrect  := self.maxresurrect;
        global.score         := self.score;
        global.maxscore      := self.maxscore;
        global.novicepoints  := self.novicepoints;
        global.quitpoints    := self.quitpoints;
        global.deathpoints   := self.deathpoints;
        global.farinpoints   := self.farinpoints;
        global.extenpoints   := self.extenpoints;
        global.closurepoints := self.closurepoints;
        global.closingpoints := self.closingpoints;
        global.endpoints     := self.endpoints;
        global.endkillpoints := self.endkillpoints;
        global.klutzpoints   := self.klutzpoints;
        global.almostpoints  := self.almostpoints;
        global.winpoints     := self.winpoints;
        global.escapepoints  := self.escapepoints;
        global.finalepoints  := self.finalepoints;
        global.treasures     := self.treasures;
        global.origtreasures := self.origtreasures;
        global.treasurelist  := self.treasurelist;
        global.pointobjs     := self.pointobjs;
        global.pointobjlist  := self.pointobjlist;
        global.allpoints     := self.allpoints;
        global.allpointlist  := self.allpointlist;
        global.dwarves       := self.dwarves;
        global.removelist    := self.removelist;
        global.movelist      := self.movelist;
        global.changeloc     := self.changeloc;
        global.moveloclist   := self.moveloclist;
        global.changeloclist := self.changeloclist;
        global.sacklist      := self.sacklist;
        global.treasures_to_find := self.treasures_to_find;
        global.bonustreasures := self.bonustreasures;
        global.bonuspointobjs := self.bonuspointobjs;
        fresh_batteries.available := self.coinsets;
        // save the bonus time.
        global.bonusorig     := global.bonustime;
        global.closingorig   := global.closingtime;
    }
;

/*   The glob0 object holds lists and values which are copied over to
     the global object when a 350-point game is selected.  */

glob0: verglob
    vnumber = 0

    maxresurrect = 2        // number of resurrections allowed

    //
    // Scoring values
    // Points for treasures are kept in the treasures themselves (as
    // oldtakepoints, olddepositpoints, defaults 2 and 12)
    score = 36              // start out with 36 points:
                            // (2 + (-1 * quitpoints) +
                            // (-1 * deathpoints) * 3)
    maxscore = 350          // maximum possible score
    novicepoints = -5       // points for playing in easy mode (neg.)
    quitpoints = -4         // points for quitting (neg.)
    deathpoints = -10       // points gained each time player dies (neg.)
    farinpoints = 25        // points for getting well into the cave
    extenpoints = 0         // points for entering extended regions

    closure = nil           // Has the endgame timer started yet?
    closurepoints = 0       // point award when it does.
    closingpoints = 25      // points for surviving until cave closing time
    endpoints = 10          // points for getting to final puzzle
    endkillpoints = 10      // points for getting killed in endgame
    klutzpoints = 15        // points for getting klutzed (blown up)
    almostpoints = 20       // points for *almost* getting final puzzle
                            // (wrong part of room blown up)
    winpoints = 35          // points for winning the final puzzle
    escapepoints = 0        // These two properties are not applicable
    finalepoints = 0        // in this version.

    //
    // NPC stuff
    //
    dwarves = 5             // number of dwarves wandering about the cave
                            // (Was 5 in original.)

    sacklist = []           // containers for automatic inventory
                                // management
;

// This object holds the settings for a 551-point game (version 1).

glob1: verglob
        vnumber = 1
        maxresurrect = 2        // number of resurrections allowed
/*
 * The following comments are adapted from the source of the scoring
 * routine in the 551-point game.   In the original Fortran version,
 * the non-treasure-related part of the score was not given in full until
 * 300 turns had been played.  This has not been implemented in the TADS
 * port.
 *
 * The scoring totals are as follows:
 *
 *    Objective:          Points:           Total possible:
 * Getting well into cave   25                    25
 * Total possible for treasures (+objects)       428
 * Surviving             (max-num)*10             30
 * Not quitting              4                     4
 * Reaching "closing"       20                    20
 * "closed": quit/killed    10
 *           klutzed        20
 *           wrong way      25
 *           success        30                    30
 * Round out the total      14                    14
 *                                      total:   551
 * (points can also be deducted for using hints.)
 * For full credit treasures must be in building, not broken, and in
 * the correct containers.  Scores depend on the basis property (default 1).
 * Optionally we can test that the container is in the
 * right place (contloc) and also the container-of-the-container
 * (outercontloc).
 *
 * Non-treasure objects: (points given for correct depositing)
 * Object       Score
 * magazine     1
 * marked rod   2
 * total        3
 *
 * Treasure scores are multiplied by 2 (range 4-10) when
 * discovered or 5 (range 5-25) when deposited correctly:
 * i.e. takepoints = 2 x basis
 *   depositpoints = 3 x basis
 *           total = 5 x basis
 * Treasure     Basis
 * book         2
 * wine         3 (in cask only - must enter via Styx to fill cask)
 * chain        4 (must enter via Styx to obtain this)
 * chest        5
 * cloak        3
 * clover       1
 * coins        5
 * crown        2
 * crystal-ball 2
 * diamonds     2
 * eggs         3
 * emerald      3
 * chalice      2
 * horn         2
 * jewels       1
 * lyre         1
 * nugget       2
 * pearl        4
 * pyramid      4
 * radium       4
 * ring         4
 * rug          3
 * sapphire     1
 * shoes        3
 * spices       1
 * sword        4
 * trident      2
 * vase         2
 * droplet      5
 * tree         5
 *      TOTAL: 85 * 5 = 425 + 3 ==> 428
 */

    //
    // Scoring values
    // Points for treasures are kept in the treasures themselves.
    //
    score = 48              // start out with 48 points:
                            // (14 + (-1 * quitpoints) +
                            // (-1 * deathpoints) * 3)
    maxscore = 551          // maximum possible score
    novicepoints = -5       // points for playing in easy mode (neg.)
    quitpoints = -4         // points for quitting (neg.)
    deathpoints = -10       // points gained each time player dies (neg.)
    farinpoints = 25        // points for getting well into the cave
    extenpoints = 0         // points for getting well into the cave

    closure = nil           // Has the endgame timer started yet?
    closurepoints = 0       // point award when it does.
    closingpoints = 20      // points for surviving until cave closing time
    endpoints = 0           // points for getting to final puzzle
    endkillpoints = 10      // points for getting killed in endgame
    klutzpoints = 20        // points for getting klutzed (blown up)
    almostpoints = 25       // points for *almost* getting final puzzle
                            // (wrong part of room blown up)
    winpoints = 30          // points for winning the final puzzle
    escapepoints = 0        // These two properties are not applicable
    finalepoints = 0        // in this version.

    sacklist = [sack,treasure_chest] // containers for automatic inventory
                                     // management
    //
    // NPC stuff
    //
    dwarves = 5             // number of dwarves wandering about the cave
                            // (somewhat reduced - was 6 on average)

    copy = {
        addword(tasty_food,&noun,'sandwich');
        addword(tasty_food,&noun,'sandwiches');
        addword(tasty_food,&noun,'watercress');
        addword(tasty_food,&adjective,'watercress');
        pass copy;
    }
;
/*   The glob2 and glob7 objects hold lists and values which are copied over to
     the global object when a 550-point or 580-poing game is selected.  */

class glob_550: verglob
    maxresurrect = 3        // number of resurrections allowed
    //
    // Scoring values
    //
    // 9 points initially, 1 point for magazines, 390 (420) points for
    // treasures, 20 points for getting far in, 30 points for reaching
    // extended areas, 100 points for endgame.
    // Total = 550 (580).
    //
    // Points for treasures are set to:
    // takepoints = 2, depositpoints = 13.
    //
    score = 9               // start out with 9 points: (-1*quitpoints)
                            // Thus, score can be negative in this version.
    novicepoints = -5       // points for playing in easy mode (neg.)
    quitpoints = -4         // points for quitting (neg.)
    deathpoints = -10       // points gained each time player dies (neg.)
    farinpoints = 20        // points for getting well into the cave
    extenpoints = 10        // points for reaching extended areas.

    closure = nil           // Has the endgame timer started yet?
    closurepoints = 20      // point award when it does.
    closingpoints = 20      // points for surviving until cave closing time
    endpoints = 20          // points for getting to final puzzle
    endkillpoints = 0       // points for getting killed in endgame
    klutzpoints = 0         // points for getting klutzed (blown up)
    almostpoints = 0        // points for *almost* getting final puzzle
                            // (wrong part of room blown up)
    winpoints = 0           // points for winning the final puzzle
    escapepoints = 20       // points for leaving the cylindrical room
    finalepoints = 20       // points for entering the treasure room in
                            // the endgame.

    sacklist = []           // containers for automatic inventory
                            // management
    //
    // NPC stuff
    //
    dwarves = 5             // number of dwarves wandering about the cave
                                // (Was 5 in original.)
;
glob2: glob_550
    maxscore = 550
    vnumber = 2
;
glob7: glob_550
    maxscore = 580
    vnumber = 7
;

/*   The glob15 and glob11 objects inherit the same set of values, and are
     used in the 701 and 701+ point games.  The difference is that the
     701+ game scores bonus points when the new extended area is reached. */

class glob_701: verglob
    maxresurrect = 3        // number of resurrections allowed
    //
    // Scoring values
    //
    // 44 points initially, 1 point for magazines, 504 points for
    // treasures, 20 points for getting far in, 30 points for reaching
    // extended areas from the 550-point game, 100 points for endgame.
    // Total = 701
    //
    // Points for all treasures are set to:
    // takepoints = 2, depositpoints = 10.
    //
    score = 46              // to round out the score
    maxscore = 701          // maximum possible score
    novicepoints = -5       // points for playing in easy mode (neg.)
    quitpoints = -4         // points for quitting (neg.)
    deathpoints = -10       // points gained each time player dies (neg.)
    farinpoints = 20        // points for getting well into the cave
    extenpoints = 10        // points for reaching extended areas.

    closure = nil           // Has the endgame timer started yet?
    closurepoints = 20      // point award when it does.
    closingpoints = 20      // points for surviving until cave closing time
    endpoints = 20          // points for getting to final puzzle
    endkillpoints = 0       // points for getting killed in endgame
    klutzpoints = 0         // points for getting klutzed (blown up)
    almostpoints = 0        // points for *almost* getting final puzzle
                            // (wrong part of room blown up)
    winpoints = 0           // points for winning the final puzzle
    escapepoints = 20       // points for leaving the cylindrical room
    finalepoints = 20       // points for entering the treasure room in
                            // the endgame.

    sacklist = [sack,treasure_chest] // containers for automatic inventory
                                     // management
    //
    // NPC stuff
    //
    dwarves = 5             // number of dwarves wandering about the cave
                            // (Was 5 in original.)

    copy = {
        inherited glob1.copy;
    }
;

glob15: glob_701
    vnumber = 15
;

glob11: glob_701
    vnumber = 11
;

/*
 *   The "version" object defines, via its "sdesc" property, the name and
 *   version number of the game.  Change this to a suitable name for your
 *   game.
 */
version: object
    sdesc = {
        "\b\(Polyadventure\)\n
        Version 3.21a - 22 Oct 2013.\b";

        "Ported by David Baggett, David Picton and Bennett
        Standeven.\b";

        if (global.vnumber = 11) {
           "You are currently playing in the 701+ point mode (with bonus points
           if you find the new extensions).\b ";
        }
        else {
           "You are currently playing in the <<global.maxscore>>-point
           mode.\b";
        }
    }
;

/*
 *   "Me" is the player's actor.  All customization is now done in the general
 *   player-character class, basicMe, which is modified in advmods.t.
 */
Me: basicMe
;

/*
 * darkTravel() is called whenever the player attempts to move from a dark
 * location into another dark location.
 *
 * This now works in the same way as the original.  Travel is successful
 * unless you fall into a pit.
 */
darkTravel: function
{
    local actor := getActor(&travelActor);
    // allow 'magic' travel with no problems.
    if(global.travelActor.nextRoute = 10) return;

    "\nProceeding ..."; P();

    // don't kill player when in Transindection areas until Wumpi chase scene
    // is finished (or prevented by discovery of green pendant)
    if (Wumpi.nodeath(actor))
        return;
    /*
     * 1 in 4 chance of death.
     */
    if (rand(4) = 1) {
        P(); I();
        "You fell into a pit and broke every bone in your body!";
        die();
        global.fellpit := true;
    }
}

/*
 *   goToSleep - carries out the task of falling asleep.  We just display
 *   a message to this effect.
 */
goToSleep: function
{
    "***\bYou wake up some time later, feeling refreshed. ";
    global.awakeTime := 0;
}

#ifdef USE_HTML_PROMPT
/*
 *   commandPrompt - this displays the command prompt.  For HTML games, we
 *   switch to the special TADS-Input font, which lets the player choose
 *   the font for command input through the preferences dialog.
 */
commandPrompt: function(code)
{
    /* display the normal prompt */
    "\b&gt;";

    /* 
     *   switch the font to TADS-Input - the standard text-only
     *   interpreter will simply ignore this, so we don't have to worry
     *   about whether we're using an HTML-enabled interpreter or not 
     */
    "<font face='TADS-Input'>";
}

/*
 *   commandAfterRead - this is called just after each command is entered.
 *   For HTML games, we'll switch back to the original font that was in
 *   effect before commandPrompt switched to TADS-Input. 
 */
commandAfterRead: function(code)
{
    "</font>";
}
#endif /* USE_HTML_PROMPT */
