/*
 * Polyadventure
 *
 * A remake of the various versions of the classic Adventure game by Don
 * Woods and Willie Crowther, based on their sources.  Currently, the 350,
 * 550, and 551-point versions are implemented.  See the file "ccr-help.t"
 * for more information.
 *
 * Please document all changes in the history so we know who did what.
 *
 * This source code is copylefted under the terms of the GNU Public
 * License.  Essentially, this means that you are free to do whatever
 * you wish with this source code, provided you do not charge any
 * money for it or for any derivative works.
 *
 * ADVENTIONS distributes this game, but you are free to do what you will
 * with it, provided you adhere to the terms in the GNU Public License.
 * Send correspondence regarding this game or original works distributed
 * by ADVENTIONS to
 *
 *      ADVENTIONS
 *      PO Box 851
 *      Columbia, MD 21044
 *
 * If you would like a catalog of releases, please enclose a SASE.  Thanks!
 *
 * Contributors (see history.t for current e-mail addresses)
 *
 *      dmb     In real life:   David M. Baggett
 *
 *      djp     In real life:   David J. Picton
 *
 *      bjs     In real life:   Bennett J. Standeven
 *
 * Modification History
 *
 * CCR
 * ===
 *
 * 1-Jan-93     dmb     rec.arts.int-fiction BETA release (source only)
 *                      For beta testing only -- not for general
 *                      distribution.
 *
 * AD551
 * =====
 *
 * 15-Jan-99    djp     Moved all endgame rooms and functions into this file
 *
 * 14-Apr-99    djp     Initial release of Adventure 551 (1.01)
 *
 * 30-Apr-99    djp     New release (1.11) of Adventure 551.
 *
 * 17-May-99    djp     New release (1.20)
 *                      Changes since 1.11:
 *                      * Added a direction to the phone booth in the
 *                        Respository.
 *
 * 15-Jul-99    djp     New release (2.00)
 *                      Changes in this version:
 *                      * Added the newgame property to some items (it is
 *                        now declared for individual items and not inherited
 *                        from classes).
 *                      * Fixed a bug which allowed you to take the wire
 *                        using YANK WIRE FROM BOOTH.
 *
 * 3-Aug-99     djp     Bugfix release - Version 2.01
 *
 *              djp     Bugfix release - Version 2.02 (not publicly released)
 *                      Changes in this version
 *                      * Fixed some bugs associated with returning
 *                        objects to their piles/bundles in the endgame.
 *                      * Fixed the start_closing code to banish the Wumpus
 *                        if it's chasing the player.
 *
 * POLYADV
 * =======
 *
 * 24-Aug-99    bjs     Pre-release version 0.00
 *              bjs     Added 550-point endgame.
 *
 *          djp+bjs     Incorporated ad551 mods up to 2.20
 *
 * 3-Mar-00     djp     Initial beta release - Version 1.00
 *                      Changes since ad551 version 2.20:
 *                      * Changed endgame.liquidcont to take advantage of the
 *                        new 'inherited' syntax.
 *                      * Changed check_for_closing to tell the player when
 *                        the last treasure has been taken.
 *
 * 4-Apr-00     djp     Version 1.01: bugfix release
 *                      Changes in this version
 *                      * Added isindoor property to Cylindrical.
 *                      * Corrections to endgame_clone and
 *                        endgame_liquidcont to ensure that verification
 *                        methods are passed up to the correct class
 *                      * 'newgame' property changed to 'game551' for the
 *                        phone booth
 *
 * 18-Sep-00    djp     Version 2.00: New version with 701-point game
 *                      Changes in this version
 *                      * Modified wordtest for 701-point game
 *              
 * 20-Dec-00    djp     Version 2.02: bugfix release
 *                      Changes in this version:
 *                      * Fixed a bug which caused the telephone to ring in the
 *                        701-point endgame.
 *
 * 17-Aug-01    djp     Version 2.06: bugfix release with e-mail address update
 *                      Changes in this version:
 *                      * (701-point mode) Fixed the Cylindrical Room code to 
 *                        remove the contents of the safe.
 *                      * (701-point mode) Put all the keys in the Pantry on 
 *                        escape from the Cylindrical room, enabling the
 *                        player to try out all the entrances.
 *
 * 22-Aug-01    djp     Version 2.07: bugfix release
 *
 * 13-Jan-03    djp     Version 2.09 rev C: bugfixes and code tidy-up
 *                      Changes in this version:
 *                      * Slight vocab adjustment for snakepit and dwarves 
 *                        (so that gonear snake, gonear dwarf will find the
 *                        normal objects)
 *                      * Stock the repository only when necessary.
 *                      * Changed the wicker cage setup procedure to
 *                        avoid the need for a directly-defined setup method.
 *                        This circumvents a TADS bug which can cause an
 *                        error when:
 *                        (i) an object *directly* defines a method e.g.
 *                        wicker_cage.setup for the production of
 *                        dynamically-created children of itself.
 *                        (ii) the construct method changes a superclass
 *                        property, e.g. wicker_cage.list.
 *                      * Implemented the createloc property for the location
 *                        of dynamically-created objects.  (endgame_clone
 *                        items no longer need a construct method).
 *                      * Changed the 'search' and 'look in' methods for
 *                        endgame piles (to fit in with updates to ccr-adv.t;
 *                        normally the number of items is not revealed except
 *                        when debug mode is enabled and 'search' is used.)
 *                      * In debug mode, make OUT work in the cylindrical
 *                        room.
 *                      * Changed the endgame code to use the
 *                        new travelTo optional argument.
 *                      * Enhancement to endgame_clone class to allow
 *                        pass_class = container.
 *                      * Changes to the cave closure code to work with the
 *                        new CCR_doorway class.
 *
 * 12-Aug-03    bjs     Version 2:11: added 580-point version
 *                      * start_closing now doubles as a security alert 
 *                        function, called by blob.summon().  Note that this 
 *                        causes a slight change to the 701-point game - a 
 *                        security alert now locks the elfin door as well as 
 *                        the grate.  The doors remain locked after the
 *                        player's resurrection, but can then be unlocked.
 *
 * 23-Aug-03    djp     Version 2:12 
 *                      * start_closing changed to save the state of the
 *                        doors so the noAutoOpen property can be reset 
 *                        later. (Further properties are saved in preparation
 *                        for Version 3, in which the unlocking of a 
 *                        certain door postpones the cave closure)
 *                      * start_closing sets global.security_alert during
 *                        a security alert.
 *
 * 23-Jul-04    djp     Version 3.00.  Addition of a new game mode (701+).
 *                      * Updated code in connection with the 701+ point mode.
 *                      * Correction to the closure code to make the ring
 *                        disappear along with the Wumpus (if it has been put
 *                        back)
 *                      * Correction to the verDoInspect method for
 *                        the endgame_clone class (which passed control to
 *                        the action method instead of the verification
 *                        method).
 *                      * Correction to the cave closure code to set the
 *                        lamp life and turn off the lamp in all cases,
 *                        avoiding unwanted messages.
 *                      * Change to cave closure procedures.  The clocks are
 *                        held at 1 if the player is in a Zarkalon room, 
 *                        not at Red level or in an isolated room (e.g. the
 *                        Large Circular Room)
 * 
 * 15-Jan-05    djp     Version 3.01: bugfix release.
 *                      * Minor changes to endgame text - King Ondral doesn't
 *                        mention Transindection until he realizes that you're
 *                        his "intruder".
 *
 * 12-Apr-05    djp     Version 3.10: Bugfixes and enhancements
 *                      * The endgame text now caters for variant gameplay 
 *                        after a throne mishap (player sits on throne with no 
 *                        crown).  If the player managed to win without
 *                        disabling the Upper Transindection Chambers, the
 *                        Mountain King talks about doing this in the future.
 *                       
 * 28-Jun-05    djp     Version 3.11: Enhancements and bugfixes
 *                      * A new function, cancel_cave_closure, has been
 *                        implemented.  It is used when certain doors are
 *                        unlocked in the 701+ point game.
 *                      * If the gold ring disappeared with the Wumpus
 *                        at cave closure, it now reappears on the Wumpus'
 *                        finger after cave closure cancellation.
 *
 * 16-Jul-06    djp     Version 3.20: Bugfixes and extensions
 *                      * Changed cancel_cave_closure to implement an optional
 *                        second argument (indicating that a new treasure has
 *                        been found, as opposed to a new area).  The first
 *                        argument is now always set to true if the player's
 *                        Transindection level is not 0 ("red") level.
 *                      * Cylindrical.roomAction now issues an appropriate
 *                        message if the SAY command is used, e.g. 
 *                        SAY "xyzzy".  
 *                      * Implementation of a new general scheme for optional
 *                        magic words in the endgame.  If the number property
 *                        is (say) -17, the word can optionally be used after
 *                        the magic word with a number property of 16.
 *                      * Alteration of the endgame messages for a special case
 *                        (it is now possible to complete the game without
 *                        transporting the Wumpi or entering the Maintenance 
 *                        Rooms)
 *                      * Modification to the 701- and 701+ point endgames to
 *                        show the player which treasures have not been
 *                        correctly deposited.
 *
 * 20-Oct-07    djp     Version 3.20 Rev. C
 *                      * Slight changes to 'treasure room' messages in
 *                        the 701- and 701+ point game modes.
 *
 * 17-Apr-09    djp     Version 3.20 Rev. D
 *                      * Changes to cancel_cave_closure.  There is now a 
 *                        third argument which is set to 'true' if the 
 *                        protagonist has managed to escape from the cave.
 *                        The player is not informed if the cancellation occurs
 *                        when the cave is merely counting down to closure.  
 *                        The closure points (global.closurepoints) are no 
 *                        longer rescinded when the cave closure is cancelled.
 *                      * Changes in check_for_closure: new code has been added
 *                        to detect the addition of bonus treasures to the list
 *                        after all normal treasures have been found, and to
 *                        inform the player when all the bonus treasures appear
 *                        to have been discovered.  (There may still be some
 *                        optional treasures to find).
 */

/*
 * This file defines a few game-specific functions for the cave closing.
 * Don't read any of this unless you want to spoil the ending!
 */

/*
 * This function is called every turn to see if it's time to start
 * closing the cave.
 * BJS: In 550-point mode, it also awards a 20-point bonus (the value
 * of the award is stored in "global.closurepoints").
 */
check_for_closing: function(parm)
{
    local toploc := toplocation(parserGetMe());
    //
    // If there are no more treasures left to be found, count
    // down the cave closing counter whenever the player is
    // well inside but not at Y2.
    //
    if (global.closingtime > 0 and global.treasures <= 0) {
        // global.closure is now used only as a flag to indicate that
        // the closure countdown is in progress
        global.closure := true; 
        // a different property is used to test whether closure points have
        // been awarded (for collecting the last treasure)
        if (not global.closurepoints_awarded) {
            // only award points once, if at all - note that 
            // cancel_cave_closure no longer deducts these points once
            // they have been awarded.
            global.closurepoints_awarded := true;
            if(global.closurepoints) {
                incscore(global.closurepoints,'for collecting
                the last treasure');
            }
            else {
                P(); I(); "*** You have collected the last
                treasure! ***";
                P();
            }
        }
        if (not toploc.notfarin and not (toploc = At_Y2)) {
            // count down to zero only when the player is in a non-isolated,
            // non-Zarkalon room at Transindection level zero.
            if ((global.closingtime != 1) or ( (toploc.analevel = 0) 
            and not toploc.isolated and not toploc.Zarkalonroom))
                global.closingtime--;
            if (global.closingtime < 1)
                start_closing(true);
        }
    }
    if (global.closingtime > 0) {
        // set a flag if extra treasures have been found after the player has
        // apparently discovered them all
        if (global.closurepoints_awarded and (global.treasures > 0))
            global.extra_treasures_found := true;
        // tell the player when the last treasure has been found
        if  (global.closurepoints_awarded and global.extra_treasures_found and 
        (global.treasures <= 0)) {
            global.extra_treasures_found := nil;
                P(); I(); "*** I think you have collected the last available
                treasure - unless you can prove me wrong! ***";
                P();
        }
    }
}

/*
 * This function is called every turn to see if it's time to start
 * the final puzzle.
 */
check_for_endgame: function(parm)
{
    local toploc := toplocation(parserGetMe());
    if (global.closed and global.bonustime > 0) {
        // count down to zero only when the player is in a non-isolated,
        // non-Zarkalon room at Transindection level zero.
        if ((global.bonustime != 1) or ( (toploc.analevel = 0) 
        and not toploc.isolated and not toploc.Zarkalonroom))
            global.bonustime--;
        if (global.bonustime < 1) {
            // set up some clocks
            if(global.newgame) {
                global.endgameclock := 0;
                global.phonetime := 40;
                global.phonewake := 47;
            }
            // start the endgame
            start_endgame();
        }
    }

    // N.B. the combined game only has the cylindrical room endgame and
    // omits the repository scene.
    if (global.fully_closed and global.newgame and not global.game701) {
        global.endgameclock++;
        if(global.endgameclock = global.phonetime) {
            P();I();
            "The phone starts to ring. ";
            Phone2.isringing := true;
        }
        else if(global.endgameclock = global.phonewake) {
            P();I();
            "The constant ringing has awakened the dwarves! ";
            end_dwarves('');
        }
    }
}


/* This function is called with clos = true when the cave closes, and clos = nil on a Security Alert. */
start_closing: function(clos)
{   local i,o,l;

    // This should contain a list of all the external doors which are
    // to be locked at closing time, or during a Security Alert.

    local doorlist := [Grate, Small_door_1, Small_door_2, Iron_door_1, 
                      Iron_door_2];
    if(clos) {
      P();
      I(); "A sepulchral voice reverberating through the cave says,
      \"Cave closing soon.  ";
    

      if (global.game550) "All adventurers please report to
      the treasure room via the alternate entrance to claim
      your treasure.\"";

      else "All adventurers exit immediately through main office.\"";

      incscore(global.closingpoints);

      global.closed := true;
    }
    else global.security_alert := true;

    CrystalBridge.vanish(parserGetMe());    // destroy the bridge
    // N.B. if the rod has been upgraded (possible in some versions),
    // it is possible to use it after cave closure.

    // Loop over all lockable external doors.
    // N.B. the closing-time condition is now handled by the new
    // CCR_lockableDoorway class; there is now no need to set the 
    // mykey property to nil.

    l := length(doorlist);
    for (i := 1; i <= l; i++) {
        o := doorlist[i];
        // Save the state of the door.
        o.waslocked := o.islocked;
        o.wasopen := o.isopen;
        o.wasAutoOpen := o.noAutoOpen;
        o.isopen := nil;            // close the door, then
        o.islocked := true;         // lock it.
        o.noAutoOpen := true;       // disable automatic opening.
    }

    if(Elevator580.isopen && (parserGetMe.isIn(S_Of_Center) || 
    parserGetMe.isIn(N_Of_Center))) 
        "With a >whooosh<, the elevator doors slide together. ";
    Elevator580.isopen := nil;
    if(parserGetMe.isIn(S_Of_Center) || parserGetMe.isIn(N_Of_Center))
        "Out from around a corner at the far end of the computer center,
         a disgruntled repair man walks over to the elevator and tapes
         onto it a sign saying, \"OUT OF ORDER - Please use the stairs\",
         and then walks away. ";
   
    if (clos) {
        Dwarves.closelist := Dwarves.loclist;
        Dwarves.loclist := [];      // nuke dwarves...
        Pirates.closelist := Pirates.loclist;
        Pirates.loclist := [];      // ...and pirate(s)
        Troll.closeloc := Troll.location;
        Troll.moveInto(nil);        // vaporize troll
        Bear.closexists := Bear.exists;
        Bear.exists := nil;         // ditto for bear
        Wumpus.closeloc := Wumpus.location;
        // fix for problem if the gold ring has been put back on the 
        // Wumpus' finger. It was still described as on the finger after
        // the Wumpus had gone.  Now it disappears with the Wumpus, but
        // reappears with the Wumpus if cave closure is cancelled. 
        if((gold_ring.location = Wumpus.location) and not gold_ring.moved)
            gold_ring.moveInto(Wumpus);
        Wumpus.moveInto(nil);       // nuke the Wumpus (he must be
                                    // dead at this stage because we've
                                    // found the gold ring.)
        Dog.closeloc := Dog.location;
        Dog.moveInto(nil);          // likewise with the dog.

        // BJS: The creatures from the 550-point version must have disappeared
        // by now.

        // This was listed in the original as being too much trouble
        // to bother with.  (The old Fortran code allowed you to reach the
        // endgame without killing the dragon, and its removal at that point
        // would have required the duplication of complex code.)  But in this
        // version, all we need is:

        DragonCorpse.closeloc := DragonCorpse.location;
        DragonCorpse.moveInto(nil);     // nuke the dragon (which must be
                                        // dead - in the TADS port you must
                                        // take all the treasures, including
                                        // the rug, to reach the endgame.)
    }
}

cancel_cave_closure: function(not_at_red,...) {
    local deduction := 0;
    local treasfind := nil, escape := nil;
    if (argcount >= 2) treasfind := getarg(2);
    if (argcount >= 3) escape := getarg(3);
    // Tell the player what is happening.  A message is now issued only if
    // the cave is actually closed.
    if(global.closed) {
        // player has managed to escape from the cave.
        if(escape) {
            P(); I();
            "You somehow sense that a sepulchral voice is reverberating through
            the cave.  I guess that it has something to do with your escape - I
            don't think anyone has managed to do that before ... \n";
        }
        // closure is at this transindection level
        else if(not not_at_red) {
            if(treasfind) {
                P(); I();
                "A sepulchral voice reverberating through the cave says,
                \"Due to the discovery of a new treasure, there will be a 
                slight postponement of the cave closure.\" ";
            }
            else {
                P(); I();
                "A sepulchral voice reverberating through the cave says,
                \"Due to the opening of a new area, the closure of the cave
                has been postponed.\" ";
            }
        }
        // closure not at this transindection level
        else if(treasfind) {
            P(); I();
            "You somehow sense that a sepulchral voice is reverberating through
            the cave at Red level.  I guess your extra treasure has caused a
            slight postponement of the cave closure. ";
        }
        else {
            P(); I();
            "You somehow sense that a sepulchral voice is reverberating through
            the cave at Red level.  I guess your discovery of
            a new area has caused a slight postponement of the cave closure. ";
        }
    }

//  Cancel the closure flag if there are now extra treasures to find.
//  (If treasfind has been set, the extra treasures have been found in this
//  turn)
    if ((global.treasures > 0) and not treasfind)
        global.closure := nil;
    global.closingtime := global.closingorig + 1;
    if(global.closed) {
        local i,o,l,doorlist := [Grate, Small_door_1, Small_door_2, 
        Iron_door_1, Iron_door_2];
        global.closed := nil;
        deduction += global.closingpoints;
        global.bonustime := global.bonusorig;
        // unlock and/or open external doors when
        // appropriate
        l := length(doorlist);
        for (i := 1; i <= l; i++) {
            o := doorlist[i];
            o.isopen := o.wasopen;
            o.islocked := o.waslocked;
            o.noAutoOpen := o.wasAutoOpen;
        }
        // reinstate NPC's
        Dwarves.loclist := Dwarves.closelist;
        Pirates.loclist := Pirates.closelist;
        Troll.moveInto(Troll.closeloc);
        Bear.exists := Bear.closexists;
        Wumpus.moveInto(Wumpus.closeloc);
        if (gold_ring.location = Wumpus) {
            gold_ring.moveInto(Wumpus.location);
            gold_ring.moved := nil;
        }
        Dog.moveInto(Dog.closeloc);
        DragonCorpse.moveInto(DragonCorpse.closeloc);
    }
    if(deduction) incscore(-deduction);
}
start_endgame: function
{
    local i,o,l,savecont := parserGetMe().contents;
    global.fully_closed := true;     // DJP
    P();
    I(); "The sepulchral voice intones, \"The cave is now
    closed.\" As the echoes fade, there is a blinding flash of
    light (and a small puff of orange smoke).\ .\ .\ .  As your
    eyes refocus, you look around and find that you're...\b";

    //
    // Vaporize everything the player's carrying, except pendants which are
    // worn or carried in hand.
    //
    l := length(savecont);
    for (i := 1; i <= l; i++) {
        o := savecont[i];
        if (not isclass(o,pendantItem))
            o.moveInto(nil);
    }
    //
    // Get rid of anything chasing the player.
    //
    for(o:=firstobj(Chaser); o<>nil; o:=nextobj(o, Chaser))
        if (o.ischasing) o.banish;

    brass_lantern.turnoff;
    brass_lantern.setlife(2500); // stop unwanted messages about the lamp
                                 // running out of power
    if (not global.game550) {    // update this statement if new versions are
                                 // added
        //
        // Stock the northeast end
        //
        bottle.setup;        // create method now makes empty container
        giant_bivalve.setup;
        black_rod.downgrade; // goes back to being rusty
        black_rod.setup;
        brass_lantern.setup;
        //
        // Stock the southwest end
        //
        // Create 'clones' of various objects (which, being dynamically
        // created, will inherit all the vocab of the parent.  Statically
        // defined objects only inherit the vocab if the parent object is
        // a class.)
        //
        wicker_cage.setup;
        black_mark_rod.setup;
        velvet_pillow.setup;
        if(global.newgame) global.newbook := new rare_book;
    }
    //
    // Move the player
    //
    if (global.game550) {
        parserGetMe().travelTo(Cylindrical,nil,nil,true);
    }

    else parserGetMe().travelTo(At_Ne_End,nil,nil,true);

    incscore(global.endpoints);
}

/*
 * Determine how the player fares in the final puzzle.
 *
 * For full points, the player must have placed any rods taken from the bundle
 * in the appropriate room, and have stood well back!
 *
 */
endpuzzle: function
{
    local rodloc,meloc;
    local bundcount,necount,swcount,boothcount,mecount;
    local cond1,cond2,oldcond1;

    meloc := At_Sw_End;

    // count the rods in different places
    bundcount := black_mark_rod.loccount(black_mark_rod_bundle);
    necount := black_mark_rod.loccount(At_Ne_End);
    swcount := black_mark_rod.loccount(At_Sw_End);
    boothcount := black_mark_rod.loccount(Phone_Booth2);
    mecount := black_mark_rod.loccount(parserGetMe());
    // if no rods have been taken, explosion uses sw end bundle
    if((necount = 0) and (swcount = 0) and (boothcount = 0) and
    (mecount = 0)) swcount := 1;

    cond1 := (necount > 0) and (mecount = 0);
    oldcond1 := cond1;
    if(global.newgame)
        cond1 := (boothcount > 0) and (necount = 0) and (mecount = 0)
        and (swcount = 0);

    cond2 := (swcount > 0) and (boothcount = 0) and (necount = 0) and
        (mecount = 0);


    if (cond1 and parserGetMe().isIn(meloc)) {
        I(); "There is a loud explosion, and a twenty-foot
        hole appears in the far wall, burying the dwarves in
        the rubble.  You march through the hole and find
        yourself in the main office, where a cheering band of
        friendly elves carry the conquering adventurer off
        into the sunset."; P();

        incscore(global.winpoints);

        win();
    }
    else if (cond2 and parserGetMe().isIn(At_Ne_End)) {
        I(); "There is a loud explosion, and a twenty-foot
        hole appears in the far wall, burying the snakes in
        the rubble.  A river of molten lava pours in through
        the hole, destroying everything in its path,
        including you!"; P();

        incscore(global.almostpoints);

        win();
    }
    else  {
        "There is a loud explosion, and you are suddenly
        splashed across the walls of the room.";
        // if the adventurer tried the 350-point solution in the
        // 551-point puzzle, issue a hint.
        if (oldcond1 and parserGetMe().isIn(meloc)) {
            P(); "That was a good try, but in the 551-point version
            you need to contain the explosion somewhat ... ";
        }
        incscore(global.klutzpoints); // DJP - CCR got a TADS
                                      // error here by referring
                                      // to an undefined property

        win();
    }
}

/*
 * The player resolves the endgame by disturbing the dwarves.
 */
end_dwarves: function(...)
{
    local sentence := 'The resulting ruckus has awakened the dwarves.';
    if (argcount > 0) sentence := getarg(1);
    P();
    I(); say(sentence); " There are now several threatening little dwarves in 
    the room with you! Most of them throw knives at you!  All of them get
    you!"; P();

    incscore(global.endkillpoints); // DJP - this was missing

    win();
}

/*
 * The player falls foul of the booby-trapped phone.
 */
end_phone: function
{
    "Whoops!  The floor has opened out from under you!  It seems you
    have fallen into a bottomless pit.  As a matter of fact, you're
    still falling!  Well, I have better things to do than wait around
    for you to strike bottom, so let's just assume you're dead.
    Sorry about that, Chief.";

    incscore(global.endkillpoints);

    win();
}

/* Endgame special object classes */
class objpile: fixeditem, qcontainer
    maxcount = 5
    // targobj is the type of object in the pile, , e.g.
    // targobj = black_rod.  It must have class 'endgame_clone' or
    // 'endgame_liquidcont' and at least maxcount + 1 must be in the pile
    // initially.
    verDoTake(actor) = {
        "Realizing that by removing the whole <<self.sdesc>> you'd
        be ruining the game for future players, you
        leave <<self.thedesc>> where it is.";
    }
    doLookin(actor) = {
        self.ldesc;
    }
    doSearch(actor) = {
#ifdef WIZARD
        pass doSearch;
#else
        self.doLookin(actor);
#endif
    }
    verIoPutIn(actor) = {}
    verIoPutOn(actor) = {}
    ioPutIn(actor,dobj) = {
        if ((dobj <> self.targobj) and (dobj.objclass <> self.targobj)) {
        caps; "<<dobj.adesc>> doesn't belong there! ";
        }
        else {
        "You return <<dobj.adesc>> to <<self.thedesc>>";
        dobj.moveInto(self);
        }
    }
    ioPutOn(actor,dobj) = {
        if ((dobj <> self.targobj) and (dobj.objclass <> self.targobj)) {
        caps; "<<dobj.adesc>> doesn't belong there! ";
        }
        else {
        "You return <<dobj.adesc>> to <<self.thedesc>>";
        dobj.moveInto(self);
        }
    }
    doCount(actor) = {targobj.doCount(actor);}
;
class endgame_clone: item
    // where the object is to be created
    createloc = self.endgame_pile
    // class for methods to be passed up to
    pass_class = item
    // endgame_pile = black_rod_pile    - where it goes in the endgame
    multisdesc = {
        if (self.location = self.endgame_pile) {
           self.adesc;" from ";self.endgame_pile.thedesc;
        }
        else self.sdesc;
    }
    // put the right number of objects in the pile (maximum + 1)
    setup = {
        local o,p,i;
        for (i := 1; i <= self.endgame_pile.maxcount + 1; i++) {
            o := new self;
            if (self = wicker_cage) {
                p := new little_bird;
                p.moveInto(o);
                p.moved := nil; 
            }
        }
    }
    // anti-greed check
    ntaken = (self.endgame_pile.maxcount + 1 -
        self.loccountin(self.endgame_pile))
    doTake(actor) = {
        if(self.isIn(self.endgame_pile) and
        (self.ntaken >= self.endgame_pile.maxcount)) {
            "Don't be so greedy!  You've already
            taken <<ntaken>> <<self.pluraldesc>>.
            Please leave some for future adventurers. ";
        }
        else if (self.pass_class = item)
            inherited item.doTake(actor);
        else if (self.pass_class = container)
            inherited container.doTake(actor);
        else if (self.pass_class = liquidcont)
            inherited liquidcont.doTake(actor);
    }
    verDoInspect(actor) = {
        // kludge to stop excessive output if we 'examine rods' etc.
        if(self.isIn(self.endgame_pile) and not
        (self = self.list[1])) self.ldesc;
        else if (self.pass_class = item)
            inherited item.verDoInspect(actor);
        else if (self.pass_class = container)
            inherited container.verDoInspect(actor);
        else if (self.pass_class = liquidcont)
            inherited liquidcont.verDoInspect(actor);
    }
    verDoLookin(actor) = {
        if(self.isIn(self.endgame_pile) and not
        (self = self.list[1])) self.ldesc;
        else if (self.pass_class = item)
            inherited item.verDoLookin(actor);
        else if (self.pass_class = container)
            inherited container.verDoLookin(actor);
        else if (self.pass_class = liquidcont)
            inherited liquidcont.verDoLookin(actor);
    }
    verDoSearch(actor) = {
        if(self.isIn(self.endgame_pile) and not
        (self = self.list[1])) self.ldesc;
        else if (self.pass_class = item)
            inherited item.verDoSearch(actor);
        else if (self.pass_class = container)
            inherited container.verDoSearch(actor);
        else if (self.pass_class = liquidcont)
            inherited liquidcont.verDoSearch(actor);
    }
    doCount(actor) = {
        if (actor.isIn(At_Ne_End) or actor.isIn(At_Sw_End))
            "You lose count of the number
            of <<self.pluraldesc>> you can see around here! ";
        else if (self.pass_class = item)
            inherited item.doCount(actor);
        else if (self.pass_class = container)
            inherited container.doCount(actor);
        else if (self.pass_class = liquidcont)
            inherited liquidcont.doCount(actor);
    }
;
// To make sure that liquidcont properties are not overridden by plain
// item properties, we define the class endgame_liquidcont to be a subclass of 
// liquidcont.  The new 'inherited' grammar is used for a 'sideways'
// inheritance of all the relevant endgame_clone properties. 

class endgame_liquidcont: liquidcont
    // where the object is to be created
    createloc = self.endgame_pile
    pass_class = liquidcont
    // endgame_pile = black_rod_pile    - where it goes in the endgame
    multisdesc = {inherited endgame_clone.multisdesc;}
    setup = {inherited endgame_clone.setup;}
    ntaken = {return inherited endgame_clone.ntaken;}
    doTake(actor) = {inherited endgame_clone.doTake(actor);}
    verDoInspect(actor) = {inherited endgame_clone.verDoInspect(actor);}
    verDoLookin(actor) = {inherited endgame_clone.verDoLookin(actor);}
    doCount(actor) = {inherited endgame_clone.doCount(actor);}
;
/*
 * Endgame locations
 *
 * We make these NoNPC rooms so that dwarves and pirate don't get
 * teleported here when (if) they get stuck trying to move around
 * the cave.  (The dwarves in here aren't real actors because they
 * kill the player immediately if they're awake.)
 *
 * The major innovation in Polyadv is to allow users to take more than
 * one object - we are told of a 'bundle' of rods or a 'row' of lamps, so
 * we ought to be able to take more than one.   But the game puts a limit
 * on the number.
 *
 */
/* 115 */
At_Ne_End: room, NoNPC
    sdesc = "At NE End"
    ldesc = {
        I(); "You are at the northeast end of an immense
        room, even larger than the giant room.  It appears to
        be a repository for the \"Adventure\" program.
        Massive torches far overhead bathe the room with
        smoky yellow light.  Scattered about you can be seen
        a pile of bottles (all of them empty), a nursery of
        young beanstalks murmuring quietly, a bed of oysters,
        a bundle of black rods with rusty stars on their
        ends, and a collection of brass lanterns.  Off to one
        side a great many dwarves are sleeping on the floor,
        snoring loudly.  A sign nearby reads: \"Do not
        disturb the dwarves!\""; P();

        I(); "An immense mirror is hanging against one wall,
        and stretches to the other end of the room, where
        various other sundry objects can be glimpsed dimly in
        the distance. ";

        if(global.newgame) {
            "An unoccupied telephone booth stands against the north
            wall. ";
        }
    }

    hasfloor = true // DJP
    in =  {
        if(global.newgame) {
            Phone_Booth2.doEnter(global.travelActor);
            return nil;
        }
        else pass in;
    }
    north =  {
        if(global.newgame) {
            Phone_Booth2.doEnter(global.travelActor);
            return nil;
        }
        else pass north;
    }
    sw = At_Sw_End
    exithints = [Phone_Booth2, &in]
;
ne_sign: fixeditem, readable
    sdesc = "sign"
    ldesc = { self.readdesc; }
    readdesc = {
        "The sign reads, \"Do not
        disturb the dwarves!\"";
    }

    noun = 'sign'
    location = At_Ne_End
;
bottle_pile: objpile
    sdesc = "pile of empty bottles"
    ldesc = "It's a pile of empty bottles like the one you found
        in the building. "
    adjective = 'pile'
    noun = 'pile'
    plural = 'bottles'
    targobj = bottle
    location = At_Ne_End
;
oyster_bed: objpile
    sdesc = "bed of oysters"
    ldesc = "It's a bed of large oysters, similar to the one you found
        in the Shell Room.  "
    maxcount = 2
    adjective = 'bundle' 'star' 'black'
    noun = 'bed'
    plural = 'oysters'
    targobj = giant_bivalve
    location = At_Ne_End
;
black_rod_bundle: objpile
    sdesc = "bundle of rods"
    ldesc = "It's a bundle of black rods with stars on their ends, like
        the one you found in the Debris Room. "
    adjective = 'bundle' 'star' 'black'
    noun = 'bundle'
    plural = 'rods'
    targobj = black_rod
    location = At_Ne_End
;
lamp_collection: objpile
    sdesc = "collection of brass lanterns"
    ldesc = "It's a collection of brass lanterns like the one you
        used earlier. "
    adjective = 'collection' 'brass'
    noun = 'collection'
    plural = 'lamps' 'lanterns'
    targobj = brass_lantern
    location = At_Ne_End
;

Mirror_2: floatingdecoration
    sdesc = "enormous mirror"
    ldesc = "It looks like an ordinary, albeit enormous, mirror."
    noun = 'mirror'
    adjective = 'enormous' 'huge' 'big' 'large' 'suspended'
        'hanging' 'vanity' 'dwarvish'

    locationOK = true       // OK for location to be method
    loclist = [At_Ne_End, At_Sw_End]
    location = {
        local actor := getActor(&currentActor);
        if (actor.isIn(At_Ne_End))
            return At_Ne_End;
        else
            return At_Sw_End;
    }

    verDoBreak(actor) = {}
    doBreak(actor) = {
        "You strike the mirror a resounding blow, whereupon
        it shatters into a myriad tiny fragments.";

        //
        // A very bad move...
        //
        end_dwarves();
    }
    verDoAttack(actor) = { self.verDoBreak(actor); }
    doAttack(actor) = { self.doBreak(actor); }
    verDoAttackWith(actor, io) = { self.verDoAttack(actor); }
    doAttackWith(actor) = { self.doAttack(actor); }
    verDoKick(actor) = { self.verDoBreak(actor); }
    doKick(actor) = { self.doBreak(actor); }
    verDoLookin(actor) = {}
    doLookin(actor) = {
        "You look about the same as always. ";
    }
;
RepositoryStuff_1: floatingdecoration
    sdesc = "collection of adventure game materials"
    ldesc = {
        "You've seen everything in here already, albeit
        in somewhat different contexts.";
    }
    loclist = [At_Ne_End, At_Sw_End]
    noun = 'stuff' 'junk' 'materials' 'objects'
    adjective = 'adventure' 'repository' 'massive' 'sundry'

    // changed a little - we're allowed to go off with a few objects but
    // not steal ALL the stuff.
    verifyRemove(actor) = {
        "Realizing that the wholesale removal of the loot here would
        ruin the game for future players, you leave the
        \"Adventure\" materials where they are.";
    }
    doCount(actor) = {
        "It would take you all year to count the items. ";
    }
;
RepositoryTorches: floatingdecoration,distantItem
    sdesc = "flaming torches"
    ldesc = "The massive flaming torches are far above your head. "

    loclist = [At_Ne_End, At_Sw_End]
    plural = 'torches'
    adjective = 'flaming' 'massive'
    doCount(actor) = {
        "It would take you all day to count those. ";
    }
;
RepositoryDwarves: fixeditem
    sdesc = "sleeping dwarves"
    adesc = { self.sdesc; }
    ldesc = {
        "I wouldn't bother the dwarves if I were you.";
    }
    location = At_Ne_End
    adjective = 'sleeping' 'snoring' 'dozing' 'snoozing' 'dwarf' 'dwarves'

    verDoWake(actor) = {}
    doWake(actor) = {
        "You prod the nearest dwarf, who wakes up grumpily,
        takes one look at you, curses, and grabs for his
        axe.";

        end_dwarves();
    }
    verDoAttack(actor) = {}
    doAttack(actor) = { self.doWake(actor); }
    verDoKick(actor) = {}
    doKick(actor) = { self.doWake(actor); }
    doCount(actor) = {
        "You start to count the dwarves.  Unfortunately, one of them
        wakes up grumpily, curses and grabs for his axe. ";
        end_dwarves();
    }
;

RepositoryPlant: fixeditem
    sdesc = "beanstalks"
    ldesc = "They look like the beanstalk which you saw earlier, but
    these plants haven't grown enough to be climbable. "
    location = At_Ne_End
    noun = 'plant' 'plants' 'beanstalk' 'beanstalks' 'nursery'
    verDoClimb(actor) = {"Unless you can find some water somewhere, you
       won't be able to climb the beanstalks. ";}
    verDoWater(actor) = {"You have nothing with which to water the
       plants. ";}
    doCount(actor) = {
        "It would take you all day to count those. ";
    }
;

/* 116 */
At_Sw_End: room, NoNPC
    sdesc = "At SW End"
    ldesc = {
        // the wording re the pillows was changed slightly here to
        // fit in with the implementation which expects the objects
        // to be in piles, bundles etc.
        I(); "You are at the southwest end of the repository.
        To one side is a pit full of fierce green snakes. On
        the other side is a row of small wicker cages, each
        of which contains a little sulking bird.  In one
        corner is a bundle of black rods with rusty marks on
        their ends.  In another corner is a large pile of
        velvet pillows. ";
        "A vast mirror stretches
        off to the northeast. At ";"%your% feet is a large steel
        grate, next to which is a sign which reads,
        \"TREASURE VAULT. Keys in main office.\"";
    }

    hasfloor = true // DJP

    ne = At_Ne_End
    down = {
        local actor := getActor(&travelActor);
        RepositoryGrate.doEnter(actor);
        return nil;
    }

;
sw_sign: fixeditem, readable
    sdesc = "sign"
    ldesc = { self.readdesc; }
    readdesc = {
        "The sign reads, \"TREASURE VAULT. Keys in main office. \"";
    }

    noun = 'sign'
    location = At_Sw_End
;
wicker_cage_row: objpile
    sdesc = "row of wicker cages"
    ldesc = "It's a row of wicker cages, similar to the one you found
        earlier.  Each one contains a little sulking bird."
    adjective = 'row' 'wicker'
    noun = 'row'
    plural = 'cages'
    targobj = wicker_cage
    location = At_Sw_End
    // Only a bird with a cage can be returned to the row.
    ioPutIn(actor,dobj) = {
        if ((dobj = self.targobj) and (length(dobj.contents) = 0)) {
            caps; "I like to keep this area tidy.  That row is for
            birds in cages - not empty cages. ";
        }
        else pass ioPutIn;
    }
    ioPutOn(actor,dobj) = {
        if ((dobj = self.targobj) and (length(dobj.contents) = 0)) {
            caps; "I like to keep this area tidy.  That row is for
            birds in cages - not empty cages. ";
        }
        else pass ioPutOn;
    }
;

black_mark_rod_bundle: objpile
    sdesc = "bundle of rods"
    ldesc = "It's a bundle of black rods, similar to the one you found
        in the Debris Room but without a star on the end.  I'll now
        refer to these rods as \"marked rods\" and to the other type
        as \"star rods\". "
    adjective = 'bundle' 'marked' 'black'
    noun = 'bundle'
    plural = 'rods'
    targobj = black_mark_rod
    location = At_Sw_End
;

velvet_pillow_pile: objpile
    sdesc = "pile of velvet pillows"
    ldesc = "It's a pile of velvet pillows, identical to the one you
        found in the Soft Room. "
    adjective = 'pile' 'velvet'
    noun = 'pile'
    plural = 'pillows'
    targobj = velvet_pillow
    location = At_Sw_End
;

RepositoryGrate: fixeditem, lockableDoorway, picklock
// Cribbed from the Grate, but we can't open it because the keys are in
// the main office ...
    isopen = nil
    islocked = true
    sdesc = "steel grate"
    ldesc = {
        "It just looks like an ordinary steel grate.";

        " It is ";
        if (self.isopen)
            "open.";
        else if (self.islocked)
            "closed and locked.";
        else
            "closed.";
    }
    noun = 'grate' 'lock' 'gate' 'grille'
    adjective = 'metal' 'strong' 'steel' 'open' 'closed' 'locked'

    location = At_Sw_End

    mykey = office_keys
    doordest = nil      // The Treasure Vault is not a game location 
    verDoLookthru(actor) = {}
    doLookthru (actor) = {
        if(brass_lantern.loccountin(actor) = 0) {
            "You peer through the grate.  A ladder leads down into
            a dark room which appears to contain rows of cabinets. ";
        }
        else {
            "You shine a lamp through the grate.  A ladder leads down to
            a dark room containing rows of locked steel cabinets, bearing
            labels like \"Diamonds\" or \"Rare Coins\". ";  
        }
        "Unfortunately the keys are in the Main Office, so you can't get
        hold of any of the treasures right now.  ";
    }

;

Snakepit: fixeditem
    sdesc = "pit of snakes"
    ldesc = {
        "They're identical to the snake you saw in the Hall of the
        Mountain King - and just as dangerous. ";
    }
    location = At_Sw_End

    verifyRemove(actor) = {
        "You can't be serious.";
    }

    noun = 'pit' 'snakes'
    adjective = 'fierce' 'green' 'snake' 'pit'
    doCount(actor) = {
        "It would take you all day to count the snakes. ";
    }
    verDoEnter(actor) = {"You can't be serious!";}
    verDoBoard(actor) = {self.verDoEnter(actor);}
;

Phone_Booth2: CCR_boothitem
    game551 = true
    noun = 'booth'
    adjective = 'phone' 'telephone'
    inldesc = "%You% %are% standing in a telephone booth at the side of
        the Repository.  Hung on the wall is an old pay telephone of
        ancient design, like the one you found in the Rotunda but
        in much better condition. "
    outldesc = "It contains a pay telephone like the one you found in
           the Rotunda. "
    location = At_Ne_End
    hasfloordesc = true
    floordesc = {
        "It isn't quite as solid as the floor elsewhere in the
        Repository. ";
        if(not booby_catch.isIn(self)) {
            "You notice an odd-looking mechanism
            on the floor. ";
            booby_catch.moveInto(self);
        }
        if(not booby_wire.isIn(self)) {
            "A cable runs down from the telephone to the
            mechanism, but something tells you that it isn't just an
            ordinary telephone cable.  Something isn't right here! ";
            booby_wire.moveInto(self);
        }
    }
    mydoor = boothdoor2
    myphone = Phone2
;
Phony_Booth2: fixeditem,distantItem
    newgame = true
    noun = 'booth'
    adjective = 'phone' 'telephone'
    location = At_Sw_End
    sdesc = "phone booth"
    ldesc = "You can't examine it closely from here. "
    heredesc = {
        if (Phone2.isringing) {
            P(); I(); "You can hear the phone ringing. ";
        }
    }
;
boothdoor2: CCR_boothdoor  // door to booth
    newgame = true
    noun = 'door'
    adjective = 'booth'
    mybooth = Phone_Booth2
    location = At_Ne_End
;
Phone2: CCR_phoneitem
    newgame = true
    sdesc = "old payphone"
    adesc = {"an ";self.sdesc;}
    ldesc = {
        "It's an old payphone of the same design as the one you
        saw earlier in the Rotunda, but in much better condition.
        A telephone cable runs down from the phone to the floor. ";
        booby_wire.moveInto(Phone_Booth2);
    }
    location = Phone_Booth2
    isringing = nil
    noun = 'phone' 'telephone' 'payphone' 'receiver' 'handset'
    adjective = 'ancient' 'old'
    // If we fiddle with the phone, the trapdoor opens.
    takemethod(actor) = {
        end_phone();
    }
    // Don't do anything more after lifting receiver.
    answermethod(actor) = {}
    dialtonemethod(actor) = {}

    // If we attack the phone, it wakes the dwarves.
    doAttack(actor) = {
        "You've hit the jackpot!!  Hundreds of coins and slugs cascade from
        the telephone's coin return slot and spill all over the floor of
        the booth. ";
        end_dwarves();
    }
    // code for yanking the wire out of the phone
    ioYankOut(actor,dobj) = {
        if(dobj = booby_wire) {
            "You yank the cable out of the phone, and ... "; P();
            end_phone();
        }
        else pass ioYankOut;
    }
;

booby_wire: fixeditem
    newgame = true
    sdesc = "odd-looking cable"
    ldesc = {
        "It looks rather odd, and you're not sure it's just a normal
        telephone cable.  It runs down from the phone to a kind of
        catch mechanism on the floor. ";
        booby_catch.moveInto(Phone_Booth2);
    }
    noun = 'cable' 'wire'
    adjective = 'odd-looking' 'strange' 'unusual'
    verDoPull(actor) = {}
    verDoYank(actor) = {}
    verDoYankOut(actor,io) = {
        if(io = Phone2 or io = booby_catch) return;
        else pass verDoYankOut;
    }
    verDoBreak(actor) = {}
    doPull(actor) = {
        "You pull the cable out of the phone, and ... "; P();
        end_phone();
    }
    doYank(actor) = {
        "You yank the cable out of the phone, and ... "; P();
        end_phone();
    }
    doBreak(actor) = {
        "You manage to break the cable, and ... "; P();
        end_phone();
    }
;

booby_catch: fixeditem
    newgame = true
    sdesc = "mechanism"
    ldesc = "It's hard to say, but it appears to be some sort of catch
        mechanism holding the floor in place!  You have a very
        bad feeling about this.  I'd be very careful what you
        do here, because it appears that the dwarves have booby-trapped
        the booth! "
    noun = 'catch' 'mechanism'
    adjective = 'catch' 'odd-looking' 'strange'
    verDoKick(actor) = {}
    doKick(actor) = {
        "You give the mechanism a mighty kick, and ..."; P();
        end_phone();
    }
    verDoAttack(actor) = {}
    doAttack(actor) = {
        "You strike the mechanism with a resounding blow, and ..."; P();
        end_phone();
    }
    ioYankOut(actor,dobj) = {
        if(dobj = booby_wire) {
            "You yank the cable out of the mechanism, and ... "; P();
            end_phone();
        }
        else pass ioYankOut;
    }
;

Cylindrical: room, NoNPC
    sdesc = "Cylindrical Room"
    ldesc = { I();
        "You are in a small cylindrical room with very smooth
        walls and a flat floor and ceiling.  There are no
        exits visible anywhere.";
    }
    wordcount = 0
    roomAction(a,v,d,p,i) = {
        if(v = sayVerb) {
            "If you want to use a magic word, just type it in! ";
            exit;
        }
        if((isclass(v, MagicWord) 
#ifdef WIZARD
        or (v = outVerb)
#endif
        ) and wordtest(v)) exit;
    }
;   // word count is used in the endgame code below.

/* Resets words in the cylindrical room. */
resetwords: function
{   local i;
    for(i := firstobj(MagicWord); i<>nil; i:=nextobj(i, MagicWord)) {
        i.tused := -2;
        i.endsaid := nil;
    }
    Cylindrical.wordcount := 0;
}
/*
 *  wordtest(word);
 *  Whenever a magic word is said in the Cylindrical Room, the code
 *  checks the word's omegapsical_order.  If it is non-nil, it looks
 *  at the preceding word's endsaid and tused properties to see if the
 *  word was used in the proper sequence.  If so, it increments the
 *  wordcount value. Otherwise, it resets everything.  If the wordcount
 *  value is equal to the number of words in the game (global.numwords)
 *  then exit!  Otherwise, continue.  This will need alteration in some
 *  extensions of the 550-point version, such as the 580-point version.
 */
wordtest: function(word)
{
    local numprop,maxcount;
    // Decide which properties to use for the word count.
    // In the 701+ point mode, we use the 701+ numbering only if the
    // steel door was unlocked.
    if (global.game701p and Trans_Room_Door.isunlocked) {
        numprop := &omegaps701p_order;
        maxcount := &numactwords701p;
    }
    else if (global.game701) {
        numprop := &omegaps701_order;
        maxcount := &numactwords701;
    }
    else if (global.game580) {
        numprop := &omegaps580_order;
        maxcount := &numactwords580;
    }
    else {
        numprop := &omegapsical_order;
        maxcount := &numactwords;
    }

    if((word.(numprop) = nil) and not (word = outVerb)) {
        "(You don't need to use that word here - undoing one command.)\n";

        if (undo())
        {
            parserGetMe().location.lookAround(true);
            scoreStatus(global.score, global.turnsofar);
        }
        else
           "No more undo information is available. ";

        abort;
    }
    else if(word.(numprop) = 1) {
        resetwords();  // Starting over.
        word.endsaid := true;
        word.tused := global.turnsofar;
        Cylindrical.wordcount := 1;
        "Ok.\n";
    }
    // prevent optional words from being used more than once. 
    else if(word.endsaid) {
        "You've already used that word. You'd best start over. ";
    }       
    else if (word = outVerb) 
        Cylindrical.wordcount := global.(maxcount); 
    else { local i, prwd; // Find the previous word in order and store
                      // it in prwd.
        for(i:=firstobj(MagicWord); i<>nil; i:=nextobj(i, MagicWord)) {
            if((i.(numprop)=word.(numprop)-1) and (word.(numprop) > 0)) 
                prwd:=i;
            // cater for optional words, provided they have not been
            // used previously
            else if((i.(numprop) = -word.(numprop) -1) and
            (word.(numprop) < 0)) 
                prwd:=i;
        }
        if(prwd.endsaid and (prwd.tused = global.turnsofar-1)) {
            word.endsaid := true;
            word.tused := global.turnsofar;
            if(word.(numprop) > 0) Cylindrical.wordcount++;
            // if word.(numprop) is negative, the word is optional and we
            // pretend that the previous word was issued in this turn
            else prwd.tused := global.turnsofar;
            "Ok.\n";
        }
        else resetwords(); // skipped prwd, or waited too long.
    }
    if(Cylindrical.wordcount = global.(maxcount)) { 
        local i;
        global.dont_rescind := true; // don't rescind deposit points.
        // suppress all the extra moveInto code for efficiency
        global.extend_moveInto := nil;

        /* move every wrongly-deposited treasure into the Troll's treasure 
         * room, and move all other items in the surface areas to the
         * treasury (which is not a game location). */
        global.wrongtreasloc := 0;
        global.trolltolls := 0;
        for(i:=firstobj(item); i<>nil; i:=nextobj(i, item)) {
            /* look for treasure items which aren't correctly deposited
               (plus discharged pendants in non-nil locations other than
               the player) 
            */
            if ((isclass(i,CCR_treasure_item) and not 
            i.awardedpointsfordepositing) or (i=tarnished_pendant) or
            (i = dull_pendant)) {
                // exclude the charged pendants if the discharged pendants
                // exist
                if ((i = pendant) and (tarnished_pendant.location != nil))
                    continue;
                if ((i = pendant2) and (dull_pendant.location != nil))
                    continue; 
                // exclude the discharged pendants if they don't exist,
                // or if they're in the player's possession
                if ((i = tarnished_pendant) or (i = dull_pendant)) {
                    if ((i.location = nil) or i.isInside(parserGetMe()))
                        continue;
                }

                // exclude deleted treasures, or not-yet-found treasures
                // in the 701+ point extensions.
                if (i.deleted or (i.bonustreasure and not i.bonusfound))
                    continue;
                if (i.isInside(Troll_Treasure))
                    global.trolltolls++;
                else {
                    i.moveInto(Troll_Treasure);
                }
                global.wrongtreasloc++; 
            }
            if (i.isfixed) continue;
            if(isclass(i.location, Outside) or (i.location = Safe)) 
                i.moveInto(nil);
        }
        // turn on the moveInto extensions in case we still need them
        global.extend_moveInto := true;
        /* In 701-point game, move keys into pantry (red herring) */
        if(global.game701) {
            small_key.moveInto(Pantry);
            set_of_keys.moveInto(Pantry);
        }
        P(); I(); ">\>Foof!<\<"; P();
        parserGetMe().travelTo(At_End_Of_Road,nil,nil,true);
        global.dont_rescind := nil; // in case pendants are dropped

        incscore(global.escapepoints);
    }
    else if (Cylindrical.wordcount=0) "Nothing happens.\n";
    // This should only happen if something resets the count.

    return true; // Don't finish the normal action routine, since it
                 // will probably print "Nothing happens" again,
                 // or do something even more inappropriate.
}

/*
 *  Hurrah! The player has won!
 */
hurrah: function
{
    incscore(global.finalepoints);
    // Print endgame text.
    "You plunge into the stream and are carried down into total blackness.\n";
    P();
    "Deeper";
    "\n    \tand";
    "\n    \t\tdeeper";
    "\n    \t\t\tyou";
    "\n    \t\t\t\tgo,";
    "\n    \t\t\t\t\ \ down";
    "\n    \t\t\t\t\ \ \ into";
    "\n    \t\t\t\t\ \ \ the";
    "\n    \t\t\t\t\ \ very";
    "\n    \t\t\t\tbowels";
    "\n    \t\t\tof";
    "\n    \t\tthe";
    "\n    \tearth,";
    "\n until";
    "\n your";
    "\n     \tlungs";
    "\n     \t\tare";
    "\n     \t\t\taching";
    "\n     \t\t\t\twith";
    "\n     \t\t\t\t\ \ the";
    "\n     \t\t\t\t\tneed";
    "\n     \t\t\t\t\t\ \ for";
    "\n     \t\t\t\t\t\ \ \ fresh";
    "\n     \t\t\t\t\t\ \ \ air.";
    "\n     \t\t\t\t\t\ \ \ Suddenly,";
    "\n     \t\t\t\t\t\ \ with";
    "\n     \t\t\t\t\t\ a";
    "\n     \t\t\t\t\t\ \ violent";
    "\n     \t\t\t\t\t>splash!!<";
    P();
    "you find yourself sitting on the edge of a pool of water in
     a vast chamber lit by dozens of flaring torches. ";
    if (global.game701)
        "Despite your dramatic entrance, your presence is not immediately
        noticed - as if everyone is too preoccupied with what they are
        doing.  You take advantage of this to have a good look round. ";
    P();
    "The floor is covered with thick layers of precious Persian rugs!";
    P();
    if(global.game701)
        "Rare coins, bars of silver, lumps of gold and platinum and gold
        rings are strewn carelessly about!";
    else
        "Rare coins, bars of silver, and lumps of gold and platinum are
        strewn carelessly about!";
    P();
    "There are diamonds, rubies, sapphires, emeralds, opals, pearls, and
    fabulous sculptures and ornaments carved out of jade and imperishable
    crystal resting on display shelves, along with rare Ming vases and
    ancient Indian turquoise beads!";
    if(global.game580) {
        P();
        "Sitting on one display shelf are a collection of rare stamps, and 
         a disk labelled \"Adventure Source Code\"!";
    }
    if (global.game701) {
        P();
        "A large pile of crystal balls is stacked against one wall.  You
        look into a few of them, fascinated by the scenes they show.  In
        one, you see the Troll's treasure chamber, lit by an orange glow
        and stacked with priceless items of all descriptions";
        if (global.wrongtreasloc > 0) {
            ": "; P();
            "\("; Troll_Treasure.sdesc; "\)";
            Troll_Treasure.nrmLkAround(true);
            P();
        }
        else
           ". ";
        if (global.wrongtreasloc = 1)
            "(You notice that the treasure which
            you failed to deposit correctly has gone there instead of
            here, losing you points.";
        else if (global.wrongtreasloc > 1)
            "(You notice that the <<global.wrongtreasloc>> treasures which
            you failed to deposit correctly have all gone there instead of
            here, losing you points.";
        if (global.wrongtreasloc > 0) {
            local totalspend := global.trolltolls + global.vendingtreasures;
            if (totalspend = 1) {
                if(global.trolltolls)
                    " This includes the treasure you paid to the Troll.) ";
                else
                    " This includes the coins you used in the Vending 
                    Machine.) ";
            }
            else if (totalspend > 1) {
                " This includes the <<totalspend>> treasures which were used
                for payments. ";
            }
            else
                ") ";
        }
        "Another ball shows a strange ruined city, illuminated
        by three moons and the golden glow of an aurora which fills the 
        entire sky!  A third ball shows a large circular room, lit by a
        dim glow - and full of sleeping Wumpi. ";
    }
    P();
    "A flotilla of ruby-encrusted toy boats is floating in the pool of
    water beside you!";
    P();
    "A network of golden chains supports a fantastic Iridium crown!";
    P();
    "There is a display case on the wall filled with a fantastic selection
    of magical swords, which are singing \"Hail to the Chief\" in perfect
    pitch and rhythm!";
    if (global.game701) {
        P();
        "A second case contains a collection of about ten elven swords, all
        gleaming like the one you took from the anvil!
        A third, much larger case holds over 130 elven crowns, all
        made of gold or mithril silver and encrusted with valuable 
        stones of all descriptions!  ";
        if (global.game701p) {
            "A plaque, affixed to the front of the cabinet, states that 114
            of the crowns were found in a recently-discovered burial vault.
            A further two vaults are believed to exist, but have not yet
            been excavated. ";
            P();
        }
        "The Mountain King is searching the case, muttering
        to himself.  You hear him say: \"I hope he realizes that the
        treasures go to the Troll when they're not left in a 'safe' 
        place.\"  He then grabs one of the crystal balls and looks into it. ";
        if (crown.awardedpointsfordepositing)
            "\"Hmmm.  I can't see it in the Troll's treasure chamber.\"  He
            looks in the case again and says: \"Ah yes! Here it is! \" He 
            retrieves his rightful property from the case, puts it on and 
            starts adjusting it in front of a mirror.  \"Hmmm.  A little 
            further to the left, perhaps ... \" ";
        else
            "\"Oh no! The Troll has it.  No use asking for it - he'd want
            ten crowns in return.\"  He then starts trying on different 
            crowns.  \"Hmmm.  Too small.  No, too large ... \"  ";

        P();
        "Off to one side there is a large closet containing silken cloaks,
        ruby slippers, and many other priceless articles of clothing! ";
    }
    P();
    "There are a dozen friendly little dwarves in the room, displaying
    their talents by deftly juggling hundreds of golden eggs!";
    P();
    "A large troll, a gigantic ogre, and a bearded pirate are tossing
    knives, axes, and clubs back and forth in a friendly demonstration
    of martial skill!";
    P();
    "A horde of cheerful little gooseberry goblins are performing
    talented acrobatics to an appreciative audience composed of a dragon,
    a large green snake, a cute little bird (which is sitting, unmolested,
    on the snake's head), a peaceful basilisk, and a large Arabian Djinn.";
    if(global.game701) {
        // Deleted reference to the Wumpus - they don't belong at Red level.
        P();
        "A Gnome is playing ancient and valuable musical
        instruments, filling the air with beautiful music.  Nearby a large
        black dog is sleeping peacefully. ";
    }
    P();
    "Everyone turns and sees you, and lets out a heart-warming cheer
    of welcome! ";
    if(pendantItem.classcount(Me) > 0) {
        local pendantword, pendantpronoun;
        if(pendantItem.classcount(Me) = 1) {
            pendantword := 'pendant';
            pendantpronoun := 'it';
        }
        else {
            pendantword := 'pendants';
            pendantpronoun := 'them';
        }
        "Then the Mountain King spots your Transindection <<pendantword>>,
        and shouts \"Where did you find the <<pendantword>>?\"  You stutter
        and stammer, but then manage to regain your composure and tell the King
        what you found. ";
    }
    else if ((global.game701p and Trans_Room_Door.isunlocked) and 
    (pendantItem.classcount(Me) = 0)) {
        "You see a flicker of consternation on the face of the Mountain
        King, and once again he looks into the crystal ball to spy upon the
        Troll's treasure chamber.  He turns to the Troll and shouts \"How did 
        you come by the Transindection pendants? \"  The Troll points to 
        you, and the King seems to know exactly what he means.  He asks you, 
        more quietly this time:  \"Where did you find the pendants?\"  You 
        tell him everything. ";
    }
    if (global.game701p and Trans_Room_Door.isunlocked) {
        "When he learns that the Upper Transindection 
        Chambers were working, he looks very worried indeed";
        if(Green_Upper_Trans_Room.isdotroom)
            if(Green_Maintenance_Room.isseen)
                ", but when you tell him how you used the Topaz to enter the
                Maintenance Rooms, he congratulates you and says
                \"Well done - Eldrand may have failed, but we can now complete
                his mission in complete safety!\" ";
            // added because it is now possible to get here without
            // entering the Maintenance Room.  However, the means to do so
            // has been obtained...
            else
                ", but when you tell him how you got hold of the 
                Eldrand-Fitzgerald Topaz, he congratulates you and says
                \"Well done!  I'm almost certain that we can get into the
                Maintenance rooms now.  We'll be able to complete
                Eldrand's mission in complete safety!\" ";
        else
            ", but when you 
            reassure him that you have disabled the Chambers, he congratulates 
            you! ";
        "Everyone lets out a loud \"Hurrah!\" ";

        P();

        "The rest of your day is occupied with discreet Transindection
        tours, conducted with the aid of your pendants and gold rings
        worn by your companions.  Rods are spun near various plaques,
        the Zarkalon tower is visited and the transmutation of lead to 
        platinum is demonstrated.  Blue-level display boards are read, and
        plans are made for the excavation of the remaining burial chambers at 
        Red level. But that's the start of a much longer story ... ";
    }
    P();
    win();
}

/*
 *  Horror! The player disturbed the Wumpi but failed to disable the Upper 
 *  Transindection Chambers, with disastrous consequences ...
 */
horror: function
{
    // Print endgame text.
    "You plunge into the stream and are carried down into total blackness.\n";
    P();
    "Deeper";
    "\n    \tand";
    "\n    \t\tdeeper";
    "\n    \t\t\tyou";
    "\n    \t\t\t\tgo,";
    "\n    \t\t\t\t\ \ down";
    "\n    \t\t\t\t\ \ \ into";
    "\n    \t\t\t\t\ \ \ the";
    "\n    \t\t\t\t\ \ very";
    "\n    \t\t\t\tbowels";
    "\n    \t\t\tof";
    "\n    \t\tthe";
    "\n    \tearth,";
    "\n until";
    "\n your";
    "\n     \tlungs";
    "\n     \t\tare";
    "\n     \t\t\taching";
    "\n     \t\t\t\twith";
    "\n     \t\t\t\t\ \ the";
    "\n     \t\t\t\t\tneed";
    "\n     \t\t\t\t\t\ \ for";
    "\n     \t\t\t\t\t\ \ \ fresh";
    "\n     \t\t\t\t\t\ \ \ air.";
    "\n     \t\t\t\t\t\ \ \ Suddenly,";
    "\n     \t\t\t\t\t\ \ with";
    "\n     \t\t\t\t\t\ a";
    "\n     \t\t\t\t\t\ \ violent";
    "\n     \t\t\t\t\t>splash!!<";

    P();
    "you find yourself sitting on the edge of a pool of water in
     a vast chamber lit by dozens of flaring torches.";

    P();
    "But something is wrong!  A battle is raging around you.  On one 
    side you see a poorly armed band of dwarves, elves and Gnomes.  On the 
    other side there is a large army of elves, all wearing uniforms made from a
    silvery fabric.  Your attention is also drawn to the gleaming swords which 
    they are all wielding, to the crowns atop their heads, and to the gold 
    rings which many of them are wearing on their fingers.  It is also clear 
    that all of them are winning easily - apparently without any 
    casualties on their side!  Axes, knives and swords either bounce 
    harmlessly off the uniforms, or veer away from the elves as if repelled
    by a powerful force-field. ";

    P();
    "You try to hide, but it is to no avail.  A large elf strikes you down
    with his sword, and you fall to the ground, mortally wounded. ";

    P();
    "As you lie dying, the remnants of the poorly armed band flee in terror.
    The elf comes up to you, and 
    exclaims, \"A Cavernizer!  Well, the cave is now outbounded 
    from ... \"  He frowns, mutters something about language drift, then puts 
    a red ring on his finger. ";

    P();
    "\"A Spelunker!  Well, the cave is now out of bounds for humans like you.
    But it seems that we have to thank someone like you for showing us the
    way in. \" He observes your reaction";
    if(pendantItem.classcount(Me) = 1)
    " and notices your pendant. ";
    else if(pendantItem.classcount(Me) > 1)
    " and notices your pendants. ";
    else ". ";
    "\"I do believe that \(you\) were our intruder!  Well, you caused us all a 
    lot of trouble, but we were very interested to find out how the Wumpi
    managed to come back from Red level.  So interested, in fact, that we
    paid them to clear a path to the Upper Transindection Chamber.  And the 
    rest is history.  Welcome to the New Order!\" ";
    P();
    "Before you lose consciousness, your mind is full of \"if only\"s. ";
    P();
    "If only ... I'd managed to complete at least part of Eldrand's mission. 
    Disabling the Upper Transindection Chambers would have been enough to 
    thwart the Elves. ";
    //N.B. the following code sections are now accessible again, due to the
    //provision of a second battery pack at White level.  It is no longer
    //necessary for the maintenance rooms to be visited for this purpose.
    if(not manual.moved) {
        P();
        "If only ... I'd searched the Control Room more thoroughly.  I'm sure 
        I could have found something useful there. ";
    }
    else if(not manual.isread) {
        P();
        "If only ... I'd read the Control Room manual. ";
    }
    else if(not Blue_Maintenance_Room.isseen) {
        P();
        "If only ... I'd found a way to use that Topaz to get into the
        Maintenance Rooms.  The damage looked very slight. ";
    }
    P();
    "But \"if only\"s won't do.  Your foolish meddling has completely wrecked 
    the balance of Colossal Cave.  The glass ball at Green level tried to warn
    you, but you did nothing about it.  Now the Green-level elves have taken 
    over one Level already, and others will likely fall in due course!   
    There's nothing we can do about it now, and no hope that Adventurers will 
    ever again be allowed into the cave.  So I'm rescinding all your game 
    points!  Get out! Out!!!!!  OUT!!!!!!!!!!!  
    OUT!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!";
    P();
    global.score := 0;
    win();
}
