#charset "us-ascii"
#include <adv3.h>
#include <en_us.h>

// Achievement.t
// by Greg Boettcher
// v1.0

// -----------------------------------------------------------------------------
// Table of Contents
// -----------------------------------------------------------------------------

// Documentation: Version history
// Documentation: Files included with this extension
// Documentation: Steps for using this extension
// Documentation: About this extension
// ModuleID info
// Classes: Achievement-scope objects: ASObject
// Classes: Achievement-scope objects: ASVariable
// Classes: Achievement-scope objects: ASInteger
// Classes: Achievement-scope objects: ASBoolean
// Classes: Achievement-scope objects: ASString
// Classes: Achievement-scope objects: ASList
// Classes: Achievement-scope objects: ASIntegerList
// Classes: Achievement-scope objects: ASBooleanList
// Classes: Achievement-scope objects: ASStringList
// Classes: Achievement-scope objects: ASAchievement
// Classes: Data validators: ASValidator
// Classes: Data validators: ASIntegerValidator
// Classes: Data validators: ASBooleanValidator
// Classes: Data validators: ASStringValidator
// Enums
// Utility objects: achievementScopeManager
// Utility objects: achievementScopeTranslator
// Utility objects: achievementScopeEncrypter
// Utility objects: achievementScopeFileManager
// Utility objects: Convenience class: AchievementScopeFileItem
// Utility objects: achievementScopePreinitObject
// Utility objects: achievementScopeUniquenessGuarantor
// Utility objects: achievementScopeDaemon
// Utility objects: rememberedThisPlaythough / rememberedThisSession
// Style tags
// Utility objects: achievementMessages
// Finish option: finishOptionAchievements
// Actions: Achievements
// Actions: Achievements On
// Actions: Achievements Off
// Actions: Achievements Notify

// -----------------------------------------------------------------------------
// Documentation: Version history
// -----------------------------------------------------------------------------

// v0.10 : 2019-04-?? Achievement-scope variables fully implemented.
//         Data successfully persists when you quit and reopen the game.
// v0.20 : 2019-05-03 Achievements now fully implemented. As part of this, 
//         achievement-scope objects have been reworked to allow more easily 
//         for more than one saveable property per object. 
// v0.21 : 2019-05-20 Minor changes made during development of Nothing but Mazes 
// v0.30 : 2019-05-22 Modified so as to no longer require dynfunc.t and reflect.t
// v1.0  : 2019-08-12 A few changes made per Nothing but Mazes beta-testing.

// -----------------------------------------------------------------------------
// Documentation: Files included with this extension
// -----------------------------------------------------------------------------

// Files included with this extension:
//
// achievement.t     : Includes all the code you need to start using achievements 
//                     and achievement-scope variables. Requires reflect.t and
//                     dynfunc.t. 
// achievementdemo.t : Demo with debugging verbs demonstrating the usage of
//                     achievement.t. Requires achievement.t. 

// -----------------------------------------------------------------------------
// Documentation: Steps for using this extension
// -----------------------------------------------------------------------------

//  1. Put achievement.t into the source files of your TADS 3 game. 
//  2. Consider modifying achievementScopeFileManager to override achievementFilename,
//     as is done by achievementdemo.t. Doing so prevents problems in the unlikely 
//     event that someone puts your .t3 game in the same folder as another .t3 game 
//     that also uses achievement.t. 
//  3. Define achievements. In doing so, consult the examples in achievementdemo.t,
//     section "Achievement examples". When you add ASAchievement objects
//     to your game, **they must be transient objects**, or achievement.t will 
//     not work correctly. 
//  4. If desired, add any achievement-scope variable objects. These can be 
//     helpful for, e.g., storing progress towards not-yet-attained achievements.
//     See achLoserCount or "Examples of objects whose properties we can manipulate"
//     in achievementdemo.t. Again, achievement-scope variables **must be transient
//     objects** for them to work as you expect. 
//  5. Consult achievementdemo.t for examples on how to achieve() and reveal()
//     achievements. 
//  6. Have fun!

// -----------------------------------------------------------------------------
// Documentation: About this extension
// -----------------------------------------------------------------------------

// This extension has two purposes:
// (1) To make it easy to save information that is remembered for the scope 
//     of not just a single playthrough, but ALL of a player's playthroughs.
// (2) As a special case, make it easy to remember Steam- or Xbox-style 
//     achievements and report them to the player.

// Traditionally there have been two "scopes" of saved information in TADS 3, 
// but this extension provides a third.
// TADS 3 provides:
// (1) "Playthrough scope": Nearly all variables are saved to this scope. 
//     This is the scope of information saved in standard *.t3v save game files. 
// (2) "Session scope": For information remembered for as long as the interpreter 
//     is open, irrespective of restoring, undoing, etc., but forgotten when 
//     the player closes the interpreter. This is the scope of TADS 3 
//     transient objects. For example, if you type HINTS OFF, the information 
//     is saved to the transient object sessionHintStatus. (See TADS 3 
//     System Manual, Object Definitions.)
// This extension provides:
// (3) "Achievement scope": A scope encompassing ALL of a player's playthroughs: 
//     Achievement.t lets you record information in this scope, saving it 
//     to a file. This is the traditional scope for achievements such as 
//     those on Steam, Xbox, Playstation, etc.

// Caveats:
// (1) This extension makes no effort to distinguish different players who are 
//     using the same installed instance of the same game. There is just one
//     achievement save file per installed instance.
// (2) This extension can't exactly stop players from deleting the achievement 
//     save file. If they do so, their achievement-scope data will be lost. 
// (3) I have done no research on how to integrate with the way achievements
//     actually work on Steam or any other digital distribution platform.
//     If you do such research and wish to suggest changes based on such,
//     feel free to contact me. 

// How to contact me:
// Feel free to send bugs/suggestions to paradoxgreg@X.com, where X = gmail. 

// This extension is freeware. If you use it, I would appreciate attribution, 
// although I'm not going to insist on it. By the same token, this comes with 
// no warranty.

// Final note:
// Some players have grown sick of achievements as an obligatory game feature 
// in almost every console game. I sympathize with that, and personally 
// I wouldn't want to see achievements become as universal as all that 
// in interactive fiction. However, if you've spent a lot of time developing 
// game features your players might miss, and if you'd like to prod them to 
// explore those features, that is when achievements may have some value. 

// Brief description for IF Archive:
// For authors who want to implement achievements in the style of Steam or 
// console games. Also provides a way to store information that persists 
// across *all* of a player's playthroughs.

// -----------------------------------------------------------------------------
// ModuleID info
// -----------------------------------------------------------------------------

ModuleID
  name = 'Achievement.t'
  byLine = 'by Greg Boettcher'
  htmlByLine = 'by Greg Boettcher'
  version = '0.21'
  //showCredit { "<<name>> <<htmlByLine>>"; }
;

// -----------------------------------------------------------------------------
// Classes: Achievement-scope objects: ASObject
// -----------------------------------------------------------------------------

// ASObject: The top-level class to which all achievement-scope objects,
// whose data is saved to achievement-scope files, belong. You won't normally 
// want to create objects that directly inherit from this class. Instead, use 
// a subclass such as ASAchievement, ASVariable, or a subclass of ASVariable. 

class ASObject: object
  // asName: Your achievement-scope objects must override this with a *unique*
  // name that can be used to identify them. See examples in achievementscopedemo.t,
  // section "Achievement examples" and section "Examples of objects whose 
  // properties we can manipulate". I like the convention of setting asName to be
  // the same as the name of the object, but that's not required; what's required
  // is uniqueness.
  asName = nil
  // asPropertyInfo: a list of properties we'll look at when it's time to save
  // achievement-scope values to disk. Subclasses should override with a list of 
  // lists, one list for each property you want to to be saved. 
  // Regarding the format of the inner list(s), see further details below under 
  // ASVariable.asPropertyInfo.
  asPropertyInfo = []
  // If your achievement-scope objects are not given a unique value for asName,
  // they will not be saved. 
  isSaveable() {
    return (achievementScopeUniquenessGuarantor.asNameIsUnique(asName));
  }
  // set(prop, val):
  // Important: Do not set the value of achievement-scope variables DIRECTLY 
  // at runtime if you want this module to work. Instead, you should call an
  // appropriate setter method, such as this one, or better yet, one of the setter methods
  // implemented by subclasses of ASObject, such as:
  // ASVariable.setValue()
  // ASAchievement.achieve()
  // ASAchievement.reveal()
  // Normally you'll want to use those subclasses' setter methods, rather than this 
  // one directly. (Also note: it's okay to override properties as a way of setting a 
  // different *initial* value at *compile-time*; it is just changing values at runtime
  // that needs to be done in a more careful way.)
  set(prop, val) {
    // Temporarily remember the property's original value:
    local origVal = self.(prop);
    
    // Get the property's validator object
    local validatorObj = getValidatorObject(prop);
    
    // Get the properly typed value using the validator object. If "val" can't be 
    // converted to a properly typed value, give up and exit. 
    local convertedVal = validatorObj.convertToValidValue(val);
    if (convertedVal == achNotConvertible) {
      if (validatorObj.badDataTypeBehavior == achThrowException) {
        throw new Exception('The value ' + toString(val) + ' can\'t be 
          converted as required for the achievement-scope object. ');
      }
      return;
    }
    
    // Change the property.
    self.(prop) = convertedVal;
    
    // If we have, in fact, changed the value of anything, then set a flag
    // saying we should write to disk. 
    if (convertedVal != origVal) {
      achievementScopeManager.scheduleSaveData();
    }
  }
  // get(prop): 
  // provided for symmetry with set(prop, val), although in truth there's no reason
  // you can't just directly retrieve the value without using this getter method.
  get(prop) {
    return self.(prop);
  }
  // getForDisplay():
  // This method is not used by anything in achievement.t, but you might find it useful.
  // E.g., returns strings as text that the user can see onscreen.
  getForDisplay(prop) {
    local validatorObj = getValidatorObject(prop);
    return validatorObj.getForDisplay(get(prop));
  }
  // getValueForDebugging():
  // This method is not used by anything in achievement.t, but is used by 
  // achievementsimpledemo.t, and you might find it useful.
  // E.g., returns strings as text surrounded by single quotes.
  getForDebugging(prop) {
    local validatorObj = getValidatorObject(prop);
    return validatorObj.getForDebugging(get(prop));
  }
  // getValidatorObject
  getValidatorObject(prop) {
    // Get the property's validator object
    local validatorObj = asValidatorsPerProperty[prop];
    if (validatorObj == nil) {
      throw new Exception('No validator found for achievement-scope property ' + prop);
    }
    return validatorObj;
  }
  
  // =================================
  // NOTE: The following properties should **NOT** be overridden either by subclasses 
  // or by objects. Instead, if you need to, override asPropertyInfo, and 
  // achievementScopePreinitObject.extractASPropertyInfo will automatically
  // populate all of these.
  // These properties, which you must **NOT** change, are:
  // - asProperties
  // - asValidatorsPerProperty
  // - asPropertiesPerName
  // - asNamesPerProperty
  
  // asProperties: a list of the achievement-scope properties
  asProperties = []
  // asPropertiesPerName: a lookup table of properties per property name
  asPropertiesPerName = new LookupTable()
  // asNamesPerProperty: a lookup table of property names per property
  asNamesPerProperty = new LookupTable()
  // asValidatorsPerProperty: a lookup table of validator objects per property
  asValidatorsPerProperty = new LookupTable()
;

// -----------------------------------------------------------------------------
// Classes: Achievement-scope objects: ASVariable
// -----------------------------------------------------------------------------

class ASVariable: ASObject
  // asPropertyInfo: a list of properties we'll look at when it's time to save
  // achievement-scope values to disk. This and other subclasses of ASObject 
  // should override with a list of lists, one list for each property you want 
  // to be saved. 
  // Each "inner list" must have three required values:
  // (1) the property to be saved (e.g. &value). 
  // (2) a single-quoted string value representing the name of the property 
  //     to be saved (e.g. 'value'). 
  // (3) an object belonging to class ASValidator or a subclass thereof, 
  //     allowing it to serve as a way to do any necessary strong typing of data for the property.
  //     If no strong typing is desired, just use new ASValidator().
  asPropertyInfo = [
    [ &value, 'value', new ASValidator() ]
  ]
  // value: the value of the variable. Objects and subclasses can override this
  // with a different initial value, but thereafter DO NOT CHANGE THIS DIRECTLY
  // or this code module will not work. Instead, use setValue() to make any changes.
  // DO NOT 
  value = nil
  // setValue(): 
  // Important: Do not set the value of achievement-scope variables DIRECTLY 
  // at runtime if you want this module to work. Instead, call this method.
  // (Note: it's okay to override properties as a way of setting a 
  // different *initial* value at *compile-time*; it is just changing values at runtime
  // that needs to be done in a more careful way.)
  setValue(val) {
    set(&value, val);
  }
  // getValue():
  // Provided for symmetry with set(prop, val), although in truth there's no reason
  // you can't just directly retrieve the value without using this getter method.
  getValue() {
    return get(&value);
  }
  // getValueForDisplay():
  // This method is not used by anything in achievement.t, but you might find it useful.
  // E.g., returns strings as text.
  getValueForDisplay() {
    return getForDisplay(&value);
  }
  // getValueForDebugging():
  // This method is not used by anything in achievement.t, but is used by 
  // achievementsimpledemo.t, and you might find it useful.
  // E.g., returns strings as text surrounded by single quotes.
  getValueForDebugging() {
    return getForDebugging(&value);
  }
;

// -----------------------------------------------------------------------------
// Classes: Achievement-scope objects: ASInteger
// -----------------------------------------------------------------------------

class ASInteger: ASVariable
  value = 0
  asPropertyInfo = [
    [ &value, 'value', new ASIntegerValidator() ]
  ]
;

// -----------------------------------------------------------------------------
// Classes: Achievement-scope objects: ASBoolean
// -----------------------------------------------------------------------------

class ASBoolean: ASVariable
  value = nil
  asPropertyInfo = [
    [ &value, 'value', new ASBooleanValidator() ]
  ]
;

// -----------------------------------------------------------------------------
// Classes: Achievement-scope objects: ASString
// -----------------------------------------------------------------------------

class ASString: ASVariable
  value = ''
  asPropertyInfo = [
    [ &value, 'value', new ASStringValidator() ]
  ]
;

// -----------------------------------------------------------------------------
// Classes: Achievement-scope objects: ASList
// -----------------------------------------------------------------------------

class ASList: ASVariable
  // =============================
  // Methods/properties to override:
  // Start with an empty list
  value = []
  // Use a non-strongly-typed validator:
  asPropertyInfo = [
    [ &value, 'value', new ASValidator() ]
  ]
  // How to display a list by default for the user? Here's one implementation.
  // Unfortunately, it'll be an empty string if it's an empty list...
  getValueForDisplay() {
    local str = '';
    for (local i = 1; i <= value.length; i++) {
      if (i > 1 && value.length >= 3) { str += ', '; }
      if (i == value.length - 1) { str += ' and '; }
      str += ASValidator.getForDisplay(value[i]);
    }
    return str;
  }
  // getValueForDebugging():
  // This method is not used by anything in achievement.t, but is used by 
  // achievementsimpledemo.t, and you might find it useful.
  // E.g., returns strings as text surrounded by single quotes.
  getValueForDebugging() {
    local str = '[';
    for (local i = 1; i <= value.length; i++) {
      if (i > 1) { str += ', '; }
      str += ASValidator.getForDebugging(value[i]);
    }
    str += ']';
    return str;
  }
  // setValue() has to be overridden here to validate input differently
  setValue(val) {
    // I could implement this. If I did, I'd have to make sure the input value
    // was a list and that all its members were valid and properly converted 
    // per convertToValidValue(). Doesn't seem worth it to me at this point.
    // If you need to replace a list, clear it and add the values one at a time.
    throw new Exception('ASList.setValue() not implemented; 
      use clearList() and addValue() instead. ');
  }
  
  // =============================
  // Clearing the list
  clearList() {
    local origValueLength = value.length;
    
    value = [];
    
    // Assuming a change was made, store to file
    if (origValueLength != 0) {
      achievementScopeManager.scheduleSaveData();
    }
  }
  
  // =============================
  // Adding to the list
  // addValue: add a value to a list. first argument should be the value to add.
  // second value (optional) can be true (to override list default and insist upon 
  // allowing duplicate values) or nil (to override list default and insist upon 
  // preventing duplicates). 
  addValue(val, [args]) {
    local allowDupes = allowDuplicates;
    if (args && args.length > 0 && 
        (dataType(args[1]) == TypeNil || dataType(args[1]) == TypeTrue))
    {
      allowDupes = args[1];
    }
    
    // For the sake of subclasses that may require that data be converted before
    // being stored, we first convert the value. If the value can't be converted, 
    // either throw an error or fail silently, per this class's configuration
    local validatorObj = getValidatorObject(&value);
    local convertedVal = validatorObj.convertToValidValue(val);
    if (convertedVal == achNotConvertible) {
      if (badDataTypeBehavior == achThrowException) {
        throw new Exception('The value ' + toString(val) + 
          ' can\'t be converted as required. ');
      }
      return;
    }
    if (value.indexOf(convertedVal) && !allowDupes) {
      if (illegalDuplicateBehavior == achThrowException) {
        throw new Exception('The value ' + toString(convertedVal) + 
          ' is already contained in the achievement-scope list. ');
      }
      return;
    }
    value = value + convertedVal;
    
    // If we reach this point, we've definitely changed the list, so set a flag
    // for writing to disk. 
    achievementScopeManager.scheduleSaveData();
  }
  // allowDuplicates:
  // For a given list, it either always allows duplicates (true) or never allows
  // duplicates (nil). Feel free to override this. 
  allowDuplicates = true
  // illegalDuplicateBehavior should be either achThrowException or achFailSilently
  illegalDuplicateBehavior = achFailSilently
  
  // =============================
  // Removing from the list
  // removeValue: add a value to a list. first argument should be the value to remove.
  // second value (optional) can be true if you want to indicate that we should remove
  // ALL instances of the value. 
  removeValue(val, [args]) {
    local removeAllInstances = nil;
    if (args && args.length > 0 && dataType(args[1]) == TypeTrue) {
      removeAllInstances = true;
    }
    
    local validatorObj = getValidatorObject(&value);
    local convertedVal = validatorObj.convertToValidValue(val);
    if (convertedVal == achNotConvertible) {
      if (badDataTypeBehavior == achThrowException) {
        throw new Exception('The value ' + toString(val) + 
          ' can\'t be converted as required. ');
      }
      return;
    }
    
    local didSomething = nil;
    local complete = nil;
    while (!complete) {
      for (local i = 1; i <= value.length; i++) {
        if (value[i] == convertedVal) {
          value = value.removeElementAt(i);
          didSomething = true;
          if (removeAllInstances) { complete = nil; }
          else { complete = true; }
          break;
        }
        if (i == value.length) {
          complete = true;
        }
      }
    }
    if (didSomething) {
      // Store to file
      achievementScopeManager.scheduleSaveData();
    }
  }
;

// -----------------------------------------------------------------------------
// Classes: Achievement-scope objects: ASIntegerList
// -----------------------------------------------------------------------------

class ASIntegerList: ASList
  asPropertyInfo = [
    [ &value, 'value', new ASIntegerValidator() ]
  ]
;

// -----------------------------------------------------------------------------
// Classes: Achievement-scope objects: ASBooleanList
// -----------------------------------------------------------------------------

class ASBooleanList: ASList
  asPropertyInfo = [
    [ &value, 'value', new ASBooleanValidator() ]
  ]
;

// -----------------------------------------------------------------------------
// Classes: Achievement-scope objects: ASStringList
// -----------------------------------------------------------------------------

class ASStringList: ASList
  asPropertyInfo = [
    [ &value, 'value', new ASStringValidator() ]
  ]
;

// -----------------------------------------------------------------------------
// Classes: Achievement-scope objects: ASAchievement
// -----------------------------------------------------------------------------

// ASAchievement:
// Unfortunately, this class can't just be called Achievement, because that is
// the name of a class in the TADS 3 adventure library. 

class ASAchievement: ASObject
  // Achievements need the following three properties, but they are not expected 
  // to change and do not need to be achievement-scope properties.
  
  // Don't forget, you need an asName property for ASAchievements as for 
  // any ASObjects. 
  
  // name: the name of the achievement, by default shown in italics. 
  // You must override this. 
  name = '' 
  // desc: the achievement's description, which follows after the achievement's name
  // when the player types ACHIEVEMENTS. You must override this.
  desc = '' 
  // listOrder: override this to control the order in which achievements are listed.
  listOrder = 99999
  
  // ASAchievements need the following three achievement-scope properties:
  achieved = nil
  achievementNotified = nil
  visible = true
  asPropertyInfo = [
    [ &achieved, 'achieved', new ASBooleanValidator() ],
    [ &achievementNotified, 'achievementNotified', new ASBooleanValidator() ],
    [ &visible, 'visible', new ASBooleanValidator() ]
  ]
  
  // Setter methods:
  // When you need to change the value of the above properties at runtime, you 
  // MUST use these setter methods (as opposed to just directly setting those
  // properties), because IF YOU DON'T, THIS MODULE WILL NOT WORK CORRECTLY!
  // (On the other hand, there is nothing wrong with overriding the properties
  // in order to change their initial value at compile-time.)
  
  // "achieved" property setter methods:
  // I couldn't decide whether "achieve()" or "unlock()" was a better name for the 
  // method to set achieved = true, so I decided to just allow both of them. I provide 
  // no specially named convenience method for "unachieving" or "locking" achievements, 
  // because that is not a normal thing to do, (although technically such a thing can 
  // be done with the syntax "setAchieved(nil)"). 
  achieve() {
    // I question whether it makes any sense to say that you've achieved an achievement
    // while still keeping it invisible
    if (!visible) {
      setVisible();
    }
    setAchieved();
  }
  unlock() { achieve(); }
  setAchieved([args]) {
    // Determine the value to set it to, either true or nil, defaulting to true
    local defaultVal = true;
    local val;
    if (args.length == 0) {
      val = defaultVal;
    }
    else if (args[1] == true) {
      val = true;
    }
    else if (args[1] == nil) {
      val = nil;
    }
    else {
      throw new Exception('Illegal input parameter for ASAchievement.setAchieved(). ');
    }
    
    // Set the value in the proper way, using the setter method, to allow for saving
    // of achievement-scope data. 
    set(&achieved, val);
  }
  // "visible" property setter methods:
  // Use "reveal()" when you want to reveal an achievement that was not previously visible.
  // In the unlikely event that you want to take a revealed achievement and make it
  // invisible, you can use the syntax "setVisible()".
  reveal() {
    setVisible();
  }
  setVisible([args]) {
    // Determine the value to set it to, either true or nil, defaulting to true
    local defaultVal = true;
    local val;
    if (args.length == 0) {
      val = defaultVal;
    }
    else if (args[1] == true) {
      val = true;
    }
    else if (args[1] == nil) {
      val = nil;
    }
    else {
      throw new Exception('Illegal input parameter for ASAchievement.setVisible(). ');
    }
    
    // Set the value in the proper way, using the setter method, to allow for saving
    // of achievement-scope data. 
    set(&visible, val);
  }

  // Getter methods, provided here for the sake of having some kind of symmetry with 
  // the setter methods. Feel free to use these if you want. You can also feel free 
  // to not use them and just read the "achieved"/"visible" properties directly if you prefer.
  isAchieved() {
    return achieved;
  }
  isVisible() {
    return visible;
  }
;

// -----------------------------------------------------------------------------
// Classes: Data validators: ASValidator
// -----------------------------------------------------------------------------

// This base class implementation does not try to do any data conversion,
// and does not declare any input data as invalid.

class ASValidator: object
  // convertToValidValue: 
  // This is the main method subclasses must override if they want to enforce 
  // strong typing. If conversion is not possible, return the value 
  // achNotConvertible.
  convertToValidValue(val) {
    return val;
  }
  // isValidValue: This is derivative of convertToValidValue, so no need to
  // override.
  isValidValue(val) {
    local convertedVal = convertToValidValue(val);
    if (convertedVal == achNotConvertible) {
      return nil;
    }
    return true;
  }
  // isNumeric:
  // At a glance, I didn't find anything in TADS 3 to define whether a string 
  // or other object is numeric. toInteger() doesn't work, as it lazily converts 
  // bogus data to 0. Here goes...
  isNumeric(val) {
    if (dataType(val) == TypeNil) return true; // I think this is a good idea...?
    if (dataType(val) == TypeTrue) return true; // I think this is a good idea...?
    if (dataType(val) == TypeInt) return true;
    if (dataType(val) == TypeSString && rexMatch(numericPattern, val) != nil) return true;
    return nil;
  }
  numericPattern = static new RexPattern('^<space>*[-]?[0-9]*(<dot>[0-9]+)?<space>*$')
  // defaultBadDataTypeBehavior: either achThrowException or achFailSilently. 
  // If not provided at the AchDataPropertyInfo level, this is what we'll do if 
  // the user tries to set a bad value. Not relevant except for strongly-typed subclasses.
  badDataTypeBehavior = achThrowException
  // getForDisplay: returns the value in a way that can be presented onscreen to a user.
  // e.g. strings are just show as they are
  getForDisplay(val) {
    return toString(val);
  }
  // getForDebugging: returns the value in a way that might make more sense for debugging.
  // e.g. strings are shown surrounded by single quotes.
  getForDebugging(val) {
    if (dataType(val) == TypeSString) {
      return '\'' + val + '\'';
    }
    else {
      return toString(val);
    }
  }
;

// -----------------------------------------------------------------------------
// Classes: Data validators: ASIntegerValidator
// -----------------------------------------------------------------------------

// This base class implementation does not try to do any data conversion,
// and does not declare any input data as invalid.

class ASIntegerValidator: ASValidator
  convertToValidValue(val) {
    if (!isNumeric(val)) {
      return achNotConvertible;
    }
    try {
      return toInteger(val);
    }
    catch (Exception ex) { }
    return achNotConvertible;
  }
;

// -----------------------------------------------------------------------------
// Classes: Data validators: ASBooleanValidator
// -----------------------------------------------------------------------------

class ASBooleanValidator: ASValidator
  convertToValidValue(val) {
    if (dataType(val) == TypeNil) return nil;
    if (dataType(val) == TypeTrue) return true;
    if (dataType(val) == TypeInt && val == 0) return nil;
    if (dataType(val) == TypeInt && val == 1) return true; // do this for other nonzero integers too?
    if (dataType(val) == TypeSString && 
        (rexMatch(nilPattern, val) != nil ||
         rexMatch(zeroPattern, val) != nil)) return nil;
    if (dataType(val) == TypeSString && 
        (rexMatch(truePattern, val) != nil ||
         rexMatch(onePattern, val) != nil)) return true;
    return achNotConvertible;
  }
  nilPattern = static new RexPattern('^<space>*nil<space>*$')
  truePattern = static new RexPattern('^<space>*true<space>*$')
  onePattern = static new RexPattern('^<space>*1<space>*$')
  zeroPattern = static new RexPattern('^<space>*0<space>*$')
  badDataTypeBehavior = achThrowException
;

// -----------------------------------------------------------------------------
// Classes: Data validators: ASStringValidator
// -----------------------------------------------------------------------------

// This base class implementation does not try to do any data conversion,
// and does not declare any input data as invalid.

class ASStringValidator: ASValidator
  convertToValidValue(val) {
    try {
      return toString(val);
    }
    catch (Exception ex) { }
    return achNotConvertible;
  }
;

// -----------------------------------------------------------------------------
// Enums
// -----------------------------------------------------------------------------

enum achFailSilently, achThrowException;
enum achNotConvertible;
//enum achUnknown;
enum condensedAchievementList, expandedAchievementList;

// -----------------------------------------------------------------------------
// Utility objects: achievementScopeManager
// -----------------------------------------------------------------------------

// achievementScopeManager:
// With luck, authors will not need to use the "utility objects" in this file.
// achievementScopeManager is the "master" of all the utility objects, with methods 
// for saving data from objects to disk, and for loading data from disks to objects.
// You should not normally need to call them directly; they are performed 
// automatically.

transient achievementScopeManager: object
  // Save data from objects to file. 
  // Note: For efficiency, instead of calling saveData(), consider calling
  // scheduleSaveData(), which makes sure data is saved, but not more than once 
  // per turn.
  saveData() {
    achievementScopeTranslator.getAchievementScopeDataFromObjects();
    achievementScopeFileManager.saveCollectedDataToFile();
    
    //// I successfully created and tested an implementation that would save to 
    //// objects with true newline characters rather than encrypted ones.
    //listBasedAchievementScopeDataTranslator.getAchievementScopeDataFromObjects();
    //achievementScopeFileManager.saveListBasedCollectedDataToFile();
  }
  // Load data from file to objects
  // Note to self: This method replaces the former 
  // achievementScopeFileManager.loadObjectDataFromFile();
  loadData() {
    achievementScopeFileManager.loadDataFromFileToMemory();
    achievementScopeTranslator.loadCollectedDataToObjects();
  }
  
  // Flag the need to save data at the end of the turn. A nice way to avoid
  // saving data to file more than once per turn. 
  scheduleSaveData() {
    saveDataIsScheduled = true;
  }
  saveDataIsScheduled = nil
;

// -----------------------------------------------------------------------------
// Utility objects: achievementScopeTranslator
// -----------------------------------------------------------------------------

// achievementScopeTranslator has the following purposes:
// (1) retrieve achievement-scope data from objects to memory 
//     (translating it from an object-friendly format to a disk-storable format)
// (2) store achievement-scope data from memory to objects
//     (translating it from a disk-storable format to an object-friendly format)

achievementScopeTranslator: object
  // =============================
  // "data" property and its setter methods
  data = ''
  clearData() {
    data = '';
  }
  addToData(val) {
    data = data + val + newlineDelimiter;
  }
  setData(d) {
    data = d;
  }
  // =============================
  // Values used as "constants"
  valueType = 'val'
  listType = 'lst'
  emptyListDummyValue = '/////EMPTY_LIST/////'
  emptyListDummyDataTypeInt = 0
  // =============================
  // delimiters
  // Warning: stored strings will be cleansed of these values
  // tabDelimiter: delimiter for different pieces of data on the same line
  tabDelimiter = '\t' 
  // newlineDelimiter: delimiter between lines
  newlineDelimiter = '\n'
  
  // =============================
  // getAchievementScopeDataFromObjects: 
  // the method to retrieve achievement-scope data from objects to memory 
  // (translating it from an object-friendly format to a disk-storable format)
  getAchievementScopeDataFromObjects() {
    local curObj;
    local curObjName;
    local curProp;
    local curPropName;
    local curPropValue;
    local curPropListOrValue;
    local curPropBaseValue;
    local curPropBaseTypeInt;
    local curSaveString;
    local listValuesAdded;
    
    clearData();
    
    // curObj: the achievement-scope object whose value(s) to save
    for (curObj = firstObj(ASObject); curObj != nil; 
         curObj = nextObj(curObj, ASObject))
    {
      // If we don't have a suitable name for the object, or otherwise can't 
      // save it, then just skip the object. 
      if (!curObj.isSaveable()) {
        continue; 
      }
      
      // curObjName: the object's name to use for saving
      curObjName = curObj.asName;
      for (local i = 1; i <= curObj.asProperties.length; i++) {
        // curProp: the property (e.g. "&value")
        curProp = curObj.asProperties[i];
        
        // curPropName: the name of the property (e.g. "value")
        //curPropName = reflectionServices.reverseSymtab_[curProp];
        curPropName = curObj.asNamesPerProperty[curProp];
        if (curPropName == nil) {
          break;
        }
        
        // curPropValue. If this is a list, we'll need to iterate through the list.
        // Otherwise, this is the value to be saved. 
        curPropValue = curObj.(curProp);
        
        if (dataType(curPropValue) != TypeList) {
          // curPropListOrValue needs to indicate that this is an atomic value, not a list
          curPropListOrValue = valueType;
          // curPropBaseValue: the value to save
          curPropBaseValue = curPropValue;
          // the integer corresponding to the value's datatype (see systype.h)
          curPropBaseTypeInt = dataType(curPropBaseValue);
          
          if (!dataTypeIsSaveable(curPropBaseTypeInt)) {
            continue;
          }
          
          curSaveString = 
            curObjName + tabDelimiter +
            curPropName + tabDelimiter +
            curPropListOrValue + tabDelimiter +
            curPropBaseTypeInt + tabDelimiter +
            toSaveableString(curPropBaseValue);
          
          addToData(curSaveString);
        }
        else {
          listValuesAdded = 0;
          // curPropListOrValue needs to indicate that this is a list, not an atomic value
          curPropListOrValue = listType;
          for (local j = 1; j <= curPropValue.length; j++) {
            // curPropBaseValue: the value to save
            curPropBaseValue = curPropValue[j];
            // the integer corresponding to the value's datatype (see systype.h)
            curPropBaseTypeInt = dataType(curPropBaseValue);
            
            if (!dataTypeIsSaveable(curPropBaseTypeInt)) {
              continue;
            }
          
            curSaveString = 
              curObjName + tabDelimiter +
              curPropName + tabDelimiter +
              curPropListOrValue + tabDelimiter +
              curPropBaseTypeInt + tabDelimiter +
              toSaveableString(curPropBaseValue);
            
            addToData(curSaveString);
            listValuesAdded++;
          }
          
          // In the special case where we're storing a list that is empty (or empty 
          // of any values that are valid enough for us to store), we must do an
          // extra step to remember that this is the case.
          if (listValuesAdded == 0) {
            curSaveString = 
              curObjName + tabDelimiter +
              curPropName + tabDelimiter +
              curPropListOrValue + tabDelimiter +
              emptyListDummyDataTypeInt + tabDelimiter +
              toSaveableString(emptyListDummyValue);
            
            addToData(curSaveString);
            listValuesAdded++;
          }
        }
        
      } // End loop of properties of achievement-scope objects
    } // End loop of achievement-scope objects
  }
  
  // =============================
  // loadCollectedDataToObjects: 
  // the method to call for loading achievement-scope data from memory to objects
  // (translating it from a disk-storable format to an object-friendly format)
  loadCollectedDataToObjects() {
    loadDataToObjects(achievementScopeTranslator.data.split(achievementScopeTranslator.newlineDelimiter));
  }
  loadListBasedCollectedDataToObjects() {
    loadDataToObjects(listBasedAchievementScopeDataTranslator.data);
  }
  loadDataToObjects(data) {
    //local objName;
    //local propName;
    //local propListOrValue;
    //local propBaseValue;
    //local propBaseTypeInt;
    local newLine = nil;
    local pendingLines = [];
    data += (
      'THE_END_OF_THE_FILE' + achievementScopeTranslator.tabDelimiter + 
      'THE_END_OF_THE_FILE' + achievementScopeTranslator.tabDelimiter + 
      achievementScopeTranslator.valueType + achievementScopeTranslator.tabDelimiter + 
      '7' + achievementScopeTranslator.tabDelimiter + 
      '1');
    local line;
    for (local i = 1; i <= data.length; i++) {
      line = data[i];
      
      // Try to convert the string values of the new line to strongly-typed values. If unsuccessful, 
      // give up and move to the next line
      newLine = nil;
      try {
        newLine = new AchievementScopeFileItem(line);
      }
      catch (Exception ex) {
        continue;
      }
      
      // If the new line indicates that we are done with the pending lines (which 
      // may be either an atomic value or a list of values), then deal with the 
      // pending lines. 
      if (pendingLines.length > 0 && 
          (pendingLines[1].objName != newLine.objName || pendingLines[1].propName != newLine.propName))
      {
        // This try/catch concerns the processing of ALL the pendingLines.
        try {
          // Get the object, or if failing to do so, throw an exception that bails out of the entire pendingLines set.
          local obj = achievementScopeUniquenessGuarantor.asObjectsPerName[pendingLines[1].objName];
          if (obj == nil) {
            throw new Exception('No unique object found for object name: ' + pendingLines[1].objName);
          }
          // Get the property, or if failing to do so, throw an exception that bails out of the entire pendingLines set.
          local prop = obj.asPropertiesPerName[pendingLines[1].propName];
          if (obj == nil) {
            throw new Exception('No property found on object for object name ' +
              '"' + pendingLines[1].objName) + '", property name "' + pendingLines[1].propName + '".';
          }
          // GREGTODO: Delete the commented-out stuff
          //local obj = Compiler.eval(pendingLines[1].objName);
          //local prop = Compiler.eval('&' + pendingLines[1].propName);
          
          // If the property is supposed to be a list, then clear it out and add 
          // all the values that the list is supposed to have. 
          if (pendingLines[1].propListOrValue == achievementScopeTranslator.listType) {
            obj.(prop) = [];
            if (pendingLines[1].propBaseTypeInt == achievementScopeTranslator.emptyListDummyDataTypeInt) {
              // don't add any values if it's supposed to be an empty list
            }
            else {
              for (local j = 1; j <= pendingLines.length; j++) {
                obj.(prop) = obj.(prop) + pendingLines[j].propBaseValue;
              }
            }
          }
          // If the property is a single value, just assign the value. 
          else {
            obj.(prop) = pendingLines[1].propBaseValue;
          }
        }
        catch (Exception ex) {
          // TODO: Consider how error logging might happen.
          
          // Do a try/catch at the level of trying to load the individual variable or 
          // list. If such a variable or list fails, it may be because the developer
          // removed or renamed an achievement-scope object or property. If that happens,
          // don't freak out, just give up gracefully and move on to additional 
          // objects/properties. 
        }
        
        // Now reset pendingLines again
        pendingLines = [];
      }
      
      // Add the most recently read line to the list of those that are pending.
      pendingLines += newLine;
    }
  }
  
  // =============================
  // Miscellaneous helper methods
  dataTypeIsSaveable(type) {
    // This could probably be rendered more dynamic by looping somehow 
    // through the subclasses of, e.g., AchievementScopeOjbect. However, 
    // due to time constraints, I'm going to go ahead and hard-code it.
    if (type == TypeNil) return true;
    if (type == TypeTrue) return true;
    if (type == TypeInt) return true;
    if (type == TypeSString) return true;
    // TypeList should not return true, because lists are handled 
    // separately, and lists within lists are not supported here. 
    return nil;
  }
  toSaveableString(val) {
    return toString(val)
      .findReplace(tabDelimiter, '', ReplaceAll)
      .findReplace(newlineDelimiter, '', ReplaceAll);
  }
  getTypeIntFromTypeName(name) {
    switch(name) {
      case 'TypeNil'        : return TypeNil;
      case 'TypeTrue'       : return TypeTrue;
      case 'TypeObject'     : return TypeObject;
      case 'TypeProp'       : return TypeProp;
      case 'TypeInt'        : return TypeInt;
      case 'TypeSString'    : return TypeSString;
      case 'TypeDString'    : return TypeDString;
      case 'TypeList'       : return TypeList;
      case 'TypeCode'       : return TypeCode;
      case 'TypeFuncPtr'    : return TypeFuncPtr;
      case 'TypeNativeCode' : return TypeNativeCode;
      case 'TypeEnum'       : return TypeEnum;
      case 'TypeBifPtr'     : return TypeBifPtr;
      default               : return nil;
    }
  }
  getTypeNameFromTypeInt(int) {
    switch(int) {
      case TypeNil        : return 'TypeNil';
      case TypeTrue       : return 'TypeTrue';
      case TypeObject     : return 'TypeObject';
      case TypeProp       : return 'TypeProp';
      case TypeInt        : return 'TypeInt';
      case TypeSString    : return 'TypeSString';
      case TypeDString    : return 'TypeDString';
      case TypeList       : return 'TypeList';
      case TypeCode       : return 'TypeCode';
      case TypeFuncPtr    : return 'TypeFuncPtr';
      case TypeNativeCode : return 'TypeNativeCode';
      case TypeEnum       : return 'TypeEnum';
      case TypeBifPtr     : return 'TypeBifPtr';
      default             : return nil;
    }
  }
;

// We may or may not want the data as a list, but if we do, 
// here's the implementation
listBasedAchievementScopeDataTranslator: achievementScopeTranslator
  // =============================
  // overriding "data" property and its setter methods
  data = []
  clearData() {
    data = [];
  }
  addToData(val) {
    data = data + val;
  }
  setData(d) {
    data = d;
  }
;

// -----------------------------------------------------------------------------
// Utility objects: achievementScopeEncrypter
// -----------------------------------------------------------------------------

achievementScopeEncrypter: object
  useEncryption = nil
  encryptionKey = 23
  encryptString(str) {
    if (!useEncryption) {
      return str;
    }
    
    local unencryptedCodes = str.toUnicode();
    local encryptedCodes = [];
    for (local i = 1; i <= unencryptedCodes.length; i++) {
      // Use the "bitwise xor" operator (^) to encrypt using the encryption key
      encryptedCodes += (unencryptedCodes[i] ^ encryptionKey);
    }
    return makeString(encryptedCodes);
  }
  encryptStringList(lst) {
    local encryptedList = [];
    for (local i = 1; i <= lst.length; i++) {
      encryptedList += encryptString(lst[i]);
    }
    return encryptedList;
  }
  // With XOR-based encryption, decryption is exactly the same as encryption
  decryptString(str) { return encryptString(str); }
  decryptStringList(lst) { return encryptStringList(lst); }
;

// -----------------------------------------------------------------------------
// Utility objects: achievementScopeFileManager
// -----------------------------------------------------------------------------

achievementScopeFileManager: object
  // =============================
  // Values used as "constants"
  
  // Consider overriding this by modifying achievementScopeFileManager and replacing
  // achievementFilename with whatever filename you prefer -- e.g., 'yourgame-ach.sav'.
  achievementFilename = 'achievement.sav'
  
  // =============================
  // Methods
  
  // saveObjectDataToFile: 
  // This method takes all the data from all achievement-scope objects 
  // and stores them to memory, then encrypts it and stores it to disk.
  saveObjectDataToFile() {
    achievementScopeTranslator.getAchievementScopeDataFromObjects();
    saveCollectedDataToFile();
    achievementScopeTranslator.clearList(); // clear the data for garbage-collection
  }
  saveListBasedObjectDataToFile() {
    listBasedAchievementScopeDataTranslator.getAchievementScopeDataFromObjects();
    saveListBasedCollectedDataToFile();
    listBasedAchievementScopeDataTranslator.clearList(); // clear the data for garbage-collection
  }
  // saveCollectedDataToFile:
  // This method takes the object data already stored to memory, encrypts it, 
  // and stores it to disk.
  saveCollectedDataToFile() {
    saveStringToTextFile(
      achievementScopeEncrypter.encryptString(achievementScopeTranslator.data), 
      achievementFilename);
  }
  saveListBasedCollectedDataToFile() {
    //saveStringListToTextFile(
    //  achievementScopeEncrypter.encryptStringList(listBasedAchievementScopeDataTranslator.data), 
    //  listBasedAchievementFilename);
  }
  // saveStringToTextFile:
  // Writes the specified text to a text file. This is just a dumb file I/O method; 
  // nothing achievement-specific.
  saveStringToTextFile(str, filename) {
    local f = File.openTextFile(filename, FileAccessWrite);
    f.writeFile(str);
    f.closeFile();
  }
  saveStringListToTextFile(lst, filename) {
    local f = File.openTextFile(filename, FileAccessWrite);
    for (local i = 1; i <= lst.length; i++) {
      f.writeFile(lst[i] + '\n');
    }
    f.closeFile();
  }
  // loadDataFromFileToMemory:
  loadDataFromFileToMemory() {
    achievementScopeTranslator.clearData();
    
    try {
      // Read file and store to memory but don't yet write it to objects.
      local f = File.openTextFile(achievementFilename, FileAccessRead);
      local entireString = '';
      local line = f.readFile();
      while (line != nil) {
        if (entireString.length > 0) { entireString += '\n'; }
        entireString += line;
        line = f.readFile();
      }
      f.closeFile();
      
      // Store to memory
      achievementScopeTranslator.setData(achievementScopeEncrypter.decryptString(entireString));
    }
    catch (Exception ex) {
      // If this fails, the file probably just isn't there, and may indicate nothing more
      // serious than that we're playing this game for the first time. Shrug and move on.
    }
  }
  // This method was tested, but is commented out now because I am not loading list-based
  // files and do not want to preserve the previously existing property listBasedAchievementFilename.
  loadListBasedDataFromFileToMemory() {
    //listBasedAchievementScopeDataTranslator.clearData();
    //
    //try {
    //  local f = File.openTextFile(listBasedAchievementFilename, FileAccessRead);
    //  local stringList = [];
    //  local line = f.readFile();
    //  while (line != nil) {
    //    // Remove any trailing newline character
    //    if (line.substr(-1) == '\n') {
    //      line = line.substr(1, line.length - 1);
    //    }
    //    stringList += line;
    //    line = f.readFile();
    //  }
    //  f.closeFile();
    //  
    //  // Store to memory
    //  listBasedAchievementScopeDataTranslator.setData(achievementScopeEncrypter.decryptStringList(stringList));
    //}
    //catch (Exception ex) {
    //  // If this fails, the file probably just isn't there, and may indicate nothing more
    //  // serious than that we're playing this game for the first time. Shrug and move on.
    //}
  }
  //// Not used:
  //listBasedAchievementFilename = 'achievementlist.sav'
;

// -----------------------------------------------------------------------------
// Utility objects: Convenience class: AchievementScopeFileItem
// -----------------------------------------------------------------------------

// AchievementScopeFileItem: a convenience class used by achievementScopeTranslator 
// when trying to convert file- data to object-friendly data.

class AchievementScopeFileItem: object
  objName = nil
  propName = nil
  propListOrValue = nil
  propBaseTypeInt = nil
  propBaseValue = nil
  construct(line) {
      local strings;
      strings = line.split(achievementScopeTranslator.tabDelimiter);
      // We require 5 pieces of data; don't try to handle bogus data
      if (strings.length != 5) {
        throw new Exception('The saved line of text contains ' + strings.length + 
          ' piece(s) of data, whereas 5 pieces are required.');
      }
      
      // Read the raw string values
      objName = strings[1];
      propName = strings[2];
      propListOrValue = strings[3];
      propBaseTypeInt = strings[4];
      propBaseValue = strings[5];
      
      // TODO: Code the conversions dynamically rather than lazily hard-coding them...
      propBaseTypeInt = toInteger(propBaseTypeInt);
      
      if (propBaseTypeInt == TypeNil) {
        if (propBaseValue == 'nil') { propBaseValue = nil; }
        else { throw new Exception('Illegal value associated with <<propBaseTypeInt>> (TypeNil). '); }
      }
      else if (propBaseTypeInt == TypeTrue) {
        if (propBaseValue == 'true') { propBaseValue = true; }
        else { throw new Exception('Illegal value associated with <<propBaseTypeInt>> (TypeTrue). '); }
      }
      else if (propBaseTypeInt == TypeInt) {
        propBaseValue = toInteger(propBaseValue);
      }
      else if (propBaseTypeInt == TypeSString) {
        // keep propBaseValue as is
      }
      else if (propBaseTypeInt == achievementScopeTranslator.emptyListDummyDataTypeInt) {
        propBaseValue = achievementScopeTranslator.emptyListDummyValue;
      }
      else {
        throw new Exception('Illegal data type for propBaseValue.');
      }
      
      if (propListOrValue != achievementScopeTranslator.valueType &&
          propListOrValue != achievementScopeTranslator.listType)
      {
        throw new Exception('Illegal value for propListOrValue.');
      }
  }
;

// -----------------------------------------------------------------------------
// Utility objects: achievementScopePreinitObject
// -----------------------------------------------------------------------------

// At startup, (1) extract the asPropertyInfo information to a more useful 
// format, and (2) load the achievement-scope data from saved file to objects.

achievementScopePreinitObject: PreinitObject
  execute() {
    // Initialize objects:
    achievementScopeUniquenessGuarantor.initialize();
    // Initialize properties:
    extractASPropertyInfo();
    // Load data from file
    achievementScopeManager.loadData();
    // Start the achievementScopeDaemon
    achievementScopeDaemon.start();
  }
  // extractASPropertyInfo:
  // At startup, extract the asPropertyInfo information into the more 
  // useful format of asProperties and asValidatorsPerProperty. 
  extractASPropertyInfo() {
    local curObj;
    for (curObj = firstObj(ASObject); curObj != nil; 
         curObj = nextObj(curObj, ASObject))
    {
      // For each object:
      // Might as well start by wiping out the following four object properties.
      // They're required to be empty at startup anyway. 
      curObj.asProperties = [];
      curObj.asPropertiesPerName = new LookupTable();
      curObj.asNamesPerProperty = new LookupTable();
      curObj.asValidatorsPerProperty = new LookupTable();
      
      try {
        for (local i = 1; i <= curObj.asPropertyInfo.length; i++) {
          // For each item in asPropertyInfo:
          // Retrieve the three required values from the individual asPropertyInfo item
          local saveProp = curObj.asPropertyInfo[i][1];
          local savePropName = curObj.asPropertyInfo[i][2];
          local validationObj = curObj.asPropertyInfo[i][3];
          // Reject them if they don't match requirements
          if (dataType(saveProp) != TypeProp) {
            throw new Exception('For ASObject "' + curObj.asName + '", asPropertyInfo[x][1] (saveProp) was not a property.');
          }
          if (dataType(savePropName) != TypeSString) {
            throw new Exception('For ASObject "' + curObj.asName + '", asPropertyInfo[x][2] (savePropName) was not a single-quoted string.');
          }
          if (dataType(validationObj) != TypeObject || !validationObj.ofKind(ASValidator)) {
            throw new Exception('For ASObject "' + curObj.asName + '", asPropertyInfo[x][3] (validationObj) was not an object belonging to class ASValidator.');
          }
          
          // Add to asProperties
          if (!curObj.asProperties.indexOf(saveProp)) {
            curObj.asProperties += saveProp;
          }
          // Add to asPropertiesPerName
          curObj.asPropertiesPerName[savePropName] = saveProp;
          // Add to asNamesPerProperty
          curObj.asNamesPerProperty[saveProp] = savePropName;
          // Add to asValidatorsPerProperty
          curObj.asValidatorsPerProperty[saveProp] = validationObj;
        }
      }
      catch (Exception ex) {
        throw new Exception('achievementScopePreinitObject failed to process an ' +
          'ASObject. Message: ' + ex.errmsg_);
      }
    }
  }
;

// -----------------------------------------------------------------------------
// Utility objects: achievementScopeUniquenessGuarantor
// -----------------------------------------------------------------------------

transient achievementScopeUniquenessGuarantor: object
  isInitialized = nil
  asObjectsPerName = new LookupTable()
  asNames = []
  // GREGTODO: TEST THIS!!!!!!
  asNameIsUnique(name) {
    initialize();
    // Return true only if the name is a key in the lookup table
    return (asObjectsPerName[name] != nil);
  }
  initialize() {
    // Populate asObjectsPerName and asNames, if not already populated. 
    // This will happen just once per session. This implementation lazy-loads it. 
    if (!isInitialized) {
      // Reinitialize the lookup table and the list just to be extra sure
      asObjectsPerName = new LookupTable();
      asNames = [];
      
      // Populate asObjectsPerName (and perhaps load up asObjectsPerName 
      // to a degree far surpassing
      local curObj;
      for (curObj = firstObj(ASObject); curObj != nil; 
           curObj = nextObj(curObj, ASObject))
      {
        // For each ASObject...
        // If it's not defined or has a bogus (non-string) datatype, ignore it
        if (curObj.asName == nil || dataType(curObj.asName) != TypeSString) {
        }
        // if it's a name we haven't seen yet, add it to the lookup table
        else if (!asNames.indexOf(curObj.asName)) {
          asObjectsPerName[curObj.asName] = curObj;
          asNames += curObj.asName;
        }
        // If it IS an object we haven't seen, remove it from the lookup table
        // (but keep it in the asNames list, just to remember that we have 
        // in fact seen it and that it remains a non-unique name)
        else {
          asObjectsPerName[curObj.asName] = nil;
        }
      }
      
      // Sterilize asNames of any remaining bad names that aren't unique
      // and should no longer be there
      for (local i = asNames.length; i >= 1; i--) {
        if (asObjectsPerName[asNames[i]] == nil) {
          asNames = asNames.removeElementAt(i);
        }
      }
      // Now that we've initialized this, we don't need to do so again...
      isInitialized = true;
    }
  }
;

// -----------------------------------------------------------------------------
// Utility objects: achievementScopeDaemon
// -----------------------------------------------------------------------------

achievementScopeDaemon: object
  daemonID = nil
  start() {
    daemonID = new PromptDaemon(self, &executeDaemon);
  }
  executeDaemon() {
    // If we need to save data to disk, do so now.
    if (achievementScopeManager.saveDataIsScheduled) {
      achievementScopeManager.saveData();
      achievementScopeManager.saveDataIsScheduled = nil;
    }
    
    // If we need to report having unlocked an
    // achievement, do so now.
    local reportedAnAchievement = nil;
    // Get the list of achievements
    local curObj;
    local achievements = [];
    for (curObj = firstObj(ASAchievement); curObj != nil; 
         curObj = nextObj(curObj, ASAchievement))
    {
      achievements += curObj;
    }
    // Sort the list
    achievements = achievements.sort(nil, { a, b: a.listOrder - b.listOrder });
    
    // Enumerate the sorted list
    local ach;
    for (local i = 1; i <= achievements.length; i++)
    {
      ach = achievements[i];
      if (ach.achieved && !ach.achievementNotified) {
        ach.set(&achievementNotified, true);
        
        // If we remember having turned off achievement notifications
        // in EITHER playthough scope or session scope, then obey
        // the "turned off" preference. 
        // The "achievementNotificationsOn" logic here must be consistent 
        // with what is in the AchievementsNotify action.
        if (rememberedThisPlaythough.achievementNotificationsOn &&
            rememberedThisSession.achievementNotificationsOn)
        {
          "<<achievementMessages.achievementUnlockedMsg(ach.name)>>";
          reportedAnAchievement = true;
        }
      }
    }
    // If one or more achievements were reported, give an additional tip
    // to the player.
    if (reportedAnAchievement &&
        !rememberedThisPlaythough.hasExplainedNotifyOff && 
        !rememberedThisSession.hasExplainedNotifyOff)
    {
      "<<achievementMessages.achievementCommandNotify(gameHasEnded)>>";
      rememberedThisPlaythough.hasExplainedNotifyOff = true;
      rememberedThisSession.hasExplainedNotifyOff = true;
    }
  }
  end() {
    if (daemonID != nil) {
      daemonID.removeEvent;
    }
    daemonID = nil;
  }
  gameHasEnded = nil
;

// -----------------------------------------------------------------------------
// Utility objects: rememberedThisPlaythough / rememberedThisSession
// -----------------------------------------------------------------------------

// I did some thinking about whether our recollection about whether we've said 
// "[To view the list of achievements..." should be stored with an achievement-scope
// variable. 
// 
// At first I decided no, it is better for it to be a conventional variable, 
// so that the "ACHIEVEMENTS OFF" notification happens once during any playthough 
// in which you are notified of scoring an achievemnt. 
// 
// Then I realized this isn't ideal if your first achievement in a playthough 
// happens when you end the game, then type UNDO or RESTORE and are basically 
// guaranteed to see the same thing again later on. 
// 
// The solution: Store the value NOT to an achievement-scope variable, but to 
// BOTH a conventional variable and a transient variable, and if EITHER records
// that we said the information, don't say it again.
// 
// I decided to use this policy also for the "ACHIEVEMENT NOTIFY ON" / 
// "ACHIEVEMENT NOTIFY OFF" preference.

rememberedThisPlaythough: object
  hasExplainedNotifyOff = nil
  achievementNotificationsOn = true
;

transient rememberedThisSession: object
  hasExplainedNotifyOff = nil
  achievementNotificationsOn = true
;

// -----------------------------------------------------------------------------
// Style tags
// -----------------------------------------------------------------------------

achievementListHeader: HtmlStyleTag
  'achievementlistheader'
  htmlOpenText = '<b>'
  htmlCloseText = '</b>'
;

achievementListItemName: HtmlStyleTag
  'achievementlistitemname'
  htmlOpenText = '<i>'
  htmlCloseText = '</i>'
;

// -----------------------------------------------------------------------------
// Utility objects: achievementMessages
// -----------------------------------------------------------------------------

achievementMessages: object
  achievementUnlockedMsg(achievementName) {
    return '<.p><i>Achievement unlocked: ' + achievementName + '.</i> ';
  }
  achievementCommandNotify(gameHasEnded) {
    return
      '<.p>[To view the list of achievements, type ' + 
        aHref('achievements', 'ACHIEVEMENTS', 'List achievements') + 
        '. If you&rsquo;d 
        prefer not to be notified about achievements, ' +
        (gameHasEnded ? 'then undo or restore the game and ' : '') +
        'type ' + 
        aHref('achievements off', 'ACHIEVEMENTS OFF', 'Turn achievement notifications off') + '.] ';
  }
  // achievementListType can be either condensedAchievementList or expandedAchievementList
  achievementListType = condensedAchievementList
  sayAchievementForList(name, desc) {
    if (achievementListType == condensedAchievementList) {
      "\n<.achievementlistitemname><<name>>:<./achievementlistitemname> <<desc>>";
    }
    else {
      "<.p><.achievementlistitemname><<name>><./achievementlistitemname>
      \n<<desc>>";
    }
  }
  // sayAchievementsForList is used in the response to typing ACHIEVEMENTS. 
  // It takes in four lists as shown.
  // Okay, so I've done a horrible job from decoupling the logical code 
  // from the language-specific code. Sue me. :-P
  sayAchievementsForList(achievedVisible, achievedInvisible, unachievedVisible, unachievedInvisible) {
    local achievedCount = achievedVisible.length + achievedInvisible.length;
    local unachievedCount = unachievedVisible.length + unachievedInvisible.length;
    local totalCount = achievedCount + unachievedCount;
    
    if (totalCount <= 0) {
      "There are no achievements in this game. ";
      return;
    }
    
    if (achievedCount > 0) {
      "<.achievementlistheader>Unlocked achievements (<<achievedCount>> of <<totalCount>>):<./achievementlistheader> ";
      "\n";
      for (local i = 1; i <= achievedVisible.length; i++) {
        local a = achievedVisible[i];
        sayAchievementForList(a.name, a.desc);
      }
      if (achievedInvisible.length > 0) {
        if (achievementListType == condensedAchievementList) { "\n"; }
        else { "<.p>"; }
        if (achievedVisible.length > 0) {
          "and ";
        }
        "<<achievedInvisible.length>> secret achievement<<sayPluralS(achievedInvisible)>> ";
      }
    }
    if (unachievedCount > 0) {
      "<.p>";
      "<.achievementlistheader>Locked achievements (<<unachievedCount>> of <<totalCount>>):<./achievementlistheader> ";
      "\n";
      for (local i = 1; i <= unachievedVisible.length; i++) {
        local a = unachievedVisible[i];
        sayAchievementForList(a.name, a.desc);
      }
      if (unachievedInvisible.length > 0) {
        if (achievementListType == condensedAchievementList) { "\n"; }
        else { "<.p>"; }
        if (unachievedVisible.length > 0) {
          "and ";
        }
        "<<unachievedInvisible.length>> secret achievement<<sayPluralS(unachievedInvisible)>> ";
      }
    }
  }
  sayPluralS(lst) {
    if (lst.length == 1) return '';
    return 's';
  }
  sayNotificationsTurnedOn(isOn) {
    "Achievement notifications are now <<isOn ? 'on' : 'off'>>. ";
  }
  sayNotificationsCurrentlyOn(isOn) {
    "Achievement notifications are currently <<isOn ? 'on' : 'off'>>. ";
  }
;

// -----------------------------------------------------------------------------
// Finish option: finishOptionAchievements
// -----------------------------------------------------------------------------

finishOptionAchievements: FinishOption
  doOption() {
    AchievementsAction.showAchievements();
    
    // Crucially, we must return true, or the game isn't really over.
    return true;
  }
;

modify finishOptionAchievements
  desc = "view the <<aHrefAlt('achievements', 'ACHIEVEMENTS', '<b>A</b>CHIEVEMENTS',
            'Show achievements')>>"
  responseKeyword = 'achievements'
  responseChar = 'a'
;

// I need to do something to announce the potential attainment of an achievement(s) 
// just after seeing "*** You have won ***" but before you get the list of 
// endgame options. 

modify finishOptionsLister
    showListPrefixWide(cnt, pov, parent)
    {
        achievementScopeDaemon.gameHasEnded = true;
        achievementScopeDaemon.executeDaemon();
        achievementScopeDaemon.gameHasEnded = nil;
        
        inherited(cnt, pov, parent);
        //"<.p>Would you like to ";
    }
;

// -----------------------------------------------------------------------------
// Actions: Achievements
// -----------------------------------------------------------------------------

DefineIAction(Achievements)
  execAction() {
    showAchievements();
  }
  showAchievements() {
    local achievements = [];
    local achievedVisible = [];
    local achievedInvisible = [];
    local unachievedVisible = [];
    local unachievedInvisible = [];
    
    // Get the list of achievements
    local curObj;
    for (curObj = firstObj(ASAchievement); curObj != nil; 
         curObj = nextObj(curObj, ASAchievement))
    {
      achievements += curObj;
    }
    // Sort the list
    achievements = achievements.sort(nil, { a, b: a.listOrder - b.listOrder });
    
    // Divide the list into four exhaustive and mutually exclusive lists
    for (local i = 1; i <= achievements.length; i++) {
      local a = achievements[i];
      if (a.achieved && a.visible) achievedVisible += a;
      if (a.achieved && !a.visible) achievedInvisible += a;
      if (!a.achieved && a.visible) unachievedVisible += a;
      if (!a.achieved && !a.visible) unachievedInvisible += a;
    }
    
    // Announce the achievements, using those four lists
    achievementMessages.sayAchievementsForList(achievedVisible, achievedInvisible, unachievedVisible, unachievedInvisible);
  }
  actionTime = 0
;

VerbRule(Achievements)
  ('achievement' | 'achievements')
  : AchievementsAction
  verbPhrase = 'show/showing achievements'
;

// -----------------------------------------------------------------------------
// Actions: Achievements On
// -----------------------------------------------------------------------------

DefineIAction(AchievementsOn)
  execAction() {
    // Turn on achievement notifications, and remember it in both 
    // playthrough-scope and session-scope.
    rememberedThisPlaythough.achievementNotificationsOn = true;
    rememberedThisSession.achievementNotificationsOn = true;
    // While we're at it, we can spare the player any future reference to
    // the "ACHIEVEMENTS OFF" command.
    rememberedThisPlaythough.hasExplainedNotifyOff = true;
    rememberedThisSession.hasExplainedNotifyOff = true;
    
    achievementMessages.sayNotificationsTurnedOn(true);
  }
  actionTime = 0
;

VerbRule(AchievementsOn)
  ('achievement' | 'achievements') ('notify' | ) 'on'
  : AchievementsOnAction
  verbPhrase = 'turn/turning achievement notifications on'
;

// -----------------------------------------------------------------------------
// Actions: Achievements Off
// -----------------------------------------------------------------------------

DefineIAction(AchievementsOff)
  execAction() {
    // Turn off achievement notifications, and remember it in both 
    // playthrough-scope and session-scope.
    rememberedThisPlaythough.achievementNotificationsOn = nil;
    rememberedThisSession.achievementNotificationsOn = nil;
    // While we're at it, we can spare the player any future reference to
    // the "ACHIEVEMENTS OFF" command.
    rememberedThisPlaythough.hasExplainedNotifyOff = true;
    rememberedThisSession.hasExplainedNotifyOff = true;
    
    achievementMessages.sayNotificationsTurnedOn(nil);
  }
  actionTime = 0
;

VerbRule(AchievementsOff)
  ('achievement' | 'achievements') ('notify' | ) 'off'
  : AchievementsOffAction
  verbPhrase = 'turn/turning achievement notifications off'
;

// -----------------------------------------------------------------------------
// Actions: Achievements Notify
// -----------------------------------------------------------------------------

DefineIAction(AchievementsNotify)
  execAction() {
    local isOn = nil;
    // This implementation is ugly but sound. The logic must be consistent
    // with what is in achievementScopeDaemon.executeDaemon().
    if (rememberedThisPlaythough.achievementNotificationsOn &&
        rememberedThisSession.achievementNotificationsOn)
    {
      isOn = true;
    }
    
    achievementMessages.sayNotificationsCurrentlyOn(isOn);
  }
  actionTime = 0
;

VerbRule(AchievementsNotify)
  ('achievement' | 'achievements') ('notify')
  : AchievementsNotifyAction
  verbPhrase = 'check/checking achievement notifications'
;

