/*  Copyright (c) 1990, 1999 by Michael J. Roberts.  All Rights Reserved. */
/*
    Ditch Day Drifter
    Interactive Fiction by Michael J. Roberts.
    2003   
    
    Developed with TADS: The Text Adventure Development System.
      RTADS
    
    This game is a sample TADS source file.  It demonstrates many of the
    features of TADS.  The TADS language is fully documented in the TADS
    Author's Manual, which is freely available in electronic format;
    you can probably get a copy of the Author's Manual from the same
    place you obtained this file.

    Please read LICENSE.DOC for information about using and copying this
    file, and about registering your copy of TADS.
*/

#define USE_HTML_STATUS
#define USE_HTML_PROMPT
/*
 *   First, insert the file containing the standard adventure definitions.
 */
#include <advr.t>
#include <gameinfo.t>
#include <stdr.t>
#include <errorru.t>
#include <extendr.t>

replace commonInit: function
{ "\H+";}



getGameInfo: function
{
    return
        ['Name', 'Ditch Day Drifter',
        'Byline', 'by Michael J. Roberts',
        'Desc', 'You\'re an undergraduate at Caltech, and you wake up
                to find it\'s Ditch Day, the traditional event when
                seniors leave the campus for the day, leaving behind
                puzzles for the underclassmen to solve in order to
                break into the seniors\' rooms.  This is the original
                TADS sample game, but is also a fully fleshed-out
                traditional text adventure game.',
        'Version', '1.0',
        'ReleaseDate', '1990-01-01',
        'LicenseType', 'Freeware',
        'CopyingRules', 'Nominal Cost Only'];
}

modify global
    maxscore = 80                                     // maximum possible score
    verbose = nil                             // we are currently in TERSE mode
    awakeTime = 0               // time that has elapsed since the player slept
    sleepTime = 600     // interval between sleeping times (longest time awake)
    lastMealTime = 0              // time that has elapsed since the player ate
    eatTime = 300         // interval between meals (longest time without food)
;

/*
 *   The player will start in the room called "startroom", by virtue
 *   of being placed there by the "init" function.
 *
 *   All rooms have an sdesc (short description), which is displayed on
 *   the status line and on entry to the room (even if the room has been
 *   seen before).  In addition, the ldesc (long description) is printed
 *   when the player enters the room for the first time or types "look."
 *   Directions (north, south, east, west, ne, nw, se, sw, in, out)
 *   specify where the exits go.  If a direction is not specified, no
 *   exit is in that direction.
 */
startroom: room
    sdesc = "<b> 3</b>"                                // short description
    ldesc = "  .    ,  
      .     ,  
     (,  )    .    . "
    west = alley1                                   // room that lies west
    out = alley1                                    // the exit
;

/*
 *   The desk is a surface, which means you can put stuff on top of it.
 *   Note that the drawer is a separate object.
 */
room3desk: surface, fixeditem
sdesc ="  "
rdesc ="  "
ddesc ="  "
vdesc ="  "
tdesc ="  "
pdesc ="  "
adjective = '' '' '' '' '' '#d'
 '' '' '' '' '' '' '#r'
noun = '' '' '' '' '' '#d'
    location = startroom
    ldesc =
    {
        "  ,       .
           ,  ,   
              
         .      (";
        if ( room3drawer.isopen ) ""; else "";
        "). ";
    }
    /*
     *   For convenience, redirect any open/close activity to the
     *   drawer.  Since the desk can't be opened and closed, we can
     *   reasonably expect that the player is really referring to the
     *   drawer if he tries to open or close the desk.
     */
    verDoOpen( actor ) = { room3drawer.verDoOpen( actor ); }
    doOpen( actor ) = { room3drawer.doOpen( actor ); }
    verDoClose( actor ) = { room3drawer.verDoClose( actor ); }
    doClose( actor ) = { room3drawer.doClose( actor ); }
isHim = true
;

/*
 *   A container can contain stuff.  An openable is a special type of container
 *   that can be opened and closed.  Though it's technically part of the desk,
 *   it's a fixeditem (==> it can't be taken), so we can just as well make it
 *   part of startroom; note that this is in fact necessary, since if it were
 *   located in the desk, it would appear to be ON the desk (since the desk is
 *   a surface).
 */
room3drawer: openable, fixeditem
isopen = nil
sdesc =""
rdesc =""
ddesc =""
vdesc =""
tdesc =""
pdesc =""
adjective = '' '' '' '' ''
noun = '' '' '' '' ''
location = startroom
isHim = true
;

/*
 *   A qcontainer is a "quiet container."  It acts like a container in all
 *   ways, except that its contents aren't displayed in a room's message.
 *   We want the player to have to think to examine the wastebasket more
 *   carefully to find out what's in it, so we'll make it a qcontainer.
 *   Which is fairly natural:  when looking around a room, you don't usually
 *   notice what's in a waste basket, even though you may notice the waste
 *   basket itself.
 *
 *   The sdesc is just the name of the object, as displayed by the game
 *   (as in "You see a wastebasket here").  The noun and adjective lists
 *   specify how the user can refer to the object; only enough need be
 *   specified by the user to uniquely identify the object for the purpose
 *   of the command.  Hence, you can specify as many words as you want without
 *   adding any burden to the user---the more words the better.  The location
 *   specifies the object (a room in this case) where the object is
 *   to be found.
 */
wastebasket: qcontainer
sdesc =" "
rdesc =" "
ddesc =" "
vdesc =" "
tdesc =" "
pdesc =" "
adjective = '' '' '' '#t'
noun = '' '' '' '' '' '#t' '#t' '' '' '' '' ''
    location = startroom
    moveInto( obj ) =
    {
        /*
         *   If this object is ever removed from its original location,
         *   turn off the "quiet" attribute.
         */
        self.isqcontainer := nil;
        pass moveInto;
    }
isHer = true
;

/*
 *   A beditem is something you can lie down on.
 */
bed: beditem
sdesc =""
rdesc =""
ddesc =""
vdesc =""
tdesc =""
pdesc =""
noun = '' '' '' '' '' ''
    location = startroom
    ldesc = "   .   
     (  , ) ,      . "
    
    /*
     *   verDoLookunder is called when the player types "look under bed"
     *   to verify that this object can be used as a direct object (Do) for
     *   that verb (Lookunder).  If no message is printed, it means that
     *   it can be used.
     */
    verDoLookunder( actor ) = {}
    
    /*
     *   ...and then, if verification succeeded, doLookunder is called to
     *   actually apply the verb (Lookunder) to this direct object (do).
     */
    doLookunder( actor ) =
    {
        if ( dollar.isfound )       // already found the dollar?
            "    . ";
        else
        {
            dollar.isfound := true;
            "      .
            (,       , 
             ,    ,     .) ";
            dollar.moveInto( Me );
        }
    }
    
    /*
     *   Verification and action for the "Make" verb.
     */
    verDoMake( actor ) = {}
    doMake( actor ) =
    {
        ", ,  ,    ,       . ";
    }
isHer = true
;

/*
 *   We wish to add the verb "make."  We need to specify the sdesc (printed
 *   by the system under certain circumstances), the vocabulary word itself
 *   (as it will be entered by the user), and the suffix of the method that
 *   will be called in direct objects.  By specifying a doAction of 'Make',
 *   we are establishing that verDoMake and doMake messages will be sent to
 *   the direct object of the verb.
 */
makeVerb: deepverb
    sdesc = ""
    verb = '' '' '' '' '' ''
    doAction = 'Make'
;

/*
 *   An "item" is the most basic class of objects that a user can carry
 *   around.  An item has no special properties.  The "iscrawlable"
 *   attribute that we're setting here is used in the north-south crawl
 *   in the steam tunnels (later in the game); setting it to "true"
 *   means that the player can carry this object through the crawl.
 */
dollar: item
sdesc =" "
rdesc =" "
ddesc =" "
vdesc =" "
tdesc =" "
pdesc =" "
adjective = '' '' '' '' '1'
noun = '' '' '' '' '' '' '' '' '' ''
'#t' '#t' '' '' '' '' '' '#t' '#d'
    iscrawlable = true
isHer = true
;

room4: room
    sdesc = "<b> 4</b>"
    ldesc = "  4,    .
       ,     . "
    east = alley1
    out = alley1
    down =
    {
        /*
         *   This is a bit of code attached to a direction.  We can do
         *   anything we want in code like this, so long as we return a
         *   room (or nil) when we're done.  In this case, we just want
         *   to display a message when the player travels this way.
         */
        "        
         ...\b";
        return( shiproom );
    }
;

/*
 *   A readable is an object that can be read, such as a note, a sign, a
 *   book, or a memo.  By default, reading the object displays its ldesc.
 *   However, if a separate readdesc is specified, "look at note" and
 *   "read note" could display different messages.
 */
winNote: readable
    iscrawlable = true
sdesc =""
rdesc =""
ddesc =""
vdesc =""
tdesc =""
pdesc =""
noun = '' '' '' '' '' '#t' 
'' '' '' '' '' '#t'
    ldesc = "!   ! ,  
       WarpMaster 2000(tm),    .
    \"  \"! "
    location = room4
isHer = true
;

shiproom: room
    sdesc = "<b>   </b>"
    ldesc = "         .
       ,    ;   
       .    . "
    north = chuteroom
    in = shipinterior
    up =
    {
        "    ,    . ";
        return( nil );
    }
;

/*
 *   The receptacle is both a fixeditem (something that can't be taken and
 *   carried around) and a container, so both superclasses are specified.
 */
shiprecept: fixeditem, container
sdesc =""
rdesc =""
ddesc =""
vdesc =""
tdesc =""
pdesc =""
noun = '' '' '' '' '' '#t' '#d'
 '' '' '' '' '' '#t' '#d'
    location = shiproom
    ioPutIn( actor, dobj ) =
    {
        /*
         *   We only want to allow the warp motivator to be put in here.
         *   Check any object going in and disallow it if it's anything else.
         */
        if ( dobj = motivator )
        {
            "  . ";
            dobj.moveInto( self );
        }
        else " . ";
    }
;

spaceship: fixeditem
sdesc =" "
rdesc =" "
ddesc =" "
vdesc =" "
tdesc =" "
pdesc =" "
adjective = '' '' '' '' ''
noun = '' '' '' '' ''  '' '' '' '' '' '' '' ''
    location = shiproom
    ldesc =
    {
        "         
         ,    .   ";
        if ( motivator.location = shiprecept )
            "      . ";
        else
            "     . 
              \"   \". ";
        "        . ";
    }
    verDoEnter( actor ) = {}
    doEnter( actor ) = { self.doBoard( actor ); }
    verDoBoard( actor ) = {}
    doBoard( actor ) =
    {
        actor.travelTo( shipinterior );
    }
isHim = true
;

shipinterior: room
    sdesc = "<b> </b>"
    out = shiproom
    ldesc = "    .     
          ,     \"\". "
;

/*
 *   This is a fake "ship" that the player can refer to while inside the
 *   spaceship.  This allows commands such as "look at ship" and "get out"
 *   to work properly while within the ship.
 */
shipship: fixeditem
    location = shipinterior
sdesc =" "
rdesc =" "
ddesc =" "
vdesc =" "
tdesc =" "
pdesc =" "
adjective = '' '' '' '' ''
noun = '' '' '' '' '' '' '' '' '' ''
    adjective = ''
    ldesc = { shipinterior.ldesc; }
    verDoUnboard( actor ) = {}
    doUnboard( actor ) =
    {
        Me.travelTo( shipinterior.out );
    }
isHim = true
;

/*
 *   A buttonitem can be pushed.  It's automatically a fixeditem, and
 *   always defines the noun 'button'.  The doPush method specifies what
 *   happens when the button is pushed.
 */
launchbutton: buttonitem
sdesc =" "
rdesc =" "
ddesc =" "
vdesc =" "
tdesc =" "
pdesc =" "
noun = '' '' '' '' ''
adjective = '' '' '#r' '#r'
    location = shipinterior
    doPush( actor ) =
    {
        if ( motivator.location = shiprecept )
        {
            incscore( 10 );
            "   . \"  \", - 
               . ";
            if ( lloyd.location = Me.location )
                "\n\t   . \",\"  , \"  
                  .      
                 ,     ,   .
                ,     \"  \". \"
                       .\n\t";
            "        .
                 .
              :
            \"  ... ... ... ... ... !\".\n\t
             ,   ,    ,
                    .
            \n\t ,      
            \"  \",    
              TADS  High Energy Software.\b";
            
            scoreRank();
            terminate();
            quit();
            abort;
        }
        else
        {
            "      :
            \":    \". ";
        }
    }
isHer = true
;

motivator: treasure
sdesc =" "
rdesc =" "
ddesc =" "
vdesc =" "
tdesc =" "
pdesc =" "
noun = '' '' '' '' 
 '#t' '' 'warpmaster'
    takevalue = 20
    adjective = '' '' '2000' '#r' '#r' '2000#r'
    location = room4
    ldesc = " WarpMaster 2000(tm),   . "
isHim = true
;

foodthing: fooditem
sdesc =""
rdesc =""
ddesc =""
vdesc =""
tdesc =""
pdesc =""
noun = '' '' '' '' '' '#t' '' '' '#r' '' '' '#t' ''
'' '' '' '' '#t' ''
adjective = '' '' '' '' '#t' '' '' '#r'
    ldesc = "     ,    . "
    location = room3drawer
isHer = true
;

/*
 *   An openable is a container, with the additional property that it can
 *   be opened and closed.  To simplify things, we don't have a separate
 *   cap; use of the cap is implied in the "open" and "close" commands.
 */
bottle: openable
sdesc ="  "
rdesc ="  "
ddesc ="  "
vdesc ="  "
tdesc ="  "
pdesc ="  "
adjective = '' '' '#t' '' '' '' '#t' ''
noun = '' '' '' '' '' '#t' 
'' '' '' '#t'
    location = wastebasket
    isopen = nil
    ldesc =
    {
        " ";
        if ( self.isfull ) "  . ";
        else ". ";

        " "; if ( self.isopen ) ". "; else ". ";
        
        if ( funnel.location = self )
            "   . ";
    }
    verDoLookin(actor)={}
    doLookin(actor)={self.ldesc;}
    ioPutIn( actor, dobj ) =
    {
        if ( not self.isopen )
        {
            "  ,   . ";
        }
        else if ( dobj = ln2 )
        {
            if ( funnel.location = self )
            {
                "      . ";
                bottle.isfull := true;
            }
            else
            {
                "         . ";
            }
        }
        else if ( dobj = funnel )
        {
            " ! ";
            funnel.moveInto( self );
        }
        else "    . ";
    }
    verIoPourIn( actor ) = {}
    ioPourIn( actor, dobj ) = { self.ioPutIn( actor, dobj ); }
    doClose( actor ) =
    {
        if ( funnel.location = self )
            "   . ";
        else if ( self.isfull )
        {
            "     ,   
                   ,   ,
               ,    . ";
            notify( self, #explodeWarning, 3 );
            self.isopen := nil;
        }
        else
        {
            ",  . ";
            self.isopen := nil;
        }
    }
    
    /*
     *   explodeWarning is sent to the bottle as a "notification," which is
     *   a message that's scheduled to occur some number of turns in the
     *   future.  In this case, closing the bottle while it has liquid
     *   nitrogen inside will set the explodeWarning notification.
     */
    explodeWarning =
    {
        if ( not self.isopen )
        {
            if ( self.isIn( Me.location ))
            {
                "\b    ,          . ";
            }
            notify( self, #explode, 3 );
        }
    }
    
    /*
     *   explode is set as a notification by explodeWarning.  This routine
     *   actually causes the explosion.  Since the bottle explodes, we will
     *   remove it from the game (by moving it into "nil").  If the bottle
     *   is in the right place, we'll also do some useful things.
     */
    explode =
    {
        if ( not self.isopen )
        {
            "\b";
            if ( self.location = banksafe )
            {
                if ( Me.location = bankvault )
                {
                    "\t    .
                              . ";
                     
                    if ( lloyd.location = bankvault )
                        "\n\t     . \", !\", -  .
                        \"        !\" ";
                }
                else
                    "  ,  . ";
                
                banksafe.isopen := true;
                banksafe.isblasted := true;
            }
            else if ( self.isIn( Me.location ))
            {
                "      
                 .    ,     ,
                  . ";
                die();
                abort;
            }
            else
            {
                "   . ";
            }
            
            self.moveInto( nil );
        }
    }
isHer = true
;

/*
 *   We're defining our own "class" of objects here.  Up to now, we've been
 *   using classes defined in "adv.t", which you will recall we included at
 *   the top of the file.  This class has some useful properties.  For one,
 *   it has an attribute (istreasure) that tells us that the object is
 *   indeed a treasure (used in the slot in room 4's door to ensure that
 *   an object can be put in the slot).  In addition, it supplements the
 *   doTake routine:  when a treasure is taken for the first time, we'll
 *   add the object's "takevalue" to the overall score.
 */
class treasure: item
    istreasure = true
    takevalue = 5       // default point value when object is taken
    putvalue = 5        // default point value when object is put in slot
    doTake( actor ) =
    {
        if ( not self.hasScored )       // have we scored yet?
        {
            incscore( self.takevalue ); // add our "takevalue" to the score
            self.hasScored := true;     // note that we have scored
        }
        pass doTake;        // continue with the normal doTake from "item"
    }
;

alley1: room
    sdesc = "<b>  </b>"
    ldesc =
    {
        "       ,   
          (       )   , 
          .   ( 3)   .   ";
        if ( alley1door.isopen ) ""; else "  ";
        " ,     .          . ";
    }
    east = startroom
    north = alley1n
    west =
    {
        /*
         *   Sometimes, we'll want to run some code when the player walks
         *   in a particular direction rather than just go to a new room.
         *   This is how.  Returning "nil" means that the player can't go
         *   this way.
         */
        if ( alley1door.isopen )
        {
            return( room4 );
        }
        else
        {
            "   .    ,    . ";
            return( nil );
        }
    }
    south = breezeway
    out = breezeway
;

alley1n: room
    sdesc = "<b>  </b>"
    ldesc = "      .      . "
    south = alley1
    west = alley1comp
;

alley1comp: room
    sdesc = "<b> </b>"
    ldesc = 
    {
        "    .  ,   
           .   . ";
        if ( not self.isseen ) notify( compstudents, #converse, 0 );
    }
    east = alley1n
    out = alley1n
;

alley1pc: fixeditem
sdesc =" "
rdesc =" "
ddesc =" "
vdesc =" "
tdesc =" "
pdesc =" "
adjective = '' '' '' '' '#t' ''
noun = '' '' '' '' '' '#t' '#d'
'' '' '' '' '' '' '' '' '' ''
'' '' '' '' ''
    location = alley1comp
    ldesc = "    .
        , ,      .
            ,  - ,     
     ,  . "
    verIoTypeOn( actor ) = {}
    ioTypeOn( actor, dobj ) =
    {
        "  .   ,     . ";
    }
    verDoTurnoff( actor ) = {}
    doTurnoff( actor ) =
    {
        "     ,      . ";
    }
isHim = true
;

compstudents: Actor
    location = alley1comp
sdesc =""
rdesc =""
ddesc =""
vdesc =""
tdesc =""
pdesc =""
noun = '' '' '' '' '' '#d'
    ldesc = "          . "
    actorAction( v, d, p, i ) =
    {
        "    . ";
        exit;
    }
    doAskAbout( actor, io ) =
    {
        "  ,  - . ";
    }
    actorDesc = "     ,  .
        . "
    state = 0
    converse =
    {
        if ( Me.location = self.location )
        {
            "\b";
            if ( self.state = 0 )
                "\"   ?\"     .     
                 ,  . ";
            else if ( self.state = 1 )
                "\"!\"    . \"   ?\"
                    . \",
                    !\" ";
            else if ( self.state = 2 )
                "  ,   ,    ,
                  - -   \"  \".
                \", !    ! 
                !\" ";
            else
                "  . ";
        
            self.state := self.state + 1;
        }
    }
isThem = true
;

class lockedDoor: fixeditem, keyedLockable
    isopen = nil
    islocked = true
sdesc =""
rdesc =""
ddesc =""
vdesc =""
tdesc =""
pdesc =""
noun = '' '' '' '#t'
    ldesc =
    {
        if ( self.isopen ) ". ";
        else if ( self.islocked ) "  . ";
        else ". ";
    }
    verIoPutIn( actor ) = { "    ! "; }
isHer = true
;

alley1door: lockedDoor
    location = alley1
    ldesc =
    {
        if ( self.isopen ) " . ";
        else
            "   .    ,
                . ";
    }
;

alley1sign: fixeditem, readable
rdesc =""
ddesc =""
vdesc =""
tdesc =""
pdesc =""
adjective = '' '' '' '#t' '#d'
noun = '' '' '' '' ''
'' '' '' '' '' '#t' '#d'
'' '' '' '#t'
'' '' '' '' ''
    location = alley1
    ldesc = " :
     \b\t\t    !
     \b  -   .   ,  
       .
        ,    , 
      ,        . 
         ,    
       .   :
     \b\t  
     \n\t  
     \n\t  
     \n\t 
     \b        .  !
     \b  : , ,  
     \"  \".       .
       ,     . "
isHer = true
;

alley1slot: fixeditem, container
sdesc =""
rdesc =""
ddesc =""
vdesc =""
tdesc =""
pdesc =""
noun = '' '' '' '#t' '' '' '' '#t' '' '' '' '' ''
    location = alley1
    itemcount = 0
    ioPutIn( actor, dobj ) =
    {
        if ( dobj.istreasure )
        {
            "<<ZA(dToS(dobj,&sdesc))>> <<glsok(0, dobj.isHim, dobj.isHer, dobj.isThem)>>  . ";
            dobj.moveInto( nil );
            incscore( dobj.putvalue );
            self.itemcount := self.itemcount + 1;
            if ( self.itemcount = 4 )
            {
                "     ,  
                    .   
                  ,    . ";
                alley1door.islocked := nil;
                alley1door.isopen := true;
            }
        }
        else
        {
            "   ,   . ";
        }
    }
isHer = true
;

breezeway: room
    sdesc = "<b></b>"
    ldesc = "   ,     
       ,    .    . "
    north = alley1
    east = courtyard
    west = orangeWalk1
;

courtyard: room
    sdesc = "<b></b>"
    ldesc = "      .
         .
         ,     . "
    east = lounge
    down = hall1
    west = breezeway
;

lounge: room
    sdesc = "<b> </b>"
    ldesc = "   .    ,     . "
    north = diningRoom
    west = courtyard
    out = courtyard
;

diningRoom: room
    sdesc = "<b></b>"
    ldesc = "   .      .
        ,       . "
    east = kitchen
    south = lounge
;

diningTable: surface, fixeditem
sdesc =" "
rdesc =" "
ddesc =" "
vdesc =" "
tdesc =" "
pdesc =" "
adjective = '' '' '' '' '#t' ''
noun = '' '' '' '' '' '#t' 
    location = diningRoom
isHim = true
;

fishfood: fooditem
sdesc =" "
rdesc =" "
ddesc =" "
vdesc =" "
tdesc =" "
pdesc =" "
adjective = '' '' '' '' '#t' ''
noun = '' '' '' '' '#t' ''
    ldesc = "    ,   
         .    
         . "
    location = diningTable
isHim = true
;

kitchen: room
    sdesc = "<b></b>"
    ldesc = "   .       .
        . "
    west = diningRoom
    out = diningRoom
;

toxicolaMachine: fixeditem, container
sdesc =" "
rdesc =" "
ddesc =" "
vdesc =" "
tdesc =" "
pdesc =" "
adjective = '' '' '#r' '#r'
noun = '' '' '' '' ''  '' '' ''
 '' '' '' '' ''  '' '' '' ''
 '' '#t' '#d'
    ldesc =
    {
        "  ,     
          .  , -,  ,      .
            ,    ,
               . ";
         if ( cup.location = self )
            "   . ";
    }
    location = kitchen
    ioPutIn( actor, dobj ) =
    {
        if ( dobj <> cup )
            "    . ";
        else pass ioPutIn;
    }
isHer = true
;

cup: container
sdesc =" "
rdesc =" "
ddesc =" "
vdesc =" "
tdesc =" "
pdesc =" "
adjective = '' '' '' '' '#t' ''
noun =  '' '' '' '' '#t' '#d' '' '' '' '' '#t' '' '' '' '' '' '' '#t' '#d' ''
plural='' ''
    isFull = nil
    ldesc =
    {
        if ( self.isFull ) "    . ";
        else " . ";
    }
    location = diningTable
    ioPutIn( actor, dobj ) =
    {
        if ( dobj = ln2 ) "     . ";
        else "    . ";
    }
verDoLookin(actor)={}
doLookin(actor)={self.ldesc;}
isHim = true
;

toxicola: fixeditem
sdesc =""
rdesc =""
ddesc =""
vdesc =""
tdesc =""
pdesc =""
noun = '' '' '' '' '' '#t'  '' '' '' '' '' '#t' '' '' '' '#t'
    ldesc = "   .   . "
    doTake( actor ) =
    {
        "   . ";
    }
    verDoDrink( actor ) = {}
    doDrink( actor ) =
    {
        "  ,    .   
              ,   ,
           .    
           .   ,
            -    ,   . ";
        cup.isFull := nil;
        self.moveInto( nil );
    }
isHer = true
;

toxicolaButton: buttonitem
    location = kitchen
sdesc =""
rdesc =""
ddesc =""
vdesc =""
tdesc =""
pdesc =""
noun = '' '' '' '' ''
    doPush( actor ) =
    {
        if ( cup.location = toxicolaMachine )
        {
            if ( cup.isFull )
                "           . ";
            else
            {
                "   ,   ,  . ";
                cup.isFull := true;
                toxicola.moveInto( cup );
            }
        }
        else
        {
            "            . ";
        }
    }
isHer = true
;

orangeWalk1: room
    sdesc = "<b>  </b>"
    ldesc = "     . 
       ,         . "
    east = breezeway
    north = orangeWalk2
;

/*
 *   We want a "class" of objects for the orange trees lining the
 *   orange walk.  They don't do anything, so they're "decoration."
 */
class orangeTree: decoration
sdesc =" "
rdesc =" "
ddesc =" "
vdesc =" "
tdesc =" "
pdesc =" "
adjective = '' '' '' ''
noun = '' '' '' '' ''
'' '' '' '' '' '#d' '#t'
ldesc = "   . "
    verDoClimb( actor ) = {}
    doClimb( actor ) =
    {
        "     ,   ,
            ,    . ";
    }
isThem=true
;

orangeTree1: orangeTree
    location = orangeWalk1
;

orangeWalk2: room
    sdesc = "<b>  </b>"
    ldesc = "      . 
                . "
    north = quad
    south = orangeWalk1
;

orangeTree2: orangeTree
    location = orangeWalk2
;

quad: room
    sdesc = "<b></b>"
    ldesc = "   -     
     .   -   ;   -;
      \"  \"  ;       .
     \n\t ,     ,  
      , ,     .    ,
       .       , 
        .
               ,    . "
    south = orangeWalk2
    nw = bookstore
    west = walkway
    ne = healthCenter
    north = BandG
;

BandG: room
    sdesc = "<b>&</b>"
    ldesc = "   \"  \".   . "
    south = quad
    out = quad
;

bngmemo: readable
sdesc =" "
rdesc =" "
ddesc =" "
vdesc =" "
tdesc =" "
pdesc =" "
noun = '' '' '' ''  '#t' ''
'' '' '' '' '#t' '#d'
 '' '' '' '' '' '#d' '#t'
'' '' '' '' '' '#t'
adjective = '' '' '' '#r' '#r' '#r'
iscrawlable = true
location = BandG
ldesc = "   ;  
  -  : 
     \b\t293 --   
     \n\t322 --   2
     \n\t612 --  
     \b\t   . "
isHim = true
;

healthCenter: room
    sdesc = "<b></b>"
    ldesc = "  .    , 
          .      
     ,     . "
    sw = quad
    out = quad
;

healthDesk: fixeditem, surface
sdesc =" "
rdesc =" "
ddesc =" "
vdesc =" "
tdesc =" "
pdesc =" "
adjective = '' '' '' '' '#t' ''
 '' '' '' '' '#t' ''
noun = '' '' '' '' '' '#t' '#d'
    location = healthCenter
isHim = true
;

healthMemo: readable
sdesc =" "
rdesc =" "
ddesc =" "
vdesc =" "
tdesc =" "
pdesc =" "
adjective = '' '' '' '' '#t' ''
noun = '' '' '' '' '' '#t' '#d'
'' '' '' '' '' '#t' '#d'
    iscrawlable = true
    plural = '' '' ''
    location = healthDesk
    ldesc =
       ":\t\t 
      \n:\t\t  
      \n:\t\t 
      \b    , 
       ,   ,  .
         , ,   , 
         .   ,  ,  ,   
        ,  ,   .
         ,    .
            
          .   ,
              
       ,   . "
isHim = true
;

students: decoration
sdesc =""
rdesc =""
ddesc =""
vdesc =""
tdesc =""
pdesc =""
noun = '' '' '' '' '' '#t' '#d'
    location = quad
    ldesc = "      
 .          
      \"  \". 
       ,   . "
isThem = true
;

presscorp: decoration
sdesc =""
rdesc =""
ddesc =""
vdesc =""
tdesc =""
pdesc =""
noun = '' '' '' '' '' '#t' '#d'
 '' '' '' '' '' '#d'
location = quad
isThem = true
;

flask: container
sdesc =""
rdesc =""
ddesc =""
vdesc =""
tdesc =""
pdesc =""
adjective = ''
noun = '' '' '' '' '#t' ''
    ldesc = ",    ; , -  ,
            . "
    location = quad
    ioPutIn( actor, dobj ) =
    {
        "   ,   "; dobj.vdesc; "     ,
           ,  . ";
    }
verDoEmpty(actor) = " . "
isHim = true
;

pourVerb: deepverb
    sdesc = ""
    verb = '' '' '' '' '' '' 
    doAction = 'Pour'
    ioAction( inPrep ) = 'PourIn'
    ioAction( onPrep ) = 'PourOn'
;

ln2: item
sdesc =" "
rdesc =" "
ddesc =" "
vdesc =" "
tdesc =" "
pdesc =" "
adjective = '' '' '' '' '#t' ''
noun = '' '' '' '' '#t' ''
    ldesc = "    -   ,       . "
    location = flask
    doTake( actor ) =
    {
        "    . ";
    }
    verDoPour( actor ) = {}
    verDoPourIn( actor, io ) = {}
    doPour( actor ) =
    {
        askio( inPrep );
    }
isHim = true
;

walkway: room
    sdesc = "<b></b>"
    ldesc = "  .       ;
         .    . "
    east = quad
    north = behaviorLab
    south = security
    west = walkway2
;

walkway2: room
    sdesc = "<b></b>"
    ldesc = "  ,   .     . "
    east = walkway
    south = explosiveLab
    north = biobuilding
;

biobuilding: room
    sdesc = "<b> </b>"
    ldesc = "   .   . "
    south = walkway2
    out = walkway2
;

bionotes: readable
sdesc =""
rdesc =""
ddesc =""
vdesc =""
tdesc =""
pdesc =""
noun = '' '' '' '' '#t' ''
    location = biobuilding
    ldesc = "      , 
       .    ,  
          ( ,   
     - )         ,
       .   ,    .
     ,      .   
              3,
         99,    . 
            ; 
       ,          . "
isHim = true
;

explosiveLab: room
    sdesc = "<b>  </b>"
    ldesc = "      .
     \" \" ,       
      .  ,     
        ,   .   . "
    north = walkway2
    out = walkway2
;

ln2doc: readable
sdesc =""
rdesc =""
ddesc =""
vdesc =""
tdesc =""
pdesc =""
noun = '' '' '' '' '#t'
    location = explosiveLab
    ldesc = "     .   
       ,    
      (..   )    
      ( ),      
      , ,     ,  ,
         -    .
         ,  ,     
      ,       ,  ,  
        .       
     ,     ,  .
     \b     : \"!    
     !        
     ;      . 
           .\" "
isHer = true
;

bookstore: room
    sdesc = "<b> </b>"
    ldesc = "   .   .  ,
        ,     ,
      ,    .   -. "
    out = { return( self.se ); }
    se =
    {
        /*
         *   We need to check that the battery has been paid for, if it's
         *   been taken.  If not, don't let 'em out.
         */
        if ( battery.isIn( Me ) and not battery.isPaid )
        {
            " ,     . \" \", - 
              ,   . ";
            return( nil );
        }
        else return( quad );
    }
;

battery: item
    location = bookstore
    isPaid = nil
sdesc =""
rdesc =""
ddesc =""
vdesc =""
tdesc =""
pdesc =""
noun = '' '' '' '' '' '#t' '#d' '' '' '' '' '' '#t'
    verDoPayfor( actor ) =
    {
        if ( actor.location <> bookstore )
            "    ! ";
    }
    doPayfor( actor ) =
    {
        clerk.doPay( actor );
    }
    verIoPayFor( actor ) = {}
    ioPayFor( actor, dobj ) =
    {
        if ( dobj <> clerk )
        {
            "    "; dobj.ddesc; "  "; self.vdesc; "! ";
        }
        else clerk.doPay( actor );
    }
isHer = true
;

payVerb: deepverb
    sdesc = ""
    verb = '' '' ''
    doAction = 'Pay'
    ioAction( forPrep ) = 'PayFor'
;

payforVerb: deepverb
    verb = ' ' ' ' ' '
    sdesc = " "
    doAction = 'Payfor'
;

money: item
sdesc =" "
rdesc =" "
ddesc =" "
vdesc =" "
tdesc =" "
pdesc =" "
adjective = '' '' '#t' '' ''
noun = '' '' '' '' '' '#t' '#d'
'' '' '' '' '' '#t' '#d'
iscrawlable = true
isHer = true
;

clerk: Actor
sdesc =""
rdesc =""
ddesc =""
vdesc =""
tdesc =""
pdesc =""
noun = '' '' '' '' '' '#t' '#d'
'' '' '' '' '' '#d' '#t'
'' '' '' '' '' '#d' '#t'
    location = bookstore
    isHer = true
    ldesc = " -  ,       
          . "
    actorDesc = "   ,      
    (      ). "
    verIoGiveTo( actor ) = {}
    ioGiveTo( actor, dobj ) =
    {
        if ( dobj = money or dobj = dollar ) self.doPay( actor );
        else if ( dobj = darbcard )
        {
            "  . \"\", -  , \"   \". ";
        }
        else if ( dobj = cup and cup.isFull )
        {
            "\"     \", -  , \" 
             !  ,    ? , 
                  ,    ,  .\" ";
        }
        else
        {
            "  . ";
        }
    }
    verDoPay( actor ) = {}
    doPay( actor ) =
    {
        if ( not battery.isIn( Me ))
        {
            " . \"  ,     \", -  .
             Ÿ        . ";
        }
        else if ( battery.isPaid )
        {
            "   : \"  !\"
            Ÿ        . ";
        }
        else if ( not money.isIn( Me ))
        {
            if ( dollar.isIn( Me ))
            {
                "  ,     .
                    .  ; \" 
                  ,    !\" ";
            }
            else
            {
                " ,    ,      . ";
            }
        }
        else
        {
            "     \",   .\"";
            battery.isPaid := true;
            money.moveInto( nil );
        }
    }
;

behaviorLab: room
    sdesc = "<b> </b>"
    ldesc = "   .    ,
         .   \" -
        \".  ,    \" \". "
    south = walkway
    out = walkway
    east = mazeview
    north =
    {
        "   .  ,    
        \"\"?";
        return( nil );
    }
;

mazeview: room
    sdesc = "<b>  </b>"
    ldesc =
    {
        "      ,       ,
        ,  , . 
         (.e. )    , ";
         
        if ( not mazeStart.isseen )
            "     ,     .
                    :\b";
        else
            "       :\b";
            
        mazeplaque.ldesc;
    }
    west = behaviorLab
;

mazewindow: fixeditem
sdesc =""
rdesc =""
ddesc =""
vdesc =""
tdesc =""
pdesc =""
noun = '' '' '' '' '' '#t' '#d'
    location = mazeview
    verDoLookthru( actor ) = {}
    doLookthru( actor ) = { self.ldesc; }
    ldesc =
    {
        "     ,    , 
         , .     ";
        if ( not mazeStart.isseen )
            ",      ,     .";
        ". ";
    }
;

mazeplaque: fixeditem, readable
sdesc =""
rdesc =""
ddesc =""
vdesc =""
tdesc =""
pdesc =""
noun = '' '' '' '' '' '#t' '#d'
    location = mazeview
    ldesc = "*** -     ***
     \b- ,   ,    
              .
         ,  ,      
        ,   ,    
      ,  ,   ,   . "
isHer = true
;

behaviorDoor: lockedDoor
    ldesc = "   .   : \"   \"."
    location = behaviorLab
;

security: room
    sdesc = "<b>  </b>"
    ldesc = "      --    . 
      ; ,       
         .      .
       . "
    north = walkway
    out = walkway
;

securityDesk: fixeditem, surface
sdesc =""
rdesc =""
ddesc =""
vdesc =""
tdesc =""
pdesc =""
noun = '' '' '' '' '' '#t' '#d'
ldesc=" "
location = security
isHim = true
;

securityMemo: readable
sdesc ="  "
rdesc ="  "
ddesc ="  "
vdesc ="  "
tdesc ="  "
pdesc ="  "
noun= '' '' '' '' '' '#t'
adjective = '' '' '#r' '#r'
iscrawlable = true
plural = '' ''
location = securityDesk
    ldesc =
       ":\t\t  
      \n:\t\t  
      \n:\t\t  
      \b      ,   
        ,    \"  \",
          .        
           ,     ,
              . 
       ,          
                 . "
isHer = true
;

flashlight: container, lightsource
sdesc =""
rdesc =""
ddesc =""
vdesc =""
tdesc =""
pdesc =""
noun = '' '' '' '' '' '#t' '#d'
'' '' '' '' '' '' '#t' '#t' '#d'
    location = security
    ioPutIn( actor, dobj ) =
    {
        if ( dobj <> battery )
        {
            "    "; dobj.vdesc; "  . ";
        }
        else pass ioPutIn;
    }
    Grab( obj ) =
    {
        /*
         *   Grab( obj ) is invoked whenever an object 'obj' that was
         *   previously located within me is removed.  If the battery is
         *   removed, the flashlight turns off.
         */
        if ( obj = battery ) self.islit := nil;
    }
    ldesc =
    {
        if ( battery.location = self )
        {
            if ( self.islit )
                " (  )   
                 ,   . ";
            else
                " (  )  . ";
        }
        else
        {
            " . ,    . ";
        }
    }
    verDoTurnon( actor ) =
    {
        if ( self.islit ) "  ! ";
    }
    doTurnon( actor ) =
    {
        if ( battery.location = self )
        {
            "  . ";
            self.islit := true;
        }
        else "     . ";
    }
    verDoTurnoff( actor ) =
    {
        if ( not self.islit ) "  . ";
    }
    doTurnoff( actor ) =
    {
        ",   . ";
        self.islit := nil;
    }
isHim = true
;

hall1: room
    sdesc = "<b></b>"
    ldesc = "     .   . "
    up = courtyard
    east = hall2
;

hall2: room
    sdesc = "<b></b>"
    ldesc = "   ,     .      . "
    west = hall1
    east = hall3
    north = hall4
;

hall3: room
    sdesc = "<b></b>"
    ldesc = "     .    . "
    west = hall2
    north = laundry
;

laundry: room
    sdesc = "<b></b>"
    ldesc = "  .     .   . "
    south = hall3
    out = hall3
;

washingMachine: fixeditem, openable
sdesc =" "
rdesc =" "
ddesc =" "
vdesc =" "
tdesc =" "
pdesc =" "
adjective = '' '' '' '#t' '#d'
noun = '' '' '' '' '' '#t' '#d'
 '' '' '' '' '' '#t' '#d'
isopen = nil
location = laundry
isHer = true
;

jeans: item
sdesc =" "
rdesc =" "
ddesc =" "
vdesc =" "
tdesc =" "
pdesc =" "
adjective = '' '' '' '' '#t'
noun = '' '' '' '' '' '#t' '#d'
'' '' '' '' '' '#t'
    location = washingMachine
    ldesc =
    {
        if ( self.isseen ) "   . ";
        else
        {
            "   ;   ,  .
              ,   ,     . ";
            masterKey.moveInto( Me.location );
            self.isseen := true;
        }
    }
    verDoLookin( actor ) = {}
    doLookin( actor ) = { self.ldesc; }
    verDoSearch(actor)={}
    doSearch(actor)={self.ldesc;}
    verDoWear( actor ) = { "  . "; }
isThem = true
;

masterKey: keyItem
    iscrawlable = true
sdesc =""
rdesc =""
ddesc =""
vdesc =""
tdesc =""
pdesc =""
adjective = '' '' '' '' '#t' ''
noun = '' '' '' '' '' '#t' '#d'
'' '' '' '' '' '#t' '#d'
    doTake( actor ) =
    {
        if ( not self.isseen )
        {
            "**         
                ,        ,
                 ,      ,
                  ,       
             ,    .   ,  - ,  
               ,           ... ";
            self.isseen := true;
        }
        pass doTake;
    }
isHer = true
;

hall4: room
    sdesc = "<b></b>"
    ldesc = "   ,     . "
    south = hall2
    north = hall5
;

hall5: room
    sdesc = "<b></b>"
    ldesc = "    ,     . "
    south = hall4
    east = hall6
;

hall6: room
    sdesc = "<b></b>"
    ldesc = "    .      \"\",   
        . "
    west = hall5
    north = storage
;

storage: room
    sdesc = "<b></b>"
    ldesc =
    {
        "   .        
          (  ,  - ). 
          .    ";
         if ( tunnelDoor.isopen ) ""; else ""; " . 
              . ";
    }
    south = hall6
    north =
    {
        if ( guard.isActive )
        {
            "      . ";
            return( nil );
        }
        else if ( not tunnelDoor.isopen )
        {
            "     . ";
            setit( tunnelDoor );       /* "it" now refers to the closed door */
            return( nil );
        }
        else return( tunnel1 );
    }
    enterRoom( actor ) =
    {
        if ( guard.isActive )
        {
            notify( guard, #patrol, 0 );
        }
        pass enterRoom;
    }
    leaveRoom( actor ) =
    {
        if ( guard.isActive )
        {
            unnotify( guard, #patrol );
        }
        pass leaveRoom;
    }
;

tunnelDoor: lockedDoor
    location = storage
    mykey = masterKey
isHer = true
;

guard: Actor
sdesc =""
rdesc =""
ddesc =""
vdesc =""
tdesc =""
pdesc =""
noun = '' '' '' '' '' '#t' '#d'
'' '' '' '' '' '#t' '#d' '' '' '' '' '' '#d' '#t'
 '' '' '' '' '' '#d' '#t'
    isHim = true
    ldesc =
    {
        " -   ,     ,
              ( , ). ";
        if ( not self.isActive )    "  ,   . ";
    }
    isActive = true
    location = storage
    actorDesc =
    {
        if ( self.isActive )
            "      .
                ,  ,   
              . ";
        else
            "        ,
                .  . ";
    }
    verIoGiveTo( actor ) =
    {
        if ( not self.isActive )
            " . ";
    }
    ioGiveTo( actor, dobj ) =
    {
        if ( dobj = dollar or dobj = money )
            "    . \"   , 
                  !\", -
              ,   . ";
        else if ( dobj <> cup or not cup.isFull )
            "  . ";
        else
        {
            self.isActive := nil;
            unnotify( self, #patrol );
            "\t    . \"!  !\", -
               ,    ,
             .      . \" !  
              ,    !\",-   .
            \n\t,        .
            \"    ...\", -    ,    
              . ";
            cup.isFull := nil;
            toxicola.moveInto( nil );
            cup.moveInto( cardtable );
            incscore( 10 );
        }
    }
    patrolMessage =
    [
        // random message 1
        '    .'
        // random message 2
        '    , ,  - .'
        // random message 3
        '   ,    -.'
        // random message 4
        '  -   ,        .'
        // random message 5
        '      ,  ,   ,    .'
    ]
    patrol =
    {
        if ( self.location = Me.location )
        {
            "\b";
            say( self.patrolMessage[rand( 5 )]);
        }
    }
;

cardtable: fixeditem, surface
sdesc =" "
rdesc =" "
ddesc =" "
vdesc =" "
tdesc =" "
pdesc =" "
adjective = '' '' '' '' ''
noun = '' '' '' '' '' '#t' '#d'
'' '' '' '' '' '#t' '#d'
ldesc="    ."
isHim = true
location = storage
;

emptyglass: container
sdesc =" "
rdesc =" "
ddesc =" "
vdesc =" "
tdesc =" "
pdesc =" "
adjective = '' '' '' '' '#t' ''
noun = '' '' '' '' '' '#t' '#d'
plural='' ''
    location = cardtable
    doTake( actor ) =
    {
        if ( guard.isActive )
            "    . \" \", -
             . ";
        else pass doTake;
    }
    ioPutIn( actor, dobj ) =
    {
        if ( dobj = ln2 ) "    . ";
        else "    . ";
    }
isHim = true
;

tunnelSounds: function( parm )
{
    if ( Me.location.istunnel )
    {
        "\b";
        say( tunnelroom.randomsound[rand( 5 )]);
    }
    setfuse( tunnelSounds, rand( 5 ), nil );
}

class tunnelroom: room
    istunnel = true
    sdesc = "<b> </b>"
    randomsound = [
    // random sound 1
      '    , ,   ,
          .'
    // random sound 2
      '  ,    
      ,  ,      ,
              .'
    // random sound 3
      '     .'
    // random sound 4
      '    .   
         .'
    // random sound 5
      '  ,    , 
      - .     .'
    ]
;

tunnelpipes: fixeditem
sdesc =""
rdesc =""
ddesc =""
vdesc =""
tdesc =""
pdesc =""
noun = '' '' '' '' '' '#t' '#d'
 '' '' '' '' '#d' '#t'
    ldesc = "       ,
      ,        .
          \" - 35 \". "
    locationOK = true
    location =
    {
        if ( Me.location.istunnel ) return( Me.location );
        else return( nil );
    }
isThem = true
;

tunnel1: tunnelroom
    ldesc = "   .     .
           ;    
          .  
      ,        .
       ,      .
     \n\t     .     . "
    east = tunnel2
    west = tunnel3
    south = storage
    enterRoom( actor ) =
    {
        if ( not self.isseen ) setfuse( tunnelSounds, rand( 5 ), nil );
        pass enterRoom;
    }
;

tunnel2: tunnelroom
    ldesc = "    . "
    west = tunnel1
;

tunnelStorage: room
    sdesc = "<b></b>"
    ldesc = "   .   . "
    north = tunnel13
;

tunnel3: tunnelroom
    ldesc = "        . "
    east = tunnel1
    west = tunnel4
;

tunnel4: tunnelroom
    ldesc = "  T-  .    ,   . "
    east = tunnel3
    north = tunnel5
    south = tunnel6
;

darktunnel: darkroom, tunnelroom
    controlon = nil
    islit =
    {
        if ( self.controlon ) return( true );
        else pass islit;
    }
;

tunnel5: darktunnel
    ldesc = "   :       . "
    west = tunnel7
    south = tunnel4
;

tunnel6: tunnelroom
    ldesc = "   :       . "
    west = tunnel8
    north = tunnel4
;

tunnel7: darktunnel
    ldesc = "        .
            , 
         . "
    east = tunnel5
    west = tunnel9
    north =
    {
        return(crawltest( tunnel12,
         '          .  
              ,    .
              .' ));
    }
;

tunnel8: tunnelroom
    controlon = true
    ldesc =
    {
        "        .    . ";
        if ( not self.controlon )
        {
            "     ;   
                  ";
             
            if (( flashlight.isIn( Me ) or flashlight.isIn( tunnel8 ))
             and flashlight.islit )
                " (  , )";
        
            ". ";
        }
    }
    east = tunnel6
    west = tunnel10
;

controlunit: fixeditem
sdesc =" "
rdesc =" "
ddesc =" "
vdesc =" "
tdesc =" "
pdesc =" "
noun = '' '' '' '' '' '#d' '#t'
adjective= '' '#r'
    location = tunnel8
    ldesc =
    {
        "    ,    .
              ,      .
            \" 2\". ";
        controldisp.ldesc;
    }
    objref =
    {
        local l;
        
        l := self.value;
        
        if ( l = 322 ) return( tunnel8 );
        else if ( l = 612 ) return( mazeroom );
        else if ( l = 293 ) return( darktunnel );
        else return( nil );
    }
    value = 322
isHim = true
;

controlpad: fixeditem
sdesc =""
rdesc =""
ddesc =""
vdesc =""
tdesc =""
pdesc =""
adjective = '' '' '' '' '#r'
noun = '' '' '' '' '' '#t'
'' '' '' '' ''
'' '' '' '' '' '#t'
 '' '' '' '' '' '#t'
'' '' '' '' '' '#t'
    location = tunnel8
    ldesc = "     ,  
       .        0  9. "
    verIoTypeOn( actor ) = {}
    ioTypeOn( actor, dobj ) =
    {
        if ( dobj <> numObj && !(dobj=strObj && cvtstr(cvtnum(strObj.value))=strObj.value))
        "     . ";
        else
        {
            "    ,   . ";
            if (dobj!=strObj) controlunit.value := numObj.value;
            else controlunit.value := cvtnum( strObj.value );
            controldisp.ldesc;
        }
    }
isThem = true
;

controlbutton: buttonitem
sdesc =""
rdesc =""
ddesc =""
vdesc =""
tdesc =""
pdesc =""
adjective = '' '' '#t' '' '' '' '#t' ''
noun = '' '' '' '' '' '#t' '#d'
    location = tunnel8
    doPush( actor ) =
    {
        local r;
        
        r := controlunit.objref;
        if ( r )
        {
            " ,    . ";
            r.controlon := not r.controlon;
            controldisp.ldesc;
            if ( r = tunnel8 )
            {
                "  ";
                if ( r.controlon )
                    " . ";
                else
                {
                    " ,     ";
                    if (( flashlight.isIn( Me ) or flashlight.isIn( tunnel8 ))
                     and flashlight.islit )
                        "( , ) ";
                    "  . ";
                }
            }
        }
        else " ,     . ";
    }
isHer = true
;

controldisp: fixeditem
sdesc =""
rdesc =""
ddesc =""
vdesc =""
tdesc =""
pdesc =""
adjective = '' '' '' '' '' '#t'
noun = '' '' '' '' '' '#t' '#d'
 '' '' '' '' '' '#t' '#d'
    location = tunnel8
    ldesc =
    {
        local l, r;
        
        l := controlunit.value;
        r := controlunit.objref;
        "   \""; say( l ); ": ";       
        if ( r )
        {
            if ( r.controlon ) ""; else "";
        }
        else "???";
        
        "\". ";
    }
;

tunnel9: darktunnel
    ldesc = "   .         . "
    east = tunnel7
    south = tunnel11
;

tunnel10: tunnelroom
    ldesc = "   .       . "
    east = tunnel8
    north = tunnel11
;

tunnel11: tunnelroom
    ldesc = "        .      . "
    north = tunnel9
    south = tunnel10
;

chute: fixeditem, container
sdesc =""
rdesc =""
ddesc =""
vdesc =""
tdesc =""
pdesc =""
noun = '' '' '' '#t' ''
 '#t' '#d' '' '' '' '' '' '#t' '#d'
'' '' '' '' '' '#t' '#d'
adjective = '''#r'
location = tunnel11
ldesc = "      ,  ,   .
      ,   ,  ,  . "
ioPutIn( actor, dobj ) =
{
 "  "; dobj.vdesc;
 "  ,  <<iao(dobj)>> <<iao(dobj)>>  .    
   . ";
  dobj.moveInto(chuteroom);
}
isHim = true
;

crawltest: function( rm, msg )
{
    local i;
    
    i := 1;
    while ( i <= length( Me.contents ))
    {
        if ( not Me.contents[i].iscrawlable )
        {
            "    ";
            Me.contents[i].vdesc; ". ";
            return( nil );
        }
        i := i + 1;
    }
    say( msg ); "\b";
    return( rm );
}

tunnel12: tunnelroom
    ldesc =
    {
        if ( self.isseen )
            "        . 
                   . ";
        else
            "    ,     
              .       .
            ,      .      ,
                    ,   
             . ";
    }
    east = tunnel13
    west = pitTop
    south =
    {
        return(crawltest( tunnel7, '  ,   
            ,    .' ));
    }
;

tunnel13: tunnelroom
    ldesc = "    .     , 
      . "
    west = tunnel12
    north = maze0
    south = tunnelStorage
;

maze0: room
    sdesc = "<b> </b>"
    ldesc =
    {
        "         . 
              \"  \".
            , ,   , - 
          . ";
         
        if ( mazeroom.controlon )
            "  .   ,     . ";
        
        if ( not self.isseen )
        {
            "\n\t  , ,     
               . ";
            if ( mazeview.isseen )
                "     ,        
                 ,    , ,  ,
                  ,       ,  
                       ,     . ";
            else
                ",      . ,  
                     ... ";
        }
    }
    south = tunnel13
    north = maze1
;

mazeequip: decoration
sdesc =""
rdesc =""
ddesc =""
vdesc =""
tdesc =""
pdesc =""
noun = '' '' '' '' '' '#t' '#d'
'' '' '' '' '' '#t' '#d'
'' '' '' '' '' '#t' '#d'
'' '' '' '' '' '#t' '#d'
'' '' '' '' '' '#t' '#d'
'' '' '' '' '' '#t' '#d' '' '' '' '' '' '#t'
'' '' '' '' '' '#t' '#d'
adjective = '' '' '' '' ''
    location = maze0
    ldesc =
    {
        "   ,  .
            ,   
          .    
                . ";
        
        if ( mazeroom.controlon )
            "  .   ,     . ";
    }
;

mazeroom: room
    controlon = true
    sdesc = "<b>-  </b>"
    lookAround( verbosity ) =
    {
        self.statusLine;
        self.nrmLkAround( self.controlon ? true : verbosity );
    }
    ldesc =
    {
        if ( self.controlon )
            "    . -    ,
                      . ";
        else
        {
            local cnt, tot, i;
            
            tot := 0;
            i := 1;
            while ( i <= 4 )
            {
                if ( self.dirlist[i] ) tot := tot + 1;
                i := i + 1;
            }

            "     .   .
                ";
            
            i := 1;
            cnt := 0;
            while ( i <= 4 )
            {
                if ( self.dirlist[i] )
                {
                    if ( cnt > 0 )
                    {
                        if ( tot = 2 ) "  ";
                        else if ( cnt+1 = tot ) "  ";
                        else ", ";
                    }
                    cnt := cnt + 1;
                    
                    say( ['' '' '' ''][i] );
                }
                i := i + 1;
            }
            ". ";
        }
    }
    mazetravel( rm ) =
    {
        if ( self.controlon )
        {
            local r;
            
            "       .  
             ,       .
                   ,   - ,
                 ...\b";
            
            /*
             *   We know we can only go one of four directions, but generate
             *   a random number up to 6; if we generate 5 or 6, we won't go
             *   anywhere, but we won't let on that this is the case.
             */
            r := rand( 6 );
            
            /*
             *   Note that we want to confuse the player in active-maze mode
             *   as much as possible, so we don't want any clues as to whether
             *   there was any travel in this direction or not.  So, return
             *   "self" rather than "nil," since we won't get any message if
             *   we return "nil," but we'll get the current room's message if
             *   we return "self;" since all the messages are the same, this
             *   won't provide any information.
             */
            if ( r < 5 )
            {
                r := self.dirlist[ r ];
                if ( r ) return( r );
                else return( self );
            }
            else return( self );
        }
        else
        {
            if ( rm )
                return( rm );
            else
            {
                "    . ";
                return( nil );
            }
        }
    }
    north = ( self.mazetravel( self.dirlist[1] ))
    south = ( self.mazetravel( self.dirlist[2] ))
    east = ( self.mazetravel( self.dirlist[3] ))
    west = ( self.mazetravel( self.dirlist[4] ))
    up = ( self.mazetravel( 0 ))
    down = ( self.mazetravel( 0 ))
    in = ( self.mazetravel( 0 ))
    out = ( self.mazetravel( 0 ))
    ne = ( self.mazetravel( 0 ))
    nw = ( self.mazetravel( 0 ))
    se = ( self.mazetravel( 0 ))
    sw = ( self.mazetravel( 0 ))
;

maze1: mazeroom
    dirlist = [ 0 maze0 maze2 0 ]
;

maze2: mazeroom
    dirlist = [ maze9 0 maze3 maze1 ]
;

maze3: mazeroom
    dirlist = [ maze8 0 maze4 maze2 ]
;

maze4: mazeroom
    dirlist = [ 0 0 maze5 maze3 ]
;

maze5: mazeroom
    dirlist = [ maze6 0 0 maze4 ]
;

maze6: mazeroom
    dirlist = [ 0 maze5 0 maze7 ]
;

maze7: mazeroom
    dirlist = [ maze30 0 maze9 maze8 ]
;

maze8: mazeroom
    dirlist = [ maze29 maze3 maze7 0 ]
;

maze9: mazeroom
    dirlist = [ 0 maze2 0 maze10 ]
;

maze10: mazeroom
    dirlist = [ 0 0 maze9 maze11 ]
;

maze11: mazeroom
    dirlist = [ 0 maze35 maze10 maze12 ]
;

maze12: mazeroom
    dirlist = [ 0 0 maze11 0 ]
;

maze13: mazeroom
    dirlist = [ maze24 maze18 0 0 ]
;

maze14: mazeroom
    dirlist = [ 0 maze17 0 maze15 ]
;

maze15: mazeroom
    dirlist = [ 0 maze16 maze14 0 ]
;

maze16: mazeroom
    dirlist = [ maze15 maze23 maze17 0 ]
;

maze17: mazeroom
    dirlist = [ maze14 maze22 maze18 maze16 ]
;

maze18: mazeroom
    dirlist = [ maze13 maze21 0 maze17 ]
;

maze19: mazeroom
    dirlist = [ 0 maze20 maze35 0 ]
;

maze20: mazeroom
    dirlist = [ maze19 0 0 maze21 ]
;

maze21: mazeroom
    dirlist = [ maze18 0 maze20 0 ]
;

maze22: mazeroom
    dirlist = [ maze17 0 0 0 ]
;

maze23: mazeroom
    dirlist = [ maze16 0 0 0 ]
;

maze24: mazeroom
    dirlist = [ 0 maze13 maze25 0 ]
;

maze25: mazeroom
    dirlist = [ maze33 0 maze26 maze24 ]
;

maze26: mazeroom
    dirlist = [ maze32 0 0 maze25 ]
;

maze27: mazeroom
    dirlist = [ maze31 0 0 0 ]
;

maze28: mazeroom
    dirlist = [ 0 0 maze29 0 ]
;

maze29: mazeroom
    dirlist = [ 0 maze8 maze30 maze28 ]
;

maze30: mazeroom
    dirlist = [ 0 maze7 0 maze29 ]
;

maze31: mazeroom
    dirlist = [ 0 maze27 0 maze32 ]
;

maze32: mazeroom
    dirlist = [ 0 maze26 maze31 0 ]
;

maze33: mazeroom
    dirlist = [ 0 maze25 0 maze34 ]
;

maze34: mazeroom
    dirlist = [ 0 0 maze33 mazeStart ]
;

maze35: mazeroom
    dirlist = [ maze11 0 0 maze19 ]
;

mazeStart: room
    sdesc = "<b> </b>"
    ldesc = "   .     ,
       ,  \"\",  . "
    east = maze34
    south = behaviorLab
;

mazedoor: fixeditem
sdesc =""
rdesc =""
ddesc =""
vdesc =""
tdesc =""
pdesc =""
noun = '' '' '' '#t' '#d' ''  '' '' '' '' 
adjective = '' '' '' '' '' '' '' ''
    verDoOpen( actor ) = { " ;    . "; }
    location = mazeStart
isHer = true
;

chuteroom: room
    sdesc = "<b>  </b>"
    ldesc = "       -
      .      . "
    nw = pitBottom
    south = shiproom
;

chute2: fixeditem, container
sdesc =""
rdesc =""
ddesc =""
vdesc =""
tdesc =""
pdesc =""
noun = '' '' '' '' '' '#t' '#d'
    location = chuteroom
    ioPutIn( actor, dobj ) =
    {
        "  ;     . ";
    }
isHim = true
;

pitTop: tunnelroom
    sdesc = "<b> </b>"
    ldesc =
    {
        "       .  
            ,  .  
          . ";
        if ( rope.tieItem = railing )
            "       . ";
    }
    east = tunnel12
    down =
    {
        if ( rope.tieItem = railing )
        {
            "   ...\b";
            return( pitBottom );
        }
        else
        {
            " ,    ,   . ";
            return( nil );
        }
    }
;

tieVerb: deepverb
    sdesc = ""
    verb = ''
    prepDefault = toPrep
    ioAction( toPrep ) = 'TieTo'
;

railing: fixeditem
sdesc =""
rdesc =""
ddesc =""
vdesc =""
tdesc =""
pdesc =""
noun = '' '' '' '' '#t' '' 
    location = pitTop
    verIoTieTo( actor ) = {}
    ioTieTo( actor, dobj ) =
    {
        "      ,     .
         ,    . ";
        rope.tieItem := self;
        rope.moveInto( pitTop );
        rope.isfixed := true;
    }
isThem = true
;

climbupVerb: deepverb
    verb = ''
    sdesc = ""
    doAction = 'Climbup'
;

climbdownVerb: deepverb
    verb = '' ' '
    sdesc = ""
    doAction = 'Climbdown'
;

rope: item
sdesc =""
rdesc =""
ddesc =""
vdesc =""
tdesc =""
pdesc =""
noun = '' '' '' '' '' '#t' '#d' '' '' '' '' '' '#t' '' '' '' '' '' '#t'
    isListed = ( not self.isfixed )
    noun = 'rope'
    location = tunnelStorage
    ldesc =
    {
        if ( self.tieItem )
        {
            "   "; self.tieItem.ddesc; ". ";
        }
        else pass ldesc;
    }
    verDoTieTo( actor, io ) =
    {
        if ( self.tieItem )
        {
            "    "; self.tieItem.ddesc; "! ";
        }
    }
    doTake( actor ) =
    {
        if ( self.tieItem )
        {
            "(   , .) ";
            self.tieItem := nil;
            self.isfixed := nil;
        }
        pass doTake;
    }
    verDoClimb( actor ) =
    {
        if ( self.tieItem = nil )
            "          . ";
    }
    doClimb( actor ) =
    {
        "   ...\b";
        Me.travelTo( pitBottom );
    }
    verDoClimbdown( actor ) = {}
    doClimbdown( actor ) = { self.doClimb( actor ); }
isHer = true
;

pitBottom: room
    sdesc = "<b> </b>"
    ldesc = "      . , ,
          ;      ,
       -  .     
         : \"   - 1982\".
              .
        .      , ,
     , , -  -. "
    up =
    {
        "   ,    - .\b";
        return( pitTop );
    }
    se = chuteroom
    east = biohall1
    south = bank
    north = cave
    sw = insOffice
    west = machineshop
;

pitplaque: fixeditem, readable
sdesc =" "
rdesc =" "
ddesc =" "
vdesc =" "
tdesc =" "
pdesc =" "
adjective = '' '' '' '' '' '' 
noun = '' '' '' '' '' '#t' '#d'
    location = pitBottom
    ldesc = "  \n\t\t\t1982"
isHer = true
;

insOffice: room
    sdesc = "<b> </b>"
    ldesc =
    {
        "   .     ,
           -  .   -. ";
        if ( not self.isseen )
        {
            "\n\t     ,   ,
                  -
            ,       ,
              . \",   -   \",-
                 .
            \",    ! ,
                .\"
            \n\t   ,     ,
                 ,       .
             - ,    .  ,
               ,   .
            \n\t\"     .     
               !\"      . "; 
            
            notify( lloyd, #offerins, 0 );
        }
    }
    ne = pitBottom
    out = pitBottom
;

/*
 *   Lloyd the Friendly Insurance Robot is a full-featured actor. He will
 *   initially just wait in his office until paid for a policy, but will
 *   thereafter follow the player around relentlessly.  Lloyd doesn't
 *   interact much, though; he just hangs around and does wacky things.
 */
lloyd: Actor
sdesc =""
rdesc =""
ddesc =""
vdesc =""
tdesc =""
pdesc =""
adjective = '' '' '' '#d' '' ''
noun = '' '' '' '' '' '#t' '#d'
'' '' '' '' '' '#t' '#d'
'' '' '' '' '' '#d' '#t'
    isHim = true
    ldesc = " -    -  
          .   
          . "
    actorAction( v, d, p, i ) =
    {
        if ( v = helloVerb )
            "\"\", -   . ";
        else if ( v = followVerb and d = Me )
        {
            if ( self.offering )
                "\",        .\" ";
            else
                "\"   \", -  . \"  
                       .
                     .\" ";
        }
        else
            "\",     .\" ";
        exit;
    }
    verDoAskAbout( actor, io ) = {}
    doAskAbout( actor, io ) =
    {
        if ( io = policy )
            "\"    ,     \",-
                 . \",     ,
                 ,     ,    .\" ";
        else
            "\",      \", -  . ";
    }
    actorDesc =
    {
        if ( self.offering )
            "   ,        . ";
        else
            "  ,   -. ";
    }
    offering = true
    offermsg =
    [
        '       .'
        '      ,     .'
        '    . "  ", -  .'
        '    . "  ,      ?"
         -    .'
        '  , ,    ,    .'
    ]
    offerins =
    {
        if ( self.location = Me.location )
        {
            "\b";
            say(self.offermsg[rand( 5 )]);
        }
    }
    followmsg =
    [
        '    ,     .'
        '      .'
        '      - .'
        '   .'
        '     .'
    ]
    follow =
    {
        if ( Me.location = machinestorage ) return;
        
        "\b";
        if ( self.location <> Me.location )
        {
            if ( Me.location = railcar )
                "   ,       . ";
            else if ( self.location = railcar )
                "   . ";
            else if ( Me.location = pitTop and self.location = pitBottom )
                ",         . ";
            else if ( Me.location = pitBottom and self.location = pitTop )
                "        . ";
            else if (( Me.location = tunnel7 and self.location = tunnel12 )
             or ( Me.location = tunnel12 and self.location = tunnel7 ))
                "      . ";
            else if ( Me.location = quad )
                "   .      ,
                  . ,   , 
                    ,  . ";
            else if ( Me.location = storage )
                "   .  , 
                      .  , ,
                 ,    . \"   \", -   .
                \"         .\" ";
            else
                "   . ";
                
            self.moveInto( Me.location );
        }
        else
        {
            say(self.followmsg[rand( 5 )]);
        }
    }
    verIoGiveTo( actor ) = {}
    ioGiveTo( actor, dobj ) =
    {
        if ( dobj = dollar )
            self.doPay( actor );
        else if ( dobj = darbcard )
            "    . \",     
             \", -  . ";
        else
            "  . ";
    }
    verDoPay( actor ) = {}
    doPay( actor ) =
    {
        if ( not self.offering )
        {
            " . \"         !\" ";
        }
        else if ( dollar.isIn( actor ))
        {
            "       .
            \"  ,     \",
             . \" :   , 
              .    ,  
             , ,   ,    
                , ,   
                 ,     
             !\" ";
            
            unnotify( self, #offerins );
            notify( self, #follow, 0 );
            dollar.moveInto( nil );
            policy.moveInto( Me );
            self.offering := nil;
            Me.isinsured := true;
        }
        else
        {
            "       . ";
        }
    }
    location = insOffice
;

policy: readable
sdesc =" "
rdesc =" "
ddesc =" "
vdesc =" "
tdesc =" "
pdesc =" "
adjective = '' '' '' '' '' '' '' ''
noun = '' '' '' '' '' '#t' '#d' 
'' '' '' '' '' '#t' '#d'

    iscrawlable = true
    location = lloyd
    ldesc =
    {
        "        .
          ,      , 
           ";
        if ( lloyd.offering )
            " ,    . ";
        else
            "       
             .     . ";
    }
isHim = true
;

machineshop: room
    sdesc = "<b></b>"
    ldesc = "  .     ,
          , 
         .      ,
          -    .
       ,    . "
    east = pitBottom
    out = pitBottom
    north = machinestorage
;

machine: fixeditem
sdesc =""
rdesc =""
ddesc =""
vdesc =""
tdesc =""
pdesc =""
noun = '' '' '' '' '' '#t' '#d'
 '' '' '' '' '' '#t' '#d'
disamvdesc=""
    location = machineshop
    ldesc = "    -   ;   
      .       ,  
         \"!\" "
isHer = true
;

machinebutton: buttonitem
    location = machineshop
sdesc =" "
rdesc =" "
ddesc =" "
vdesc =" "
tdesc =" "
pdesc =" "
adjective = '' '' ''
noun = '' '' '' '' '' '#t' '#d'
    doPush( actor ) =
    {
        "   ,     ,
              
           .     ,
            .  ,      , 
           ,        ,
            . ";
        
        if ( lloyd.location = self.location )
        {
            "\n\t";
            if ( self.isscored )
                "    . \"   ,
                  ,  . ,      
                 ,     .\", -  
                 . \"  ,   .  
                ,   \", -  . ";
            else
            {
                self.isscored := true;
                "     . \" \", -
                 ,    ,   .
                        .
                \"     ... ,
                , .      .\"  
                      ,
                    ( ). ";
                money.moveInto( Me );
            }
        }
    }
isHer = true
;

machinestorage: darkroom
    sdesc = "<b></b>"
    ldesc = "   .    . "
    south = machineshop
    out = machineshop
;

happygear: treasure
    location = machinestorage
sdesc ="  "
rdesc ="  "
ddesc ="  "
vdesc ="  "
tdesc ="  "
pdesc ="  "
adjective = '' '' '' '' '#r' '' ''
noun = '' '' '' '' '' '#t' '' '' '' '' '' '#t'
    ldesc = "     .        . "
isHer = true
;

bank: room
    sdesc = "<b></b>"
    ldesc = " ,        .
       .        . "
    north = pitBottom
    south = bankvault
    out = pitBottom
;

bankvault: room
    sdesc = "<b></b>"
    ldesc = "    .
         . "
    north = bank
    out = bank
;

banksafe: fixeditem, openable
sdesc =""
rdesc =""
ddesc =""
vdesc =""
tdesc =""
pdesc =""
adjective = '' '' ''
noun = '' '' '' '' '' '#t' '#d'
 '' '' '' '#t' '' '' '' '' '#t' '#d'
    location = bankvault
    isopen = nil
    ldesc =
    {
        if ( self.isblasted )
        {
            "  ,     .
                ,    . ";
            pass ldesc;
        }
        else
        {
            " ,  ,     .
                ( ),
              ,  \" Ҹ \". ";
        }
    }
    doOpen( actor ) =
    {
        if ( self.isblasted ) pass doOpen;
        else
            "        . ";
    }
isHim = true
;

darbcard: treasure
sdesc =" "
rdesc =" "
ddesc =" "
vdesc =" "
tdesc =" "
pdesc =" "
adjective = ''
noun = '' '' '' '' '' '#t' '#d'
location = banksafe
isHer=true
;

bankslot: fixeditem, container
sdesc ="  "
rdesc ="  "
ddesc ="  "
vdesc ="  "
tdesc ="  "
pdesc ="  "
noun = '' '' '' '' '' '' '' '' '' '' '' '' '#d' '#t'
 '' '' '' '#t' '#d'
adjective = '' '' '#r' '#r'
    location = bankvault
    ldesc = "  ,   ,   
      .  ,     ,   
     ,    ,   . "
    ioPutIn( actor, dobj ) =
    {
        "<<ZA(dToS(dobj,&sdesc))>> <<glsok(2, dobj.isHim, dobj.isHer, dobj.isThem)>>  . ";
        dobj.moveInto( banksafe );
    }
;

cave: room
    sdesc = "<b></b>"
    ldesc = "    -,      . "
    north = railway1
    down = railway1
    south = pitBottom
    up = pitBottom
;

railway1: room
    sdesc = "<b> </b>"
    ldesc = "       .   
      <<displayLink( '> -  ./.     . . ', '' )>>. ,           .
     ,     .  ,    
           .    . "
    south = cave
    destination = railway2
    tunneltime = 3
    east =
    {
        "  .       . ";
        return( nil );
    }
;

class rtunnel: fixeditem
sdesc = "<b></b>"
rdesc =""
ddesc =""
vdesc =""
tdesc =""
pdesc =""
noun = '' '' '' '' '' '#t' '#d'
    verDoEnter( actor ) = {}
    doEnter( actor ) =
    {
        railway1.east;
    }
;

rtunnel1: rtunnel
    location = railway1
;

rtunnel2: rtunnel
    location = railway2
;

railway2: room
    sdesc = "<b> </b>"
    ldesc = "   .         
      .  (  )   . 
          . "
    south = computercenter
    destination = railway1
    tunneltime = 3
    west =
    {
        "  .       . ";
        return( nil );
    }
;

computercenter: room
    sdesc = "<b> </b>"
    ldesc = "   .  ,  
           .
       . "
    north = railway2
;

compequip: decoration
sdesc =" "
rdesc =" "
ddesc =" "
vdesc =" "
tdesc =" "
pdesc =" "
adjective = '' '' '' '' ''
noun = '' '' '' '' ''
'#t' '#d' '' '' '' '' '' 
'' '' '' '' '' 
    location = computercenter
    ldesc = "  .     . "
;

randombook: treasure, readable
sdesc =""
rdesc =""
ddesc =""
vdesc =""
tdesc =""
pdesc =""
noun = '' '' '' '' '' '#t' '#d' '"  "'
adjective = '' '' '' '' '' '#r' '#r' '#r' 
    ldesc = "  \"  \".  
      ,       ,
          0  999,999 ( 
        ). "
    location = computercenter
isHer=true
;

railtunnel: room
    sdesc = "<b></b>"
    ldesc = "   ,    . "
;

railcar: fixeditem, vehicle, container
    location = railway1
    isdroploc = true        // stuff dropped while on board ends up in railcar
    sdesc =""
    rdesc =""
    ddesc =""
    vdesc =""
    tdesc =""
    pdesc =""
    noun = '' '' '' '' '' '#t' '#d'
     '' '' '' '' '' '#t' '#d'
    ldesc =
    {
        if (Me=self) "   ,     . ";
        else "  ,     . ";
        "   ,     ,
               \"\". ";
         if ( funnel.location = railhole )
            "   . ";
         railmeter.ldesc;
    }
    travel =
    {
        if ( self.location <> railtunnel )
        {
            "\b       . 
                       
              . ";
            self.destination := self.location.destination;
            self.tunneltime := self.location.tunneltime;
            self.moveInto( railtunnel );
        }
        else if ( self.tunneltime > 0 )
        {
            "\b      . ";
            self.tunneltime := self.tunneltime - 1;
        }
        else
        {
            "\b   .   
                  .   
                  . ";
            self.isActive := nil;
            self.moveInto( self.destination );
            unnotify( self, #travel );
        }
    }
    checkunboard =
    {
        if ( self.isActive )
        {
            ",     , 
                \" \". (  ,
                 \" \",   ,
              .) ";
            return( nil );
        }
        else return( true );
    }
    out =
    {
        if ( self.checkunboard ) pass out;
        else return( nil );
    }
    doUnboard( actor ) =
    {
        if ( self.checkunboard ) pass doUnboard;
    }
isHer = true
;

railmeter: fixeditem
    location = railcar
sdesc =""
rdesc =""
ddesc =""
vdesc =""
tdesc =""
pdesc =""
noun = '' '' '' '' '' '#t' '#d' '' '' '' '' ''
    ldesc =
    {
        "      ,  
             ,    . 
            ";
        if ( railcar.iscooled ) ""; else "";
        "  . ";
    }
isHer = true
;

railhole: fixeditem, container
    location = railcar
sdesc =""
rdesc =""
ddesc =""
vdesc =""
tdesc =""
pdesc =""
adjective = '' '' ''
noun = '' '' '' '' '' '' '' '' '' '' '#t' '#d' '' '' '' '' '' '#t' '#d' 
    ldesc =
    {
        if ( funnel.location = self )
            "  . ";
        else if ( railcar.iscooled )
            "˸     . ";
        else
            "  \"\".     . ";
    }
    verIoPourIn( actor ) = {}
    ioPourIn( actor, dobj ) = { self.ioPutIn( actor, dobj ); }
    ioPutIn( actor, dobj ) =
    {
        if ( dobj = funnel )
        {
            " ! ";
            funnel.moveInto( self );
        }
        else if ( dobj = ln2 )
        {
            if ( funnel.location = self )
            {
                if ( railcar.iscooled )
                    "    ! ";
                else
                {
                    "      .
                      . ";
                    railcar.iscooled := true;
                }
            }
            else
            {
                "  ,    ,      
                ,    . ";
            }
        }
        else
        {
            "  . ";
        }
    }
isHer = true
;

railbutton: buttonitem
    location = railcar
sdesc =" "
rdesc =" "
ddesc =" "
vdesc =" "
tdesc =" "
pdesc =" "
adjective = '' '' ''
noun = '' '' '' '' '' '#t' '#d'
    doPush( actor ) =
    {
        if ( Me.location <> railcar )
        {
            "   . ";
        }
        else if ( railcar.isbroken )
        {
            "  . ";
        }
        else if ( railcar.isActive )
        {
            "  .   ,
                  . ";
        }
        else if ( railcar.iscooled )
        {
            "   - . 
                 . ";
            notify( railcar, #travel, 0 );
            railcar.isActive := true;
        }
        else
        {
            "   - .    ,
                 .    
             . ,     , 
                  . , 
              . ";
            railcar.isbroken := true;
        }
    }
ldesc=" "
isHer = true
;

biohall1: room
    sdesc = "<b></b>"
    ldesc = "  ,     .  \"-\"   
       . "
    west = pitBottom
    east = biohall2
    south = biolab
;

biolab: room
    sdesc = "<b>-</b>"
    ldesc =
    {
        "  -.      .
             ,    -   (";
        if ( biocabinet.isopen ) ""; else "";
        ").    . ";
    }
    north = biohall1
;

bioEquipment: decoration
sdesc =" "
rdesc =" "
ddesc =" "
vdesc =" "
tdesc =" "
pdesc =" "
adjective = '' '' '' '' ''
noun = '' '' '' ''
'' '#t' '#d'
'' '' '' '' '' '#t' 
    location = biolab
    ldesc = "     . "
isThem = true
;

biobench: fixeditem, surface
sdesc =" "
rdesc =" "
ddesc =" "
vdesc =" "
tdesc =" "
pdesc =" "
adjective = '' '' '' '' ''
noun = '' '' '' '' '' '#t' '#d'
    location = biolab
    ldesc =
    {
        "        ,
        , ,    . ";
        pass ldesc;
    }
isHim = true
;

funnel: container
sdesc =""
rdesc =""
ddesc =""
vdesc =""
tdesc =""
pdesc =""
noun = '' '' '' '' '' '#t' '#d'
    location = biobench
    ioPutIn( actor, dobj ) =
    {
        if ( dobj = ln2 )
        {
            if ( self.location = railhole or self.location = bottle )
                self.location.ioPutIn( actor, dobj );
            else
                "    ,   . ";
        }
        else
        {
            "  ";
            dobj.vdesc; "  ,     . ";
        }
    }
    ldesc =
    {
        if ( self.location = bottle or self.location = railhole )
        {
            "   "; self.location.vdesc; ". ";
        }
        else
            "       
              ,     . ";
    }
    verIoPourIn( actor ) = {}
    ioPourIn( actor, dobj ) = { self.ioPutIn( actor, dobj ); }
isHer = true
;

biohall2: room
    sdesc = ""
    ldesc =
    {
        "    ,     .   . ";
        if ( not self.isseen )
        {
            notify( biocreature, #menace, 0 );
        }
    }
    west = biohall1
    east =
    {
        if ( biocreature.location = self )
        {
            if ( slime.location = nil )
            {
                "  ,   ,      . ";
                slime.moveInto( Me );
                slime.isworn := true;
            }
            else "    . ";
            return( nil );
        }
        else return( biooffice );
    }
;

slime: clothingItem
sdesc =" "
rdesc =" "
ddesc =" "
vdesc =" "
tdesc =" "
pdesc =" "
noun = '' '' '' '' '' '#t' '#d' '' '' '#t' ''
adjective = '' '#r'
    doWear( actor ) =
    {
        ", . ";
    }
isHer = true
;

biocreature: Actor
    location = biohall2
sdesc =""
rdesc =""
ddesc =""
vdesc =""
tdesc =""
pdesc =""
noun = '' '' '' '' '' '#t' '#d'
'' '' '' '' '' '#t' '#d'
'' '' '' '' '#d' '#t'
    ldesc = "     ,
       (  ,    ). "
    actorDesc = "     .
       (,    ,   ,   ,  \"\", -
       ), ,   -  .  
       -        . "
    menaceMessage =
    [
        '     .'
        '  .'
        '  ,     
           .     .  
                .  
         ,     "."'
        '    .'
        '   ,     .'
    ]
    menace =
    {
        if ( self.location = Me.location )
        {
            "\b";
            say( self.menaceMessage[rand( 5 )]);
        }
    }
;

biooffice: room
    sdesc = "<b> </b>"
    ldesc = "   .    .
       . "
    west = biohall2
;

biodesk: fixeditem, surface
sdesc =" "
rdesc =" "
ddesc =" "
vdesc =" "
tdesc =" "
pdesc =" "
adjective = '' '' '' '' ''
noun = '' '' '' '' '' '#t' '#d'
    location = biooffice
isHim = true
;

biocabinet: openable, fixeditem
sdesc =" "
rdesc =" "
ddesc =" "
vdesc =" "
tdesc =" "
pdesc =" "
adjective = '' '' '' '' ''
noun = '' '' '' '' '' '#t' '#d'
 '' '' '' '' '' '#t' '#d'
    location = biolab
    isopen = nil
isHim = true
;

class chemitem: item
    location = biocabinet
sdesc =" "
rdesc =" "
ddesc =" "
vdesc =" "
tdesc =" "
pdesc =" "
adjective = '' '' '' '' '' 
noun = '' '' '' '' '' '#t' '#d'
    plural = '' '' '' '' '' ''
 '' '#t' '#d'
    ldesc =
    {
        " -   .   - 
           \""; self.sdesc; "\". ";
    }
;
    
gfxq3: chemitem
adjective = '-3' '3' '' '-3#r' '3#r' '#r'
sdesc = " -3"
rdesc =" -3"
ddesc =" -3"
vdesc =" -3"
tdesc =" -3"
pdesc =" -3"
;

gfxq9: chemitem
adjective = '-9' '9' '' '-9#r' '9#r' '#r'
sdesc =" -9"
rdesc =" -9"
ddesc =" -9"
vdesc =" -9"
tdesc =" -9"
pdesc =" -9"
;

polyred: chemitem
sdesc =" "
rdesc =" "
ddesc =" "
vdesc =" "
tdesc =" "
pdesc =" "
adjective = '' '' '' '' ''
noun = '' '' '' '' '' '#t' '#d'
isHim = true
;


polyblue: chemitem
sdesc =" "
rdesc =" "
ddesc =" "
vdesc =" "
tdesc =" "
pdesc =" "
adjective = '' '' '' '' ''
noun = '' '' '' '' '' '#t' '#d'
isHim = true
;

compoundT99: chemitem
    adjective = '99' '99#r'
    noun = ''
sdesc = " T99"
rdesc =" T99"
ddesc =" T99"
vdesc =" T99"
tdesc =" T99"
pdesc =" T99"
;

compoundT30: chemitem
   adjective  = '30' '30#r'
   noun = ''
   sdesc = " 30"
rdesc =" 30"
ddesc =" 30"
vdesc =" 30"
tdesc =" 30"
pdesc =" 30"
;

clonemaster: container
sdesc =""
rdesc =""
ddesc =""
vdesc =""
tdesc =""
pdesc =""
adjective = '' '' ''
noun =  '' '' '' '' '' '#t' '#d'
 '' '' '' '' '' '#t' '#d'
    location = biobench
    ldesc =
    {
        " -  .    
         \"\",    . ";
         clonerecept.ldesc;
    }
    ioPutIn( actor, dobj ) = { clonerecept.ioPutIn( actor, dobj ); }
    verDoTakeOut( actor, io ) = { clonerecept.verIoTakeOut( actor, io ); }
isHim = true
;

clonerecept: fixeditem, container
sdesc =""
rdesc =""
ddesc =""
vdesc =""
tdesc =""
pdesc =""
adjective = '' '' '' '' ''
noun = '' '' '' '' '' '#t' '' '' '' '' '' '#t' '#d'
    location = clonemaster
    ldesc =
    {
        "    . ";
        pass ldesc;
    }
isHim = true
;

clonebutton: buttonitem
sdesc =" "
rdesc =" "
ddesc =" "
vdesc =" "
tdesc =" "
pdesc =" "
adjective = '' '#r' '' '#r'
noun = '' '' '' '' '' '#t' '#d'
    doPush( actor ) =
    {
        if ( slime.location = clonerecept )
        {
            "     . ";
            if ( gfxq3.location = clonerecept and
             polyblue.location = clonerecept and
             compoundT99.location = clonerecept
             and length( clonerecept.contents ) = 4 )
            {
                "   (,     
                 )    ";
                if ( Me.location = biocreature.location )
                {
                    ".         
                      .        
                       .  , 
                        . ";
                    biocreature.moveInto( nil );
                    unnotify( biocreature, #menace );
                }
                else
                    ", , ,    , . ";
            }
            else
                "   ,      ,
                  .     , 
                    . ";
                
            slime.moveInto( nil );
        }
        else "  . ";
    }
    location = clonemaster
isHer = true
;

omega: treasure
sdesc ="  "
rdesc ="  "
ddesc ="  "
vdesc ="  "
tdesc ="  "
pdesc ="  "
adjective = '' '' ''  '' '#r' '' '' '' 
noun = '' '' '' '#t'
    location = biodesk
    ldesc = "         .
        ,    ,   
     \"\".     \"  \". "
isHer=true
;

rope2: fixeditem
sdesc =""
rdesc =""
ddesc =""
vdesc =""
tdesc =""
pdesc =""
noun = '' '' '' '' '' '#t' '#d'
    location = pitBottom
    ldesc = "      . "
    verDoTake( actor ) =
    {
        ",    -  (   ,
             , -       ). ";
    }
    verDoClimbup( actor ) = {}
    doClimbup( actor ) = { self.doClimb( actor ); }
    verDoClimb( actor ) = {}
    doClimb( actor ) =
    {
        "   ,    - .\b";
        Me.travelTo( pitTop );
    }
isHer = true
;

modify HelpVerb
action(actor) =
    {
"<b></b>\n
\t\" \"(Ditch Day) -     , 
       \"\"    ,
    - ,        .
 ,          .
    \"\"    . ,
     ,      
     , ,  ,    .
    ,        
 .          . \b
<b>   ?</b>\n
         ,    .\n
   ,     ,    ,  ,  .\n 
     ,   , .\n
        ,
  , ,   .
 ,  ,  .   .
:\n
<i>
\n
\n
  \n
    \n
 \n
 007  </i>\n
 ..\b
<b>  :</b>\n
, ()\n
 -    \n
 - \n
 -     .  -   .\n
 -     \n
 -    \n
 -   \n
 -   \n
 () -     .\n
         .\n
LINKS -   ( -   )\n \b
 ,    ,    
<<displayLink( 'http://www.rtads.h-type.com', 'http://www.rtads.h-type.com' )>>.\n\b
     ().\n
      ,     ,
            .\b

     !\n \b";
}
;



//     ,   stdr.t



replace scoreRank: function
{
    local s;
    s := global.score;

 "    "; say(global.turnsofar);
    " ,     ";
    say(global.score); " ";
    if ((global.score>20) or (global.score<5))
    switch (global.score-(global.score/10)*10)
    { case 1: "";break;
      case 2: "";break;
      case 3: "";break;
      case 4: "";break;
    }
    "   ";
    say(global.maxscore);
",     ";
    
    if ( s < 10 ) "  ";
    else if ( s < 25 ) "";
    else if ( s < 40 ) "";
    else if ( s < 60 ) "";
    else if ( s < global.maxscore ) "";
    else "";
    
    ". ";
}

replace init: function
{
#ifdef USE_HTML_STATUS
    if (systemInfo(__SYSINFO_SYSINFO) != true
        || systemInfo(__SYSINFO_VERSION) < '2.2.4')
    {
        "\b\b\(WARNING! This game requires the TADS run-time version
        2.2.4 or higher.  You appear to be using an older version of the
        TADS run-time.  You can still attempt to run this game, but the
        game's screen display may not work properly.  If you experience
        any problems, you should try upgrading to the latest version of
        the TADS run-time.\)\b\b";
    }
#endif
    commonInit();
    "\t      .     ,
         , -  ,   
    .   : ,   
        , 
     .        
       ,     
      (   ,    ), 
        .\b";
    version.sdesc;              
    setdaemon(turncount, nil);                 // start the turn counter daemon
    setdaemon(sleepDaemon, nil);                      // start the sleep daemon
    setdaemon(eatDaemon, nil);                       // start the hunger daemon
    parserGetMe().location := startroom;     // move player to initial location
    startroom.lookAround(true);                      // show player where he is
    startroom.isseen := true;                  // note that we've seen the room
    scoreStatus(0, 0);                          // initialize the score display
}

modify version
    sdesc = "\(   \) (Ditch Day Drifter)
     \nInteractive Fiction by Michael J.\ Roberts
     \bRelease 1.0
     \nCopyright (c) 1990, 2002 Michael J.\ Roberts. All Rights Reserved.
     \nCopyright (c) 2003    ( 01.10.03)
     \nDeveloped with TADS: The Text Adventure Development System.
     \n  RTADS.\b "
;

replace commandPrompt: function(code)
{
"<font face='TADS-Input'>";
"\b&gt;"; 
}

replace die: function
{
    "\b***  ! ***\b";
    scoreRank();
    if ( Me.isinsured )
        "\b ,         
          .   ,     . ";
    "\b:   ,  , .\n    .\n";
    while (true)
    {
        local resp;

        "\n  , , ,  : >";
        resp := upper(input());
        if ((resp = '') or (resp='restore') or (resp=''))
        {
            resp := askfile('  :',
                            ASKFILE_PROMPT_OPEN, FILE_TYPE_SAVE);
            if (resp = nil)
                "  . ";
            else if (restore(resp))
                "  . ";
            else
            {
                parserGetMe().location.lookAround(true);
                scoreStatus(global.score, global.turnsofar);
                abort;
            }
        }
        else if ((resp = '') or (resp=''))
        {
            scoreStatus(0, 0);
            restart();
        }
        else if ((resp = '') or (resp=''))
        {
            terminate();
            quit();
            abort;
        }
        else if ((resp = '') or (resp=''))
        {
            if (undo())
            {
                "(  )\b";
                parserGetMe().location.lookAround(true);
                scoreStatus(global.score, global.turnsofar);
                abort;
            }
            else
                " ,   . ";
        }
    }
}

replace terminate: function
{
    "\b    \( !\)\n";
}
