module AR::Game

    class Menu

      attr_accessor :options
      attr_accessor :prompt_class
      attr_accessor :prompt_tick
      attr_accessor :name

      def initialize prompt_class = AR::Input::CommandPrompt
        self.name = ""
        self.options = []
        self.prompt_class = prompt_class
      end

      #present should return resulting events from choices
      def present 
        option = prompt.listen
        while option.nil? #should just mean invalid input 
          option = prompt(true).listen
        end
        option.chosen.call
      end

      def prompt no_menu_display=false
        prompt_text = no_menu_display ? "" : menu_display
        prompt_proc = Proc.new do |input|
          if input.class ==  Fixnum
            input = input.chr
          else
            input = input.chomp
            input = input.strip
          end
          if input == ""
             
          elsif invalid? input
            AR.put_s "invalid choice '#{input}'... try again..."
            sleep(1.0)
            nil
          elsif ambiguous? input 
            clarify suggessions_from input
          else
            option_from input
          end
        end 

        @cmd_prompt = prompt_class.new prompt_text, prompt_proc,(prompt_tick || "[#{name}]> ")
        @cmd_prompt
      end

      def option_from input
        suggestions_from(input).last
      end

      def suggestions_from choice
        options.select{|o| o.ids.include? choice}
      end

      def menu_display 
        title = self.name.split(" ").map(&:capitalize).join(" ")
        header = "\n=========[#{title}]=========="
        header + "\n" + 
          options.map{|o| "#{o.usage} -> #{o.description}"}.join("\n") + "\n" + 
          "=" * header.size 
      end

      
      def clarify suggestions
        narrow_down = proc.new do |input|
          num = input.to_i 
          suggestions[num-1]
        end

        prompt_text = "what are you trying to do?  (enter a number)" 
        prompt_text += suggestions.map{|s,i| "#{i}. #{s.id}"}.join("\n")
        prompt_tick = "[#{self.name}]> "
        prompt = AR::Input::KeyPushPrompt.new prompt_text, narrow_down, prompt_tick 
        prompt.listen
      end

      def invalid? input
        suggestions_from(input).empty?
      end

      def ambiguous? input
        suggestions_from(input).size > 1
      end

    end

  class MainMenu < Menu 
    def initialize
      super
      quit = AR::Game::QuitOption.new 
      new_game = AR::Game::NewGameOption.new
      load_game= AR::Game::LoadGameOption.new
      self.options = [new_game]
      
      self.options.push load_game if File.exists? "saves/default_world_data.yml"
      self.options.push quit 
      self.name = "main menu"
    end 
  end 

  class ExplorationMenu < Menu
    attr_accessor :player
    def initialize player
      super AR::Input::CommandPrompt
      self.player = player 

      action_classnames = ["Wait", "Attack",  "Inspect", "Talk", "Travel", "Grab", "Put", "Give"]
      self.options = action_classnames.map do |acn|
        if Object.const_get("AR::Game::#{acn}Action").can_fill_params?(player)
          Object.const_get("AR::Game::#{acn}Choice").new player
        end
      end.compact
=begin
      wait = AR::Game::WaitChoice.new player
      inspect = AR::Game::InspectChoice.new player
      travel = AR::Game::TravelChoice.new player
      grab= AR::Game::GrabChoice.new player
      put= AR::Game::PutChoice.new player
      give= AR::Game::GiveChoice.new player
      self.options = [wait, inspect, travel, grab, put, give]
=end
    end 

    def present
      self.name = "action menu | ap: #{player.action_points}"
      super
    end
  end 

  class PlayerBreakMenu < Menu
    attr_accessor :player
    def initialize player
      super AR::Input::KeyPushPrompt
      self.player = player 

      save = Option.new do |opt| 
        opt.ids = ["s"]
        opt.usage = "(s) to save"
        opt.chosen = Proc.new do 
          AR.put_s "Attempting to save your world's data..."
          AR::Events::GameSaveEvent.new({world: player.world})
        end
      end

      quit = Option.new do |opt| 
        opt.ids = ["q"]
        opt.usage = "(q) to quit"
        opt.chosen = Proc.new do 
          key = nil
          while key.nil? 
            AR.put_s "Really quit? (Y/y N/n)"
            key = (AR::Input.take_key.chr.downcase rescue nil)
            key = nil unless ["n", "y"].include? key
          end

          AR::Events::GameOverEvent.new if key == "y"
        end
      end

      self.prompt_tick = ""
      self.options = [save, quit] 
    end


    #same as subclass, but only listens one time
    def present 
      AR.print_s "\n"
      option = prompt.listen
      ret = option.chosen.call unless option.nil?
      ret
    end

    def prompt no_menu_display=false
      prompt_text = no_menu_display ? "" : menu_display
      prompt_proc = Proc.new do |input|
        if input.class ==  Fixnum
          input = input.chr
        else
          input = input.chomp
          input = input.strip
        end
        option_from input
      end 

      @cmd_prompt = prompt_class.new prompt_text, prompt_proc,(prompt_tick || "[#{name}]> ")
      @cmd_prompt
    end




    def menu_display 
      title = self.name.split(" ").map(&:capitalize).join(" ")
      "(any key) to continue  |  #{options.map{|o| o.usage}.join("  |  ")}"
    end
  end 


end

