/*
    Mines

    Copyright (c) 1979, 2005-2011 James L. Dean

    Version 2011.01.17.01 released 1/17/2011

    This C program and associated executable may be distributed or used without
payment to its author.  Derivative works must credit its author.

    "Mines" lets you explore mines. The mine you explore is determined by a
mine number specified at the beginning of a game.

    The object of a game is to visit all of the rooms and return all of the
treasures to the entrance without making too many moves.

    In a mine, the passages are straight.  So, for example, if you go North to
leave a room, you can go South to reenter it.  The rooms are not evenly spaced.
However, the distance between adjacent rooms is always a multiple of the minimum
distance between adjacent rooms.
*/

#include "fxlib.h"

#define TRUE  1
#define FALSE 0

#define NULL 0

#define NUM_ROOMS        99
#define NUM_TREASURES    15

#define SENTENCE_LEN  1024 /* must be larger than the length of the longest sentence */
#define PARAGRAPH_LEN 8192 /* must be at least 5*SENTENCE_LEN */

typedef char      int8;
typedef short int int16;
typedef int       intC;
typedef int       int32;

typedef struct RoomNode
          {
            char  *szDescription;
            int16 nChokepoint;
            int8  bMined;
            int8  bVisited;
            union
              {
                struct
                  {
                    int16 pPassageNorth;
                    int16 pPassageSouth;
                    int16 pPassageEast;
                    int16 pPassageWest;
                    int16 pPassageUp;
                    int16 pPassageDown;
                  }                Passage;
                int16 pPassage [6];
              }   Adjacent;
           } RoomRec;

typedef struct TreasureNode
          {
            char   *szTreasure;
            char   *szGuardian;
            char   *szWeapon;
            int8   bSlain;
            int16  pRoomTreasure;
            int16  pRoomWeapon;
          } TreasureRec;

typedef struct PassageNode
          {
            int8  bBlocked;
            int16 pTreasureGuardian;
            int16 pRoom1;
            int16 pRoom2;
          } PassageRec;

typedef struct PathStackNode
          {
            int16 pRoom;
            int16 pPassageUsedToEnterRoom;
            int8  nDirectionUsedToEnterRoom;
            int16 pNext;
          } PathStackRec;

typedef struct FindPathLocalsStackNode
          {
            int8  bRoomAlreadyInStack;
            int8  nDirection1;
            int8  nDirection2;
            int8  nDirection3;
            int8  nDirectionRandom [6];
            int16 nPathLen;
            int16 pPathStack;
            int16 pNext;
          } FindPathLocalsStackRec;

typedef struct FindPathParametersStackNode
          {
            int16 pRoom;
            int8  nDirectionUsedToEnterRoom;
            int8  nReturnAddress;
            int16 pNext;
          } FindPathParametersStackRec;

       int   AddIn_main(int,unsigned short);
static void  Carry(int16,int8,int8 *);
static intC  Command(void);
static void  DisplayHelp(int8 *);
static void  DisplayLargeFontParagraph(char *);
static void  DisplayParagraph(char *);
static void  DisplaySmallFontParagraph(char *);
static void  DisplayText(char *,int8 *);
static void  ExcavateMine(int16,int16,int8,int16 *,int16 *,int8 *);
static void  FindPathToEntrance(int16,int16,char **);
static void  HideTreasuresAndWeapons(int16,int16 *,int8);
static void  InitializeGlobals(void);
static void  Inventory(int16,int8,int8 *);
static void  IToA(int16,char *);
static void  Leave(int16,int8,int8 *);
static void  ListCommands(int8 *);
static void  ListPassages(int16,int8 *);
static void  ListTreasuresAndWeapons(int16,int8,int8 *);
static void  Move(int16 *,int8,int16 *,int8 *);
static void  PlayGame(int16,int16,int16 *,int16,int8,int8 *);
static void  Points(int16,int16,int16 *,int8,int16,int8,int8 *);
static int16 RandomNumber(void);
static intC  ReadChar(void);
static void  SeedRandomNumberGenerator(void);
static void  ShuffleRoomDescriptions(int16,char **);
static void  StrCat(char *,char *);
static void  StrCpy(char *,char *);
static int16 StrLen(char *);
static intC  ToUpper(intC);
static void  WayOut(int16,int16,int16,int16 *,int8,int8 *);
static void  WriteChar(char);

static int8                           m_bUseSmallFont = 0;
static FindPathLocalsStackRec         *m_FindPathLocal;
static FindPathParametersStackRec     *m_FindPathParameter;
static int8                           m_nAddIndex = 0;
static int8                           m_nDirectionOpposite [6] = {1,0,3,2,5,4};
static int16                          m_nFindPathLocals = 0;
static int16                          m_nFindPathParameters = 0;
static int16                          m_nLinesPerPage = 8;
static int8                           m_nLinesWritten = 0;
static int16                          m_nMaxLineLen = 21;
static int16                          m_nPassages = 0;
static int16                          m_nPaths = 0;
static int8                           m_nReplaceIndex = 0;
static int16                          m_nRooms = 0;
static int32                          m_nRN [8];
static int32                          m_nRNPartialSum = 0;
static int32                          m_nXMax = 127;
static PassageRec                     *m_Passage;
static PathStackRec                   *m_Path;
static int16                          *m_pRoomWithName;
static RoomRec                        *m_Room;
static char                           *m_szDirection[6]
 = {"north","south","east","west","up","down"};
static char                           *m_szMsg;
static char                           *m_szParagraph;
static char                           **m_szRoom;
static char                           *m_szWayOut;
static TreasureRec                    m_Treasure [NUM_TREASURES]
 = {
     {
       "bag full of gold coins",
       "gorgon",
       "mirror",
       0,
       -1,
       -1
     },
     {
       "large blue-white diamond",
       "grizzly bear",
       "pepper spray dispenser",
       0,
       -1,
       -1
     },
     {
       "sixty carat emerald",
       "vampire",
       "wooden stake",
       0,
       -1,
       -1
     },
     {
       "platinum crucifix",
       "crocodile",
       "roll of duct tape",
       0,
       -1,
       -1
     },
     {
       "string of pearls",
       "giant crayfish",
       "jar of Zatarain's Crab Boil",
       0,
       -1,
       -1
     },
     {
       "Ming vase",
       "troll",
       "elfin sword",
       0,
       -1,
       -1
     },
     {
       "list of the next ten winners of the Kentucky Derby",
       "werewolf",
       "silver bullet",
       0,
       -1,
       -1
     },
     {
       "pile of rubies",
       "harpy",
       "crossbow",
       0,
       -1,
       -1
     },
     {
       "previously undiscovered Rembrandt painting",
       "cobra",
       "flute fashioned from a dried-out pumpkin",
       0,
       -1,
       -1
     },
     {
       "ounce of antimatter in magnetic containment",
       "ferocious snail",
       "bag of salt",
       0,
       -1,
       -1
     },
     {
       "stack of silver bars",
       "vicious Doberman pinscher",
       "soporific-laced dog biscuit",
       0,
       -1,
       -1
     },
     {
       "set of ivory tusks",
       "colossal cockroach",
       "block of boric acid",
       0,
       -1,
       -1
     },
     {
       "Holland and Holland double rifle chambered for .600 Nitro Express",
       "giant",
       "slingshot",
       0,
       -1,
       -1
     },
     {
       "ancient Greek statue",
       "Trekkie",
       "recording of Leonard Nimoy singing \"Proud Mary\"",
       0,
       -1,
       -1
     },
     {
       "five kilograms of plutonium in a lead container",
       "titanic ant",
       "Fresnel lens",
       0,
       -1,
       -1
     }
   };
static m_ucCharWidth [256] =
         {
           6,  6,  6,  6,  6,  6,  6,  6,  6,  6,  6,  6,  6,  6,  6,  6,
           6,  6,  6,  6,  6,  6,  6,  6,  6,  6,  6,  6,  6,  6,  6,  6,
           4,  2,  4,  6,  6,  5,  5,  3,  3,  3,  6,  4,  3,  4,  4,  4,
           4,  4,  4,  4,  4,  4,  4,  4,  4,  4,  3,  3,  4,  4,  4,  4,
           6,  4,  4,  4,  4,  4,  4,  4,  4,  4,  4,  5,  4,  6,  6,  4,
           4,  6,  4,  4,  4,  4,  4,  6,  4,  4,  4,  3,  4,  3,  4,  4,
           3,  4,  4,  4,  4,  4,  4,  4,  4,  2,  4,  4,  4,  6,  5,  4,
           4,  4,  5,  4,  4,  4,  4,  6,  4,  4,  4,  4,  2,  4,  5,  6,
           6,  6,  6,  6,  6,  6,  6,  6,  6,  6,  6,  6,  6,  6,  6,  6,
           6,  6,  6,  6,  6,  6,  6,  6,  6,  6,  6,  6,  6,  6,  6,  6,
           6,  6,  6,  6,  6,  6,  6,  6,  6,  6,  6,  6,  6,  6,  6,  6,
           6,  6,  6,  6,  6,  6,  6,  6,  6,  6,  6,  6,  6,  6,  6,  6,
           6,  6,  6,  6,  6,  6,  6,  6,  6,  6,  6,  6,  6,  6,  6,  6,
           6,  6,  6,  6,  6,  6,  6,  6,  6,  6,  6,  6,  6,  6,  6,  6,
           6,  6,  6,  6,  6,  6,  6,  6,  6,  6,  6,  6,  6,  6,  6,  6,
           6,  6,  6,  6,  6,  6,  6,  6,  6,  6,  6,  6,  6,  6,  6,  6
         };

static void InitializeGlobals(
 void)
  {
    m_nLinesWritten=(int8) 0;
    m_nFindPathLocals=(int16) 0;
    m_nFindPathParameters=(int16) 0;
    m_nPassages=(int16) 0;
    m_nPaths=(int16) 0;
    m_nReplaceIndex=(int8) 0;
    m_nRooms=(int16) 0;
    m_nRNPartialSum=(int32) 0;
    m_szRoom[0]="You\'re in a spherical room.";
    m_szRoom[1]="You\'re in the hall of bones.  Dinosaur bones are everywhere.";
    m_szRoom[2]="You\'re in a subway tunnel.  Don't touch that third rail!";
    m_szRoom[3]="You\'re in a railroad tunnel.  A whistle wails in the distance.";
    m_szRoom[4]="You\'re in an elfin quiche factory.  The elves are out mowing spinach.";
    m_szRoom[5]="You\'re in an abandoned Plymouth plant.  Beware of Road Runners and Barracudas.";
    m_szRoom[6]="You\'re in an abandoned Dodge plant.  There is a Dart embedded in the North wall.";
    m_szRoom[7]="You\'re in a mouse's nest.  You'd best exhale; this is a small room.";
    m_szRoom[8]="You\'re in a giant crayfish hole.  An immense chicken neck is hanging from a rope.";
    m_szRoom[9]="You\'re in an abandoned coal mine.  Beware of methane.";
    m_szRoom[10]="You\'re in the hall of winds.  Presently, the wind is from the south.";
    m_szRoom[11]="You\'re in a stove pipe!";
    m_szRoom[12]="You\'re in a totally darkened room.  Although you can see nothing, the sound of dripping water echoes from the walls.";
    m_szRoom[13]="You\'re in an industrial waste site.  Hold your breath and don't touch anything.";
    m_szRoom[14]="You\'re in the warehouse for an extremely popular brand of home computer.  Tacky plastic enclosures abound.";
    m_szRoom[15]="You\'re in a hobbit's bedroom.  The hobbit does not look pleased!";
    m_szRoom[16]="You\'re in a troll sewerage processing plant.  The air quality is not good.";
    m_szRoom[17]="You\'re in a rabbit hole.  There is a jar of marmalade on a shelf in the wall.";
    m_szRoom[18]="You\'re in the giant's oven.  Fortunately, it hasn't been used for years.";
    m_szRoom[19]="You\'re in a hobbit's drying room.  Tobacco leaves abound.";
    m_szRoom[20]="You\'re in a large circular room.  It is dark in here.";
    m_szRoom[21]="You\'re in the Crystal Palace.  Quartz crystals cover the walls.";
    m_szRoom[22]="You\'re in the Emerald Room.  Green crystals cover the ceiling.";
    m_szRoom[23]="You\'re in a curtained room.";
    m_szRoom[24]="You\'re in an air conditioning duct!";
    m_szRoom[25]="You\'re in a giant kiln.  Smoke stains the walls.";
    m_szRoom[26]="You\'re in the Hall of Mists.  Wisps of white vapor rise from the floor.";
    m_szRoom[27]="You\'re in an Aztec pyramid.  A mummy lies in the northwest corner.";
    m_szRoom[28]="You\'re in the Room of the Eternal Flame.  A large natural gas flame burns in the center of the room.  Coming from the west wall you can barely hear the words, 'Fee Fye Foe Fum'.";
    m_szRoom[29]="You\'re in the Giant's store room.  You are surrounded by old rugs, books, chairs, etc.";
    m_szRoom[30]="You\'re in the Leprechaun's Treasure Room.  Unfortunately, the leprechaun's treasure was stolen years ago.";
    m_szRoom[31]="You\'re in a large tiled room.  A girl is inquiring after a rabbit.  Feel free to ignore her.";
    m_szRoom[32]="You\'re in a former nuclear test site.  There is a large pile of rubble in the center of the floor.  The walls are streaked with a multitude of colors.";
    m_szRoom[33]="You\'re in a drainage system holding tank.  Either it is no longer used, or it hasn't rained in a long time; the room is dry.";
    m_szRoom[34]="You\'re in Mayor Schiro's bomb shelter.";
    m_szRoom[35]="You\'re in a room with a waterfall.  It is not very impressive; it looks like someone left a water faucet running.";
    m_szRoom[36]="You\'re in an abandoned Neanderthal home.";
    m_szRoom[37]="You\'re in a volcanic chimney.  The air is getting warmer.";
    m_szRoom[38]="You\'re in a pit full of snakes.  Let's get out of here!!!!";
    m_szRoom[39]="You\'re in a salt dome.";
    m_szRoom[40]="You\'re in Eleanor Roosevelt's privy.  Wendell Wilkie buttons cover the wall.";
    m_szRoom[41]="You\'re in Napoleon's wine cellar.  German wines line the shelves.";
    m_szRoom[42]="You\'re in the space behind the giant's bathroom wall. Large razor blades litter the floor.  Step lightly.";
    m_szRoom[43]="You\'re in the room where all old toe nail clipping come to rest.  Used corn pads litter the floor.";
    m_szRoom[44]="You\'re in the Den of the Horta.  The walls are covered with a sticky fluid.  Don't touch it; it is extremely corrosive.";
    m_szRoom[45]="You\'re in a damp room.  A small creek runs into a crack in the West wall.";
    m_szRoom[46]="You\'re in what appears to be a NOPSI manhole.";
    m_szRoom[47]="You\'re in the cafeteria of Mammoth Cave.  The aroma of rancid corned beef assails your nostrils.";
    m_szRoom[48]="You\'re in a small room with a large table.  On the table is a bottle that says, 'Drink me.'";
    m_szRoom[49]="You\'re in a Confederate States of America bank vault.  Once worthless currency papers the walls.";
    m_szRoom[50]="You\'re in an abandoned subway station.";
    m_szRoom[51]="You\'re in a mine shaft.  In the distance you can hear seven high pitched voices singing, 'Hi Ho, Hi Ho, ...'";
    m_szRoom[52]="You\'re in a Minuteman missile silo.";
    m_szRoom[53]="You\'re in the giant's mouse trap.  Fortunately, you are small enough to escape.";
    m_szRoom[54]="You\'re in Adolph Hitler's summer bunker.";
    m_szRoom[55]="You\'re in a dwarf work site.  A sign says, \"Under construction. Enter at your own risk.\"";
    m_szRoom[56]="You\'re in the giant's refrigerator.  Dwarf bodies hang from hooks.";
    m_szRoom[57]="You\'re in the Dragon's Lair.  Slightly melted suits of armor litter the floor.";
    m_szRoom[58]="You\'re in a nuclear waste depository.  The walls glow faintly.";
    m_szRoom[59]="You\'re in Millard Fillmore's tomb.  It is dull.";
    m_szRoom[60]="You\'re in an abandoned corridor of the Strategic Air Command Headquarters.  A graffito reads, \"Beware of Bat Guano.\"";
    m_szRoom[61]="You\'re in a gnome's workshop.  Half-completed whoopee cushions line the tables.";
    m_szRoom[62]="You\'re in the Mummy's Tomb.  You've triggered some mechanism and the ceiling is slowly descending.";
    m_szRoom[63]="You\'re in the Underground Gourmet's retreat.  Twinkie and King Don wrappers are piled knee deep.";
    m_szRoom[64]="You\'re in a Hoola Hoop warehouse.  The words \"Shoop Shoop\" echo from the walls.";
    m_szRoom[65]="You\'re in the first circle of hell.  The living are not allowed here.";
    m_szRoom[66]="You\'re in the hall of the pixies.  The furniture appears to have been made from cradles.";
    m_szRoom[67]="You\'re in a sulfur mine.  Molten sulfur churns in a nearby pit. It is becoming difficult to breath.";
    m_szRoom[68]="You\'re in a fairy mushroom farm.  Brilliantly colored mushrooms cover the floor.";
    m_szRoom[69]="You\'re in an ice cave.  Along the western wall, a brontosaurus is defrosting.";
    m_szRoom[70]="You\'re in the giant's stove.  Fortunately, the giant now owns a microwave oven.";
    m_szRoom[71]="You\'re in the rib cage of a long deceased whale.";
    m_szRoom[72]="You\'re in a room with six walls.  The walls appear to be made of wax.  A loud buzzing noise can be heard.";
    m_szRoom[73]="You\'re in the tomb of a Pharaoh.  It has obviously been visited by others; the tomb is in a state of total disarray.";
    m_szRoom[74]="You\'re in a coal bin.  There is a fossilized fern stump here.";
    m_szRoom[75]="You\'re in a diamond mine.  It is uncomfortably hot here.";
    m_szRoom[76]="You\'re in the bottom of an oil well.  The walls are slick.";
    m_szRoom[77]="You\'re in the lowest level of Project Mohole.  The funding bubble burst before the earth did.";
    m_szRoom[78]="You\'re in the giant's cesspool.  Fortunately, the giant was connected to the city sewerage system years ago.";
    m_szRoom[79]="You\'re in an eighteenth century sewer.  The walls are crumbling brick.  Small alligators snap at your feet.";
    m_szRoom[80]="You\'re in the lair of a giant trapdoor spider.";
    m_szRoom[81]="You\'re in a giant gopher tunnel.";
    m_szRoom[82]="You\'re in a shell -- the sole remains of a prehistoric turtle.";
    m_szRoom[83]="You\'re in a small chamber.  The walls are made of earth.  The air smells of formic acid.  A strange squeaking noise can be heard in the distance.";
    m_szRoom[84]="You\'re in a chamber of columns.  The stalagmites and stalactites join here.";
    m_szRoom[85]="You\'re in a service tunnel.  Ducts, pipes, and cables are everywhere.";
    m_szRoom[86]="You\'re in a gas tank below an abandoned service station.  No smoking!";
    m_szRoom[87]="You\'re in a huge dark chamber.  To one side, a dark, muddy river moves sluggishly.  A sign written in ancient Greek says, \"Ferry out of order.\"";
    m_szRoom[88]="You\'re in a small chamber. It is brightly lit by a peculiar lichen growing on the walls and ceiling.  The floor is rocky and steeply sloping.  Nearby, a cold, clear creek boils into the floor and out of sight.";
    m_szRoom[89]="You\'re in the nest of a very large pack rat.  There are discarded aluminum Christmas trees, broken steel utility poles, and other shiny, worthless items here.";
    m_szRoom[90]="You\'re in a dungeon.  An iron maiden, a rack, a piano, and other machines of torture can be seen.";
    m_szRoom[91]="You\'re in the hall of bats.  Thousands of bats hang from the ceiling.  Watch your step; the droppings are quite deep in places.";
    m_szRoom[92]="You\'re in a hobgoblin's hideaway.";
    m_szRoom[93]="You\'re in an electrical substation.  A transformer is humming loudly.  Nearby, cables crackle with high voltage.";
    m_szRoom[94]="You\'re in the \"gold\" room.  The walls are covered with iron pyrite.";
    m_szRoom[95]="You\'re in a room with one of Dracula's emergency coffins.  The Count is out.";
    m_szRoom[96]="You\'re in a saltpeter mine.  To one side there is a huge wooden evaporation vat.  Small crystals of saltpeter cause the walls to sparkle.";
    m_szRoom[97]="You\'re in the basement cafeteria of a local hospital.  Some say that there has been nothing edible here for years.";
    m_szRoom[98]="You\'re in a troll arsenal.  Kegs of gun powder surround you.";
    return;
  }


static intC ReadChar(
 void)
  {
    unsigned int  uiKey;
    unsigned char uc [2];

    GetKey(&uiKey);
    uc[0]=(unsigned char) uiKey;
    uc[1]='\0';
    Print(&uc[0]);
    return (intC) uiKey;
  }

static void WriteChar(
 char c)
  {
    unsigned char uc[2];

    uc[0]=(unsigned char) c;
    uc[1]='\0';
    Print(&uc[0]);
    return;
  }

static intC ToUpper(
 intC n)
  {
    intC nResult;

    if (n >= (intC) 'a')
      if (n <= (intC) 'z')
        nResult=(n-((intC) 'a'))+((intC) 'A');
      else
        nResult=n;
    else
      nResult=n;
    return nResult;
  }

static int16 StrLen(
 char *sz)
  {
    int16 nResult;
    char  *pc;

    nResult=0;
    pc=sz;
    while (*pc)
      {
        ++nResult;
        ++pc;
      }
    return nResult;
  }

static void StrCpy(
 char *szDestination,
 char *szSource)
  {
    char *pcDestination;
    char *pcSource;

    pcDestination=szDestination;
    pcSource=szSource;
    while (*pcSource)
      {
        *pcDestination=*pcSource;
        ++pcDestination;
        ++pcSource;
      }
    *pcDestination='\0';
    return;
  }

static void StrCat(
 char *szDestination,
 char *szSource)
  {
    char *pcDestination;

    pcDestination=szDestination;
    while (*pcDestination)
      ++pcDestination;
    StrCpy(pcDestination,szSource);
    return;
  }

static void IToA(
 int16 n,
 char  *sz)
  {
    char  c [10];
    char  cDigit [10] = {'0','1','2','3','4','5','6','7','8','9'};
    int8  nChars;
    int16 nDividend;
    int16 nQuotient;
    int16 nRemainder;
    char  *pc;

    nChars=0;
    nDividend=n;
    do
      {
        nQuotient=nDividend/10;
        nRemainder=nDividend-10*nQuotient;
        c[nChars]=cDigit[nRemainder];
        ++nChars;
        nDividend=nQuotient;
      }
    while (nDividend);
    pc=sz;
    while (nChars)
      {
        --nChars;
        *pc=c[nChars];
        ++pc;
      }
    *pc='\0';
    return;
  }

static int16 TokenLen(
 char *szText,
 int16 nTokenStart,
 int16 nTokenStop)
  {
    int16 nResult;
    int16 nToken;
    char  *pc;

    nResult=(int16) 0;
    pc=szText+nTokenStart;
    nToken=nTokenStart;
    while (nToken < nTokenStop)
      {
        nResult+=m_ucCharWidth[*pc];
        ++pc;
        ++nToken;
      }
    return nResult;
  }

static void DisplaySmallFontParagraph(
 char *szText)
  {
    int8          nX;
    int16         nPendingSpaces;
    int16         nTokenLen;
    int16         nTokenStart;
    int16         nTokenStop;
    unsigned int  uiKey;
    unsigned char uc [2];

    uc[1]=(unsigned char) '\0';
    nX=0;
    nPendingSpaces=0;
    nTokenStart=0;
    while (szText[nTokenStart] && (szText[nTokenStart] == ' '))
      {
        uc[0]=(unsigned char) ' ';
        PrintMini(nX,6*m_nLinesWritten,&uc[0],MINI_OVER);
        nX+=m_ucCharWidth[' '];
        if (nX > m_nXMax)
          {
            ++m_nLinesWritten;
            nX=0;
            if (m_nLinesWritten >= (m_nLinesPerPage-1))
              {
                PrintMini(nX,6*m_nLinesWritten,
                 (unsigned char *) "(press a key to continue)",MINI_OVER);
                GetKey(&uiKey);
                Bdisp_AllClr_DDVRAM();
                m_nLinesWritten=0;
              }
          }
        ++nTokenStart;
      }
    while (szText[nTokenStart])
      {
        while (szText[nTokenStart] && (szText[nTokenStart] == ' '))
          ++nTokenStart;
        if (nX+nPendingSpaces*m_ucCharWidth[' '] > m_nXMax)
          {
            ++m_nLinesWritten;
            nPendingSpaces=0;
            nX=0;
            if (m_nLinesWritten >= (m_nLinesPerPage-1))
              {
                PrintMini(nX,6*m_nLinesWritten,
                 (unsigned char *) "(press a key to continue)",MINI_OVER);
                GetKey(&uiKey);
                Bdisp_AllClr_DDVRAM();
                m_nLinesWritten=0;
              }
          }
        else
          while (nPendingSpaces)
            {
              uc[0]=(unsigned char) ' ';
              PrintMini(nX,6*m_nLinesWritten,&uc[0],MINI_OVER);
              nX+=m_ucCharWidth[' '];
              --nPendingSpaces;
            }
        if (szText[nTokenStart])
          {
            nTokenStop=nTokenStart;
            while (szText[nTokenStop] && (szText[nTokenStop] != ' '))
              ++nTokenStop;
            nTokenLen=TokenLen(szText,nTokenStart,nTokenStop);
            if (nX+nTokenLen > m_nXMax)
              if (nTokenLen > m_nXMax)
                {
                  while (nX < m_nXMax)
                    {
                      uc[0]=(unsigned char) (szText[nTokenStart]);
                      PrintMini(nX,6*m_nLinesWritten,&uc[0],MINI_OVER);
                      nX+=m_ucCharWidth[szText[nTokenStart]];
                      ++nTokenStart;
                    }
                  ++m_nLinesWritten;
                  nX=0;
                  if (m_nLinesWritten >= (m_nLinesPerPage-1))
                    {
                      PrintMini(nX,6*m_nLinesWritten,
                       (unsigned char *) "(press a key to continue)",MINI_OVER);
                      GetKey(&uiKey);
                      Bdisp_AllClr_DDVRAM();
                      m_nLinesWritten=0;
                    }
                  while (nTokenStart < nTokenStop)
                    {
                      uc[0]=(unsigned char) (szText[nTokenStart]);
                      PrintMini(nX,6*m_nLinesWritten,&uc[0],MINI_OVER);
                      nX+=m_ucCharWidth[szText[nTokenStart]];
                      if (nX > m_nXMax)
                        {
                          ++m_nLinesWritten;
                          nX=0;
                          if (m_nLinesWritten >= (m_nLinesPerPage-1))
                            {
                              PrintMini(nX,6*m_nLinesWritten,
                               (unsigned char *) "(press a key to continue)",
                               MINI_OVER);
                              GetKey(&uiKey);
                              Bdisp_AllClr_DDVRAM();
                              m_nLinesWritten=0;
                            }
                        }
                      ++nTokenStart;
                    }
                }
              else
                {
                  ++m_nLinesWritten;
                  nX=0;
                  if (m_nLinesWritten >= (m_nLinesPerPage-1))
                    {
                      PrintMini(nX,6*m_nLinesWritten,
                       (unsigned char *) "(press a key to continue)",MINI_OVER);
                      GetKey(&uiKey);
                      Bdisp_AllClr_DDVRAM();
                      m_nLinesWritten=0;
                    }
                  while (nTokenStart < nTokenStop)
                    {
                      uc[0]=(unsigned char) (szText[nTokenStart]);
                      PrintMini(nX,6*m_nLinesWritten,&uc[0],MINI_OVER);
                      nX+=m_ucCharWidth[szText[nTokenStart]];
                      ++nTokenStart;
                    }
                }
            else
              while (nTokenStart < nTokenStop)
                {
                  uc[0]=(unsigned char) (szText[nTokenStart]);
                  PrintMini(nX,6*m_nLinesWritten,&uc[0],MINI_OVER);
                  nX+=m_ucCharWidth[szText[nTokenStart]];
                  ++nTokenStart;
                }
            switch (szText[nTokenStop-1])
              {
                case '.':
                  nPendingSpaces=2;
                  break;
                case '?':
                  nPendingSpaces=2;
                  break;
                case '!':
                  nPendingSpaces=2;
                  break;
                default:
                  nPendingSpaces=1;
                  break;
              }
          }
      }
    if (nX > 1)
      {
        ++m_nLinesWritten;
        nX=0;
      }
    return;
  }

static void DisplayLargeFontParagraph(
 char *szText)
  {
    int8         nColumn;
    int16        nPendingSpaces;
    int16        nTokenLen;
    int16        nTokenStart;
    int16        nTokenStop;
    unsigned int uiKey;

    nColumn=1;
    nPendingSpaces=0;
    nTokenStart=0;
    while (szText[nTokenStart] && (szText[nTokenStart] == ' '))
      {
        WriteChar(' ');
        if (++nColumn > m_nMaxLineLen)
          {
            ++m_nLinesWritten;
            locate(1,m_nLinesWritten+1);
            nColumn=1;
            if (m_nLinesWritten >= (m_nLinesPerPage-1))
              {
                Print((unsigned char *) "             (cont\'d)");
                GetKey(&uiKey);
                Bdisp_AllClr_DDVRAM();
                m_nLinesWritten=0;
                locate(1,1);
              }
          }
        ++nTokenStart;
      }
    while (szText[nTokenStart])
      {
        while (szText[nTokenStart] && (szText[nTokenStart] == ' '))
          ++nTokenStart;
        if (nColumn+nPendingSpaces > m_nMaxLineLen)
          {
            ++m_nLinesWritten;
            locate(1,m_nLinesWritten+1);
            nPendingSpaces=0;
            nColumn=1;
            if (m_nLinesWritten >= (m_nLinesPerPage-1))
              {
                Print((unsigned char *) "             (cont\'d)");
                GetKey(&uiKey);
                Bdisp_AllClr_DDVRAM();
                m_nLinesWritten=0;
                locate(1,1);
              }
          }
        else
          while (nPendingSpaces)
            {
              WriteChar(' ');
              ++nColumn;
              --nPendingSpaces;
            }
        if (szText[nTokenStart])
          {
            nTokenStop=nTokenStart;
            while (szText[nTokenStop] && (szText[nTokenStop] != ' '))
              ++nTokenStop;
            nTokenLen=nTokenStop-nTokenStart;
            if (nColumn+nTokenLen > m_nMaxLineLen)
              if (nTokenLen > m_nMaxLineLen)
                {
                  while (nColumn < m_nMaxLineLen)
                    {
                      WriteChar(szText[nTokenStart]);
                      ++nColumn;
                      ++nTokenStart;
                    }
                  ++m_nLinesWritten;
                  locate(1,m_nLinesWritten+1);
                  if (m_nLinesWritten >= (m_nLinesPerPage-1))
                    {
                      Print((unsigned char *) "             (cont\'d)");
                      GetKey(&uiKey);
                      Bdisp_AllClr_DDVRAM();
                      m_nLinesWritten=0;
                      locate(1,1);
                    }
                  nColumn=1;
                  while (nTokenStart < nTokenStop)
                    {
                      WriteChar(szText[nTokenStart]);
                      if (++nColumn > m_nMaxLineLen)
                        {
                          ++m_nLinesWritten;
                          locate(1,m_nLinesWritten+1);
                          if (m_nLinesWritten >= (m_nLinesPerPage-1))
                            {
                              Print((unsigned char *) "             (cont\'d)");
                              GetKey(&uiKey);
                              Bdisp_AllClr_DDVRAM();
                              m_nLinesWritten=0;
                              locate(1,1);
                            }
                          nColumn=1;
                        }
                      ++nTokenStart;
                    }
                }
              else
                {
                  ++m_nLinesWritten;
                  locate(1,m_nLinesWritten+1);
                  if (m_nLinesWritten >= (m_nLinesPerPage-1))
                    {
                      Print((unsigned char *) "             (cont\'d)");
                      GetKey(&uiKey);
                      Bdisp_AllClr_DDVRAM();
                      m_nLinesWritten=0;
                      locate(1,1);
                    }
                  nColumn=1;
                  while (nTokenStart < nTokenStop)
                    {
                      WriteChar(szText[nTokenStart]);
                      ++nColumn;
                      ++nTokenStart;
                    }
                }
            else
              while (nTokenStart < nTokenStop)
                {
                  WriteChar(szText[nTokenStart]);
                  ++nColumn;
                  ++nTokenStart;
                }
            switch (szText[nTokenStop-1])
              {
                case '.':
                  nPendingSpaces=2;
                  break;
                case '?':
                  nPendingSpaces=2;
                  break;
                case '!':
                  nPendingSpaces=2;
                  break;
                default:
                  nPendingSpaces=1;
                  break;
              }
          }
      }
    if (nColumn > 1)
      {
        ++m_nLinesWritten;
        locate(1,m_nLinesWritten+1);
      }
    return;
  }

static void DisplayParagraph(
 char *szText)
  {
    if (m_bUseSmallFont)
      DisplaySmallFontParagraph(szText);
    else
      DisplayLargeFontParagraph(szText);
    return;
  }

static void DisplayText(
 char *szText,
 int8 *bErr)
  {
    int16 nBytes;

    nBytes=StrLen(&m_szParagraph[0]);
    if (m_szParagraph[0])
      nBytes+=StrLen(" ");
    nBytes+=StrLen(szText);
    if (nBytes < PARAGRAPH_LEN)
      {
        if (m_szParagraph[0])
          StrCat(&m_szParagraph[0]," ");
        StrCat(&m_szParagraph[0],szText);
      }
    else
      {
        *bErr=TRUE;
        StrCpy(&m_szParagraph[0],"Fatal error:  the paragraph buffer is too small.");
      }
    return;
  }

static void ListCommands(
 int8 *bErr)
  {
    DisplayText(
     "Commands in the game are \"N\", \"E\", \"S\", \"W\", \"U\", and "
     "\"D\" to move North, East, South, West, Up, or Down, "
     "respectively.  Other commands are \"C\" to carry things, \"I\" to "
     "inventory what you are carrying, \"L\" to leave treasures, \"P\" to "
     "get the points you\'ve scored, \"O\" for help getting out of the "
     "mine, \"F\" to switch fonts, \"H\" for help, and \"Q\" to quit.",bErr);
    return;
  }

static void DisplayHelp(
 int8 *bErr)
  {
    ListCommands(bErr);
    if (! *bErr)
      DisplayText(
       "In a mine, the passages are straight.  So, for example, if "
       "you go North to leave a room, you can go South to reenter it.  "
       "The rooms are not evenly spaced.  However, the distance between "
       "adjacent rooms is always a multiple of the minimum distance "
       "between adjacent rooms.",bErr);
    return;
  }

static int16 RandomNumber(
 void)
  {
    int32 nResult;
   /*
       Each pseudo-random number is the modulo sum of the
     previous eight pseudo-random numbers.  A prime modulus
     makes it likely that the pseudo-random numbers will be
     uniformly distributed.  To speed computation, a partial
     sum of 7 of the 8 previous pseudo-random numbers is maintained.
          For a given set of initial values m_nRN[i], i=0,1,2,...,7,
     this random number generator should produce the same sequence
     of random numbers, no matter what 32-bit C Compiler it is compiled
     under.
   */
    do
      {
        nResult=m_nRNPartialSum+m_nRN[m_nAddIndex];
        if (nResult >= 32771) nResult-=32771;
        m_nRNPartialSum=nResult-m_nRN[m_nReplaceIndex];
        if (m_nRNPartialSum < 0) m_nRNPartialSum+=32771;
        m_nRN[m_nReplaceIndex]=nResult;
        m_nAddIndex=m_nReplaceIndex;
        if (++m_nReplaceIndex >= 8)
          m_nReplaceIndex=0;
      }
    while (nResult > 32767);
    return (int16) nResult;
  }

static void SeedRandomNumberGenerator(
 void)
  {
    char  *pc;
    int16 nKeyIndex;
    int16 nKeyStart;
    int16 nKeyLength;
    intC  nTem;
    char  szMine [9];

    nKeyLength=0;
    Bdisp_AllClr_DDVRAM();
    m_nLinesWritten=0;
    locate(1,1);
    m_szParagraph[0]='\0';
    Print((unsigned char *) "Mine number? ");
    do
      {
        nTem=ReadChar();
        if ((nTem != (intC) 13)
        &&  (nTem != (intC) 30004))
          {
            if (nTem == 8)
              --nKeyLength;
            else
              {
                if (nKeyLength < 8)
                  szMine[nKeyLength]=(char) (nTem%128);
                ++nKeyLength;
              }
          }
      }
    while ((nTem != (intC) 13)
    &&     (nTem != (intC) 30004));
    ++m_nLinesWritten;
    locate(1,m_nLinesWritten+1);
    if (nKeyLength > 8)
      nKeyLength=8;
    szMine[nKeyLength]='\0';
    nKeyIndex=0;
    if (nKeyLength < 8)
      {
        for (nKeyStart=nKeyLength; nKeyStart < 8; ++nKeyStart)
          m_nRN[nKeyIndex++]=1+(int) '0';
        pc=szMine;
      }
    else
      pc=szMine+nKeyLength-8;
    while (*pc)
      m_nRN[nKeyIndex++]=1+(int) *pc++;
    m_nRNPartialSum=0;
    for (nKeyIndex=7; nKeyIndex > 0; --nKeyIndex)
      {
        m_nRNPartialSum+=m_nRN[nKeyIndex];
        if (m_nRNPartialSum >= 32771)
          m_nRNPartialSum-=32771;
      }
    m_nReplaceIndex=1;
    m_nAddIndex=0;
    for (nKeyIndex=256; nKeyIndex--;)
      RandomNumber();
    return;
  }

static void ShuffleRoomDescriptions(
 int16 nRooms,
 char  **pszRoom)
  {
    int16 nRoom1;
    int16 nRoom2;
    char  *sz;

    for (nRoom1=nRooms-1; nRoom1 > 0; -- nRoom1)
      {
        nRoom2=RandomNumber()%nRoom1;
        sz=pszRoom[nRoom1];
        pszRoom[nRoom1]=pszRoom[nRoom2];
        pszRoom[nRoom2]=sz;
      }
    return;
  }

static void ExcavateMine(
 int16 nRooms,
 int16 pRoomEntrance,
 int8  nTreasures,
 int16 *nChokepoints,
 int16 *ppRoomWithName,
 int8  *bErr)
  {
    int16 nChokepoint;
    int8  nDirectionOpposite;
    int8  nDirection1;
    int8  nDirection2;
    int16 nDistance;
    int8  nHeight;
    int8  nLen;
    int8  nMax;
    int16 nRoom;
    int16 nStep;
    int8  nTreasure;
    int16 nUnblocked;
    int8  nWidth;
    int8  nX;
    int8  nXFirstRoom;
    int8  nY;
    int8  nYFirstRoom;
    int8  nZ;
    int8  nZFirstRoom;
    int16 pPassage;
    int16 pRoom;
    int16 pRoomDown;
    int16 pRoomNext;
    int16 pRoomStart;
    int16 pRoomSouth;
    int16 pRoomSouthBase;
    int16 pRoomWest;
    int16 pRoomWestBase;
    int16 pRoomWestBaseBase;
    int32 nRN;

    /* nMax=(int) (10.0+exp(log((double) nRooms)/3.0)); */
    nMax=0;
    while (nMax*nMax*nMax < nRooms)
      ++nMax;
    if (nMax*nMax*nMax == nRooms)
      nMax+=10;
    else
      nMax+=9;
    do
      {
        nRN=RandomNumber();
        nLen=nRN%(nMax-3)+3;
        nRN=RandomNumber();
        nWidth=nRN%(nMax-3)+3;
        nRN=RandomNumber();
        nHeight=nRN%(nMax-3)+3;
      }
    while ((nLen*nWidth*nHeight < 2*(nRooms-1))
    ||     (nLen*nWidth*nHeight > 3*(nRooms-1)));
    nXFirstRoom=0;
    nYFirstRoom=RandomNumber()%nLen;
    nZFirstRoom=RandomNumber()%nHeight;
    pRoomWestBase=-1;
    pRoomWestBaseBase=-1;
    for (nX=0; ((! *bErr) && (nX < nWidth)); ++nX)
      {
        pRoomSouthBase=-1;
        for (nY=0; ((! *bErr) && (nY < nLen)); ++nY)
          {
            pRoomWest=pRoomWestBase;
            pRoomSouth=pRoomSouthBase;
            pRoomDown=-1;
            for (nZ=0; ((! *bErr) && (nZ < nHeight)); ++nZ)
              {
                if (m_nRooms < 3*NUM_ROOMS-2)
                  {
                    pRoom=m_nRooms;
                    ++m_nRooms;
                    m_Room[pRoom].szDescription=(char *) NULL;
                    m_Room[pRoom].bMined=FALSE;
                    m_Room[pRoom].nChokepoint=-1;
                    m_Room[pRoom].bVisited=FALSE;
                    m_Room[pRoom].Adjacent.Passage.pPassageNorth=-1;
                    m_Room[pRoom].Adjacent.Passage.pPassageSouth=-1;
                    m_Room[pRoom].Adjacent.Passage.pPassageEast=-1;
                    m_Room[pRoom].Adjacent.Passage.pPassageWest=-1;
                    m_Room[pRoom].Adjacent.Passage.pPassageUp=-1;
                    m_Room[pRoom].Adjacent.Passage.pPassageDown=-1;
/*
(X,Y,Z) gets linked to (X-1,Y,Z); (X,Y-1,Z); and (X,Y,Z-1)
                       pRoomWest  pRoomSouth     pRoomDown
*/
                    if (nZ == 0)
                      {
                        pRoomSouthBase=pRoom;
                        if (nY == 0)
                          pRoomWestBaseBase=pRoom;
                      }
                    if (pRoomDown != -1)
                      {
                      /* link room to one below it */
                        if (m_nPassages < (9*NUM_ROOMS-8))
                          {
                            pPassage=m_nPassages;
                            ++m_nPassages;
                            m_Passage[pPassage].bBlocked=TRUE;
                            m_Passage[pPassage].pTreasureGuardian=-1;
                            m_Passage[pPassage].pRoom1=pRoom;
                            m_Passage[pPassage].pRoom2=pRoomDown;
                            m_Room[pRoom].Adjacent.Passage.pPassageDown=pPassage;
                            m_Room[pRoomDown].Adjacent.Passage.pPassageUp=pPassage;
                          }
                        else
                          {
                            DisplayText("Fatal error:  memory was exhausted at locaton 1.",bErr);
                            *bErr=TRUE;
                          }
                      }
                    if (pRoomSouth != -1)
                      {
                        /* link to room to the south */
                        if (m_nPassages < 9*NUM_ROOMS-8)
                          {
                            pPassage=m_nPassages;
                            ++m_nPassages;
                            m_Passage[pPassage].bBlocked=TRUE;
                            m_Passage[pPassage].pTreasureGuardian=-1;
                            m_Passage[pPassage].pRoom1=pRoom;
                            m_Passage[pPassage].pRoom2=pRoomSouth;
                            m_Room[pRoom].Adjacent.Passage.pPassageSouth=pPassage;
                            m_Room[pRoomSouth].Adjacent.Passage.pPassageNorth=pPassage;
                          }
                        else
                          {
                            DisplayText("Fatal error:  memory was exhausted at locaton 2.",bErr);
                            *bErr=TRUE;
                          }
                        if (m_Room[pRoomSouth].Adjacent.Passage.pPassageUp != -1)
                          if (m_Passage[m_Room[pRoomSouth].Adjacent.Passage.pPassageUp].pRoom1
                           == pRoomSouth)
                            pRoomSouth=m_Passage[m_Room[pRoomSouth].Adjacent.Passage.pPassageUp].pRoom2;
                          else
                            pRoomSouth=m_Passage[m_Room[pRoomSouth].Adjacent.Passage.pPassageUp].pRoom1;
                        else
                          pRoomSouth=-1;
                      }
                    if (pRoomWest != -1)
                      {
                      /* link to the room to the west */
                        if (m_nPassages < 9*NUM_ROOMS-8)
                          {
                            pPassage=m_nPassages;
                            ++m_nPassages;
                            m_Passage[pPassage].bBlocked=TRUE;
                            m_Passage[pPassage].pTreasureGuardian=-1;
                            m_Passage[pPassage].pRoom1=pRoom;
                            m_Passage[pPassage].pRoom2=pRoomWest;
                            m_Room[pRoom].Adjacent.Passage.pPassageWest=pPassage;
                            m_Room[pRoomWest].Adjacent.Passage.pPassageEast=pPassage;
                          }
                        else
                          {
                            DisplayText("Fatal error:  memory was exhausted at locaton 3.",bErr);
                            *bErr=TRUE;
                          }
                        if (m_Room[pRoomWest].Adjacent.Passage.pPassageUp != -1)
                          if (m_Passage[m_Room[pRoomWest].Adjacent.Passage.pPassageUp].pRoom1
                           == pRoomWest)
                            pRoomWest=m_Passage[m_Room[pRoomWest].Adjacent.Passage.pPassageUp].pRoom2;
                          else
                            pRoomWest=m_Passage[m_Room[pRoomWest].Adjacent.Passage.pPassageUp].pRoom1;
                        else
                          pRoomWest=-1;

                      }
                    /* If this is the first room, connect it to the entrance. */
                    if ((nX == nXFirstRoom)
                    &&  (nY == nYFirstRoom)
                    &&  (nZ == nZFirstRoom))
                      {
                        if (m_nPassages < 9*NUM_ROOMS-8)
                          {
                            pPassage=m_nPassages;
                            ++m_nPassages;
                            m_Passage[pPassage].bBlocked=FALSE;
                            m_Passage[pPassage].pTreasureGuardian=-1;
                            m_Passage[pPassage].pRoom1=pRoom;
                            m_Passage[pPassage].pRoom2=pRoomEntrance;
                            m_Room[pRoom].Adjacent.Passage.pPassageWest=pPassage;
                            m_Room[pRoomEntrance].Adjacent.Passage.pPassageEast
                             =pPassage;
                          }
                        else
                          {
                            DisplayText("Fatal error:  memory was exhausted at locaton 4.",bErr);
                            *bErr=TRUE;
                          }
                      }
                    pRoomDown=pRoom;
                  }
                else
                  {
                    DisplayText("Fatal error:  memory was exhausted at locaton 5.",bErr);
                    *bErr=TRUE;
                  }
              }
            if (pRoomWestBase != -1)
              {
                if (m_Room[pRoomWestBase].Adjacent.Passage.pPassageNorth != -1)
                  {
                    if (m_Passage[m_Room[pRoomWestBase].Adjacent.Passage.pPassageNorth].pRoom1
                     == pRoomWestBase)
                      pRoomWestBase=m_Passage[m_Room[pRoomWestBase].Adjacent.Passage.pPassageNorth].pRoom2;
                    else
                      pRoomWestBase=m_Passage[m_Room[pRoomWestBase].Adjacent.Passage.pPassageNorth].pRoom1;
                  }
              }
          }
        pRoomWestBase=pRoomWestBaseBase;
      }
    nChokepoint=-1;
    nTreasure=0;
    pRoom=pRoomEntrance;
    nRoom=0;
    while (nRoom < nRooms)
      {
        pRoomStart=pRoom;
        nDirection1=RandomNumber()%6;
        nDirectionOpposite=m_nDirectionOpposite[nDirection1];
        nDistance=RandomNumber()%3+1;
        nStep=0;
        while ((m_Room[pRoom].Adjacent.pPassage[nDirection1] != -1)
        &&     (nStep < nDistance))
          {
            pPassage=m_Room[pRoom].Adjacent.pPassage[nDirection1];
            if (m_Passage[pPassage].pRoom1 == pRoom)
              pRoomNext=m_Passage[pPassage].pRoom2;
            else
              pRoomNext=m_Passage[pPassage].pRoom1;
            if (m_Passage[pPassage].bBlocked)
              if ((m_Room[pRoomNext].nChokepoint == nChokepoint)
              ||  (! m_Room[pRoomNext].bMined))
                {
                  pRoom=pRoomNext;
                  m_Room[pRoom].bMined=TRUE;
                  m_Room[pRoom].nChokepoint=nChokepoint;
                  m_Passage[pPassage].bBlocked=FALSE;
                  if (m_Room[pRoom].szDescription)
                    ++nStep;
                  else
                    {
                      nUnblocked=0;
                      for (nDirection2=6;
                       ((nUnblocked < 2) && (nDirection2--));)
                        if (nDirection2 != nDirectionOpposite)
                          if (m_Room[pRoom].Adjacent.pPassage[nDirection2] != -1)
                            if (! m_Passage[m_Room[pRoom].Adjacent.pPassage[nDirection2]].bBlocked)
                              ++nUnblocked;
                      if (nUnblocked < 2)
                        ++nStep;
                      else
                        nStep=nDistance;
                    }
                }
              else
                nStep=nDistance;
            else
              {
                ++nStep;
                pRoom=pRoomNext;
                nChokepoint=m_Room[pRoom].nChokepoint;
              }
          }
        if (nStep)
          {
            if (m_Room[pRoom].szDescription == NULL)
              {
                m_Room[pRoom].szDescription=m_szRoom[nRoom];
                nDirection1=0;
                nUnblocked=0;
                for (nDirection2=6;
                 ((nUnblocked < 2) && (nDirection2--));)
                  if (m_Room[pRoom].Adjacent.pPassage[nDirection2] != -1)
                    if (! m_Passage[m_Room[pRoom].Adjacent.pPassage[nDirection2]].bBlocked)
                      {
                        nDirection1=nDirection2;
                        ++nUnblocked;
                      }
                if (nUnblocked == 1)
                  {
                    if (((int32) (nRooms-nRoom))*RandomNumber() < 32768*((int32) (nTreasures-nTreasure)))
                      {
                        nChokepoint=nTreasure;
                        m_Passage[m_Room[pRoom].Adjacent.pPassage[nDirection1]].
                         pTreasureGuardian=nTreasure;
                        ++nTreasure;
                      }
                  }
                m_Room[pRoom].nChokepoint=nChokepoint;
                ppRoomWithName[nRoom]=pRoom;
                ++nRoom;
              }
          }
      }
    ppRoomWithName[nRoom]=pRoomEntrance;
    *nChokepoints=nTreasure;
    return;
  }

static void FindPathToEntrance(
 int16 pRoomStart,
 int16 pRoomEntrance,
 char  **pszWayOut)
  {
    int8  bOutOfMemory;
    int8  bReenter;
    int8  bResume;
    int16 nIterations;
    int8  nReturnAddress;
    int16 pFindPathLocals;
    int16 pFindPathLocalsHead;
    int16 pFindPathParameters;
    int16 pFindPathParametersHead;
    int16 pPathStackHead;

    bOutOfMemory=FALSE;
    pFindPathLocals=-1;
    pFindPathLocalsHead=-1;
    pFindPathParameters=-1;
    pFindPathParametersHead=-1;
    if (m_nFindPathParameters < 3*NUM_ROOMS-2)
      {
        pFindPathParameters=m_nFindPathParameters;
        ++m_nFindPathParameters;
        m_FindPathParameter[pFindPathParameters].pNext=pFindPathParametersHead;
        m_FindPathParameter[pFindPathParameters].pRoom=pRoomStart;
        m_FindPathParameter[pFindPathParameters].nDirectionUsedToEnterRoom=0;
        m_FindPathParameter[pFindPathParameters].nReturnAddress=0;
        pFindPathParametersHead=pFindPathParameters;
        pPathStackHead=-1;
        bResume=FALSE;
        nIterations=0;
        do
          {
            ++nIterations;
            bReenter=FALSE;
            if (bResume)
              {
                for (;
                 ((! bOutOfMemory)
                 && (! bReenter)
                 && (! *pszWayOut)
                 && ((m_FindPathLocal[pFindPathLocalsHead].nDirection1) > 0));)
                  {
                    --(m_FindPathLocal[pFindPathLocalsHead].nDirection1);
                    m_FindPathLocal[pFindPathLocalsHead].nDirection2
                     =(m_FindPathLocal[pFindPathLocalsHead].nDirectionRandom)[
                     m_FindPathLocal[pFindPathLocalsHead].nDirection1];
                    if (m_Room[m_FindPathParameter[pFindPathParametersHead].pRoom].Adjacent.pPassage[
                     m_FindPathLocal[pFindPathLocalsHead].nDirection2] != -1)
                      {
                        if (! m_Passage[m_Room[m_FindPathParameter[pFindPathParametersHead].pRoom].
                         Adjacent.pPassage[
                         m_FindPathLocal[pFindPathLocalsHead].nDirection2]].bBlocked)
                          {
                            if (m_Passage[m_Room[m_FindPathParameter[pFindPathParametersHead].pRoom].
                             Adjacent.pPassage[
                             m_FindPathLocal[pFindPathLocalsHead].nDirection2]].pRoom1
                             == m_FindPathParameter[pFindPathParametersHead].pRoom)
                              if (m_nFindPathParameters < 3*NUM_ROOMS-2)
                                {
                                  pFindPathParameters=m_nFindPathParameters;
                                  ++m_nFindPathParameters;
                                  m_FindPathParameter[pFindPathParameters].pNext=pFindPathParametersHead;
                                  m_FindPathParameter[pFindPathParameters].pRoom
                                   =m_Passage[m_Room[m_FindPathParameter[pFindPathParametersHead
                                   ].pRoom].Adjacent.pPassage[m_FindPathLocal[pFindPathLocalsHead
                                   ].nDirection2]].pRoom2;
                                  m_FindPathParameter[pFindPathParameters].nDirectionUsedToEnterRoom=m_FindPathLocal[pFindPathLocalsHead].nDirection2;
                                  m_FindPathParameter[pFindPathParameters].nReturnAddress=1;
                                  pFindPathParametersHead=pFindPathParameters;
                                  bReenter=TRUE;
                                }
                              else
                                bOutOfMemory=TRUE;
                            else
                              if (m_nFindPathParameters < 3*NUM_ROOMS-2)
                                {
                                  pFindPathParameters=m_nFindPathParameters;
                                  ++m_nFindPathParameters;
                                  m_FindPathParameter[pFindPathParameters].pNext=pFindPathParametersHead;
                                  m_FindPathParameter[pFindPathParameters].pRoom
                                   =m_Passage[m_Room[m_FindPathParameter[pFindPathParametersHead].pRoom
                                   ].Adjacent.pPassage[m_FindPathLocal[pFindPathLocalsHead
                                   ].nDirection2]].pRoom1;
                                  m_FindPathParameter[pFindPathParameters].nDirectionUsedToEnterRoom=m_FindPathLocal[pFindPathLocalsHead].nDirection2;
                                  m_FindPathParameter[pFindPathParameters].nReturnAddress=1;
                                  pFindPathParametersHead=pFindPathParameters;
                                  bReenter=TRUE;
                                }
                              else
                                bOutOfMemory=TRUE;
                          }
                      }
                  }
                if (! bOutOfMemory)
                  {
                    if (! bReenter)
                      {
                        pPathStackHead=m_Path[m_FindPathLocal[pFindPathLocalsHead].pPathStack].pNext;
                        m_FindPathLocal[pFindPathLocalsHead].pPathStack=-1;
                        pFindPathLocals=pFindPathLocalsHead;
                        pFindPathLocalsHead=m_FindPathLocal[pFindPathLocals].pNext;
                        --m_nFindPathLocals;
                        pFindPathLocals=-1;
                      }
                  }
              }
            else
              bReenter=TRUE;
            if (! bOutOfMemory)
              {
                if (bReenter)
                  do
                    {
                      bReenter=FALSE;
                      if (m_nFindPathLocals < 3*NUM_ROOMS-2)
                        {
                          pFindPathLocals=m_nFindPathLocals;
                          ++m_nFindPathLocals;
                          m_FindPathLocal[pFindPathLocals].pNext=pFindPathLocalsHead;
                          pFindPathLocalsHead=pFindPathLocals;
                          if (m_FindPathParameter[pFindPathParametersHead].pRoom == pRoomEntrance)
                            {
                              m_FindPathLocal[pFindPathLocalsHead].nPathLen=0;
                              m_FindPathLocal[pFindPathLocalsHead].pPathStack=pPathStackHead;
                              while (m_FindPathLocal[pFindPathLocalsHead].pPathStack != -1)
                                {
                                  if (m_Room[m_Path[m_FindPathLocal[pFindPathLocalsHead].pPathStack].pRoom].szDescription)
                                    ++(m_FindPathLocal[pFindPathLocalsHead].nPathLen);
                                  m_FindPathLocal[pFindPathLocalsHead].pPathStack=m_Path[m_FindPathLocal[pFindPathLocalsHead].pPathStack].pNext;
                                }
                              if (1+(m_FindPathLocal[pFindPathLocalsHead].nPathLen) < NUM_ROOMS)
                                {
                                  *pszWayOut=&m_szWayOut[0];
                                  (*pszWayOut)[m_FindPathLocal[pFindPathLocalsHead].nPathLen]='\0';
                                  (*pszWayOut)[--(m_FindPathLocal[pFindPathLocalsHead].nPathLen)]=(char) ToUpper((intC) *m_szDirection[m_FindPathParameter[pFindPathParametersHead].nDirectionUsedToEnterRoom]);
                                  m_FindPathLocal[pFindPathLocalsHead].pPathStack=pPathStackHead;
                                  while (m_FindPathLocal[pFindPathLocalsHead].pPathStack != -1)
                                    {
                                      if (m_Path[m_FindPathLocal[pFindPathLocalsHead].pPathStack].pNext != -1)
                                        {
                                          if (m_Room[m_Path[m_FindPathLocal[pFindPathLocalsHead].pPathStack].pRoom].szDescription)
                                            (*pszWayOut)[--(m_FindPathLocal[pFindPathLocalsHead].nPathLen)]
                                             =(char) ToUpper((intC) *m_szDirection[m_Path[m_FindPathLocal[pFindPathLocalsHead].pPathStack].nDirectionUsedToEnterRoom]);
                                        }
                                      m_FindPathLocal[pFindPathLocalsHead].pPathStack=m_Path[m_FindPathLocal[pFindPathLocalsHead].pPathStack].pNext;
                                    }
                                }
                              else
                                bOutOfMemory=TRUE;
                            }
                          else
                            {
                              m_FindPathLocal[pFindPathLocalsHead].bRoomAlreadyInStack=FALSE;
                              m_FindPathLocal[pFindPathLocalsHead].pPathStack=pPathStackHead;
                              while ((m_FindPathLocal[pFindPathLocalsHead].pPathStack != -1)
                              &&     (! (m_FindPathLocal[pFindPathLocalsHead].bRoomAlreadyInStack)))
                                {
                                  m_FindPathLocal[pFindPathLocalsHead].bRoomAlreadyInStack=(m_Path[m_FindPathLocal[pFindPathLocalsHead].pPathStack].pRoom == m_FindPathParameter[pFindPathParametersHead].pRoom);
                                  m_FindPathLocal[pFindPathLocalsHead].pPathStack=m_Path[m_FindPathLocal[pFindPathLocalsHead].pPathStack].pNext;
                                }
                              if (! (m_FindPathLocal[pFindPathLocalsHead].bRoomAlreadyInStack))
                                {
                                  if (m_nPaths < 6*NUM_ROOMS)
                                    {
                                      m_FindPathLocal[pFindPathLocalsHead].pPathStack=m_nPaths;
                                      ++m_nPaths;
                                      m_Path[m_FindPathLocal[pFindPathLocalsHead].pPathStack].pRoom=m_FindPathParameter[pFindPathParametersHead].pRoom;
                                      m_Path[m_FindPathLocal[pFindPathLocalsHead].pPathStack].nDirectionUsedToEnterRoom=m_FindPathParameter[pFindPathParametersHead].nDirectionUsedToEnterRoom;
                                      m_Path[m_FindPathLocal[pFindPathLocalsHead].pPathStack].pNext=pPathStackHead;
                                      pPathStackHead=m_FindPathLocal[pFindPathLocalsHead].pPathStack;
                                      for (m_FindPathLocal[pFindPathLocalsHead].nDirection1=6; (m_FindPathLocal[pFindPathLocalsHead].nDirection1)--;)
                                        (m_FindPathLocal[pFindPathLocalsHead].nDirectionRandom)[m_FindPathLocal[pFindPathLocalsHead].nDirection1]=m_FindPathLocal[pFindPathLocalsHead].nDirection1;
                                      for (m_FindPathLocal[pFindPathLocalsHead].nDirection1=5; m_FindPathLocal[pFindPathLocalsHead].nDirection1 > 0; --(m_FindPathLocal[pFindPathLocalsHead].nDirection1))
                                        {
                                          m_FindPathLocal[pFindPathLocalsHead].nDirection2=RandomNumber()%(m_FindPathLocal[pFindPathLocalsHead].nDirection1);
                                          m_FindPathLocal[pFindPathLocalsHead].nDirection3=(m_FindPathLocal[pFindPathLocalsHead].nDirectionRandom)[m_FindPathLocal[pFindPathLocalsHead].nDirection1];
                                          (m_FindPathLocal[pFindPathLocalsHead].nDirectionRandom)[m_FindPathLocal[pFindPathLocalsHead].nDirection1]=(m_FindPathLocal[pFindPathLocalsHead].nDirectionRandom)[m_FindPathLocal[pFindPathLocalsHead].nDirection2];
                                          (m_FindPathLocal[pFindPathLocalsHead].nDirectionRandom)[m_FindPathLocal[pFindPathLocalsHead].nDirection2]=m_FindPathLocal[pFindPathLocalsHead].nDirection3;
                                        }
                                      for (m_FindPathLocal[pFindPathLocalsHead].nDirection1=6; ((! bOutOfMemory) && (! bReenter) && (! *pszWayOut) && ((m_FindPathLocal[pFindPathLocalsHead].nDirection1) > 0));)
                                        {
                                          --(m_FindPathLocal[pFindPathLocalsHead].nDirection1);
                                          m_FindPathLocal[pFindPathLocalsHead].nDirection2=(m_FindPathLocal[pFindPathLocalsHead].nDirectionRandom)[m_FindPathLocal[pFindPathLocalsHead].nDirection1];
                                          if (m_Room[m_FindPathParameter[pFindPathParametersHead].pRoom].Adjacent.pPassage[m_FindPathLocal[pFindPathLocalsHead].nDirection2] != -1)
                                            {
                                              if (! m_Passage[m_Room[m_FindPathParameter[pFindPathParametersHead].pRoom
                                               ].Adjacent.pPassage[m_FindPathLocal[pFindPathLocalsHead].nDirection2
                                               ]].bBlocked)
                                                {
                                                  if (m_Passage[m_Room[m_FindPathParameter[pFindPathParametersHead].pRoom
                                                   ].Adjacent.pPassage[m_FindPathLocal[pFindPathLocalsHead
                                                   ].nDirection2]].pRoom1 == m_FindPathParameter[pFindPathParametersHead].pRoom)
                                                    if (m_nFindPathParameters < 3*NUM_ROOMS-2)
                                                      {
                                                        pFindPathParameters=m_nFindPathParameters;
                                                        ++m_nFindPathParameters;
                                                        m_FindPathParameter[pFindPathParameters].pNext=pFindPathParametersHead;
                                                        m_FindPathParameter[pFindPathParameters].pRoom
                                                         =m_Passage[m_Room[m_FindPathParameter[pFindPathParametersHead].pRoom
                                                         ].Adjacent.pPassage[m_FindPathLocal[pFindPathLocalsHead
                                                         ].nDirection2]].pRoom2;
                                                        m_FindPathParameter[pFindPathParameters].nDirectionUsedToEnterRoom=m_FindPathLocal[pFindPathLocalsHead].nDirection2;
                                                        m_FindPathParameter[pFindPathParameters].nReturnAddress=1;
                                                        pFindPathParametersHead=pFindPathParameters;
                                                        bReenter=TRUE;
                                                      }
                                                    else
                                                      bOutOfMemory=TRUE;
                                                  else
                                                    if (m_nFindPathParameters < 3*NUM_ROOMS-2)
                                                      {
                                                        pFindPathParameters=m_nFindPathParameters;
                                                        ++m_nFindPathParameters;
                                                        m_FindPathParameter[pFindPathParameters].pNext=pFindPathParametersHead;
                                                        m_FindPathParameter[pFindPathParameters].pRoom
                                                         =m_Passage[m_Room[m_FindPathParameter[pFindPathParametersHead].pRoom
                                                         ].Adjacent.pPassage[m_FindPathLocal[pFindPathLocalsHead
                                                         ].nDirection2]].pRoom1;
                                                        m_FindPathParameter[pFindPathParameters].nDirectionUsedToEnterRoom=m_FindPathLocal[pFindPathLocalsHead].nDirection2;
                                                        m_FindPathParameter[pFindPathParameters].nReturnAddress=1;
                                                        pFindPathParametersHead=pFindPathParameters;
                                                        bReenter=TRUE;
                                                      }
                                                    else
                                                      bOutOfMemory=TRUE;
                                                }
                                            }
                                        }
                                      if (! bReenter)
                                        {
                                          pPathStackHead=m_Path[m_FindPathLocal[pFindPathLocalsHead].pPathStack].pNext;
                                          m_FindPathLocal[pFindPathLocalsHead].pPathStack=-1;
                                        }
                                    }
                                  else
                                    bOutOfMemory=TRUE;
                                }
                            }
                          if (! bReenter)
                            {
                              pFindPathLocals=pFindPathLocalsHead;
                              pFindPathLocalsHead=m_FindPathLocal[pFindPathLocals].pNext;
                              --m_nFindPathLocals;
                              pFindPathLocals=-1;
                            }
                        }
                      else
                        bOutOfMemory=TRUE;
                    }
                  while ((! bOutOfMemory) && bReenter);
              }
            if (! bOutOfMemory)
              {
                nReturnAddress=m_FindPathParameter[pFindPathParametersHead].nReturnAddress;
                pFindPathParameters=pFindPathParametersHead;
                pFindPathParametersHead=m_FindPathParameter[pFindPathParameters].pNext;
                --m_nFindPathParameters;
                pFindPathParameters=-1;
                bResume=TRUE;
              }
          }
        while ((! bOutOfMemory) && nReturnAddress && (nIterations < 1000));
      }
    else
      bOutOfMemory=TRUE;
    m_nPaths=0;
    m_nFindPathLocals=0;
    m_nFindPathParameters=0;
    return;
  }

static void WayOut(
 int16 nRooms,
 int16 pRoom,
 int16 pRoomEntrance,
 int16 *ppRoomWithName,
 int8 nTreasures,
 int8 *bErr)
  {
    int8  bCarryingTreasure;
    int16 nRoom;
    int8  nTreasure;
    char  *szText;
    char  *szWayOut;

    if (pRoom == pRoomEntrance)
      DisplayText("You\'re already at the entrance.",bErr);
    else
      {
        bCarryingTreasure=FALSE;
        for (nTreasure=nTreasures; ((! bCarryingTreasure) && (nTreasure--));)
          bCarryingTreasure=(m_Treasure[nTreasure].pRoomTreasure == -1);
        if (bCarryingTreasure)
          {
            szWayOut=(char *) NULL;
            FindPathToEntrance(pRoom,pRoomEntrance,&szWayOut);
            if (szWayOut)
              {
                do
                  nRoom=RandomNumber()%nRooms;
                while (ppRoomWithName[nRoom] == pRoom);
                m_Treasure[nTreasure].pRoomTreasure=ppRoomWithName[nRoom];
                if (1+StrLen("The pirate takes one of your treasures.  "
                 "As he leaves, he shouts the letters \"")
                 +StrLen(szWayOut)+StrLen("\".") < SENTENCE_LEN)
                  {
                    szText=&m_szMsg[0];
                    StrCpy(szText,"The pirate takes one of your treasures.  ");
                    if (StrLen(szWayOut) == 1)
                      StrCat(szText,"As he leaves, he shouts the letter \"");
                    else
                      StrCat(szText,"As he leaves, he shouts the letters \"");
                    StrCat(szText,szWayOut);
                    StrCat(szText,"\".");
                    DisplayText(szText,bErr);
                  }
                else
                  {
                    DisplayText("Fatal error:  memory was exhausted at locaton 6.",bErr);
                    *bErr=TRUE;
                  }
              }
            else
              DisplayText("Nothing happens.",bErr);
          }
        else
          DisplayText("Nothing happens.",bErr);
      }
    return;
  }

static void HideTreasuresAndWeapons(
 int16 nRooms,
 int16 *ppRoomWithName,
 int8  nTreasures)
  {
    int8  bChokepointFound;
    int16 nRoom;
    int8  nTreasure;

    for (nTreasure=0; nTreasure < nTreasures; ++nTreasure)
      {
        bChokepointFound=FALSE;
        nRoom=0;
        while ((! bChokepointFound) && (nRoom < nRooms))
          if (m_Room[ppRoomWithName[nRoom]].nChokepoint == nTreasure)
            bChokepointFound=TRUE;
          else
            ++nRoom;
        if (bChokepointFound)
          {
            do
              nRoom=RandomNumber()%nRooms;
            while (m_Room[ppRoomWithName[nRoom]].nChokepoint != nTreasure);
            m_Treasure[nTreasure].pRoomTreasure=ppRoomWithName[nRoom];
            do
              nRoom=RandomNumber()%(nRooms+1);
            while (m_Room[ppRoomWithName[nRoom]].nChokepoint >= nTreasure);
            m_Treasure[nTreasure].pRoomWeapon=ppRoomWithName[nRoom];
          }
        else
          {
            nRoom=RandomNumber()%nRooms;
            m_Treasure[nTreasure].pRoomTreasure=ppRoomWithName[nRoom];
            m_Treasure[nTreasure].pRoomWeapon=ppRoomWithName[nRooms];
          }
      }
    return;
  }

static intC Command(
 void)
  {
    intC         nResult;
    unsigned int uiKey;

    if (m_bUseSmallFont)
      PrintMini(0,6*m_nLinesWritten,(unsigned char *) "Command? ",MINI_OVER);
    else
      Print((unsigned char *) "Command? ");
    GetKey(&uiKey);
    nResult=(intC) uiKey;
    return nResult;
  }

static void ListPassages(
 int16 pRoom,
 int8  *bErr)
  {
    int8  nDirection;
    intC  nFirstChar;
    int16 nPassage;
    int16 nPassages;
    int16 pRoomAlongPassage;
    int16 pTreasure;
    char  *szDescription;
    char  szPassages [] = "There are passages North, South, East, West, Up, and Down.";
    char  *szText;

    nPassages=0;
    for (nDirection=6; nDirection--;)
      if ((m_Room[pRoom].Adjacent.pPassage[nDirection] != -1)
      &&  (! m_Passage[m_Room[pRoom].Adjacent.pPassage[nDirection]].bBlocked))
        ++nPassages;
    if (nPassages > 1)
      StrCpy(&szPassages[0],"There are passages");
    else
      StrCpy(&szPassages[0],"There is a passage");
    nPassage=0;
    for (nDirection=0; nDirection < 6; ++nDirection)
      if ((m_Room[pRoom].Adjacent.pPassage[nDirection] != -1)
      &&  (! m_Passage[m_Room[pRoom].Adjacent.pPassage[nDirection]].bBlocked))
        {
          if (++nPassage > 1)
            if (nPassages != 2)
              StrCat(&szPassages[0],",");
          StrCat(&szPassages[0]," ");
          if (nPassage == nPassages)
            if (nPassages > 1)
              StrCat(&szPassages[0],"and ");
          StrCat(&szPassages[0],m_szDirection[nDirection]);
        }
    StrCat(&szPassages[0],".");
    DisplayText(&szPassages[0],bErr);
    for (nDirection=0; (! *bErr) && (nDirection < 6); ++nDirection)
      {
        szDescription=(char *) NULL;
        pRoomAlongPassage=pRoom;
        while ((! *bErr)
        && (m_Room[pRoomAlongPassage].Adjacent.pPassage[nDirection] != -1)
        && (! m_Passage[m_Room[pRoomAlongPassage].Adjacent.pPassage[nDirection]].bBlocked)
        && (szDescription == NULL))
          {
            if ((pTreasure=m_Passage[m_Room[pRoomAlongPassage].Adjacent.pPassage[nDirection]].pTreasureGuardian) != -1)
              {
                nFirstChar=ToUpper((intC) *(m_Treasure[pTreasure].szGuardian));
                if ((nFirstChar == 'A') || (nFirstChar == 'E') || (nFirstChar == 'I') || (nFirstChar == 'O') || (nFirstChar == 'U'))
                  if (1+StrLen("The passage ")+StrLen(m_szDirection[nDirection])+StrLen(" is guarded by an ")+StrLen(m_Treasure[pTreasure].szGuardian)+StrLen(".") < SENTENCE_LEN)
                    {
                      szText=&m_szMsg[0];
                      StrCpy(szText,"The passage ");
                      StrCat(szText,m_szDirection[nDirection]);
                      StrCat(szText," is guarded by an ");
                      StrCat(szText,m_Treasure[pTreasure].szGuardian);
                      StrCat(szText,".");
                      DisplayText(szText,bErr);
                    }
                  else
                    {
                      DisplayText("Fatal error:  memory was exhausted at locaton 7.",bErr);
                      *bErr=TRUE;
                    }
                else
                  if (1+StrLen("The passage ")+StrLen(m_szDirection[nDirection])+StrLen(" is guarded by a ")+StrLen(m_Treasure[pTreasure].szGuardian)+StrLen(".") < SENTENCE_LEN)
                    {
                      szText=&m_szMsg[0];
                      StrCpy(szText,"The passage ");
                      StrCat(szText,m_szDirection[nDirection]);
                      StrCat(szText," is guarded by a ");
                      StrCat(szText,m_Treasure[pTreasure].szGuardian);
                      StrCat(szText,".");
                      DisplayText(szText,bErr);
                    }
                  else
                    {
                      DisplayText("Fatal error:  memory was exhausted at locaton 8.",bErr);
                      *bErr=TRUE;
                    }
              }
            if (m_Passage[m_Room[pRoomAlongPassage].Adjacent.pPassage[nDirection]].pRoom1 == pRoomAlongPassage)
              pRoomAlongPassage=m_Passage[m_Room[pRoomAlongPassage].Adjacent.pPassage[nDirection]].pRoom2;
            else
              pRoomAlongPassage=m_Passage[m_Room[pRoomAlongPassage].Adjacent.pPassage[nDirection]].pRoom1;
            szDescription=m_Room[pRoomAlongPassage].szDescription;
          }
      }
    return;
  }

static void ListTreasuresAndWeapons(
 int16 pRoom,
 int8  nTreasures,
 int8  *bErr)
  {
    int16 nBytes;
    intC  nFirstChar;
    int8  nItem;
    int8  nItems;
    int8  nTreasure;
    char  *szText;

    nItems=0;
    for (nTreasure=nTreasures; nTreasure--;)
      {
        if (m_Treasure[nTreasure].pRoomTreasure == pRoom)
          ++nItems;
        if (m_Treasure[nTreasure].pRoomWeapon == pRoom)
          ++nItems;
      }
    if (nItems)
      {
        nBytes=0;
        if (nItems == 1)
          nBytes+=StrLen("There is ");
        else
          nBytes+=StrLen("There are ");
        nItem=0;
        for (nTreasure=nTreasures; nTreasure--;)
          {
            if (m_Treasure[nTreasure].pRoomTreasure == pRoom)
              {
                nFirstChar=ToUpper((intC) *(m_Treasure[nTreasure].szTreasure));
                ++nItem;
                if (nItem < nItems)
                  {
                    if ((nFirstChar == 'A') || (nFirstChar == 'E') || (nFirstChar == 'I') || (nFirstChar == 'O') || (nFirstChar == 'U'))
                      nBytes+=StrLen("an ");
                    else
                      nBytes+=StrLen("a ");
                    nBytes+=StrLen(m_Treasure[nTreasure].szTreasure);
                    if (nItems == 2)
                      nBytes+=StrLen(" ");
                    else
                      nBytes+=StrLen(", ");
                  }
                else
                  {
                    if (nItems > 1)
                      nBytes+=StrLen("and ");
                    if ((nFirstChar == 'A') || (nFirstChar == 'E') || (nFirstChar == 'I') || (nFirstChar == 'O') || (nFirstChar == 'U'))
                      nBytes+=StrLen("an ");
                    else
                      nBytes+=StrLen("a ");
                    nBytes+=StrLen(m_Treasure[nTreasure].szTreasure);
                    nBytes+=StrLen(" here.");
                  }
              }
            if (m_Treasure[nTreasure].pRoomWeapon == pRoom)
              {
                nFirstChar=ToUpper((intC) *(m_Treasure[nTreasure].szWeapon));
                ++nItem;
                if (nItem < nItems)
                  {
                    if ((nFirstChar == 'A') || (nFirstChar == 'E') || (nFirstChar == 'I') || (nFirstChar == 'O') || (nFirstChar == 'U'))
                      nBytes+=StrLen("an ");
                    else
                      nBytes+=StrLen("a ");
                    nBytes+=StrLen(m_Treasure[nTreasure].szWeapon);
                    if (nItems == 2)
                      nBytes+=StrLen(" ");
                    else
                      nBytes+=StrLen(", ");
                  }
                else
                  {
                    if (nItems > 1)
                      nBytes+=StrLen("and ");
                    if ((nFirstChar == 'A') || (nFirstChar == 'E') || (nFirstChar == 'I') || (nFirstChar == 'O') || (nFirstChar == 'U'))
                      nBytes+=StrLen("an ");
                    else
                      nBytes+=StrLen("a ");
                    nBytes+=StrLen(m_Treasure[nTreasure].szWeapon);
                    nBytes+=StrLen(" here.");
                  }
              }
          }
        if (nBytes < SENTENCE_LEN)
          {
            szText=&m_szMsg[0];
            if (nItems == 1)
              StrCpy(szText,"There is ");
            else
              StrCpy(szText,"There are ");
            nItem=0;
            for (nTreasure=nTreasures; nTreasure--;)
              {
                if (m_Treasure[nTreasure].pRoomTreasure == pRoom)
                  {
                    nFirstChar=ToUpper((intC) *(m_Treasure[nTreasure].szTreasure));
                    ++nItem;
                    if (nItem < nItems)
                      {
                        if ((nFirstChar == 'A') || (nFirstChar == 'E') || (nFirstChar == 'I') || (nFirstChar == 'O') || (nFirstChar == 'U'))
                          StrCat(szText,"an ");
                        else
                          StrCat(szText,"a ");
                        StrCat(szText,m_Treasure[nTreasure].szTreasure);
                        if (nItems == 2)
                          StrCat(szText," ");
                        else
                          StrCat(szText,", ");
                      }
                    else
                      {
                        if (nItems > 1)
                          StrCat(szText,"and ");
                        if ((nFirstChar == 'A') || (nFirstChar == 'E') || (nFirstChar == 'I') || (nFirstChar == 'O') || (nFirstChar == 'U'))
                          StrCat(szText,"an ");
                        else
                          StrCat(szText,"a ");
                        StrCat(szText,m_Treasure[nTreasure].szTreasure);
                        StrCat(szText," here.");
                      }
                  }
                if (m_Treasure[nTreasure].pRoomWeapon == pRoom)
                  {
                    nFirstChar=ToUpper((intC) *(m_Treasure[nTreasure].szWeapon));
                    ++nItem;
                    if (nItem < nItems)
                      {
                        if ((nFirstChar == 'A') || (nFirstChar == 'E') || (nFirstChar == 'I') || (nFirstChar == 'O') || (nFirstChar == 'U'))
                          StrCat(szText,"an ");
                        else
                          StrCat(szText,"a ");
                        StrCat(szText,m_Treasure[nTreasure].szWeapon);
                        if (nItems == 2)
                          StrCat(szText," ");
                        else
                          StrCat(szText,", ");
                      }
                    else
                      {
                        if (nItems > 1)
                          StrCat(szText,"and ");
                        if ((nFirstChar == 'A') || (nFirstChar == 'E') || (nFirstChar == 'I') || (nFirstChar == 'O') || (nFirstChar == 'U'))
                          StrCat(szText,"an ");
                        else
                          StrCat(szText,"a ");
                        StrCat(szText,m_Treasure[nTreasure].szWeapon);
                        StrCat(szText," here.");
                      }
                  }
              }
            DisplayText(szText,bErr);
          }
        else
          {
            DisplayText("Fatal error:  memory was exhausted at location 22.",bErr);
            *bErr=TRUE;
          }
      }
    return;
  }

static void Carry(
 int16 pRoom,
 int8  nTreasures,
 int8  *bErr)
  {
    int8 nPickedUp;
    int8 nTreasure;

    nPickedUp=0;
    for (nTreasure=nTreasures; nTreasure--;)
      {
        if (m_Treasure[nTreasure].pRoomWeapon == pRoom)
          {
            ++nPickedUp;
            m_Treasure[nTreasure].pRoomWeapon=-1;
          }
        if (m_Treasure[nTreasure].pRoomTreasure == pRoom)
          {
            ++nPickedUp;
            m_Treasure[nTreasure].pRoomTreasure=-1;
          }
      }
    if (nPickedUp == 0)
      DisplayText("There is nothing to carry.",bErr);
    return;
  }

static void Leave(
 int16 pRoom,
 int8  nTreasures,
 int8  *bErr)
  {
    int8 nLeft;
    int8 nTreasure;

    nLeft=0;
    for (nTreasure=nTreasures; nTreasure--;)
      if (m_Treasure[nTreasure].pRoomTreasure == -1)
        {
          ++nLeft;
          m_Treasure[nTreasure].pRoomTreasure=pRoom;
        }
    if (nLeft == 0)
      DisplayText("You aren\'t carrying anything to leave.",bErr);
    return;
  }

static void Inventory(
 int16 nChokepoints,
 int8  nTreasures,
 int8  *bErr)
  {
    int16 nBytes;
    intC  nFirstChar;
    int8  nItem;
    int8  nItems;
    int8  nTreasure;
    char  *szText;

    nItems=0;
    for (nTreasure=nTreasures; nTreasure--;)
      if (m_Treasure[nTreasure].pRoomTreasure == -1)
        ++nItems;
    for (nTreasure=nChokepoints; nTreasure--;)
      if ((m_Treasure[nTreasure].pRoomWeapon == -1) && (! m_Treasure[nTreasure].bSlain))
        ++nItems;
    if (nItems)
      {
        nBytes=0;
        nItem=0;
        szText=&m_szMsg[0];
        nBytes+=StrLen("You are carrying ");
        for (nTreasure=nTreasures; nTreasure--;)
          if (m_Treasure[nTreasure].pRoomTreasure == -1)
            {
              nFirstChar=ToUpper((intC) *(m_Treasure[nTreasure].szTreasure));
              ++nItem;
              if (nItem < nItems)
                {
                  if ((nFirstChar == 'A') || (nFirstChar == 'E') || (nFirstChar == 'I') || (nFirstChar == 'O') || (nFirstChar == 'U'))
                    nBytes+=StrLen("an ");
                  else
                    nBytes+=StrLen("a ");
                  nBytes+=StrLen(m_Treasure[nTreasure].szTreasure);
                  if (nItems == 2)
                    nBytes+=StrLen(" ");
                  else
                    nBytes+=StrLen(", ");
                }
              else
                {
                  if (nItems > 1)
                    nBytes+=StrLen("and ");
                  if ((nFirstChar == 'A') || (nFirstChar == 'E') || (nFirstChar == 'I') || (nFirstChar == 'O') || (nFirstChar == 'U'))
                    nBytes+=StrLen("an ");
                  else
                    nBytes+=StrLen("a ");
                  nBytes+=StrLen(m_Treasure[nTreasure].szTreasure);
                  nBytes+=StrLen(".");
                }
            }
        for (nTreasure=nChokepoints; nTreasure--;)
          if ((m_Treasure[nTreasure].pRoomWeapon == -1) && (! m_Treasure[nTreasure].bSlain))
            {
              nFirstChar=ToUpper((intC) *(m_Treasure[nTreasure].szWeapon));
              ++nItem;
              if (nItem < nItems)
                {
                  if ((nFirstChar == 'A') || (nFirstChar == 'E') || (nFirstChar == 'I') || (nFirstChar == 'O') || (nFirstChar == 'U'))
                    nBytes+=StrLen("an ");
                  else
                    nBytes+=StrLen("a ");
                  nBytes+=StrLen(m_Treasure[nTreasure].szWeapon);
                  if (nItems == 2)
                    nBytes+=StrLen(" ");
                  else
                    nBytes+=StrLen(", ");
                }
              else
                {
                  if (nItems > 1)
                    nBytes+=StrLen("and ");
                  if ((nFirstChar == 'A') || (nFirstChar == 'E') || (nFirstChar == 'I') || (nFirstChar == 'O') || (nFirstChar == 'U'))
                    nBytes+=StrLen("an ");
                  else
                    nBytes+=StrLen("a ");
                  nBytes+=StrLen(m_Treasure[nTreasure].szWeapon);
                  nBytes+=StrLen(".");
                }
            }
        if (nBytes < SENTENCE_LEN)
          {
            nItem=0;
            szText=&m_szMsg[0];
            StrCpy(szText,"You are carrying ");
            for (nTreasure=nTreasures; nTreasure--;)
              if (m_Treasure[nTreasure].pRoomTreasure == -1)
                {
                  nFirstChar=ToUpper((intC) *(m_Treasure[nTreasure].szTreasure));
                  ++nItem;
                  if (nItem < nItems)
                    {
                      if ((nFirstChar == 'A') || (nFirstChar == 'E') || (nFirstChar == 'I') || (nFirstChar == 'O') || (nFirstChar == 'U'))
                        StrCat(szText,"an ");
                      else
                        StrCat(szText,"a ");
                      StrCat(szText,m_Treasure[nTreasure].szTreasure);
                      if (nItems == 2)
                        StrCat(szText," ");
                      else
                        StrCat(szText,", ");
                    }
                  else
                    {
                      if (nItems > 1)
                        StrCat(szText,"and ");
                      if ((nFirstChar == 'A') || (nFirstChar == 'E') || (nFirstChar == 'I') || (nFirstChar == 'O') || (nFirstChar == 'U'))
                        StrCat(szText,"an ");
                      else
                        StrCat(szText,"a ");
                      StrCat(szText,m_Treasure[nTreasure].szTreasure);
                      StrCat(szText,".");
                    }
                }
            for (nTreasure=nChokepoints; nTreasure--;)
              if ((m_Treasure[nTreasure].pRoomWeapon == -1) && (! m_Treasure[nTreasure].bSlain))
                {
                  nFirstChar=ToUpper((intC) *(m_Treasure[nTreasure].szWeapon));
                  ++nItem;
                  if (nItem < nItems)
                    {
                      if ((nFirstChar == 'A') || (nFirstChar == 'E') || (nFirstChar == 'I') || (nFirstChar == 'O') || (nFirstChar == 'U'))
                        StrCat(szText,"an ");
                      else
                        StrCat(szText,"a ");
                      StrCat(szText,m_Treasure[nTreasure].szWeapon);
                      if (nItems == 2)
                        StrCat(szText," ");
                      else
                        StrCat(szText,", ");
                    }
                  else
                    {
                      if (nItems > 1)
                        StrCat(szText,"and ");
                      if ((nFirstChar == 'A') || (nFirstChar == 'E') || (nFirstChar == 'I') || (nFirstChar == 'O') || (nFirstChar == 'U'))
                        StrCat(szText,"an ");
                      else
                        StrCat(szText,"a ");
                      StrCat(szText,m_Treasure[nTreasure].szWeapon);
                      StrCat(szText,".");
                    }
                }
            DisplayText(szText,bErr);
          }
        else
          {
            DisplayText("Fatal error:  memory was exhausted at location 21.",bErr);
            *bErr=TRUE;
          }
      }
    else
      DisplayText("You aren\'t carrying anything.",bErr);
    return;
  }

static void Move(
 int16 *nMoves,
 int8  nDirection,
 int16 *pRoom,
 int8  *bErr)
  {
    int8  bGuarded;
    int16 pRoomAlongPassage;
    char  *szDescription;
    char  *szText;

    if ((m_Room[(*pRoom)].Adjacent.pPassage[nDirection] != -1)
    && (! m_Passage[m_Room[(*pRoom)].Adjacent.pPassage[nDirection]].bBlocked))
      {
        bGuarded=FALSE;
        szDescription=(char *) NULL;
        pRoomAlongPassage=*pRoom;
        while ((! *bErr)
        &&     (m_Room[pRoomAlongPassage].Adjacent.pPassage[nDirection] != -1)
        &&     (szDescription == NULL))
          {
            if (m_Passage[m_Room[pRoomAlongPassage].Adjacent.pPassage[nDirection]].pTreasureGuardian != -1)
              if (m_Treasure[m_Passage[m_Room[pRoomAlongPassage].Adjacent.pPassage[nDirection]].pTreasureGuardian].pRoomWeapon != -1)
                if (1+StrLen("You carry nothing to overcome the ")
                 +StrLen(m_Treasure[m_Passage[m_Room[pRoomAlongPassage].Adjacent.pPassage[nDirection]].pTreasureGuardian].szGuardian)+StrLen(".") < SENTENCE_LEN)
                  {
                    szText=&m_szMsg[0];
                    bGuarded=TRUE;
                    StrCpy(szText,"You carry nothing to overcome the ");
                    StrCat(szText,m_Treasure[m_Passage[m_Room[pRoomAlongPassage].Adjacent.pPassage[nDirection]].pTreasureGuardian].szGuardian);
                    StrCat(szText,".");
                    DisplayText(szText,bErr);
                  }
                else
                  {
                    DisplayText("Fatal error:  memory was exhausted at locaton 15.",bErr);
                    *bErr=TRUE;
                  }
              else
                if (1+StrLen("Your ")
                 +StrLen(m_Treasure[m_Passage[m_Room[pRoomAlongPassage].Adjacent.pPassage[nDirection]].pTreasureGuardian].szWeapon)
                 +StrLen(" overcomes the ")
                 +StrLen(m_Treasure[m_Passage[m_Room[pRoomAlongPassage].Adjacent.pPassage[nDirection]].pTreasureGuardian].szGuardian)+StrLen(".") < SENTENCE_LEN)
                  {
                    szText=&m_szMsg[0];
                    StrCpy(szText,"Your ");
                    StrCat(szText,m_Treasure[m_Passage[m_Room[pRoomAlongPassage].Adjacent.pPassage[nDirection]].pTreasureGuardian].szWeapon);
                    StrCat(szText," overcomes the ");
                    StrCat(szText,m_Treasure[m_Passage[m_Room[pRoomAlongPassage].Adjacent.pPassage[nDirection]].pTreasureGuardian].szGuardian);
                    StrCat(szText,".");
                    DisplayText(szText,bErr);
                    m_Treasure[m_Passage[m_Room[pRoomAlongPassage].Adjacent.pPassage[nDirection]].pTreasureGuardian].bSlain=TRUE;
                    m_Passage[m_Room[pRoomAlongPassage].Adjacent.pPassage[nDirection]].pTreasureGuardian=-1;
                  }
                else
                  {
                    DisplayText("Fatal error:  memory was exhausted at locaton 16.",bErr);
                    *bErr=TRUE;
                  }
            if (m_Passage[m_Room[pRoomAlongPassage].Adjacent.pPassage[nDirection]].pRoom1 == pRoomAlongPassage)
              pRoomAlongPassage=m_Passage[m_Room[pRoomAlongPassage].Adjacent.pPassage[nDirection]].pRoom2;
            else
              pRoomAlongPassage=m_Passage[m_Room[pRoomAlongPassage].Adjacent.pPassage[nDirection]].pRoom1;
            szDescription=m_Room[pRoomAlongPassage].szDescription;
          }
        if (! bGuarded)
          {
            ++(*nMoves);
            m_Room[pRoomAlongPassage].bVisited=TRUE;
            *pRoom=pRoomAlongPassage;
          }
      }
    else
      DisplayText("You can\'t go that way.",bErr);
    return;
  }

static void Points(
 int16 nRooms,
 int16 pRoomEntrance,
 int16 *ppRoomWithName,
 int8 nTreasures,
 int16 nMoves,
 int8 bRank,
 int8 *bErr)
  {
    int16 nRoom;
    int16 nRoomsVisited;
    int16 nScore;
    int8  nTreasure;
    int16 nTreasuresCarried;
    int16 nTreasuresRecovered;
    char  szMoves [12];
    char  szRooms [12];
    char  szRoomsVisited [12];
    char  szScore [12];
    char  *szText;
    char  szTreasures [12];
    char  szTreasuresCarried [12];
    char  szTreasuresRecovered [12];

    IToA(nRooms+1,&szRooms[0]);
    nScore=0;
    nTreasuresRecovered=0;
    nTreasuresCarried=0;
    nRoomsVisited=0;
    for (nRoom=0; nRoom <= nRooms; ++nRoom)
      if (m_Room[ppRoomWithName[nRoom]].bVisited)
        ++nRoomsVisited;
    IToA(nRoomsVisited,&szRoomsVisited[0]);
    IToA(nMoves,&szMoves[0]);
    if (1+StrLen("You have moved ")+StrLen(&szMoves[0])+StrLen(" times to visit ")+StrLen(&szRoomsVisited[0])+StrLen(" of ")+StrLen(&szRooms[0])+StrLen(" locations.") < SENTENCE_LEN)
      {
        szText=&m_szMsg[0];
        StrCpy(szText,"You have moved ");
        StrCat(szText,&szMoves[0]);
        StrCat(szText," times to visit ");
        StrCat(szText,&szRoomsVisited[0]);
        StrCat(szText," of ");
        StrCat(szText,&szRooms[0]);
        StrCat(szText," locations.");
        DisplayText(szText,bErr);
      }
    else
      {
        DisplayText("Fatal error:  memory was exhausted at locaton 17.",bErr);
        *bErr=TRUE;
      }
    if (! *bErr)
      {
        IToA(nTreasures,&szTreasures[0]);
        for (nTreasure=nTreasures; nTreasure--;)
          if (m_Treasure[nTreasure].pRoomTreasure == -1)
            ++nTreasuresCarried;
        IToA(nTreasuresCarried,&szTreasuresCarried[0]);
        if (1+StrLen("You hold ")+StrLen(&szTreasuresCarried[0])+StrLen(" of ")+StrLen(&szTreasures[0])+StrLen(" treasures.") < SENTENCE_LEN)
          {
            szText=&m_szMsg[0];
            StrCpy(szText,"You hold ");
            StrCat(szText,&szTreasuresCarried[0]);
            StrCat(szText," of ");
            StrCat(szText,&szTreasures[0]);
            StrCat(szText," treasures.");
            DisplayText(szText,bErr);
          }
        else
          {
            DisplayText("Fatal error:  memory was exhausted at locaton 18.",bErr);
            *bErr=TRUE;
          }
      }
    if (! *bErr)
      {
        for (nTreasure=nTreasures; nTreasure--;)
          if (m_Treasure[nTreasure].pRoomTreasure == pRoomEntrance)
            ++nTreasuresRecovered;
        IToA(nTreasuresRecovered,&szTreasuresRecovered[0]);
        if (1+StrLen("You have returned ")+StrLen(&szTreasuresRecovered[0])+StrLen(" of ")+StrLen(&szTreasures[0])+StrLen(" treasures to the entrance of the mine.") < SENTENCE_LEN)
          {
            szText=&m_szMsg[0];
            StrCpy(szText,"You have returned ");
            StrCat(szText,&szTreasuresRecovered[0]);
            StrCat(szText," of ");
            StrCat(szText,&szTreasures[0]);
            StrCat(szText," treasures to the entrance of the mine.");
            DisplayText(szText,bErr);
          }
        else
          {
            DisplayText("Fatal error:  memory was exhausted at locaton 19.",bErr);
            *bErr=TRUE;
          }
      }
    if (! *bErr)
      {
        nScore=25*nRoomsVisited/(nRooms+1)+75*nTreasuresRecovered/nTreasures+45*nTreasuresCarried/nTreasures;
        if (nRoomsVisited > 5*nRooms)
          {
            nScore=nScore-nRoomsVisited/(5*nRooms);
            if (nScore < 0)
              nScore=0;
          }
        IToA(nScore,&szScore[0]);
        if (1+StrLen("You have scored ")+StrLen(&szScore[0])+StrLen(" of 100 points.") < SENTENCE_LEN)
          {
            szText=&m_szMsg[0];
            StrCpy(szText,"You have scored ");
            StrCat(szText,&szScore[0]);
            StrCat(szText," of 100 points.");
            DisplayText(szText,bErr);
          }
        else
          {
            DisplayText("Fatal error:  memory was exhausted at locaton 20.",bErr);
            *bErr=TRUE;
          }
      }
    if (! *bErr)
      {
        if (bRank)
          {
            if (nScore < 25)
              DisplayText("Your score ranks you as a beginner.",bErr);
            else if (nScore < 50)
              DisplayText("Your score ranks you as a novice adventurer.",bErr);
            else if (nScore < 75)
              DisplayText("Your score ranks you as a seasoned explorer.",bErr);
            else if (nScore < 100)
              DisplayText("Your score ranks you as a grissly old prospector.",bErr);
            else
              DisplayText("Your score ranks you as an expert treasure hunter; there is no higher rating.",bErr);
          }
      }
    return;
  }

static void PlayGame(
 int16 nRooms,
 int16 pRoomEntrance,
 int16 *ppRoomWithName,
 int16 nChokepoints,
 int8  nTreasures,
 int8  *bErr)
  {
    intC  nCommand;
    int16 nMoves;
    int16 pRoom;

    nMoves=0;
    pRoom=pRoomEntrance;
    nCommand=(int) 'Q';
    Bdisp_AllClr_DDVRAM();
    m_nLinesWritten=0;
    locate(1,1);
    m_szParagraph[0]='\0';
    do
      {
        DisplayText(m_Room[pRoom].szDescription,bErr);
        if (! *bErr)
          ListTreasuresAndWeapons(pRoom,nTreasures,bErr);
        if (! *bErr)
          ListPassages(pRoom,bErr);
        if (! *bErr)
          {
            DisplayParagraph(&m_szParagraph[0]);
            nCommand=Command();
            Bdisp_AllClr_DDVRAM();
            m_nLinesWritten=0;
            if (! m_bUseSmallFont)
              locate(1,1);
            m_szParagraph[0]='\0';
            switch (nCommand)
              {
                case 56:
                case (int) 'N':
                  Move(&nMoves,0,&pRoom,bErr);
                  break;
                case 169:
                case (int) 'S':
                  Move(&nMoves,1,&pRoom,bErr);
                  break;
                case 130:
                case (int) 'E':
                  Move(&nMoves,2,&pRoom,bErr);
                  break;
                case 51:
                case (int) 'W':
                  Move(&nMoves,3,&pRoom,bErr);
                  break;
                case 49:
                case (int) 'U':
                  Move(&nMoves,4,&pRoom,bErr);
                  break;
                case 129:
                case (int) 'D':
                  Move(&nMoves,5,&pRoom,bErr);
                  break;
                case 133:
                case (int) 'C':
                  Carry(pRoom,nTreasures,bErr);
                  break;
                case 40:
                case (int) 'I':
                  Inventory(nChokepoints,nTreasures,bErr);
                  break;
                case 14:
                case (int) 'L':
                  Leave(pRoom,nTreasures,bErr);
                  break;
                case 52:
                case (int) 'P':
                  Points(nRooms,pRoomEntrance,ppRoomWithName,nTreasures,nMoves,FALSE,bErr);
                  break;
                case 57:
                case (int) 'O':
                  WayOut(nRooms,pRoom,pRoomEntrance,ppRoomWithName,nTreasures,bErr);
                  break;
                case 53:
                case (int) 'Q':
                  nCommand='Q';
                  Points(nRooms,pRoomEntrance,ppRoomWithName,nTreasures,nMoves,TRUE,bErr);
                  break;
                case 30046:
                case (int) 'H':
                  DisplayHelp(bErr);
                  break;
                case 131:
                case (int) 'F':
                  if (m_bUseSmallFont)
                    {
                      m_nLinesPerPage=8;
                      m_bUseSmallFont=FALSE;
                    }
                  else
                    {
                      m_nLinesPerPage=10;
                      m_bUseSmallFont=TRUE;
                    }
                  break;
                default:
                  DisplayText("I don\'t recognize that command.",bErr);
                  ListCommands(bErr);
                  break;
              }
          }
      }
    while ((! *bErr) && (nCommand != (int) 'Q'));
    DisplayParagraph(&m_szParagraph[0]);
    return;
  }

int AddIn_main(
 int            isAppli,
 unsigned short OptionNum)
  {
    int8         bErr;
    int          bResult;
    int16        nChokepoints;
    int16        nRooms;
    int8         nTreasures;
    int16        pTreasure;
    int16        pRoomEntrance;
    int16        *ppRoomWithName;
    char         szProductID [12];
    unsigned int uiKey;

    bErr=FALSE;
    locate(1,1);
    m_nLinesWritten=0;
    m_FindPathLocal=(FindPathLocalsStackRec *) NULL;
    m_FindPathParameter=(FindPathParametersStackRec *) NULL;
    m_Passage=(PassageRec *) NULL;
    m_Path=(PathStackRec *) NULL;
    m_pRoomWithName=(int16 *) NULL;
    m_Room=(RoomRec *) NULL;
    m_szMsg=(char *) NULL;
    m_szParagraph=(char *) NULL;
    m_szRoom=(char **) NULL;
    m_szWayOut=(char *) NULL;
    if ((m_FindPathLocal=(FindPathLocalsStackRec *)
     malloc((3*NUM_ROOMS-2)*sizeof(FindPathLocalsStackRec))) == NULL)
      {
        bErr=TRUE;
        Print((unsigned char *) "Error:  out of memory");
      }
    if (! bErr)
      {
        if ((m_FindPathParameter=(FindPathParametersStackRec *)
         malloc((3*NUM_ROOMS-2)*sizeof(FindPathParametersStackRec))) == NULL)
          {
            bErr=TRUE;
            Print((unsigned char *) "Error:  out of memory");
          }
      }
    if (! bErr)
      {
        if ((m_Passage=(PassageRec *)
         malloc((9*NUM_ROOMS-8)*sizeof(PassageRec))) == NULL)
          {
            bErr=TRUE;
            Print((unsigned char *) "Error:  out of memory");
          }
      }
    if (! bErr)
      {
        if ((m_Path=(PathStackRec *)
         malloc((6*NUM_ROOMS)*sizeof(PathStackRec))) == NULL)
          {
            bErr=TRUE;
            Print((unsigned char *) "Error:  out of memory");
          }
      }
    if (! bErr)
      {
        if ((m_pRoomWithName=(int16 *)
         malloc((NUM_ROOMS+1)*sizeof(int16))) == NULL)
          {
            bErr=TRUE;
            Print((unsigned char *) "Error:  out of memory");
          }
      }
    if (! bErr)
      {
        if ((m_Room=(RoomRec *)
         malloc((3*NUM_ROOMS-2)*sizeof(RoomRec))) == NULL)
          {
            bErr=TRUE;
            Print((unsigned char *) "Error:  out of memory");
          }
      }
    if (! bErr)
      {
        if ((m_szMsg=(char *)
         malloc(SENTENCE_LEN)) == NULL)
          {
            bErr=TRUE;
            Print((unsigned char *) "Error:  out of memory");
          }
      }
    if (! bErr)
      {
        if ((m_szParagraph=(char *)
         malloc(PARAGRAPH_LEN)) == NULL)
          {
            bErr=TRUE;
            Print((unsigned char *) "Error:  out of memory");
          }
      }
    if (! bErr)
      {
        if ((m_szRoom=(char **)
         malloc(NUM_ROOMS*sizeof(char *))) == NULL)
          {
            bErr=TRUE;
            Print((unsigned char *) "Error:  out of memory");
          }
      }
    if (! bErr)
      {
        if ((m_szWayOut=(char *)
         malloc(NUM_ROOMS*sizeof(char))) == NULL)
          {
            bErr=TRUE;
            Print((unsigned char *) "Error:  out of memory");
          }
      }
    if (! bErr)
      {
        m_szParagraph[0]='\0';
        m_bUseSmallFont=TRUE;
        m_nLinesPerPage=10;
        m_nMaxLineLen=21;
        m_nXMax=127;
        InitializeGlobals();
        m_nRooms=0;
        pRoomEntrance=m_nRooms;
        ++m_nRooms;
        m_Room[pRoomEntrance].szDescription="You\'re in the entrance to the mine.";
        m_Room[pRoomEntrance].bMined=TRUE;
        m_Room[pRoomEntrance].nChokepoint=-1;
        m_Room[pRoomEntrance].bVisited=TRUE;
        m_Room[pRoomEntrance].Adjacent.Passage.pPassageNorth=-1;
        m_Room[pRoomEntrance].Adjacent.Passage.pPassageSouth=-1;
        m_Room[pRoomEntrance].Adjacent.Passage.pPassageEast=-1;
        m_Room[pRoomEntrance].Adjacent.Passage.pPassageWest=-1;
        m_Room[pRoomEntrance].Adjacent.Passage.pPassageUp=-1;
        m_Room[pRoomEntrance].Adjacent.Passage.pPassageDown=-1;
        SeedRandomNumberGenerator();
        nRooms=NUM_ROOMS;
        if (! bErr)
          ShuffleRoomDescriptions(nRooms,&m_szRoom[0]);
        nTreasures=NUM_TREASURES;
        pTreasure=0;
        nChokepoints=0;
        if (! bErr)
          {
            ppRoomWithName=&m_pRoomWithName[0];
            ExcavateMine(nRooms,pRoomEntrance,nTreasures,&nChokepoints,
             ppRoomWithName,&bErr);
            if (! bErr)
              HideTreasuresAndWeapons(nRooms,ppRoomWithName,nTreasures);
            if (! bErr)
              PlayGame(nRooms,pRoomEntrance,ppRoomWithName,nChokepoints,
               nTreasures,&bErr);
          }
      }
    if (m_bUseSmallFont)
      PrintMini(0,6*m_nLinesWritten,(unsigned char *) "Press a key to exit.",MINI_OVER);
    else
      Print((unsigned char *) "Press a key to exit.");
    GetKey(&uiKey);
    if (m_szWayOut)
      free((void *) m_szWayOut);
    if (m_szRoom)
      free((void *) m_szRoom);
    if (m_szParagraph)
      free((void *) m_szParagraph);
    if (m_szMsg)
      free((void *) m_szMsg);
    if (m_Room)
      free((void *) m_Room);
    if (m_pRoomWithName)
      free((void *) m_pRoomWithName);
    if (m_Path)
      free((void *) m_Path);
    if (m_Passage)
      free((void *) m_Passage);
    if (m_FindPathParameter)
      free((void *) m_FindPathParameter);
    if (m_FindPathLocal)
      free((void *) m_FindPathLocal);
    if (bErr)
      bResult=0;
    else
      bResult=1;
    ResetCalculator();
    return bResult;
  }

//****************************************************************************
//**************                                              ****************
//**************                 Notice!                      ****************
//**************                                              ****************
//**************  Please do not change the following source.  ****************
//**************                                              ****************
//****************************************************************************


#pragma section _BR_Size
unsigned long BR_Size;
#pragma section


#pragma section _TOP

//****************************************************************************
//  InitializeSystem
//
//  param   :   isAppli   : 1 = Application / 0 = eActivity
//              OptionNum : Option Number (only eActivity)
//
//  retval  :   1 = No error / 0 = Error
//
//****************************************************************************
int InitializeSystem(int isAppli, unsigned short OptionNum)
{
    return INIT_ADDIN_APPLICATION(isAppli, OptionNum);
}

#pragma section



